// gcc -g -o bigfits -I $HOME/include -L$HOME/lib bigfits.c -lfuntools
// gcc -g -o bigfits -I $HOME/include -L$HOME/lib bigfits.c -lfuntools -lm -lsocket -ldl -lnsl

#include <iostream>
using namespace std;

#include <limits.h>
#include <funtools.h>
 
void doImage(char**);
void doTable(char* fn, int);

int lsb()
{
  return (*(short *)"\001\000" & 0x0001);
}

void swap2(char* src, char* dest) {
  *(dest  ) = *(src+1);
  *(dest+1) = *(src  );
}

void swap4(char* src, char* dest) {
  *(dest  ) = *(src+3);
  *(dest+1) = *(src+2);
  *(dest+2) = *(src+1);
  *(dest+3) = *(src  );
}

void swap8(char* src, char* dest) {
  *(dest  ) = *(src+7);
  *(dest+1) = *(src+6);
  *(dest+2) = *(src+5);
  *(dest+3) = *(src+4);
  *(dest+4) = *(src+3);
  *(dest+5) = *(src+2);
  *(dest+6) = *(src+1);
  *(dest+7) = *(src  );
}

int main(int argc, char **argv)
{
  if( argc < 4 ){
    cerr << "usage: " << argv[0] 
	 << " outfile [image width height bitpix | table rowsize]" 
	 << endl;
    exit(1);
  }

  if (!strcmp("image",argv[2]))
    doImage(argv);
  else
    doTable(argv[1], atoi(argv[3]));
}

void doImage(char** argv)
{
#define FT_BLOCK 2880

  FILE* out = fopen(argv[1],"w");
  int ww = atoi(argv[3]);
  int hh = atoi(argv[4]);
  int bitpix = atoi(argv[5]);

  char hdu[FT_BLOCK];
  memset(hdu,' ',FT_BLOCK);
  memcpy(hdu+   0,"SIMPLE  = T",11);
  memcpy(hdu+  80,"BITPIX  = ",10);
  memcpy(hdu+  90, argv[5], strlen(argv[5]));
  memcpy(hdu+ 160,"NAXIS   = 2",11);
  memcpy(hdu+ 240,"NAXIS1  = ",10);
  memcpy(hdu+ 250, argv[3], strlen(argv[3]));
  memcpy(hdu+ 320,"NAXIS2  = ",10);
  memcpy(hdu+ 330, argv[4], strlen(argv[4]));
  memcpy(hdu+ 400,"END",3);

  fwrite(hdu,FT_BLOCK,1,out);

  // create sample row
  char* row;
  int size;
  switch (bitpix) {
  case 8:
    size = sizeof(unsigned char);
    row = new char[ww*size];
    for (int ii=0; ii<ww; ii++)
      row[ii] = ii+1;
    break;
  case 16: 
    {
      size = sizeof(short);
      row = new char[ww*size];
      for (int ii=0; ii<ww; ii++) {
	short vv = ii+1;
	if (lsb())
	  swap2((char*)&vv, row+(ii*size));
	else
	  memcpy(row+(ii*size),(char*)&vv,size);
      }
    }
    break;
  case -16:
    {
      size = sizeof(unsigned short);
      row = new char[ww*size];
      for (int ii=0; ii<ww; ii++) {
	unsigned short vv = ii+1;
	if (lsb())
	  swap2((char*)&vv, row+(ii*size));
	else
	  memcpy(row+(ii*size),(char*)&vv,size);
      }
    }
    break;
  case 32:
    {
      size = sizeof(int);
      row = new char[ww*size];
      for (int ii=0; ii<ww; ii++) {
	int vv = ii+1;
	if (lsb())
	  swap4((char*)&vv, row+(ii*size));
	else
	  memcpy(row+(ii*size),(char*)&vv,size);
      }
    }
    break;
  case 64:
    {
      size = sizeof(long long);
      row = new char[ww*size];
      for (int ii=0; ii<ww; ii++) {
	long long vv = ii+1;
	if (lsb())
	  swap8((char*)&vv, row+(ii*size));
	else
	  memcpy(row+(ii*size),(char*)&vv,size);
      }
    }
    break;
  case -32:
    {
      size = sizeof(float);
      row = new char[ww*size];
      for (int ii=0; ii<ww; ii++) {
	float vv = ii+1;
	if (lsb())
	  swap4((char*)&vv, row+(ii*size));
	else
	  memcpy(row+(ii*size),(char*)&vv,size);
      }
    }
    break;
  case -64:
    {
      size = sizeof(double);
      row = new char[ww*size];
      for (int ii=0; ii<ww; ii++) {
	double vv = ii+1;
	if (lsb())
	  swap8((char*)&vv, row+(ii*size));
	else
	  memcpy(row+(ii*size),(char*)&vv,size);
      }
    }
    break;
  }

  // write all lines
  for (int jj=0; jj<hh; jj++)
    fwrite(row,ww,size,out);
  delete row;

  // write any padding
  memset(hdu,0,FT_BLOCK);
  int npad = FT_BLOCK - ((ww*hh*size) % FT_BLOCK);
  if (npad == FT_BLOCK)
    npad = 0;
  if( npad > 0 )
    fwrite(hdu, npad, 1, out);

  // done
  fclose(out);
}

void doTable(char* fn, int n)
{
#define MAXX   100
#define MAXY   100
#define MAXREC (MAXX*MAXY)

  Fun out;
  // open output file
  if( !(out = FunOpen(fn,"w", NULL)) ) {
    cerr << "Could not open the output file: " << fn << endl;
    exit(1);
  }

  int put;
  unsigned int total=0;

  typedef struct EvStruct{
    unsigned int n;
    short x, y;
    unsigned short block;
  } *Event, EventRec;
  EventRec events[MAXREC];
  Event ev;

  cerr << "writing table 10 bytes * " << MAXREC 
       << " groups of records * " << n << " times" << endl;

  // fill in the records
  for(int jj=0; jj<MAXY; jj++)
    for(int ii=0; ii<MAXX; ii++){
      ev = &events[jj*MAXY+ii];
      ev->x = ii;
      ev->y = jj;
      ev->n = total++;
    }

  // set up the (hardwired) columns
  FunColumnSelect( out, sizeof(EventRec), NULL,
                   "x",    "I:0:99",   "w", FUN_OFFSET(Event, x),
                   "y",    "I:0:99",   "w", FUN_OFFSET(Event, y),
                   "block","I",         "w", FUN_OFFSET(Event, block),
                   "n",    "J",         "w", FUN_OFFSET(Event, n),
                   NULL);


  // write n * MAXREC records
  for(int ii=0; ii<n; ii++){

    // set block count
    for(int jj=0; jj<MAXREC; jj++){
        ev = &events[jj];
        ev->block = ii;
    }
    if( (put=FunTableRowPut(out, events, MAXREC, 0, NULL)) != MAXREC ){
      cerr << "expected to write " << MAXREC 
	   << " rows; only wrote " << put << endl;
    }
  }

  FunClose(out);
}

