# text.tcl --
#
#	The wiki export plugin. Generation of plain text, ready for
#	use by the Tcler' Wiki
#
# Copyright (c) 2009 Andreas Kupries <andreas_kupries@sourceforge.net>
#
# See the file "license.terms" for information on usage and redistribution
# of this file, and for a DISCLAIMER OF ALL WARRANTIES.
# 
# RCS: @(#) $Id: export_wiki.tcl,v 1.1.1.1 2013/04/16 20:40:31 joye Exp $

# This package is a plugin for the the doctools::idx v2 system.  It
# takes the list serialization of a keyword index and produces text in
# text format.

# ### ### ### ######### ######### #########
## Requisites

# @mdgen NODEP: doctools::idx::export::plugin

package require Tcl 8.4
package require doctools::idx::export::plugin ; # Presence of this
						# pseudo package
						# indicates execution
						# inside of a properly
						# initialized plugin
						# interpreter.
package require doctools::idx::structure ; # Verification that the
					   # input is proper.
package require doctools::text           ; # Text assembly package

doctools::text::import ;# -> ::text

# ### ### ### ######### ######### #########
## API. 

proc export {serial configuration} {

    # Phase I. Check that we got a canonical index serialization. That
    #          makes the unpacking easier, as we can mix it with the
    #          generation of the output, knowing that everything is
    #          already sorted as it should be.

    ::doctools::idx::structure verify-as-canonical $serial

    # ### ### ### ######### ######### #########
    # Configuration ...
    # * Standard entries
    #   - user   = person running the application doing the formatting
    #   - format = name of this format
    #   - file   = name of the file the index came from. Optional.
    #   - map    = maps symbolic references to actual file path. Optional.

    # Format specific
    #   - style  = string in { list, table }, list default.


    # //possible parameters to influence the output.
    # //* symbolic mapping off/on

    # Import the configuration and initialize the internal state

    array set config {
	style list
    }
    array set config $configuration
    array set map    {}
    if {[info exists config(map)]} {
	array set map $config(map)
    }

    # ### ### ### ######### ######### #########

    # Phase II. Generate the output, taking the configuration into
    #           account.

    # Unpack the serialization.
    array set idx $serial
    array set idx $idx(doctools::idx)
    unset     idx(doctools::idx)
    array set r $idx(references)

    # FUTURE :: Create wiki package on top of the text generator,
    #           providing encapsulated wiki commands.

    text::begin
    text::+ "**[Header]**"
    text::newline 2

    switch -exact -- $config(style) {
	list {
	    text::+ <<TOC>>
	    text::newline 2

	    # Iterate over the keys and their references
	    foreach {keyword references} $idx(keywords) {
		text::+ "***$keyword***"
		text::newline

		text::indented 3 {
		    # Iterate over the references
		    foreach id $references {
			text::+ "* [FormatReference $id]"
			text::newline
		    }
		}
		if {[llength $references]} {
		    text::newline
		}
	    }
	}
	table {
	    foreach {keyword references} $idx(keywords) {
		text::+ "%|'''$keyword'''| |%"
		text::newline
		foreach id $references {
		    text::+ "&| | [FormatReference $id] |&"
		    text::newline
		}
	    }
	}
	default {
	    return -code error "bad style \"$style\""
	}
    }

    # Last formatting, joining the lines together.
    return [text::done]
}

# ### ### ### ######### ######### #########

proc FormatReference {id} {
    upvar 1 r r map map
    foreach {type label} $r($id) break
    return "\[[Map $type $id]%|%$label\]"
}

proc Header {} {
    upvar 1 idx(label) label idx(title) title
    if {($label ne {}) && ($title ne {})} {
	return "$label -- $title"
    } elseif {$label ne {}} {
	return $label
    } elseif {$title ne {}} {
	return $title
    }
    return -code error {Reached the unreachable}
}

proc Map {type id} {
    if {$type eq "url"} { return $id }
    upvar 1 map map
    if {![info exists map($id)]} { return $id }
    return $map($id)
}

# ### ### ### ######### ######### #########
## Ready

package provide doctools::idx::export::wiki 0.2
return
