/*
 *  Copyright (C) 2002-2021  The DOSBox Team
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */


#include "dosbox.h"
#include "logging.h"
#include "mem.h"
#include "cpu.h"
#include "inout.h"
#include "int10.h"
#include "callback.h"
#include "jfont.h"

bool int10_vga_bios_vector = true;
bool rom_bios_8x8_cga_font = true;
bool VGA_BIOS_dont_duplicate_CGA_first_half = false;
bool VIDEO_BIOS_always_carry_14_high_font = true;
bool VIDEO_BIOS_always_carry_16_high_font = true;
bool VIDEO_BIOS_enable_CGA_8x8_second_half = false;
bool VIDEO_BIOS_disable = false;

static uint8_t static_functionality[0x10]=
{
 /* 0 */ 0xff,  // All modes supported #1
 /* 1 */ 0xff,  // All modes supported #2
 /* 2 */ 0x0f,  // All modes supported #3
 /* 3 */ 0x00, 0x00, 0x00, 0x00,  // reserved
 /* 7 */ 0x07,  // 200, 350, 400 scan lines
 /* 8 */ 0x04,  // total number of character blocks available in text modes
 /* 9 */ 0x02,  // maximum number of active character blocks in text modes
 /* a */ 0xff,  // Misc Flags Everthing supported 
 /* b */ 0x0e,  // Support for Display combination, intensity/blinking and video state saving/restoring
 /* c */ 0x00,  // reserved
 /* d */ 0x00,  // reserved
 /* e */ 0x00,  // Change to add new functions
 /* f */ 0x00   // reserved
};

static uint16_t map_offset[8]={
	0x0000,0x4000,0x8000,0xc000,
	0x2000,0x6000,0xa000,0xe000
};

void INT10_LoadFont(PhysPt font,bool reload,uint16_t count,Bitu offset,Bitu map,uint8_t height) {
    unsigned char m64k;

	if (IS_VGA_ARCH || (IS_EGA_ARCH && vga.mem.memsize >= 0x20000))
        m64k=0x02;
    else
        m64k = 0x00;

    PhysPt ftwhere = PhysMake(0xa000, map_offset[map & 0x7] + (uint16_t)(offset * 32));
	uint16_t base=real_readw(BIOSMEM_SEG,BIOSMEM_CRTC_ADDRESS);
	bool mono=(base==VGAREG_MDA_CRTC_ADDRESS);
	
	//Put video adapter in planar mode
	IO_Write(0x3c4,0x02);IO_Write(0x3c5,0x04); // select plane 2 for writing
	IO_Write(0x3c4,0x04);IO_Write(0x3c5,0x05|m64k); // odd/even off in SEQ
	IO_Write(0x3ce,0x04);IO_Write(0x3cf,0x02); // select plane 2 for reading
	IO_Write(0x3ce,0x05);IO_Write(0x3cf,0x00); // write mode 0, odd/even off in GFX
	IO_Write(0x3ce,0x06);IO_Write(0x3cf,0x04); // CPU memory window A0000-AFFFF
	
	//Load character patterns
	for (uint16_t i=0;i<count;i++) {
		MEM_BlockCopy(ftwhere+i*32u,font,height);
		font+=height;
	}
	//Load alternate character patterns
	if (map & 0x80) {
		while (Bitu chr=(Bitu)mem_readb(font++)) {
			MEM_BlockCopy(ftwhere+(PhysPt)chr*32u,font,height);
			font+=height;
		}
	}

	//Return to normal text mode
	IO_Write(0x3c4,0x02);IO_Write(0x3c5,0x03); // select planes 0&1 for writing
	IO_Write(0x3c4,0x04);IO_Write(0x3c5,0x01|m64k); // odd/even on in SEQ
	IO_Write(0x3ce,0x04);IO_Write(0x3cf,0x00); // select plane 0 for reading
	IO_Write(0x3ce,0x05);IO_Write(0x3cf,0x10); // write mode 0, odd/even on in GFX
	IO_Write(0x3ce,0x06);IO_Write(0x3cf,mono?0x0a:0x0e); // Bx000-BxFFF, odd/even on

	/* Reload tables and registers with new values based on this height */
	if (reload) {
		//Max scanline 
		IO_Write(base,0x9);
		IO_Write(base+1u,(IO_Read(base+1u) & 0xe0u)|((unsigned int)height-1u));
		//Vertical display end
		Bitu rows=CurMode->sheight/height;
		Bitu vdend=rows*height*((CurMode->sheight==200)?2:1)-1;
		IO_Write(base,0x12);
		IO_Write(base+1u,(uint8_t)vdend);
		//Underline location
		if (CurMode->mode==7) {
			IO_Write(base,0x14);
			IO_Write(base+1u,(IO_Read(base+1u) & ~0x1fu)|(height-1u));
		}
		//Rows setting in bios segment
		real_writeb(BIOSMEM_SEG,BIOSMEM_NB_ROWS,(uint8_t)(rows-1));
		real_writeb(BIOSMEM_SEG,BIOSMEM_CHAR_HEIGHT,(uint8_t)height);
		//Page size
		Bitu pagesize=rows*real_readb(BIOSMEM_SEG,BIOSMEM_NB_COLS)*2;
		pagesize+=0x100; // bios adds extra on reload
		real_writew(BIOSMEM_SEG,BIOSMEM_PAGE_SIZE,(uint16_t)pagesize);
		//Cursor shape
		if (height>=14) height--; // move up one line on 14+ line fonts
		INT10_SetCursorShape(height-2,height-1);
        //Clip the cursor within range in case it is now beyond the new height. To fix 28.COM and 50.COM (issue #1324 and #1325).
        //FIXME: Remove this code *IF* it turns out BIOSes do not clip the row value when calling INT 10h AX=1111h/1113h
        //       If BIOSes are inconsistent about it, note which ones do and don't and try to match behavior with machine= type.
        //       At the very least, if the IBM PS/2 VGA BIOS does not, then this code should be coded NOT to clip the cursor
        //       when machine=vgaonly.
        {
            uint8_t page=real_readb(BIOSMEM_SEG,BIOSMEM_CURRENT_PAGE);
            uint8_t cur_row=CURSOR_POS_ROW(page);
            uint8_t cur_col=CURSOR_POS_COL(page);

            if (cur_row >= rows)
                INT10_SetCursorPos((uint8_t)(rows-1),cur_col,page);
        }
	}
}

// FIXME: ripsaw8080 noted that while the svgaCard==SVGA_None conditions are appropriate for DOSBox SVN
// because the vgaonly machine type is alone in displaying 9-dot character cells, but in DOSBox-X (as in
// DOSBox SVN Daum) the char9= setting should be respected. As it is now, users will not get the wider
// symbols when char9= is true and machine= is an SVGA machine type.
// SVN commit reference: https://sourceforge.net/p/dosbox/code-0/3932/
void INT10_ReloadFont(void) {
	Bitu map=0;
	switch(CurMode->cheight) {
	case 8:
		INT10_LoadFont(Real2Phys(int10.rom.font_8_first),false,256,0,map,8);
		break;
	case 14:
		if (IS_VGA_ARCH && svgaCard==SVGA_None && CurMode->mode==7) map=0x80;
		INT10_LoadFont(Real2Phys(int10.rom.font_14),false,256,0,map,14);
		break;
	case 19:
		if (IS_EGA_ARCH && svgaCard == SVGA_None) map = 0x80;
		INT10_LoadFont(Real2Phys(int10.rom.font_19), false, 256, 0, map, 19);
		break;
	case 16:
	default:
		if (IS_VGA_ARCH && svgaCard==SVGA_None) map=0x80;
		INT10_LoadFont(Real2Phys(int10.rom.font_16),false,256,0,map,16);
		break;
	}
}

extern Bitu VGA_BIOS_Size;

static Bitu VGA_ROM_BIOS_ENTRY_cb = 0;

Bitu VGA_ROM_BIOS_ENTRY_callback_func(void) {
    LOG_MSG("VGA ROM BIOS init callback");

    reg_eax = 3;		// 80x25 text
    CALLBACK_RunRealInt(0x10);

    return CBRET_NONE;
}

bool MEM_unmap_physmem(Bitu start,Bitu end);

void INT10_RemoveVGABIOS(void) { /* PC-98 does not have VGA BIOS */
    if (VGA_BIOS_Size != 0) {
        for (unsigned int i=0;i < VGA_BIOS_Size;i++)
            phys_writeb(0xC0000+i,0xFF);

        MEM_unmap_physmem(0xC0000,0xC0000+VGA_BIOS_Size-1);
        VGA_BIOS_Size = 0;
    }
}

RealPt GetSystemBiosINT10Vector(void);

extern bool VGA_BIOS_use_rom;

void INT10_SetupRomMemory(void) {
	/* if no space allocated for video BIOS (such as machine=cga) then return immediately */
	if (VGA_BIOS_Size == 0) {
		int10.rom.font_8_first=RealMake(0xF000,0xFA6E); /* why duplicate data? use the copy in the ROM BIOS */
		int10.rom.font_8_second=0;
		int10.rom.static_state=0;
		int10.rom.font_14=0;
		int10.rom.font_16=0;

        /* ref: [http://www.ctyme.com/intr/rb-6173.htm] */
        if (IS_TANDY_ARCH)
            RealSetVec(0x44,int10.rom.font_8_first);
        else
            RealSetVec(0x43,int10.rom.font_8_first);

		RealSetVec(0x1F,int10.rom.font_8_second);

        if (machine == MCH_MCGA) {
            Bitu ROMBIOS_GetMemory(Bitu bytes,const char *who,Bitu alignment,Bitu must_be_at);

            Bitu base = ROMBIOS_GetMemory((Bitu)(256*16),"MCGA 16-line font",1,0u);
            if (base == 0) E_Exit("Unable to alloc MCGA 16x font");

            for (unsigned int i=0;i<256*16;i++)
                phys_writeb((PhysPt)base+i,int10_font_16[i]);

            int10.rom.font_16 = RealMake((uint16_t)(base >> 4u),(uint16_t)(base & 0xF));

            // MCGA has the pointer at 40:A8 (BIOSMEM_VS_POINTER), confirmed on real hardware.
            // It points into the BIOS, because MCGA systems do not have a BIOS at C000:0000
            Bitu vptr = ROMBIOS_GetMemory((Bitu)(0x600),"MCGA video save pointer and structs",1,0u);
            Bitu vptrseg = vptr >> 4;
            Bitu vptroff = vptr & 0xF;
            vptr -= vptroff;
            Bitu vptroff_limit = vptroff + 0x600;

            int10.rom.video_parameter_table=RealMake((uint16_t)vptrseg, (uint16_t)vptroff);
            vptroff+=INT10_SetupVideoParameterTable((PhysPt)(vptr+vptroff));

            // The dynamic save area should be in RAM, it cannot exist in ROM
            int10.rom.video_dynamic_save_area=0;

            int10.rom.video_save_pointers=RealMake((uint16_t)vptrseg, (uint16_t)vptroff);
            phys_writed((PhysPt)(vptr+vptroff),int10.rom.video_parameter_table);
            vptroff+=4;
            phys_writed((PhysPt)(vptr+vptroff),int10.rom.video_dynamic_save_area);		// dynamic save area pointer
            vptroff+=4;
            phys_writed((PhysPt)(vptr+vptroff),0);		// alphanumeric character set override
            vptroff+=4;
            phys_writed((PhysPt)(vptr+vptroff),0);		// graphics character set override
            vptroff+=4;
            phys_writed((PhysPt)(vptr+vptroff),0);		// secondary save pointer table
            vptroff+=4;
            phys_writed((PhysPt)(vptr+vptroff),0); vptroff+=4;
            phys_writed((PhysPt)(vptr+vptroff),0); vptroff+=4;

            if (vptroff > vptroff_limit) E_Exit("MCGA ptr overrun");
        }

		return;
	}

/* This should fill up certain structures inside the Video Bios Rom Area */
	PhysPt rom_base=PhysMake(0xc000,0);
	Bitu i;
	int10.rom.used=3;
	if (IS_EGAVGA_ARCH) {
		// set up the start of the ROM

        // we must make valid boot code at seg:3. return value is callback index
        if (VGA_ROM_BIOS_ENTRY_cb == 0) {
            VGA_ROM_BIOS_ENTRY_cb = CALLBACK_Allocate();
            CALLBACK_Setup(VGA_ROM_BIOS_ENTRY_cb,VGA_ROM_BIOS_ENTRY_callback_func,CB_RETF,"VGA ROM BIOS boot up entry point");
        }

        // ROM signature
		phys_writew(rom_base+0,0xaa55);
		phys_writeb(rom_base+2,(uint8_t)(VGA_BIOS_Size >> 9u));
        // entry point
        phys_writeb(rom_base+3,0xFE); // Callback instruction
        phys_writeb(rom_base+4,0x38);
        phys_writew(rom_base+5,(uint16_t)VGA_ROM_BIOS_ENTRY_cb);
        phys_writeb(rom_base+7,0xCB); // RETF

        // VGA BIOS copyright
		if (IS_VGA_ARCH) phys_writes(rom_base+0x1e, "IBM compatible VGA BIOS", 24);
		else phys_writes(rom_base+0x1e, "IBM compatible EGA BIOS", 24);

		if (IS_VGA_ARCH) {
			// SVGA card-specific ROM signatures
			switch (svgaCard) {
			case SVGA_S3Trio:
                if(!VGA_BIOS_use_rom) {
                    // S3 ROM signature
                    phys_writes(rom_base+0x003f, "S3 86C764", 10);
                }
				break;
			case SVGA_TsengET4K:
			case SVGA_TsengET3K:
                if(!VGA_BIOS_use_rom) {
                    // Tseng ROM signature
                    phys_writes(rom_base+0x0075, " Tseng ", 8);
                }
				break;
			case SVGA_ParadisePVGA1A:
				phys_writeb(rom_base+0x0048,' ');
				phys_writeb(rom_base+0x0049,'W');
				phys_writeb(rom_base+0x004a,'E');
				phys_writeb(rom_base+0x004b,'S');
				phys_writeb(rom_base+0x004c,'T');
				phys_writeb(rom_base+0x004d,'E');
				phys_writeb(rom_base+0x004e,'R');
				phys_writeb(rom_base+0x004f,'N');
				phys_writeb(rom_base+0x0050,' ');
				phys_writeb(rom_base+0x007d,'V');
				phys_writeb(rom_base+0x007e,'G');
				phys_writeb(rom_base+0x007f,'A');
				phys_writeb(rom_base+0x0080,'=');
				break;
			case SVGA_None:
				break;
			}
		}

		// JMP to INT 10h in the system BIOS.
		//
		// SuperCalc 3 and 4 fails to detect EGA/VGA if INT 10h points at a location higher than segment 0xFE00 in the system BIOS.
		// It will also not attempt EGA/VGA detect unless there is an adapter ROM signature (AA55h) at 0xC000:0x0000.
		// The check does not attempt EGA/VGA detection if ((segment >> 4) + offset) >= 0xFE00.
		//
		// The idea is to point INT 10h at this JMP so that DOS programs like SuperCalc will think there is in fact an EGA/VGA BIOS
		// and that INT 10h is provided by the EGA/VGA BIOS so it can function normally.
		//
		// [https://github.com/joncampbell123/dosbox-x/issues/1473]
		if (int10_vga_bios_vector) {
			const RealPt biosint10 = GetSystemBiosINT10Vector();

			if (biosint10 != 0) {
				LOG(LOG_MISC,LOG_DEBUG)("Redirecting INT 10h to point at the VGA BIOS");

				phys_writeb(rom_base+0xEE,0xEA); // JMP FAR
				phys_writew(rom_base+0xEF,(uint16_t)(biosint10 & 0xFFFFu));
				phys_writew(rom_base+0xF1,(uint16_t)((biosint10 >> 16u) & 0xFFFFu));

				/* WARNING: This overwrites the INT 10 startup code's vector successfully only because this
				 *          code is called AFTER it has initialized the INT 10h vector. If initialization
				 *          order changes this could stop working. */
				RealSetVec(0x10,RealMake(0xC000,0x00EE));
			}
		}

        // and then other data follows
		int10.rom.used=0x100;
	}
	
	if (IS_VGA_ARCH && svgaCard==SVGA_S3Trio) INT10_SetupVESA();
	
	if (VGA_BIOS_dont_duplicate_CGA_first_half) {
		int10.rom.font_8_first=RealMake(0xF000,0xFA6E); /* why duplicate data? use the copy in the ROM BIOS */
	}
	else {
		int10.rom.font_8_first=RealMake(0xC000,int10.rom.used);
		for (i=0;i<128*8;i++) {
			phys_writeb(rom_base+int10.rom.used++,int10_font_08[i]);
		}
	}
	if (IS_EGAVGA_ARCH || VIDEO_BIOS_enable_CGA_8x8_second_half) {
		int10.rom.font_8_second=RealMake(0xC000,int10.rom.used);
		for (i=0;i<128*8;i++) {
			phys_writeb(rom_base+int10.rom.used++,int10_font_08[i+128*8]);
		}
	}
	else {
		int10.rom.font_8_second=0;
	}
	if (IS_EGAVGA_ARCH || VIDEO_BIOS_always_carry_14_high_font) {
		int10.rom.font_14=RealMake(0xC000,int10.rom.used);
		for (i=0;i<256*14;i++) {
			phys_writeb(rom_base+int10.rom.used++,int10_font_14[i]);
		}
		int10.rom.font_14_alternate=RealMake(0xC000,int10.rom.used);
		for (i=0;i<20*15+1;i++) {
			phys_writeb(rom_base+int10.rom.used++,int10_font_14_alternate[i]);
	}
	}
	else {
		int10.rom.font_14=0; /* why write the 14-high version if not emulating EGA/VGA? */
		int10.rom.font_14_alternate=0;
	}	
	if (IS_VGA_ARCH || VIDEO_BIOS_always_carry_16_high_font) {
		int10.rom.font_16=RealMake(0xC000,int10.rom.used);
		for (i=0;i<256*16;i++) {
			phys_writeb(rom_base+int10.rom.used++,int10_font_16[i]);
		}
		int10.rom.font_16_alternate=RealMake(0xC000,int10.rom.used);
		for (i=0;i<19*17+1;i++) {
			phys_writeb(rom_base+int10.rom.used++,int10_font_16_alternate[i]);
		}
	}
    else if (IS_JEGA_ARCH) {
        int10.rom.font_19 = RealMake(0xC000, int10.rom.used);
        for (i = 0; i < 256 * 19; i++)
            phys_writeb(rom_base + int10.rom.used++, jfont_sbcs_19[i]);
    }
	else {
		int10.rom.font_16=0; /* why write the 16-high version if not emulating VGA? */
		int10.rom.font_16_alternate=0;
		/* FIXME: Does the EGA BIOS have the 16-high font, or just 14-high? */
	}
	if (IS_VGA_ARCH) { /* according to sources this static state stuff only applies to VGA, right? */
		int10.rom.static_state=RealMake(0xC000,int10.rom.used);
		for (i=0;i<0x10;i++) {
			phys_writeb(rom_base+int10.rom.used++,static_functionality[i]);
		}
	}
	else {
		int10.rom.static_state=0;
	}
	RealSetVec(0x1F,int10.rom.font_8_second);

	if (IS_EGAVGA_ARCH) {
		int10.rom.video_parameter_table=RealMake(0xC000,int10.rom.used);
		int10.rom.used+=INT10_SetupVideoParameterTable(rom_base+int10.rom.used);

		if (IS_VGA_ARCH) {
			int10.rom.video_dcc_table=RealMake(0xC000,int10.rom.used);
			phys_writeb(rom_base+int10.rom.used++,0x10);	// number of entries
			phys_writeb(rom_base+int10.rom.used++,1);		// version number
			phys_writeb(rom_base+int10.rom.used++,8);		// maximum display code
			phys_writeb(rom_base+int10.rom.used++,0);		// reserved
			// display combination codes
			phys_writew(rom_base+int10.rom.used,0x0000);	int10.rom.used+=2;
			phys_writew(rom_base+int10.rom.used,0x0100);	int10.rom.used+=2;
			phys_writew(rom_base+int10.rom.used,0x0200);	int10.rom.used+=2;
			phys_writew(rom_base+int10.rom.used,0x0102);	int10.rom.used+=2;
			phys_writew(rom_base+int10.rom.used,0x0400);	int10.rom.used+=2;
			phys_writew(rom_base+int10.rom.used,0x0104);	int10.rom.used+=2;
			phys_writew(rom_base+int10.rom.used,0x0500);	int10.rom.used+=2;
			phys_writew(rom_base+int10.rom.used,0x0502);	int10.rom.used+=2;
			phys_writew(rom_base+int10.rom.used,0x0600);	int10.rom.used+=2;
			phys_writew(rom_base+int10.rom.used,0x0601);	int10.rom.used+=2;
			phys_writew(rom_base+int10.rom.used,0x0605);	int10.rom.used+=2;
			phys_writew(rom_base+int10.rom.used,0x0800);	int10.rom.used+=2;
			phys_writew(rom_base+int10.rom.used,0x0801);	int10.rom.used+=2;
			phys_writew(rom_base+int10.rom.used,0x0700);	int10.rom.used+=2;
			phys_writew(rom_base+int10.rom.used,0x0702);	int10.rom.used+=2;
			phys_writew(rom_base+int10.rom.used,0x0706);	int10.rom.used+=2;

			int10.rom.video_save_pointer_table=RealMake(0xC000,int10.rom.used);
			phys_writew(rom_base+int10.rom.used,0x1a);	// length of table
			int10.rom.used+=2;
			phys_writed(rom_base+int10.rom.used,int10.rom.video_dcc_table);
			int10.rom.used+=4;
			phys_writed(rom_base+int10.rom.used,0);		// alphanumeric charset override
			int10.rom.used+=4;
			phys_writed(rom_base+int10.rom.used,0);		// user palette table
			int10.rom.used+=4;
			phys_writed(rom_base+int10.rom.used,0);		int10.rom.used+=4;
			phys_writed(rom_base+int10.rom.used,0);		int10.rom.used+=4;
			phys_writed(rom_base+int10.rom.used,0);		int10.rom.used+=4;
		}

        // The dynamic save area should be in RAM, it cannot exist in ROM
        int10.rom.video_dynamic_save_area=0;

		int10.rom.video_save_pointers=RealMake(0xC000,int10.rom.used);
		phys_writed(rom_base+int10.rom.used,int10.rom.video_parameter_table);
		int10.rom.used+=4;
		phys_writed(rom_base+int10.rom.used,int10.rom.video_dynamic_save_area);		// dynamic save area pointer
		int10.rom.used+=4;
		phys_writed(rom_base+int10.rom.used,0);		// alphanumeric character set override
		int10.rom.used+=4;
		phys_writed(rom_base+int10.rom.used,0);		// graphics character set override
		int10.rom.used+=4;
		if (IS_VGA_ARCH) {
			phys_writed(rom_base+int10.rom.used,int10.rom.video_save_pointer_table);
		} else {
			phys_writed(rom_base+int10.rom.used,0);		// secondary save pointer table
		}
		int10.rom.used+=4;
		phys_writed(rom_base+int10.rom.used,0);		int10.rom.used+=4;
		phys_writed(rom_base+int10.rom.used,0);		int10.rom.used+=4;
	}
	
	INT10_SetupRomMemoryChecksum();

	if (IS_TANDY_ARCH) {
		RealSetVec(0x44,RealMake(0xf000,0xfa6e));
	}
}

void INT10_ReloadRomFonts(void) {
	// 16x8 font
	PhysPt font16pt=Real2Phys(int10.rom.font_16);
	for (unsigned int i=0;i<256*16;i++) {
		phys_writeb(font16pt+i,int10_font_16[i]);
	}
	phys_writeb(Real2Phys(int10.rom.font_16_alternate),0x1d);
	// 14x8 font
	PhysPt font14pt=Real2Phys(int10.rom.font_14);
	for (unsigned int i=0;i<256*14;i++) {
		phys_writeb(font14pt+i,int10_font_14[i]);
	}
	phys_writeb(Real2Phys(int10.rom.font_14_alternate),0x1d);
	// 8x8 fonts
	PhysPt font8pt=Real2Phys(int10.rom.font_8_first);
	for (unsigned int i=0;i<128*8;i++) {
		phys_writeb(font8pt+i,int10_font_08[i]);
	}
	font8pt=Real2Phys(int10.rom.font_8_second);
	for (unsigned int i=0;i<128*8;i++) {
		phys_writeb(font8pt+i,int10_font_08[i+128*8]);
	}
	if (IS_JEGA_ARCH) {
		PhysPt font19pt = Real2Phys(int10.rom.font_19);
		for (Bitu i = 0; i < 256 * 19; i++) {
			phys_writeb(font19pt + i, jfont_sbcs_19[i]);
		}
	}
	INT10_SetupRomMemoryChecksum();
}

void INT10_SetupRomMemoryChecksum(void) {
	if (IS_EGAVGA_ARCH) { //EGA/VGA. Just to be safe
		/* Sum of all bytes in rom module 256 should be 0 */
		uint8_t sum = 0;
		PhysPt rom_base = PhysMake(0xc000,0);
		unsigned int last_rombyte = (unsigned int)VGA_BIOS_Size - 1;		//32 KB romsize
		for (unsigned int i = 0;i < last_rombyte;i++)
			sum += phys_readb(rom_base + i);	//OVERFLOW IS OKAY
		sum = (uint8_t)((256 - (Bitu)sum)&0xff);
		phys_writeb(rom_base + last_rombyte,sum);
	}
}


uint8_t int10_font_08[256 * 8] = {
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x7e, 0x81, 0xa5, 0x81, 0xbd, 0x99, 0x81, 0x7e,
  0x7e, 0xff, 0xdb, 0xff, 0xc3, 0xe7, 0xff, 0x7e,
  0x6c, 0xfe, 0xfe, 0xfe, 0x7c, 0x38, 0x10, 0x00,
  0x10, 0x38, 0x7c, 0xfe, 0x7c, 0x38, 0x10, 0x00,
  0x38, 0x7c, 0x38, 0xfe, 0xfe, 0x7c, 0x38, 0x7c,
  0x10, 0x10, 0x38, 0x7c, 0xfe, 0x7c, 0x38, 0x7c,
  0x00, 0x00, 0x18, 0x3c, 0x3c, 0x18, 0x00, 0x00,
  0xff, 0xff, 0xe7, 0xc3, 0xc3, 0xe7, 0xff, 0xff,
  0x00, 0x3c, 0x66, 0x42, 0x42, 0x66, 0x3c, 0x00,
  0xff, 0xc3, 0x99, 0xbd, 0xbd, 0x99, 0xc3, 0xff,
  0x0f, 0x07, 0x0f, 0x7d, 0xcc, 0xcc, 0xcc, 0x78,
  0x3c, 0x66, 0x66, 0x66, 0x3c, 0x18, 0x7e, 0x18,
  0x3f, 0x33, 0x3f, 0x30, 0x30, 0x70, 0xf0, 0xe0,
  0x7f, 0x63, 0x7f, 0x63, 0x63, 0x67, 0xe6, 0xc0,
  0x99, 0x5a, 0x3c, 0xe7, 0xe7, 0x3c, 0x5a, 0x99,
  0x80, 0xe0, 0xf8, 0xfe, 0xf8, 0xe0, 0x80, 0x00,
  0x02, 0x0e, 0x3e, 0xfe, 0x3e, 0x0e, 0x02, 0x00,
  0x18, 0x3c, 0x7e, 0x18, 0x18, 0x7e, 0x3c, 0x18,
  0x66, 0x66, 0x66, 0x66, 0x66, 0x00, 0x66, 0x00,
  0x7f, 0xdb, 0xdb, 0x7b, 0x1b, 0x1b, 0x1b, 0x00,
  0x3e, 0x63, 0x38, 0x6c, 0x6c, 0x38, 0xcc, 0x78,
  0x00, 0x00, 0x00, 0x00, 0x7e, 0x7e, 0x7e, 0x00,
  0x18, 0x3c, 0x7e, 0x18, 0x7e, 0x3c, 0x18, 0xff,
  0x18, 0x3c, 0x7e, 0x18, 0x18, 0x18, 0x18, 0x00,
  0x18, 0x18, 0x18, 0x18, 0x7e, 0x3c, 0x18, 0x00,
  0x00, 0x18, 0x0c, 0xfe, 0x0c, 0x18, 0x00, 0x00,
  0x00, 0x30, 0x60, 0xfe, 0x60, 0x30, 0x00, 0x00,
  0x00, 0x00, 0xc0, 0xc0, 0xc0, 0xfe, 0x00, 0x00,
  0x00, 0x24, 0x66, 0xff, 0x66, 0x24, 0x00, 0x00,
  0x00, 0x18, 0x3c, 0x7e, 0xff, 0xff, 0x00, 0x00,
  0x00, 0xff, 0xff, 0x7e, 0x3c, 0x18, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x30, 0x78, 0x78, 0x30, 0x30, 0x00, 0x30, 0x00,
  0x6c, 0x6c, 0x6c, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x6c, 0x6c, 0xfe, 0x6c, 0xfe, 0x6c, 0x6c, 0x00,
  0x30, 0x7c, 0xc0, 0x78, 0x0c, 0xf8, 0x30, 0x00,
  0x00, 0xc6, 0xcc, 0x18, 0x30, 0x66, 0xc6, 0x00,
  0x38, 0x6c, 0x38, 0x76, 0xdc, 0xcc, 0x76, 0x00,
  0x60, 0x60, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x30, 0x60, 0x60, 0x60, 0x30, 0x18, 0x00,
  0x60, 0x30, 0x18, 0x18, 0x18, 0x30, 0x60, 0x00,
  0x00, 0x66, 0x3c, 0xff, 0x3c, 0x66, 0x00, 0x00,
  0x00, 0x30, 0x30, 0xfc, 0x30, 0x30, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x30, 0x60,
  0x00, 0x00, 0x00, 0xfc, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x30, 0x00,
  0x06, 0x0c, 0x18, 0x30, 0x60, 0xc0, 0x80, 0x00,
  0x7c, 0xc6, 0xce, 0xde, 0xf6, 0xe6, 0x7c, 0x00,
  0x30, 0x70, 0x30, 0x30, 0x30, 0x30, 0xfc, 0x00,
  0x78, 0xcc, 0x0c, 0x38, 0x60, 0xcc, 0xfc, 0x00,
  0x78, 0xcc, 0x0c, 0x38, 0x0c, 0xcc, 0x78, 0x00,
  0x1c, 0x3c, 0x6c, 0xcc, 0xfe, 0x0c, 0x1e, 0x00,
  0xfc, 0xc0, 0xf8, 0x0c, 0x0c, 0xcc, 0x78, 0x00,
  0x38, 0x60, 0xc0, 0xf8, 0xcc, 0xcc, 0x78, 0x00,
  0xfc, 0xcc, 0x0c, 0x18, 0x30, 0x30, 0x30, 0x00,
  0x78, 0xcc, 0xcc, 0x78, 0xcc, 0xcc, 0x78, 0x00,
  0x78, 0xcc, 0xcc, 0x7c, 0x0c, 0x18, 0x70, 0x00,
  0x00, 0x30, 0x30, 0x00, 0x00, 0x30, 0x30, 0x00,
  0x00, 0x30, 0x30, 0x00, 0x00, 0x30, 0x30, 0x60,
  0x18, 0x30, 0x60, 0xc0, 0x60, 0x30, 0x18, 0x00,
  0x00, 0x00, 0xfc, 0x00, 0x00, 0xfc, 0x00, 0x00,
  0x60, 0x30, 0x18, 0x0c, 0x18, 0x30, 0x60, 0x00,
  0x78, 0xcc, 0x0c, 0x18, 0x30, 0x00, 0x30, 0x00,
  0x7c, 0xc6, 0xde, 0xde, 0xde, 0xc0, 0x78, 0x00,
  0x30, 0x78, 0xcc, 0xcc, 0xfc, 0xcc, 0xcc, 0x00,
  0xfc, 0x66, 0x66, 0x7c, 0x66, 0x66, 0xfc, 0x00,
  0x3c, 0x66, 0xc0, 0xc0, 0xc0, 0x66, 0x3c, 0x00,
  0xf8, 0x6c, 0x66, 0x66, 0x66, 0x6c, 0xf8, 0x00,
  0xfe, 0x62, 0x68, 0x78, 0x68, 0x62, 0xfe, 0x00,
  0xfe, 0x62, 0x68, 0x78, 0x68, 0x60, 0xf0, 0x00,
  0x3c, 0x66, 0xc0, 0xc0, 0xce, 0x66, 0x3e, 0x00,
  0xcc, 0xcc, 0xcc, 0xfc, 0xcc, 0xcc, 0xcc, 0x00,
  0x78, 0x30, 0x30, 0x30, 0x30, 0x30, 0x78, 0x00,
  0x1e, 0x0c, 0x0c, 0x0c, 0xcc, 0xcc, 0x78, 0x00,
  0xe6, 0x66, 0x6c, 0x78, 0x6c, 0x66, 0xe6, 0x00,
  0xf0, 0x60, 0x60, 0x60, 0x62, 0x66, 0xfe, 0x00,
  0xc6, 0xee, 0xfe, 0xfe, 0xd6, 0xc6, 0xc6, 0x00,
  0xc6, 0xe6, 0xf6, 0xde, 0xce, 0xc6, 0xc6, 0x00,
  0x38, 0x6c, 0xc6, 0xc6, 0xc6, 0x6c, 0x38, 0x00,
  0xfc, 0x66, 0x66, 0x7c, 0x60, 0x60, 0xf0, 0x00,
  0x78, 0xcc, 0xcc, 0xcc, 0xdc, 0x78, 0x1c, 0x00,
  0xfc, 0x66, 0x66, 0x7c, 0x6c, 0x66, 0xe6, 0x00,
  0x78, 0xcc, 0xe0, 0x70, 0x1c, 0xcc, 0x78, 0x00,
  0xfc, 0xb4, 0x30, 0x30, 0x30, 0x30, 0x78, 0x00,
  0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xfc, 0x00,
  0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0x78, 0x30, 0x00,
  0xc6, 0xc6, 0xc6, 0xd6, 0xfe, 0xee, 0xc6, 0x00,
  0xc6, 0xc6, 0x6c, 0x38, 0x38, 0x6c, 0xc6, 0x00,
  0xcc, 0xcc, 0xcc, 0x78, 0x30, 0x30, 0x78, 0x00,
  0xfe, 0xc6, 0x8c, 0x18, 0x32, 0x66, 0xfe, 0x00,
  0x78, 0x60, 0x60, 0x60, 0x60, 0x60, 0x78, 0x00,
  0xc0, 0x60, 0x30, 0x18, 0x0c, 0x06, 0x02, 0x00,
  0x78, 0x18, 0x18, 0x18, 0x18, 0x18, 0x78, 0x00,
  0x10, 0x38, 0x6c, 0xc6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff,
  0x30, 0x30, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x78, 0x0c, 0x7c, 0xcc, 0x76, 0x00,
  0xe0, 0x60, 0x60, 0x7c, 0x66, 0x66, 0xdc, 0x00,
  0x00, 0x00, 0x78, 0xcc, 0xc0, 0xcc, 0x78, 0x00,
  0x1c, 0x0c, 0x0c, 0x7c, 0xcc, 0xcc, 0x76, 0x00,
  0x00, 0x00, 0x78, 0xcc, 0xfc, 0xc0, 0x78, 0x00,
  0x38, 0x6c, 0x60, 0xf0, 0x60, 0x60, 0xf0, 0x00,
  0x00, 0x00, 0x76, 0xcc, 0xcc, 0x7c, 0x0c, 0xf8,
  0xe0, 0x60, 0x6c, 0x76, 0x66, 0x66, 0xe6, 0x00,
  0x30, 0x00, 0x70, 0x30, 0x30, 0x30, 0x78, 0x00,
  0x0c, 0x00, 0x0c, 0x0c, 0x0c, 0xcc, 0xcc, 0x78,
  0xe0, 0x60, 0x66, 0x6c, 0x78, 0x6c, 0xe6, 0x00,
  0x70, 0x30, 0x30, 0x30, 0x30, 0x30, 0x78, 0x00,
  0x00, 0x00, 0xcc, 0xfe, 0xfe, 0xd6, 0xc6, 0x00,
  0x00, 0x00, 0xf8, 0xcc, 0xcc, 0xcc, 0xcc, 0x00,
  0x00, 0x00, 0x78, 0xcc, 0xcc, 0xcc, 0x78, 0x00,
  0x00, 0x00, 0xdc, 0x66, 0x66, 0x7c, 0x60, 0xf0,
  0x00, 0x00, 0x76, 0xcc, 0xcc, 0x7c, 0x0c, 0x1e,
  0x00, 0x00, 0xdc, 0x76, 0x66, 0x60, 0xf0, 0x00,
  0x00, 0x00, 0x7c, 0xc0, 0x78, 0x0c, 0xf8, 0x00,
  0x10, 0x30, 0x7c, 0x30, 0x30, 0x34, 0x18, 0x00,
  0x00, 0x00, 0xcc, 0xcc, 0xcc, 0xcc, 0x76, 0x00,
  0x00, 0x00, 0xcc, 0xcc, 0xcc, 0x78, 0x30, 0x00,
  0x00, 0x00, 0xc6, 0xd6, 0xfe, 0xfe, 0x6c, 0x00,
  0x00, 0x00, 0xc6, 0x6c, 0x38, 0x6c, 0xc6, 0x00,
  0x00, 0x00, 0xcc, 0xcc, 0xcc, 0x7c, 0x0c, 0xf8,
  0x00, 0x00, 0xfc, 0x98, 0x30, 0x64, 0xfc, 0x00,
  0x1c, 0x30, 0x30, 0xe0, 0x30, 0x30, 0x1c, 0x00,
  0x18, 0x18, 0x18, 0x00, 0x18, 0x18, 0x18, 0x00,
  0xe0, 0x30, 0x30, 0x1c, 0x30, 0x30, 0xe0, 0x00,
  0x76, 0xdc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x10, 0x38, 0x6c, 0xc6, 0xc6, 0xfe, 0x00,
  0x78, 0xcc, 0xc0, 0xcc, 0x78, 0x18, 0x0c, 0x78,
  0x00, 0xcc, 0x00, 0xcc, 0xcc, 0xcc, 0x7e, 0x00,
  0x1c, 0x00, 0x78, 0xcc, 0xfc, 0xc0, 0x78, 0x00,
  0x7e, 0xc3, 0x3c, 0x06, 0x3e, 0x66, 0x3f, 0x00,
  0xcc, 0x00, 0x78, 0x0c, 0x7c, 0xcc, 0x7e, 0x00,
  0xe0, 0x00, 0x78, 0x0c, 0x7c, 0xcc, 0x7e, 0x00,
  0x30, 0x30, 0x78, 0x0c, 0x7c, 0xcc, 0x7e, 0x00,
  0x00, 0x00, 0x78, 0xc0, 0xc0, 0x78, 0x0c, 0x38,
  0x7e, 0xc3, 0x3c, 0x66, 0x7e, 0x60, 0x3c, 0x00,
  0xcc, 0x00, 0x78, 0xcc, 0xfc, 0xc0, 0x78, 0x00,
  0xe0, 0x00, 0x78, 0xcc, 0xfc, 0xc0, 0x78, 0x00,
  0xcc, 0x00, 0x70, 0x30, 0x30, 0x30, 0x78, 0x00,
  0x7c, 0xc6, 0x38, 0x18, 0x18, 0x18, 0x3c, 0x00,
  0xe0, 0x00, 0x70, 0x30, 0x30, 0x30, 0x78, 0x00,
  0xc6, 0x38, 0x6c, 0xc6, 0xfe, 0xc6, 0xc6, 0x00,
  0x30, 0x30, 0x00, 0x78, 0xcc, 0xfc, 0xcc, 0x00,
  0x1c, 0x00, 0xfc, 0x60, 0x78, 0x60, 0xfc, 0x00,
  0x00, 0x00, 0x7f, 0x0c, 0x7f, 0xcc, 0x7f, 0x00,
  0x3e, 0x6c, 0xcc, 0xfe, 0xcc, 0xcc, 0xce, 0x00,
  0x78, 0xcc, 0x00, 0x78, 0xcc, 0xcc, 0x78, 0x00,
  0x00, 0xcc, 0x00, 0x78, 0xcc, 0xcc, 0x78, 0x00,
  0x00, 0xe0, 0x00, 0x78, 0xcc, 0xcc, 0x78, 0x00,
  0x78, 0xcc, 0x00, 0xcc, 0xcc, 0xcc, 0x7e, 0x00,
  0x00, 0xe0, 0x00, 0xcc, 0xcc, 0xcc, 0x7e, 0x00,
  0x00, 0xcc, 0x00, 0xcc, 0xcc, 0x7c, 0x0c, 0xf8,
  0xc3, 0x18, 0x3c, 0x66, 0x66, 0x3c, 0x18, 0x00,
  0xcc, 0x00, 0xcc, 0xcc, 0xcc, 0xcc, 0x78, 0x00,
  0x18, 0x18, 0x7e, 0xc0, 0xc0, 0x7e, 0x18, 0x18,
  0x38, 0x6c, 0x64, 0xf0, 0x60, 0xe6, 0xfc, 0x00,
  0xcc, 0xcc, 0x78, 0xfc, 0x30, 0xfc, 0x30, 0x30,
  0xf8, 0xcc, 0xcc, 0xfa, 0xc6, 0xcf, 0xc6, 0xc7,
  0x0e, 0x1b, 0x18, 0x3c, 0x18, 0x18, 0xd8, 0x70,
  0x1c, 0x00, 0x78, 0x0c, 0x7c, 0xcc, 0x7e, 0x00,
  0x38, 0x00, 0x70, 0x30, 0x30, 0x30, 0x78, 0x00,
  0x00, 0x1c, 0x00, 0x78, 0xcc, 0xcc, 0x78, 0x00,
  0x00, 0x1c, 0x00, 0xcc, 0xcc, 0xcc, 0x7e, 0x00,
  0x00, 0xf8, 0x00, 0xf8, 0xcc, 0xcc, 0xcc, 0x00,
  0xfc, 0x00, 0xcc, 0xec, 0xfc, 0xdc, 0xcc, 0x00,
  0x3c, 0x6c, 0x6c, 0x3e, 0x00, 0x7e, 0x00, 0x00,
  0x38, 0x6c, 0x6c, 0x38, 0x00, 0x7c, 0x00, 0x00,
  0x30, 0x00, 0x30, 0x60, 0xc0, 0xcc, 0x78, 0x00,
  0x00, 0x00, 0x00, 0xfc, 0xc0, 0xc0, 0x00, 0x00,
  0x00, 0x00, 0x00, 0xfc, 0x0c, 0x0c, 0x00, 0x00,
  0xc3, 0xc6, 0xcc, 0xde, 0x33, 0x66, 0xcc, 0x0f,
  0xc3, 0xc6, 0xcc, 0xdb, 0x37, 0x6f, 0xcf, 0x03,
  0x18, 0x18, 0x00, 0x18, 0x18, 0x18, 0x18, 0x00,
  0x00, 0x33, 0x66, 0xcc, 0x66, 0x33, 0x00, 0x00,
  0x00, 0xcc, 0x66, 0x33, 0x66, 0xcc, 0x00, 0x00,
  0x22, 0x88, 0x22, 0x88, 0x22, 0x88, 0x22, 0x88,
  0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa,
  0xdb, 0x77, 0xdb, 0xee, 0xdb, 0x77, 0xdb, 0xee,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0xf8, 0x18, 0x18, 0x18,
  0x18, 0x18, 0xf8, 0x18, 0xf8, 0x18, 0x18, 0x18,
  0x36, 0x36, 0x36, 0x36, 0xf6, 0x36, 0x36, 0x36,
  0x00, 0x00, 0x00, 0x00, 0xfe, 0x36, 0x36, 0x36,
  0x00, 0x00, 0xf8, 0x18, 0xf8, 0x18, 0x18, 0x18,
  0x36, 0x36, 0xf6, 0x06, 0xf6, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x00, 0x00, 0xfe, 0x06, 0xf6, 0x36, 0x36, 0x36,
  0x36, 0x36, 0xf6, 0x06, 0xfe, 0x00, 0x00, 0x00,
  0x36, 0x36, 0x36, 0x36, 0xfe, 0x00, 0x00, 0x00,
  0x18, 0x18, 0xf8, 0x18, 0xf8, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xf8, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x1f, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x18, 0x18, 0xff, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xff, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x1f, 0x18, 0x18, 0x18,
  0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x18, 0x18, 0xff, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x1f, 0x18, 0x1f, 0x18, 0x18, 0x18,
  0x36, 0x36, 0x36, 0x36, 0x37, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x37, 0x30, 0x3f, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3f, 0x30, 0x37, 0x36, 0x36, 0x36,
  0x36, 0x36, 0xf7, 0x00, 0xff, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xff, 0x00, 0xf7, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x37, 0x30, 0x37, 0x36, 0x36, 0x36,
  0x00, 0x00, 0xff, 0x00, 0xff, 0x00, 0x00, 0x00,
  0x36, 0x36, 0xf7, 0x00, 0xf7, 0x36, 0x36, 0x36,
  0x18, 0x18, 0xff, 0x00, 0xff, 0x00, 0x00, 0x00,
  0x36, 0x36, 0x36, 0x36, 0xff, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xff, 0x00, 0xff, 0x18, 0x18, 0x18,
  0x00, 0x00, 0x00, 0x00, 0xff, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x3f, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x1f, 0x18, 0x1f, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x1f, 0x18, 0x1f, 0x18, 0x18, 0x18,
  0x00, 0x00, 0x00, 0x00, 0x3f, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0xff, 0x36, 0x36, 0x36,
  0x18, 0x18, 0xff, 0x18, 0xff, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0xf8, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x1f, 0x18, 0x18, 0x18,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
  0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0,
  0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f,
  0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x76, 0xdc, 0xc8, 0xdc, 0x76, 0x00,
  0x00, 0x78, 0xcc, 0xf8, 0xcc, 0xf8, 0xc0, 0xc0,
  0x00, 0xfc, 0xcc, 0xc0, 0xc0, 0xc0, 0xc0, 0x00,
  0x00, 0xfe, 0x6c, 0x6c, 0x6c, 0x6c, 0x6c, 0x00,
  0xfc, 0xcc, 0x60, 0x30, 0x60, 0xcc, 0xfc, 0x00,
  0x00, 0x00, 0x7e, 0xd8, 0xd8, 0xd8, 0x70, 0x00,
  0x00, 0x66, 0x66, 0x66, 0x66, 0x7c, 0x60, 0xc0,
  0x00, 0x76, 0xdc, 0x18, 0x18, 0x18, 0x18, 0x00,
  0xfc, 0x30, 0x78, 0xcc, 0xcc, 0x78, 0x30, 0xfc,
  0x38, 0x6c, 0xc6, 0xfe, 0xc6, 0x6c, 0x38, 0x00,
  0x38, 0x6c, 0xc6, 0xc6, 0x6c, 0x6c, 0xee, 0x00,
  0x1c, 0x30, 0x18, 0x7c, 0xcc, 0xcc, 0x78, 0x00,
  0x00, 0x00, 0x7e, 0xdb, 0xdb, 0x7e, 0x00, 0x00,
  0x06, 0x0c, 0x7e, 0xdb, 0xdb, 0x7e, 0x60, 0xc0,
  0x38, 0x60, 0xc0, 0xf8, 0xc0, 0x60, 0x38, 0x00,
  0x78, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0x00,
  0x00, 0xfc, 0x00, 0xfc, 0x00, 0xfc, 0x00, 0x00,
  0x30, 0x30, 0xfc, 0x30, 0x30, 0x00, 0xfc, 0x00,
  0x60, 0x30, 0x18, 0x30, 0x60, 0x00, 0xfc, 0x00,
  0x18, 0x30, 0x60, 0x30, 0x18, 0x00, 0xfc, 0x00,
  0x0e, 0x1b, 0x1b, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0xd8, 0xd8, 0x70,
  0x30, 0x30, 0x00, 0xfc, 0x00, 0x30, 0x30, 0x00,
  0x00, 0x76, 0xdc, 0x00, 0x76, 0xdc, 0x00, 0x00,
  0x38, 0x6c, 0x6c, 0x38, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00,
  0x0f, 0x0c, 0x0c, 0x0c, 0xec, 0x6c, 0x3c, 0x1c,
  0x78, 0x6c, 0x6c, 0x6c, 0x6c, 0x00, 0x00, 0x00,
  0x70, 0x18, 0x30, 0x60, 0x78, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3c, 0x3c, 0x3c, 0x3c, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

uint8_t int10_font_14[256 * 14] = {
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x7e, 0x81, 0xa5, 0x81, 0x81, 0xbd, 0x99, 0x81,
  0x7e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7e, 0xff,
  0xdb, 0xff, 0xff, 0xc3, 0xe7, 0xff, 0x7e, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x6c, 0xfe, 0xfe,
  0xfe, 0xfe, 0x7c, 0x38, 0x10, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x10, 0x38, 0x7c, 0xfe, 0x7c,
  0x38, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x3c, 0x3c, 0xe7, 0xe7, 0xe7, 0x18, 0x18,
  0x3c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x3c,
  0x7e, 0xff, 0xff, 0x7e, 0x18, 0x18, 0x3c, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18,
  0x3c, 0x3c, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xe7, 0xc3, 0xc3,
  0xe7, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00,
  0x00, 0x00, 0x3c, 0x66, 0x42, 0x42, 0x66, 0x3c,
  0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
  0xc3, 0x99, 0xbd, 0xbd, 0x99, 0xc3, 0xff, 0xff,
  0xff, 0xff, 0x00, 0x00, 0x1e, 0x0e, 0x1a, 0x32,
  0x78, 0xcc, 0xcc, 0xcc, 0x78, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3c, 0x66, 0x66, 0x66, 0x3c, 0x18,
  0x7e, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x3f, 0x33, 0x3f, 0x30, 0x30, 0x30, 0x70, 0xf0,
  0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7f, 0x63,
  0x7f, 0x63, 0x63, 0x63, 0x67, 0xe7, 0xe6, 0xc0,
  0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0xdb, 0x3c,
  0xe7, 0x3c, 0xdb, 0x18, 0x18, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x80, 0xc0, 0xe0, 0xf8, 0xfe, 0xf8,
  0xe0, 0xc0, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x02, 0x06, 0x0e, 0x3e, 0xfe, 0x3e, 0x0e, 0x06,
  0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x3c,
  0x7e, 0x18, 0x18, 0x18, 0x7e, 0x3c, 0x18, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x66, 0x66, 0x66, 0x66,
  0x66, 0x66, 0x00, 0x66, 0x66, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7f, 0xdb, 0xdb, 0xdb, 0x7b, 0x1b,
  0x1b, 0x1b, 0x1b, 0x00, 0x00, 0x00, 0x00, 0x7c,
  0xc6, 0x60, 0x38, 0x6c, 0xc6, 0xc6, 0x6c, 0x38,
  0x0c, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xfe, 0xfe, 0xfe, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x18, 0x3c, 0x7e, 0x18,
  0x18, 0x18, 0x7e, 0x3c, 0x18, 0x7e, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x3c, 0x7e, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x7e, 0x3c,
  0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x0c, 0xfe, 0x0c, 0x18, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x60,
  0xfe, 0x60, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0xc0, 0xc0,
  0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x28, 0x6c, 0xfe, 0x6c, 0x28, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10,
  0x38, 0x38, 0x7c, 0x7c, 0xfe, 0xfe, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0xfe, 0x7c,
  0x7c, 0x38, 0x38, 0x10, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x3c, 0x3c, 0x3c, 0x18, 0x18, 0x00, 0x18,
  0x18, 0x00, 0x00, 0x00, 0x00, 0x66, 0x66, 0x66,
  0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x6c, 0x6c, 0xfe, 0x6c,
  0x6c, 0x6c, 0xfe, 0x6c, 0x6c, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x7c, 0xc6, 0xc2, 0xc0, 0x7c, 0x06,
  0x86, 0xc6, 0x7c, 0x18, 0x18, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc2, 0xc6, 0x0c, 0x18, 0x30, 0x66,
  0xc6, 0x00, 0x00, 0x00, 0x00, 0x00, 0x38, 0x6c,
  0x6c, 0x38, 0x76, 0xdc, 0xcc, 0xcc, 0x76, 0x00,
  0x00, 0x00, 0x00, 0x30, 0x30, 0x30, 0x60, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x0c, 0x18, 0x30, 0x30, 0x30, 0x30,
  0x30, 0x18, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x30, 0x18, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x18,
  0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x66, 0x3c, 0xff, 0x3c, 0x66, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18,
  0x7e, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x18, 0x30, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xfe, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x02, 0x06, 0x0c, 0x18,
  0x30, 0x60, 0xc0, 0x80, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0xce, 0xde, 0xf6, 0xe6,
  0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x38, 0x78, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x7e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6,
  0x06, 0x0c, 0x18, 0x30, 0x60, 0xc6, 0xfe, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 0x06, 0x06,
  0x3c, 0x06, 0x06, 0xc6, 0x7c, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x0c, 0x1c, 0x3c, 0x6c, 0xcc, 0xfe,
  0x0c, 0x0c, 0x1e, 0x00, 0x00, 0x00, 0x00, 0x00,
  0xfe, 0xc0, 0xc0, 0xc0, 0xfc, 0x06, 0x06, 0xc6,
  0x7c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x38, 0x60,
  0xc0, 0xc0, 0xfc, 0xc6, 0xc6, 0xc6, 0x7c, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xfe, 0xc6, 0x06, 0x0c,
  0x18, 0x30, 0x30, 0x30, 0x30, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0x7c, 0xc6,
  0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x7c, 0xc6, 0xc6, 0xc6, 0x7e, 0x06, 0x06, 0x0c,
  0x78, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18,
  0x18, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00,
  0x00, 0x00, 0x18, 0x18, 0x30, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x06, 0x0c, 0x18, 0x30, 0x60, 0x30,
  0x18, 0x0c, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x7e, 0x00, 0x00, 0x7e, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0x30,
  0x18, 0x0c, 0x06, 0x0c, 0x18, 0x30, 0x60, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 0xc6, 0x0c,
  0x18, 0x18, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xde, 0xde, 0xde,
  0xdc, 0xc0, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x10, 0x38, 0x6c, 0xc6, 0xc6, 0xfe, 0xc6, 0xc6,
  0xc6, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfc, 0x66,
  0x66, 0x66, 0x7c, 0x66, 0x66, 0x66, 0xfc, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x3c, 0x66, 0xc2, 0xc0,
  0xc0, 0xc0, 0xc2, 0x66, 0x3c, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xf8, 0x6c, 0x66, 0x66, 0x66, 0x66,
  0x66, 0x6c, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00,
  0xfe, 0x66, 0x62, 0x68, 0x78, 0x68, 0x62, 0x66,
  0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0x66,
  0x62, 0x68, 0x78, 0x68, 0x60, 0x60, 0xf0, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x3c, 0x66, 0xc2, 0xc0,
  0xc0, 0xde, 0xc6, 0x66, 0x3a, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0xc6, 0xc6, 0xc6, 0xfe, 0xc6,
  0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x3c, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x3c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1e, 0x0c,
  0x0c, 0x0c, 0x0c, 0x0c, 0xcc, 0xcc, 0x78, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xe6, 0x66, 0x6c, 0x6c,
  0x78, 0x6c, 0x6c, 0x66, 0xe6, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xf0, 0x60, 0x60, 0x60, 0x60, 0x60,
  0x62, 0x66, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00,
  0xc6, 0xee, 0xfe, 0xfe, 0xd6, 0xc6, 0xc6, 0xc6,
  0xc6, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc6, 0xe6,
  0xf6, 0xfe, 0xde, 0xce, 0xc6, 0xc6, 0xc6, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x38, 0x6c, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0x6c, 0x38, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xfc, 0x66, 0x66, 0x66, 0x7c, 0x60,
  0x60, 0x60, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0xd6, 0xde, 0x7c,
  0x0c, 0x0e, 0x00, 0x00, 0x00, 0x00, 0xfc, 0x66,
  0x66, 0x66, 0x7c, 0x6c, 0x66, 0x66, 0xe6, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 0xc6, 0x60,
  0x38, 0x0c, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7e, 0x7e, 0x5a, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00, 0x00,
  0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
  0x7c, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0xc6, 0x6c, 0x38, 0x10, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xc6, 0xc6, 0xc6, 0xc6,
  0xd6, 0xd6, 0xfe, 0x7c, 0x6c, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0xc6, 0x6c, 0x38, 0x38, 0x38,
  0x6c, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x66, 0x66, 0x66, 0x66, 0x3c, 0x18, 0x18, 0x18,
  0x3c, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0xc6,
  0x8c, 0x18, 0x30, 0x60, 0xc2, 0xc6, 0xfe, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x3c, 0x30, 0x30, 0x30,
  0x30, 0x30, 0x30, 0x30, 0x3c, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x80, 0xc0, 0xe0, 0x70, 0x38, 0x1c,
  0x0e, 0x06, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x3c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c,
  0x3c, 0x00, 0x00, 0x00, 0x10, 0x38, 0x6c, 0xc6,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00,
  0x30, 0x30, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x78, 0x0c, 0x7c, 0xcc, 0xcc,
  0x76, 0x00, 0x00, 0x00, 0x00, 0x00, 0xe0, 0x60,
  0x60, 0x78, 0x6c, 0x66, 0x66, 0x66, 0x7c, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7c,
  0xc6, 0xc0, 0xc0, 0xc6, 0x7c, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x1c, 0x0c, 0x0c, 0x3c, 0x6c, 0xcc,
  0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x7c, 0xc6, 0xfe, 0xc0, 0xc6,
  0x7c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x38, 0x6c,
  0x64, 0x60, 0xf0, 0x60, 0x60, 0x60, 0xf0, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x76,
  0xcc, 0xcc, 0xcc, 0x7c, 0x0c, 0xcc, 0x78, 0x00,
  0x00, 0x00, 0xe0, 0x60, 0x60, 0x6c, 0x76, 0x66,
  0x66, 0x66, 0xe6, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x00, 0x38, 0x18, 0x18, 0x18, 0x18,
  0x3c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x06,
  0x00, 0x0e, 0x06, 0x06, 0x06, 0x06, 0x66, 0x66,
  0x3c, 0x00, 0x00, 0x00, 0xe0, 0x60, 0x60, 0x66,
  0x6c, 0x78, 0x6c, 0x66, 0xe6, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x38, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0xec, 0xfe, 0xd6, 0xd6, 0xd6,
  0xc6, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0xdc, 0x66, 0x66, 0x66, 0x66, 0x66, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7c,
  0xc6, 0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xdc, 0x66, 0x66,
  0x66, 0x7c, 0x60, 0x60, 0xf0, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x76, 0xcc, 0xcc, 0xcc, 0x7c,
  0x0c, 0x0c, 0x1e, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0xdc, 0x76, 0x66, 0x60, 0x60, 0xf0, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7c,
  0xc6, 0x70, 0x1c, 0xc6, 0x7c, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x10, 0x30, 0x30, 0xfc, 0x30, 0x30,
  0x30, 0x36, 0x1c, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc,
  0x76, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x66, 0x66, 0x66, 0x66, 0x3c, 0x18, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc6,
  0xc6, 0xd6, 0xd6, 0xfe, 0x6c, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xc6, 0x6c, 0x38,
  0x38, 0x6c, 0xc6, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0xc6, 0xc6, 0xc6, 0xc6, 0x7e,
  0x06, 0x0c, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0xfe, 0xcc, 0x18, 0x30, 0x66, 0xfe, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x0e, 0x18, 0x18, 0x18,
  0x70, 0x18, 0x18, 0x18, 0x0e, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x18, 0x18, 0x18, 0x00, 0x18,
  0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x70, 0x18, 0x18, 0x18, 0x0e, 0x18, 0x18, 0x18,
  0x70, 0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0xdc,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x38,
  0x6c, 0xc6, 0xc6, 0xfe, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3c, 0x66, 0xc2, 0xc0, 0xc0, 0xc2,
  0x66, 0x3c, 0x0c, 0x06, 0x7c, 0x00, 0x00, 0x00,
  0xcc, 0xcc, 0x00, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc,
  0x76, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x18, 0x30,
  0x00, 0x7c, 0xc6, 0xfe, 0xc0, 0xc6, 0x7c, 0x00,
  0x00, 0x00, 0x00, 0x10, 0x38, 0x6c, 0x00, 0x78,
  0x0c, 0x7c, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xcc, 0xcc, 0x00, 0x78, 0x0c, 0x7c,
  0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00, 0x60,
  0x30, 0x18, 0x00, 0x78, 0x0c, 0x7c, 0xcc, 0xcc,
  0x76, 0x00, 0x00, 0x00, 0x00, 0x38, 0x6c, 0x38,
  0x00, 0x78, 0x0c, 0x7c, 0xcc, 0xcc, 0x76, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3c, 0x66,
  0x60, 0x66, 0x3c, 0x0c, 0x06, 0x3c, 0x00, 0x00,
  0x00, 0x10, 0x38, 0x6c, 0x00, 0x7c, 0xc6, 0xfe,
  0xc0, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00,
  0xcc, 0xcc, 0x00, 0x7c, 0xc6, 0xfe, 0xc0, 0xc6,
  0x7c, 0x00, 0x00, 0x00, 0x00, 0x60, 0x30, 0x18,
  0x00, 0x7c, 0xc6, 0xfe, 0xc0, 0xc6, 0x7c, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x66, 0x66, 0x00, 0x38,
  0x18, 0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00,
  0x00, 0x18, 0x3c, 0x66, 0x00, 0x38, 0x18, 0x18,
  0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00, 0x60,
  0x30, 0x18, 0x00, 0x38, 0x18, 0x18, 0x18, 0x18,
  0x3c, 0x00, 0x00, 0x00, 0x00, 0xc6, 0xc6, 0x10,
  0x38, 0x6c, 0xc6, 0xc6, 0xfe, 0xc6, 0xc6, 0x00,
  0x00, 0x00, 0x38, 0x6c, 0x38, 0x00, 0x38, 0x6c,
  0xc6, 0xc6, 0xfe, 0xc6, 0xc6, 0x00, 0x00, 0x00,
  0x18, 0x30, 0x60, 0x00, 0xfe, 0x66, 0x60, 0x7c,
  0x60, 0x66, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xcc, 0x76, 0x36, 0x7e, 0xd8, 0xd8,
  0x6e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3e, 0x6c,
  0xcc, 0xcc, 0xfe, 0xcc, 0xcc, 0xcc, 0xce, 0x00,
  0x00, 0x00, 0x00, 0x10, 0x38, 0x6c, 0x00, 0x7c,
  0xc6, 0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0xc6, 0x00, 0x7c, 0xc6, 0xc6,
  0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x60,
  0x30, 0x18, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6,
  0x7c, 0x00, 0x00, 0x00, 0x00, 0x30, 0x78, 0xcc,
  0x00, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0x76, 0x00,
  0x00, 0x00, 0x00, 0x60, 0x30, 0x18, 0x00, 0xcc,
  0xcc, 0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0xc6, 0x00, 0xc6, 0xc6, 0xc6,
  0xc6, 0x7e, 0x06, 0x0c, 0x78, 0x00, 0x00, 0xc6,
  0xc6, 0x38, 0x6c, 0xc6, 0xc6, 0xc6, 0xc6, 0x6c,
  0x38, 0x00, 0x00, 0x00, 0x00, 0xc6, 0xc6, 0x00,
  0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x7c, 0x00,
  0x00, 0x00, 0x00, 0x18, 0x18, 0x3c, 0x66, 0x60,
  0x60, 0x66, 0x3c, 0x18, 0x18, 0x00, 0x00, 0x00,
  0x00, 0x38, 0x6c, 0x64, 0x60, 0xf0, 0x60, 0x60,
  0x60, 0xe6, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x66, 0x66, 0x3c, 0x18, 0x7e, 0x18, 0x7e, 0x18,
  0x18, 0x00, 0x00, 0x00, 0x00, 0xf8, 0xcc, 0xcc,
  0xf8, 0xc4, 0xcc, 0xde, 0xcc, 0xcc, 0xc6, 0x00,
  0x00, 0x00, 0x00, 0x0e, 0x1b, 0x18, 0x18, 0x18,
  0x7e, 0x18, 0x18, 0x18, 0x18, 0xd8, 0x70, 0x00,
  0x00, 0x18, 0x30, 0x60, 0x00, 0x78, 0x0c, 0x7c,
  0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00, 0x0c,
  0x18, 0x30, 0x00, 0x38, 0x18, 0x18, 0x18, 0x18,
  0x3c, 0x00, 0x00, 0x00, 0x00, 0x18, 0x30, 0x60,
  0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0x7c, 0x00,
  0x00, 0x00, 0x00, 0x18, 0x30, 0x60, 0x00, 0xcc,
  0xcc, 0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x76, 0xdc, 0x00, 0xdc, 0x66, 0x66,
  0x66, 0x66, 0x66, 0x00, 0x00, 0x00, 0x76, 0xdc,
  0x00, 0xc6, 0xe6, 0xf6, 0xfe, 0xde, 0xce, 0xc6,
  0xc6, 0x00, 0x00, 0x00, 0x00, 0x3c, 0x6c, 0x6c,
  0x3e, 0x00, 0x7e, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x38, 0x6c, 0x6c, 0x38, 0x00,
  0x7c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x30, 0x30, 0x00, 0x30, 0x30, 0x60,
  0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xfe, 0xc0, 0xc0, 0xc0,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xfe, 0x06, 0x06, 0x06, 0x00, 0x00,
  0x00, 0x00, 0x00, 0xc0, 0xc0, 0xc6, 0xcc, 0xd8,
  0x30, 0x60, 0xdc, 0x86, 0x0c, 0x18, 0x3e, 0x00,
  0x00, 0xc0, 0xc0, 0xc6, 0xcc, 0xd8, 0x30, 0x66,
  0xce, 0x9e, 0x3e, 0x06, 0x06, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x00, 0x18, 0x18, 0x3c, 0x3c, 0x3c,
  0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x36, 0x6c, 0xd8, 0x6c, 0x36, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xd8, 0x6c,
  0x36, 0x6c, 0xd8, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x11, 0x44, 0x11, 0x44, 0x11, 0x44, 0x11, 0x44,
  0x11, 0x44, 0x11, 0x44, 0x11, 0x44, 0x55, 0xaa,
  0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa,
  0x55, 0xaa, 0x55, 0xaa, 0xdd, 0x77, 0xdd, 0x77,
  0xdd, 0x77, 0xdd, 0x77, 0xdd, 0x77, 0xdd, 0x77,
  0xdd, 0x77, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0xf8,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0xf8, 0x18, 0xf8, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0xf6, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0xfe, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xf8, 0x18, 0xf8,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x36, 0x36,
  0x36, 0x36, 0x36, 0xf6, 0x06, 0xf6, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe,
  0x06, 0xf6, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0xf6, 0x06, 0xfe,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0xfe, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x18, 0x18,
  0x18, 0xf8, 0x18, 0xf8, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0xf8, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x1f,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0xff, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0xff, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x1f, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0xff, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x1f, 0x18, 0x1f, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x37, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x37, 0x30, 0x3f,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x3f, 0x30, 0x37, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0xf7, 0x00, 0xff, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff,
  0x00, 0xf7, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x37, 0x30, 0x37,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x00, 0x00,
  0x00, 0x00, 0x00, 0xff, 0x00, 0xff, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x36, 0x36, 0x36, 0x36,
  0x36, 0xf7, 0x00, 0xf7, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x18, 0x18, 0x18, 0x18, 0x18, 0xff,
  0x00, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0xff,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0xff, 0x00, 0xff, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0xff, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x3f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x1f, 0x18, 0x1f,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x1f, 0x18, 0x1f, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x3f, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0xff, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x18, 0x18, 0x18, 0x18, 0x18, 0xff, 0x18, 0xff,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0xf8, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x1f, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0xf0,
  0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0,
  0xf0, 0xf0, 0xf0, 0xf0, 0x0f, 0x0f, 0x0f, 0x0f,
  0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f,
  0x0f, 0x0f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0xdc, 0xd8,
  0xd8, 0xdc, 0x76, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0xfc, 0xc6, 0xc6, 0xfc,
  0xc0, 0xc0, 0x40, 0x00, 0x00, 0x00, 0xfe, 0xc6,
  0xc6, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0x6c,
  0x6c, 0x6c, 0x6c, 0x6c, 0x6c, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xfe, 0xc6, 0x60, 0x30, 0x18, 0x30,
  0x60, 0xc6, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x7e, 0xd8, 0xd8, 0xd8, 0xd8,
  0x70, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x66, 0x66, 0x66, 0x66, 0x7c, 0x60, 0x60, 0xc0,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0xdc,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7e, 0x18, 0x3c, 0x66, 0x66, 0x66,
  0x3c, 0x18, 0x7e, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x38, 0x6c, 0xc6, 0xc6, 0xfe, 0xc6, 0xc6, 0x6c,
  0x38, 0x00, 0x00, 0x00, 0x00, 0x00, 0x38, 0x6c,
  0xc6, 0xc6, 0xc6, 0x6c, 0x6c, 0x6c, 0xee, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x1e, 0x30, 0x18, 0x0c,
  0x3e, 0x66, 0x66, 0x66, 0x3c, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x7e, 0xdb, 0xdb,
  0x7e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x03, 0x06, 0x7e, 0xdb, 0xdb, 0xf3, 0x7e, 0x60,
  0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1c, 0x30,
  0x60, 0x60, 0x7c, 0x60, 0x60, 0x30, 0x1c, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0xfe, 0x00, 0x00, 0xfe, 0x00,
  0x00, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x18, 0x18, 0x7e, 0x18, 0x18, 0x00, 0x00,
  0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x18,
  0x0c, 0x06, 0x0c, 0x18, 0x30, 0x00, 0x7e, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x0c, 0x18, 0x30, 0x60,
  0x30, 0x18, 0x0c, 0x00, 0x7e, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x0e, 0x1b, 0x1b, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0xd8, 0xd8,
  0x70, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18,
  0x18, 0x00, 0x7e, 0x00, 0x18, 0x18, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0xdc,
  0x00, 0x76, 0xdc, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x38, 0x6c, 0x6c, 0x38, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x0f, 0x0c, 0x0c, 0x0c, 0x0c,
  0x0c, 0xec, 0x6c, 0x3c, 0x1c, 0x00, 0x00, 0x00,
  0x00, 0xd8, 0x6c, 0x6c, 0x6c, 0x6c, 0x6c, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x70,
  0xd8, 0x30, 0x60, 0xc8, 0xf8, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x7c, 0x7c, 0x7c, 0x7c, 0x7c, 0x7c, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

uint8_t int10_font_16[256 * 16] = {
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7e, 0x81, 0xa5, 0x81, 0x81, 0xbd,
  0x99, 0x81, 0x81, 0x7e, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7e, 0xff, 0xdb, 0xff, 0xff, 0xc3,
  0xe7, 0xff, 0xff, 0x7e, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x6c, 0xfe, 0xfe, 0xfe,
  0xfe, 0x7c, 0x38, 0x10, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x10, 0x38, 0x7c, 0xfe,
  0x7c, 0x38, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x18, 0x3c, 0x3c, 0xe7, 0xe7,
  0xe7, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x18, 0x3c, 0x7e, 0xff, 0xff,
  0x7e, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x3c,
  0x3c, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe7, 0xc3,
  0xc3, 0xe7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x3c, 0x66, 0x42,
  0x42, 0x66, 0x3c, 0x00, 0x00, 0x00, 0x00, 0x00,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xc3, 0x99, 0xbd,
  0xbd, 0x99, 0xc3, 0xff, 0xff, 0xff, 0xff, 0xff,
  0x00, 0x00, 0x1e, 0x0e, 0x1a, 0x32, 0x78, 0xcc,
  0xcc, 0xcc, 0xcc, 0x78, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3c, 0x66, 0x66, 0x66, 0x66, 0x3c,
  0x18, 0x7e, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3f, 0x33, 0x3f, 0x30, 0x30, 0x30,
  0x30, 0x70, 0xf0, 0xe0, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7f, 0x63, 0x7f, 0x63, 0x63, 0x63,
  0x63, 0x67, 0xe7, 0xe6, 0xc0, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x18, 0x18, 0xdb, 0x3c, 0xe7,
  0x3c, 0xdb, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x80, 0xc0, 0xe0, 0xf0, 0xf8, 0xfe, 0xf8,
  0xf0, 0xe0, 0xc0, 0x80, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x02, 0x06, 0x0e, 0x1e, 0x3e, 0xfe, 0x3e,
  0x1e, 0x0e, 0x06, 0x02, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x3c, 0x7e, 0x18, 0x18, 0x18,
  0x7e, 0x3c, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66,
  0x66, 0x00, 0x66, 0x66, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7f, 0xdb, 0xdb, 0xdb, 0x7b, 0x1b,
  0x1b, 0x1b, 0x1b, 0x1b, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x7c, 0xc6, 0x60, 0x38, 0x6c, 0xc6, 0xc6,
  0x6c, 0x38, 0x0c, 0xc6, 0x7c, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0xfe, 0xfe, 0xfe, 0xfe, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x3c, 0x7e, 0x18, 0x18, 0x18,
  0x7e, 0x3c, 0x18, 0x7e, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x3c, 0x7e, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x7e, 0x3c, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x0c, 0xfe,
  0x0c, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x60, 0xfe,
  0x60, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0xc0,
  0xc0, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x6c, 0xfe,
  0x6c, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x10, 0x38, 0x38, 0x7c,
  0x7c, 0xfe, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xfe, 0xfe, 0x7c, 0x7c,
  0x38, 0x38, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x3c, 0x3c, 0x3c, 0x18, 0x18,
  0x18, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x66, 0x66, 0x66, 0x24, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x6c, 0x6c, 0xfe, 0x6c, 0x6c,
  0x6c, 0xfe, 0x6c, 0x6c, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x7c, 0xc6, 0xc2, 0xc0, 0x7c, 0x06,
  0x06, 0x86, 0xc6, 0x7c, 0x18, 0x18, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xc2, 0xc6, 0x0c, 0x18,
  0x30, 0x60, 0xc6, 0x86, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x38, 0x6c, 0x6c, 0x38, 0x76, 0xdc,
  0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x30, 0x30, 0x30, 0x60, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x0c, 0x18, 0x30, 0x30, 0x30, 0x30,
  0x30, 0x30, 0x18, 0x0c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x30, 0x18, 0x0c, 0x0c, 0x0c, 0x0c,
  0x0c, 0x0c, 0x18, 0x30, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x66, 0x3c, 0xff,
  0x3c, 0x66, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x7e,
  0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x18, 0x18, 0x18, 0x30, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x02, 0x06, 0x0c, 0x18,
  0x30, 0x60, 0xc0, 0x80, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x38, 0x6c, 0xc6, 0xc6, 0xd6, 0xd6,
  0xc6, 0xc6, 0x6c, 0x38, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x38, 0x78, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x7e, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0x06, 0x0c, 0x18, 0x30,
  0x60, 0xc0, 0xc6, 0xfe, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0x06, 0x06, 0x3c, 0x06,
  0x06, 0x06, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x0c, 0x1c, 0x3c, 0x6c, 0xcc, 0xfe,
  0x0c, 0x0c, 0x0c, 0x1e, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xfe, 0xc0, 0xc0, 0xc0, 0xfc, 0x06,
  0x06, 0x06, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x38, 0x60, 0xc0, 0xc0, 0xfc, 0xc6,
  0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xfe, 0xc6, 0x06, 0x06, 0x0c, 0x18,
  0x30, 0x30, 0x30, 0x30, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0x7c, 0xc6,
  0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0x7e, 0x06,
  0x06, 0x06, 0x0c, 0x78, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00,
  0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00,
  0x00, 0x18, 0x18, 0x30, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x06, 0x0c, 0x18, 0x30, 0x60,
  0x30, 0x18, 0x0c, 0x06, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x7e, 0x00, 0x00,
  0x7e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x60, 0x30, 0x18, 0x0c, 0x06,
  0x0c, 0x18, 0x30, 0x60, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0xc6, 0x0c, 0x18, 0x18,
  0x18, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xde, 0xde,
  0xde, 0xdc, 0xc0, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x10, 0x38, 0x6c, 0xc6, 0xc6, 0xfe,
  0xc6, 0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xfc, 0x66, 0x66, 0x66, 0x7c, 0x66,
  0x66, 0x66, 0x66, 0xfc, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3c, 0x66, 0xc2, 0xc0, 0xc0, 0xc0,
  0xc0, 0xc2, 0x66, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xf8, 0x6c, 0x66, 0x66, 0x66, 0x66,
  0x66, 0x66, 0x6c, 0xf8, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xfe, 0x66, 0x62, 0x68, 0x78, 0x68,
  0x60, 0x62, 0x66, 0xfe, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xfe, 0x66, 0x62, 0x68, 0x78, 0x68,
  0x60, 0x60, 0x60, 0xf0, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3c, 0x66, 0xc2, 0xc0, 0xc0, 0xde,
  0xc6, 0xc6, 0x66, 0x3a, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0xc6, 0xc6, 0xc6, 0xfe, 0xc6,
  0xc6, 0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3c, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x1e, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c,
  0xcc, 0xcc, 0xcc, 0x78, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xe6, 0x66, 0x66, 0x6c, 0x78, 0x78,
  0x6c, 0x66, 0x66, 0xe6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xf0, 0x60, 0x60, 0x60, 0x60, 0x60,
  0x60, 0x62, 0x66, 0xfe, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0xee, 0xfe, 0xfe, 0xd6, 0xc6,
  0xc6, 0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0xe6, 0xf6, 0xfe, 0xde, 0xce,
  0xc6, 0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xfc, 0x66, 0x66, 0x66, 0x7c, 0x60,
  0x60, 0x60, 0x60, 0xf0, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
  0xc6, 0xd6, 0xde, 0x7c, 0x0c, 0x0e, 0x00, 0x00,
  0x00, 0x00, 0xfc, 0x66, 0x66, 0x66, 0x7c, 0x6c,
  0x66, 0x66, 0x66, 0xe6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0xc6, 0x60, 0x38, 0x0c,
  0x06, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7e, 0x7e, 0x5a, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
  0xc6, 0x6c, 0x38, 0x10, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0xc6, 0xc6, 0xc6, 0xd6, 0xd6,
  0xd6, 0xfe, 0xee, 0x6c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0xc6, 0x6c, 0x7c, 0x38, 0x38,
  0x7c, 0x6c, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x66, 0x66, 0x66, 0x66, 0x3c, 0x18,
  0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xfe, 0xc6, 0x86, 0x0c, 0x18, 0x30,
  0x60, 0xc2, 0xc6, 0xfe, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3c, 0x30, 0x30, 0x30, 0x30, 0x30,
  0x30, 0x30, 0x30, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x80, 0xc0, 0xe0, 0x70, 0x38,
  0x1c, 0x0e, 0x06, 0x02, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c,
  0x0c, 0x0c, 0x0c, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x10, 0x38, 0x6c, 0xc6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00,
  0x30, 0x30, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x78, 0x0c, 0x7c,
  0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xe0, 0x60, 0x60, 0x78, 0x6c, 0x66,
  0x66, 0x66, 0x66, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 0xc0,
  0xc0, 0xc0, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x1c, 0x0c, 0x0c, 0x3c, 0x6c, 0xcc,
  0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 0xfe,
  0xc0, 0xc0, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x38, 0x6c, 0x64, 0x60, 0xf0, 0x60,
  0x60, 0x60, 0x60, 0xf0, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0xcc, 0xcc,
  0xcc, 0xcc, 0xcc, 0x7c, 0x0c, 0xcc, 0x78, 0x00,
  0x00, 0x00, 0xe0, 0x60, 0x60, 0x6c, 0x76, 0x66,
  0x66, 0x66, 0x66, 0xe6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x18, 0x00, 0x38, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x06, 0x06, 0x00, 0x0e, 0x06, 0x06,
  0x06, 0x06, 0x06, 0x06, 0x66, 0x66, 0x3c, 0x00,
  0x00, 0x00, 0xe0, 0x60, 0x60, 0x66, 0x6c, 0x78,
  0x78, 0x6c, 0x66, 0xe6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x38, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xec, 0xfe, 0xd6,
  0xd6, 0xd6, 0xd6, 0xc6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xdc, 0x66, 0x66,
  0x66, 0x66, 0x66, 0x66, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xdc, 0x66, 0x66,
  0x66, 0x66, 0x66, 0x7c, 0x60, 0x60, 0xf0, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0xcc, 0xcc,
  0xcc, 0xcc, 0xcc, 0x7c, 0x0c, 0x0c, 0x1e, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xdc, 0x76, 0x66,
  0x60, 0x60, 0x60, 0xf0, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 0x60,
  0x38, 0x0c, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x10, 0x30, 0x30, 0xfc, 0x30, 0x30,
  0x30, 0x30, 0x36, 0x1c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xcc, 0xcc, 0xcc,
  0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x66, 0x66, 0x66,
  0x66, 0x66, 0x3c, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xc6, 0xc6, 0xd6,
  0xd6, 0xd6, 0xfe, 0x6c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xc6, 0x6c, 0x38,
  0x38, 0x38, 0x6c, 0xc6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xc6, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0x7e, 0x06, 0x0c, 0xf8, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0xcc, 0x18,
  0x30, 0x60, 0xc6, 0xfe, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x0e, 0x18, 0x18, 0x18, 0x70, 0x18,
  0x18, 0x18, 0x18, 0x0e, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x18, 0x18, 0x18, 0x00, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x70, 0x18, 0x18, 0x18, 0x0e, 0x18,
  0x18, 0x18, 0x18, 0x70, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x76, 0xdc, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x10, 0x38, 0x6c, 0xc6,
  0xc6, 0xc6, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3c, 0x66, 0xc2, 0xc0, 0xc0, 0xc0,
  0xc2, 0x66, 0x3c, 0x0c, 0x06, 0x7c, 0x00, 0x00,
  0x00, 0x00, 0xcc, 0x00, 0x00, 0xcc, 0xcc, 0xcc,
  0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x0c, 0x18, 0x30, 0x00, 0x7c, 0xc6, 0xfe,
  0xc0, 0xc0, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x10, 0x38, 0x6c, 0x00, 0x78, 0x0c, 0x7c,
  0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xcc, 0x00, 0x00, 0x78, 0x0c, 0x7c,
  0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x60, 0x30, 0x18, 0x00, 0x78, 0x0c, 0x7c,
  0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x38, 0x6c, 0x38, 0x00, 0x78, 0x0c, 0x7c,
  0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x3c, 0x66, 0x60, 0x60,
  0x66, 0x3c, 0x0c, 0x06, 0x3c, 0x00, 0x00, 0x00,
  0x00, 0x10, 0x38, 0x6c, 0x00, 0x7c, 0xc6, 0xfe,
  0xc0, 0xc0, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0x00, 0x00, 0x7c, 0xc6, 0xfe,
  0xc0, 0xc0, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x60, 0x30, 0x18, 0x00, 0x7c, 0xc6, 0xfe,
  0xc0, 0xc0, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x66, 0x00, 0x00, 0x38, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x18, 0x3c, 0x66, 0x00, 0x38, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x60, 0x30, 0x18, 0x00, 0x38, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0xc6, 0x00, 0x10, 0x38, 0x6c, 0xc6, 0xc6,
  0xfe, 0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00,
  0x38, 0x6c, 0x38, 0x00, 0x38, 0x6c, 0xc6, 0xc6,
  0xfe, 0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x30, 0x60, 0x00, 0xfe, 0x66, 0x60, 0x7c,
  0x60, 0x60, 0x66, 0xfe, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xcc, 0x76, 0x36,
  0x7e, 0xd8, 0xd8, 0x6e, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3e, 0x6c, 0xcc, 0xcc, 0xfe, 0xcc,
  0xcc, 0xcc, 0xcc, 0xce, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x10, 0x38, 0x6c, 0x00, 0x7c, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0x00, 0x00, 0x7c, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x60, 0x30, 0x18, 0x00, 0x7c, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x30, 0x78, 0xcc, 0x00, 0xcc, 0xcc, 0xcc,
  0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x60, 0x30, 0x18, 0x00, 0xcc, 0xcc, 0xcc,
  0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0x00, 0x00, 0xc6, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0x7e, 0x06, 0x0c, 0x78, 0x00,
  0x00, 0xc6, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0xc6, 0x00, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x18, 0x18, 0x3c, 0x66, 0x60, 0x60, 0x60,
  0x66, 0x3c, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x38, 0x6c, 0x64, 0x60, 0xf0, 0x60, 0x60,
  0x60, 0x60, 0xe6, 0xfc, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x66, 0x66, 0x3c, 0x18, 0x7e, 0x18,
  0x7e, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0xf8, 0xcc, 0xcc, 0xf8, 0xc4, 0xcc, 0xde,
  0xcc, 0xcc, 0xcc, 0xc6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x0e, 0x1b, 0x18, 0x18, 0x18, 0x7e, 0x18,
  0x18, 0x18, 0x18, 0x18, 0xd8, 0x70, 0x00, 0x00,
  0x00, 0x18, 0x30, 0x60, 0x00, 0x78, 0x0c, 0x7c,
  0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x0c, 0x18, 0x30, 0x00, 0x38, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x18, 0x30, 0x60, 0x00, 0x7c, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x18, 0x30, 0x60, 0x00, 0xcc, 0xcc, 0xcc,
  0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x76, 0xdc, 0x00, 0xdc, 0x66, 0x66,
  0x66, 0x66, 0x66, 0x66, 0x00, 0x00, 0x00, 0x00,
  0x76, 0xdc, 0x00, 0xc6, 0xe6, 0xf6, 0xfe, 0xde,
  0xce, 0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x3c, 0x6c, 0x6c, 0x3e, 0x00, 0x7e, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x38, 0x6c, 0x6c, 0x38, 0x00, 0x7c, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x30, 0x30, 0x00, 0x30, 0x30, 0x60,
  0xc0, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0xc0,
  0xc0, 0xc0, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0x06,
  0x06, 0x06, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0xc0, 0xc0, 0xc2, 0xc6, 0xcc, 0x18, 0x30,
  0x60, 0xdc, 0x86, 0x0c, 0x18, 0x3e, 0x00, 0x00,
  0x00, 0xc0, 0xc0, 0xc2, 0xc6, 0xcc, 0x18, 0x30,
  0x66, 0xce, 0x9e, 0x3e, 0x06, 0x06, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x18, 0x00, 0x18, 0x18, 0x18,
  0x3c, 0x3c, 0x3c, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x36, 0x6c, 0xd8,
  0x6c, 0x36, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xd8, 0x6c, 0x36,
  0x6c, 0xd8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x11, 0x44, 0x11, 0x44, 0x11, 0x44, 0x11, 0x44,
  0x11, 0x44, 0x11, 0x44, 0x11, 0x44, 0x11, 0x44,
  0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa,
  0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa,
  0xdd, 0x77, 0xdd, 0x77, 0xdd, 0x77, 0xdd, 0x77,
  0xdd, 0x77, 0xdd, 0x77, 0xdd, 0x77, 0xdd, 0x77,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0xf8,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0xf8, 0x18, 0xf8,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0xf6,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xf8, 0x18, 0xf8,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x36, 0x36, 0x36, 0x36, 0x36, 0xf6, 0x06, 0xf6,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0x06, 0xf6,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0xf6, 0x06, 0xfe,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0xfe,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x18, 0x18, 0x18, 0xf8, 0x18, 0xf8,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf8,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x1f,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0xff,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x1f,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0xff,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x1f, 0x18, 0x1f,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x37,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x37, 0x30, 0x3f,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x3f, 0x30, 0x37,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0xf7, 0x00, 0xff,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0xf7,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x37, 0x30, 0x37,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0xff,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x36, 0x36, 0x36, 0x36, 0x36, 0xf7, 0x00, 0xf7,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x18, 0x18, 0x18, 0x18, 0x18, 0xff, 0x00, 0xff,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0xff,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0xff,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x3f,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x1f, 0x18, 0x1f,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x1f, 0x18, 0x1f,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3f,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0xff,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x18, 0x18, 0x18, 0x18, 0x18, 0xff, 0x18, 0xff,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0xf8,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1f,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0,
  0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0,
  0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f,
  0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0xdc, 0xd8,
  0xd8, 0xd8, 0xdc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x78, 0xcc, 0xcc, 0xcc, 0xd8, 0xcc,
  0xc6, 0xc6, 0xc6, 0xcc, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xfe, 0xc6, 0xc6, 0xc0, 0xc0, 0xc0,
  0xc0, 0xc0, 0xc0, 0xc0, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xfe, 0x6c, 0x6c, 0x6c,
  0x6c, 0x6c, 0x6c, 0x6c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0xfe, 0xc6, 0x60, 0x30, 0x18,
  0x30, 0x60, 0xc6, 0xfe, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x7e, 0xd8, 0xd8,
  0xd8, 0xd8, 0xd8, 0x70, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x66, 0x66, 0x66, 0x66,
  0x66, 0x7c, 0x60, 0x60, 0xc0, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x76, 0xdc, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x7e, 0x18, 0x3c, 0x66, 0x66,
  0x66, 0x3c, 0x18, 0x7e, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x38, 0x6c, 0xc6, 0xc6, 0xfe,
  0xc6, 0xc6, 0x6c, 0x38, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x38, 0x6c, 0xc6, 0xc6, 0xc6, 0x6c,
  0x6c, 0x6c, 0x6c, 0xee, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x1e, 0x30, 0x18, 0x0c, 0x3e, 0x66,
  0x66, 0x66, 0x66, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x7e, 0xdb, 0xdb,
  0xdb, 0x7e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x03, 0x06, 0x7e, 0xdb, 0xdb,
  0xf3, 0x7e, 0x60, 0xc0, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x1c, 0x30, 0x60, 0x60, 0x7c, 0x60,
  0x60, 0x60, 0x30, 0x1c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xfe, 0x00, 0x00, 0xfe,
  0x00, 0x00, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x7e, 0x18,
  0x18, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x30, 0x18, 0x0c, 0x06, 0x0c,
  0x18, 0x30, 0x00, 0x7e, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x0c, 0x18, 0x30, 0x60, 0x30,
  0x18, 0x0c, 0x00, 0x7e, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x0e, 0x1b, 0x1b, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0xd8, 0xd8, 0xd8, 0x70, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x7e,
  0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0xdc, 0x00,
  0x76, 0xdc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x38, 0x6c, 0x6c, 0x38, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18,
  0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x0f, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0xec,
  0x6c, 0x6c, 0x3c, 0x1c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0xd8, 0x6c, 0x6c, 0x6c, 0x6c, 0x6c, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x70, 0xd8, 0x30, 0x60, 0xc8, 0xf8, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x7c, 0x7c, 0x7c, 0x7c,
  0x7c, 0x7c, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

uint8_t int10_font_19[256 * 19] = {
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x7E,0x81,0x81,0xA5,0x81,0x81,0x81,0xBD,0x99,0x81,0x81,
	0x7E,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x7E,0xFF,0xFF,
	0xDB,0xFF,0xFF,0xFF,0xC3,0xE7,0xFF,0xFF,0x7E,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x6C,0xFE,0xFE,
	0xFE,0xFE,0xFE,0x7C,0x38,0x10,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x10,0x38,0x7C,0xFE,0xFE,0x7C,
	0x38,0x10,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x18,0x3C,0x3C,0x3C,0xE7,0xE7,0xE7,0xE7,0x18,0x18,0x3C,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x18,0x18,0x3C,
	0x7E,0xFF,0xFF,0xFF,0x7E,0x18,0x18,0x3C,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x18,0x3C,0x3C,
	0x3C,0x18,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xFF,0xFF,
	0xFF,0xFF,0xFF,0xFF,0xFF,0xE7,0xC3,0xC3,0xC3,0xE7,0xFF,
	0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0x00,0x00,0x00,0x00,0x00,
	0x00,0x3C,0x66,0x42,0x42,0x42,0x66,0x3C,0x00,0x00,0x00,
	0x00,0x00,0x00,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xC3,0x99,
	0xBD,0xBD,0xBD,0x99,0xC3,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,
	0x00,0x00,0x00,0x1E,0x06,0x0E,0x1A,0x30,0x78,0xCC,0xCC,
	0xCC,0xCC,0xCC,0x78,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x3C,0x66,0x66,0x66,0x66,0x66,0x66,0x3C,0x18,0x7E,0x18,
	0x18,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x3F,0x33,0x33,
	0x3F,0x30,0x30,0x30,0x30,0x30,0x70,0xF0,0xE0,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x7F,0x63,0x63,0x7F,0x63,0x63,
	0x63,0x63,0x63,0x67,0xE7,0xE6,0xC0,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x18,0x18,0xDB,0x3C,0xE7,0xE7,0x3C,
	0xDB,0x18,0x18,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,
	0xC0,0xE0,0xF0,0xF8,0xFE,0xFE,0xF8,0xF0,0xE0,0xC0,0x80,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x02,0x06,0x0E,0x1E,
	0x3E,0xFE,0xFE,0x3E,0x1E,0x0E,0x06,0x02,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x18,0x3C,0x7E,0x18,0x18,0x18,0x18,
	0x18,0x18,0x7E,0x3C,0x18,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x66,0x66,0x66,0x66,0x66,0x66,0x66,0x66,0x66,0x00,
	0x66,0x66,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x7F,0xDB,
	0xDB,0xDB,0xDB,0x7B,0x1B,0x1B,0x1B,0x1B,0x1B,0x1B,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x7C,0xC6,0x60,0x38,0x6C,
	0xC6,0xC6,0x6C,0x38,0x0C,0xC6,0x7C,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0xFE,0xFE,0xFE,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x18,0x3C,0x7E,0x18,0x18,0x18,0x18,0x18,0x18,0x7E,0x3C,
	0x18,0x7E,0x00,0x00,0x00,0x00,0x00,0x00,0x18,0x3C,0x7E,
	0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x18,0x18,0x18,0x18,0x18,0x18,
	0x18,0x18,0x18,0x7E,0x3C,0x18,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x18,0x0C,0xFE,0x0C,0x18,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x30,0x60,0xFE,0x60,0x30,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0xC0,0xC0,0xC0,0xC0,0xC0,0xFE,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x24,0x66,0xFF,
	0x66,0x24,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x10,0x10,0x38,0x38,0x7C,0x7C,0xFE,0xFE,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0xFE,0xFE,0x7C,0x7C,0x38,0x38,0x10,0x10,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x18,0x3C,0x3C,0x3C,0x3C,0x18,0x18,0x18,
	0x18,0x00,0x18,0x18,0x00,0x00,0x00,0x00,0x00,0x00,0x66,
	0x66,0x66,0x24,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x6C,0x6C,0x6C,
	0xFE,0x6C,0x6C,0x6C,0x6C,0xFE,0x6C,0x6C,0x6C,0x00,0x00,
	0x00,0x00,0x00,0x18,0x18,0x7C,0xC6,0xC2,0xC0,0xC0,0x7C,
	0x06,0x06,0x06,0x86,0xC6,0x7C,0x18,0x18,0x00,0x00,0x00,
	0x00,0x00,0xC6,0xC6,0xCC,0x0C,0x18,0x18,0x30,0x30,0x60,
	0x66,0xC6,0xC6,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x38,
	0x6C,0x6C,0x6C,0x38,0x76,0xDC,0xDC,0xCC,0xCC,0xCC,0x76,
	0x00,0x00,0x00,0x00,0x00,0x00,0x18,0x18,0x18,0x30,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x0C,0x18,0x30,0x30,0x30,0x30,0x30,
	0x30,0x30,0x30,0x18,0x0C,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x30,0x18,0x0C,0x0C,0x0C,0x0C,0x0C,0x0C,0x0C,0x0C,
	0x18,0x30,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x66,0x3C,0xFF,0x3C,0x66,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x18,0x18,
	0x18,0xFF,0x18,0x18,0x18,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x18,0x18,0x18,0x30,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0xFE,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x18,0x18,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x06,0x06,0x0C,0x0C,0x18,0x18,
	0x30,0x30,0x60,0x60,0xC0,0xC0,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x7C,0xC6,0xC6,0xC6,0xD6,0xD6,0xD6,0xD6,0xC6,
	0xC6,0xC6,0x7C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x18,
	0x38,0x78,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x7E,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x7C,0xC6,0x06,0x06,
	0x0C,0x18,0x30,0x60,0xC0,0xC0,0xC6,0xFE,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x7C,0xC6,0x06,0x06,0x06,0x3C,0x06,
	0x06,0x06,0x06,0xC6,0x7C,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x1C,0x1C,0x3C,0x3C,0x6C,0x6C,0xCC,0xFE,0x0C,0x0C,
	0x0C,0x1E,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xFE,0xC0,
	0xC0,0xC0,0xC0,0xFC,0x06,0x06,0x06,0x06,0xC6,0x7C,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x38,0x60,0xC0,0xC0,0xC0,
	0xFC,0xC6,0xC6,0xC6,0xC6,0xC6,0x7C,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0xFE,0xC6,0x06,0x06,0x06,0x0C,0x18,0x18,
	0x30,0x30,0x30,0x30,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x7C,0xC6,0xC6,0xC6,0xC6,0x7C,0xC6,0xC6,0xC6,0xC6,0xC6,
	0x7C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x7C,0xC6,0xC6,
	0xC6,0xC6,0x7E,0x06,0x06,0x06,0x06,0x0C,0x78,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x18,0x18,0x00,0x00,
	0x00,0x00,0x00,0x18,0x18,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x18,0x18,0x00,0x00,0x00,0x00,0x00,
	0x18,0x18,0x30,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x06,0x0C,0x18,0x30,0x60,0x60,0x30,0x18,0x0C,0x06,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0xFE,0x00,0x00,0xFE,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x60,0x30,0x18,0x0C,0x06,0x06,
	0x0C,0x18,0x30,0x60,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x7C,0xC6,0xC6,0x06,0x06,0x0C,0x18,0x18,0x18,0x00,
	0x18,0x18,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x7C,
	0xC6,0xC6,0xC6,0xDE,0xDE,0xDE,0xDC,0xC0,0xC0,0x7C,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x10,0x38,0x6C,0xC6,0xC6,
	0xC6,0xFE,0xC6,0xC6,0xC6,0xC6,0xC6,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0xFC,0x66,0x66,0x66,0x66,0x7C,0x66,0x66,
	0x66,0x66,0x66,0xFC,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x3C,0x66,0xC2,0xC0,0xC0,0xC0,0xC0,0xC0,0xC0,0xC2,0x66,
	0x3C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xF8,0x6C,0x66,
	0x66,0x66,0x66,0x66,0x66,0x66,0x66,0x6C,0xF8,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0xFE,0x66,0x62,0x60,0x68,0x78,
	0x68,0x60,0x60,0x62,0x66,0xFE,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0xFE,0x66,0x62,0x60,0x68,0x78,0x68,0x60,0x60,
	0x60,0x60,0xF0,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x3C,
	0x66,0xC2,0xC0,0xC0,0xC0,0xDE,0xC6,0xC6,0xC6,0x66,0x3A,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xC6,0xC6,0xC6,0xC6,
	0xC6,0xFE,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x3C,0x18,0x18,0x18,0x18,0x18,0x18,
	0x18,0x18,0x18,0x18,0x3C,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x0F,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0xC6,
	0xC6,0x7C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xE6,0x66,
	0x66,0x6C,0x6C,0x78,0x7C,0x6C,0x6C,0x66,0x66,0xE6,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0xF0,0x60,0x60,0x60,0x60,
	0x60,0x60,0x60,0x60,0x62,0x66,0xFE,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0xC6,0xEE,0xFE,0xFE,0xD6,0xD6,0xC6,0xC6,
	0xC6,0xC6,0xC6,0xC6,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0xC6,0xC6,0xE6,0xE6,0xF6,0xF6,0xDE,0xDE,0xCE,0xCE,0xC6,
	0xC6,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x38,0x6C,0xC6,
	0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0x6C,0x38,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0xFC,0x66,0x66,0x66,0x66,0x7C,
	0x60,0x60,0x60,0x60,0x60,0xF0,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x7C,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,
	0xD6,0xDE,0x7C,0x0C,0x0E,0x00,0x00,0x00,0x00,0x00,0xFC,
	0x66,0x66,0x66,0x66,0x7C,0x6C,0x6C,0x66,0x66,0x66,0xE6,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x7C,0xC6,0xC6,0xC0,
	0x60,0x38,0x0C,0x06,0x06,0xC6,0xC6,0x7C,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x7E,0x7E,0x5A,0x18,0x18,0x18,0x18,
	0x18,0x18,0x18,0x18,0x3C,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,
	0xC6,0x7C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xC6,0xC6,
	0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0x6C,0x38,0x10,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0xC6,0xC6,0xC6,0xC6,0xC6,
	0xC6,0xD6,0xD6,0xD6,0xFE,0x6C,0x6C,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0xC6,0xC6,0xC6,0x6C,0x6C,0x38,0x38,0x6C,
	0x6C,0xC6,0xC6,0xC6,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x66,0x66,0x66,0x66,0x66,0x3C,0x18,0x18,0x18,0x18,0x18,
	0x3C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xFE,0xC6,0x86,
	0x06,0x0C,0x18,0x30,0x60,0xC0,0xC2,0xC6,0xFE,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x3C,0x30,0x30,0x30,0x30,0x30,
	0x30,0x30,0x30,0x30,0x30,0x3C,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0xC0,0xC0,0x60,0x60,0x30,0x30,0x18,0x18,0x0C,
	0x0C,0x06,0x06,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x3C,
	0x0C,0x0C,0x0C,0x0C,0x0C,0x0C,0x0C,0x0C,0x0C,0x0C,0x3C,
	0x00,0x00,0x00,0x00,0x00,0x10,0x38,0x6C,0xC6,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0xFF,0x00,0x00,0x00,0x00,
	0x30,0x18,0x0C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x78,0x0C,0x0C,0x7C,0xCC,0xCC,0xCC,0x76,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0xE0,0x60,0x60,0x60,0x78,
	0x6C,0x66,0x66,0x66,0x66,0x66,0xDC,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x7C,0xC6,0xC0,0xC0,
	0xC0,0xC0,0xC6,0x7C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x1C,0x0C,0x0C,0x0C,0x3C,0x6C,0xCC,0xCC,0xCC,0xCC,0xCC,
	0x76,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x7C,0xC6,0xC6,0xFE,0xC0,0xC0,0xC6,0x7C,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x38,0x6C,0x64,0x60,0x60,0xF0,
	0x60,0x60,0x60,0x60,0x60,0xF0,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x76,0xCC,0xCC,0xCC,0xCC,
	0xCC,0x7C,0x0C,0x0C,0xCC,0x78,0x00,0x00,0x00,0x00,0xE0,
	0x60,0x60,0x60,0x6C,0x76,0x66,0x66,0x66,0x66,0x66,0xE6,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x18,0x18,0x00,0x00,
	0x38,0x18,0x18,0x18,0x18,0x18,0x18,0x3C,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x06,0x06,0x00,0x00,0x0E,0x06,0x06,
	0x06,0x06,0x06,0x06,0x06,0x66,0x66,0x3C,0x00,0x00,0x00,
	0x00,0xE0,0x60,0x60,0x60,0x66,0x66,0x6C,0x78,0x78,0x6C,
	0x66,0xE6,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x38,0x18,
	0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x3C,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xEC,
	0xFE,0xD6,0xD6,0xD6,0xD6,0xC6,0xC6,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xDC,0x66,0x66,0x66,
	0x66,0x66,0x66,0x66,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x7C,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,
	0x7C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0xDC,0x66,0x66,0x66,0x66,0x66,0x7C,0x60,0x60,0x60,
	0xF0,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x76,0xCC,
	0xCC,0xCC,0xCC,0xCC,0x7C,0x0C,0x0C,0x0C,0x1E,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0xDC,0x76,0x66,0x60,0x60,
	0x60,0x60,0xF0,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x7C,0xC6,0x60,0x38,0x0C,0x06,0xC6,0x7C,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x10,0x30,0x30,0x30,
	0xFC,0x30,0x30,0x30,0x30,0x30,0x36,0x1C,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xCC,0xCC,0xCC,
	0xCC,0xCC,0xCC,0xCC,0x76,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x66,0x66,0x66,0x66,0x66,0x66,
	0x3C,0x18,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0xC6,0xC6,0xC6,0xD6,0xD6,0xD6,0xFE,0x6C,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xC6,
	0xC6,0x6C,0x38,0x38,0x6C,0xC6,0xC6,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xC6,0xC6,0xC6,0xC6,
	0xC6,0xC6,0x7E,0x06,0x06,0x0C,0x78,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0xFE,0xC6,0x0C,0x18,0x30,0x60,0xC6,
	0xFE,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0E,0x18,0x18,
	0x18,0x18,0x70,0x70,0x18,0x18,0x18,0x18,0x0E,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x18,0x18,0x18,0x18,0x18,0x18,
	0x18,0x18,0x18,0x18,0x18,0x18,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x70,0x18,0x18,0x18,0x18,0x0E,0x0E,0x18,0x18,
	0x18,0x18,0x70,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x76,
	0xDC,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x10,
	0x38,0x6C,0xC6,0xC6,0xC6,0xC6,0xFE,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x3C,0x66,0xC2,0xC0,0xC0,0xC0,0xC0,
	0xC0,0xC0,0xC2,0x66,0x3C,0x18,0x0C,0x38,0x00,0x00,0x00,
	0x00,0x00,0xCC,0xCC,0x00,0xCC,0xCC,0xCC,0xCC,0xCC,0xCC,
	0xCC,0x76,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0C,0x18,
	0x30,0x00,0x7C,0xC6,0xC6,0xFE,0xC0,0xC0,0xC6,0x7C,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x10,0x38,0x6C,0x00,0x78,
	0x0C,0x0C,0x7C,0xCC,0xCC,0xCC,0x76,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0xCC,0xCC,0x00,0x78,0x0C,0x0C,0x7C,
	0xCC,0xCC,0xCC,0x76,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x60,0x30,0x18,0x00,0x78,0x0C,0x0C,0x7C,0xCC,0xCC,0xCC,
	0x76,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x38,0x6C,0x38,
	0x00,0x78,0x0C,0x0C,0x7C,0xCC,0xCC,0xCC,0x76,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x7C,0xC6,
	0xC0,0xC0,0xC0,0xC0,0xC6,0x7C,0x18,0x0C,0x38,0x00,0x00,
	0x00,0x00,0x10,0x38,0x6C,0x00,0x7C,0xC6,0xC6,0xFE,0xC0,
	0xC0,0xC6,0x7C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0xCC,0xCC,0x00,0x7C,0xC6,0xC6,0xFE,0xC0,0xC0,0xC6,0x7C,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x60,0x30,0x18,0x00,
	0x7C,0xC6,0xC6,0xFE,0xC0,0xC0,0xC6,0x7C,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x66,0x66,0x00,0x38,0x18,0x18,
	0x18,0x18,0x18,0x18,0x3C,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x18,0x3C,0x66,0x00,0x38,0x18,0x18,0x18,0x18,0x18,
	0x18,0x3C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x60,0x30,
	0x18,0x00,0x38,0x18,0x18,0x18,0x18,0x18,0x18,0x3C,0x00,
	0x00,0x00,0x00,0xC6,0xC6,0x00,0x10,0x38,0x6C,0xC6,0xC6,
	0xC6,0xFE,0xC6,0xC6,0xC6,0xC6,0xC6,0x00,0x00,0x00,0x00,
	0x38,0x6C,0x38,0x00,0x38,0x6C,0xC6,0xC6,0xC6,0xFE,0xC6,
	0xC6,0xC6,0xC6,0xC6,0x00,0x00,0x00,0x00,0x0C,0x18,0x00,
	0xFE,0x66,0x62,0x60,0x68,0x78,0x68,0x60,0x60,0x62,0x66,
	0xFE,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0xEC,0x36,0x36,0x7E,0xD8,0xD8,0xD8,0x6E,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x3E,0x6C,0xCC,0xCC,0xCC,0xFE,
	0xCC,0xCC,0xCC,0xCC,0xCC,0xCE,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x10,0x38,0x6C,0x00,0x7C,0xC6,0xC6,0xC6,0xC6,
	0xC6,0xC6,0x7C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0xC6,0xC6,0x00,0x7C,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0x7C,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x60,0x30,0x18,0x00,
	0x7C,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0x7C,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x30,0x78,0xCC,0x00,0xCC,0xCC,0xCC,
	0xCC,0xCC,0xCC,0xCC,0x76,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x60,0x30,0x18,0x00,0xCC,0xCC,0xCC,0xCC,0xCC,0xCC,
	0xCC,0x76,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xC6,
	0xC6,0x00,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0x7E,0x06,0x06,
	0x0C,0x78,0x00,0xC6,0xC6,0x00,0x38,0x6C,0xC6,0xC6,0xC6,
	0xC6,0xC6,0xC6,0xC6,0xC6,0x6C,0x38,0x00,0x00,0x00,0x00,
	0xC6,0xC6,0x00,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,
	0xC6,0xC6,0xC6,0x7C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x18,0x18,0x3C,0x66,0x60,0x60,0x60,0x60,0x66,0x3C,0x18,
	0x18,0x00,0x00,0x00,0x00,0x00,0x00,0x38,0x6C,0x64,0x60,
	0x60,0xF0,0x60,0x60,0x60,0x60,0x60,0xE6,0xFC,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x66,0x66,0x66,0x3C,0x18,0x7E,
	0x18,0x18,0x7E,0x18,0x18,0x18,0x00,0x00,0x00,0x00,0x00,
	0x00,0xF8,0xCC,0xCC,0xCC,0xF8,0xC4,0xCC,0xDE,0xCC,0xCC,
	0xCC,0xCC,0xC6,0x00,0x00,0x00,0x00,0x00,0x00,0x0E,0x1B,
	0x18,0x18,0x18,0x18,0x7E,0x18,0x18,0x18,0x18,0x18,0x18,
	0xD8,0x70,0x00,0x00,0x00,0x00,0x00,0x0C,0x18,0x30,0x00,
	0x78,0x0C,0x0C,0x7C,0xCC,0xCC,0xCC,0x76,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x0C,0x18,0x30,0x00,0x38,0x18,0x18,
	0x18,0x18,0x18,0x18,0x3C,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x0C,0x18,0x30,0x00,0x7C,0xC6,0xC6,0xC6,0xC6,0xC6,
	0xC6,0x7C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0C,0x18,
	0x30,0x00,0xCC,0xCC,0xCC,0xCC,0xCC,0xCC,0xCC,0x76,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x76,0xDC,0x00,0xDC,
	0x66,0x66,0x66,0x66,0x66,0x66,0x66,0x00,0x00,0x00,0x00,
	0x76,0xDC,0x00,0xC6,0xC6,0xE6,0xE6,0xF6,0xF6,0xDE,0xDE,
	0xCE,0xCE,0xC6,0xC6,0x00,0x00,0x00,0x00,0x00,0x00,0x3C,
	0x6C,0x6C,0x6C,0x3E,0x00,0x7E,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x38,0x6C,0x6C,0x6C,
	0x38,0x00,0x7C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x30,0x30,0x00,0x30,0x30,0x30,
	0x60,0xC0,0xC0,0xC6,0xC6,0x7C,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xFE,0xC0,0xC0,0xC0,
	0xC0,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0xFE,0x06,0x06,0x06,0x06,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0xC0,0xC0,0xC0,0xC2,0xC6,
	0xCC,0x18,0x30,0x60,0xDC,0xA6,0x0C,0x18,0x30,0x3E,0x00,
	0x00,0x00,0x00,0xC0,0xC0,0xC0,0xC2,0xC6,0xCC,0x18,0x30,
	0x60,0xCC,0x9C,0x3C,0x7E,0x0C,0x0C,0x00,0x00,0x00,0x00,
	0x00,0x18,0x18,0x00,0x00,0x18,0x18,0x18,0x18,0x3C,0x3C,
	0x3C,0x18,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x33,0x66,0xCC,0xCC,0x66,0x33,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xCC,
	0x66,0x33,0x33,0x66,0xCC,0x00,0x00,0x00,0x00,0x00,0x00,
	0x11,0x44,0x11,0x44,0x11,0x44,0x11,0x44,0x11,0x44,0x11,
	0x44,0x11,0x44,0x11,0x44,0x11,0x44,0x11,0x55,0xAA,0x55,
	0xAA,0x55,0xAA,0x55,0xAA,0x55,0xAA,0x55,0xAA,0x55,0xAA,
	0x55,0xAA,0x55,0xAA,0x55,0xDD,0x77,0xDD,0x77,0xDD,0x77,
	0xDD,0x77,0xDD,0x77,0xDD,0x77,0xDD,0x77,0xDD,0x77,0xDD,
	0x77,0xDD,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,
	0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,
	0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0xF8,0x18,0x18,
	0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,
	0x18,0x18,0x18,0x18,0xF8,0x18,0xF8,0x18,0x18,0x18,0x18,
	0x18,0x18,0x18,0x18,0x36,0x36,0x36,0x36,0x36,0x36,0x36,
	0x36,0x36,0xF6,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,
	0x36,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xFE,
	0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0xF8,0x18,0xF8,0x18,0x18,
	0x18,0x18,0x18,0x18,0x18,0x18,0x36,0x36,0x36,0x36,0x36,
	0x36,0x36,0x36,0xF6,0x06,0xF6,0x36,0x36,0x36,0x36,0x36,
	0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,
	0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xFE,0x06,0xF6,
	0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,
	0x36,0x36,0x36,0x36,0x36,0xF6,0x06,0xFE,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x36,0x36,0x36,0x36,0x36,0x36,
	0x36,0x36,0x36,0xFE,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0xF8,
	0x18,0xF8,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xF8,0x18,0x18,
	0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,
	0x18,0x18,0x18,0x18,0x18,0x1F,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x18,0x18,0x18,0x18,0x18,0x18,0x18,
	0x18,0x18,0xFF,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xFF,
	0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,
	0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x1F,0x18,0x18,0x18,
	0x18,0x18,0x18,0x18,0x18,0x18,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0xFF,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,
	0x18,0xFF,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,
	0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x1F,0x18,0x1F,
	0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x36,0x36,0x36,
	0x36,0x36,0x36,0x36,0x36,0x36,0x37,0x36,0x36,0x36,0x36,
	0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,
	0x36,0x36,0x37,0x30,0x3F,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x3F,
	0x30,0x37,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,
	0x36,0x36,0x36,0x36,0x36,0x36,0x36,0xF7,0x00,0xFF,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0xFF,0x00,0xF7,0x36,0x36,0x36,0x36,
	0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,
	0x36,0x37,0x30,0x37,0x36,0x36,0x36,0x36,0x36,0x36,0x36,
	0x36,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xFF,0x00,
	0xFF,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x36,0x36,
	0x36,0x36,0x36,0x36,0x36,0x36,0xF7,0x00,0xF7,0x36,0x36,
	0x36,0x36,0x36,0x36,0x36,0x36,0x18,0x18,0x18,0x18,0x18,
	0x18,0x18,0x18,0xFF,0x00,0xFF,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,
	0x36,0xFF,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xFF,0x00,0xFF,
	0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0xFF,0x36,0x36,0x36,0x36,
	0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,
	0x36,0x36,0x36,0x3F,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x1F,
	0x18,0x1F,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x1F,0x18,0x1F,0x18,
	0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x3F,0x36,0x36,0x36,0x36,0x36,
	0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,
	0x36,0x36,0xFF,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,
	0x36,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0xFF,0x18,
	0xFF,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,
	0x18,0x18,0x18,0x18,0x18,0x18,0x18,0xF8,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x1F,0x18,0x18,0x18,0x18,0x18,0x18,
	0x18,0x18,0x18,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,
	0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xFF,0xFF,
	0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xF0,0xF0,0xF0,
	0xF0,0xF0,0xF0,0xF0,0xF0,0xF0,0xF0,0xF0,0xF0,0xF0,0xF0,
	0xF0,0xF0,0xF0,0xF0,0xF0,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,
	0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,
	0x0F,0x0F,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x76,0xDC,0xD8,0xD8,0xD8,
	0xD8,0xDC,0x76,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x78,
	0xCC,0xCC,0xCC,0xCC,0xD8,0xCC,0xC6,0xC6,0xC6,0xC6,0xDC,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xFE,0xC6,0xC6,0xC0,
	0xC0,0xC0,0xC0,0xC0,0xC0,0xC0,0xC0,0xC0,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xFE,0x6C,0x6C,
	0x6C,0x6C,0x6C,0x6C,0x6C,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0xFE,0xC6,0xC0,0x60,0x30,0x18,0x18,0x30,0x60,0xC0,
	0xC6,0xFE,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x7E,0xD8,0xD8,0xD8,0xD8,0xD8,0xD8,0x70,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x66,
	0x66,0x66,0x66,0x66,0x66,0x66,0x7C,0x60,0x60,0xC0,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x76,0xDC,0x18,0x18,
	0x18,0x18,0x18,0x18,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x3C,0x18,0x3C,0x66,0x66,0x66,0x66,0x66,0x66,0x3C,0x18,
	0x3C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x38,0x6C,0xC6,
	0xC6,0xC6,0xFE,0xC6,0xC6,0xC6,0xC6,0x6C,0x38,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x38,0x6C,0xC6,0xC6,0xC6,0xC6,
	0xC6,0xC6,0x6C,0x6C,0x6C,0xEE,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x1E,0x30,0x18,0x0C,0x3E,0x66,0x66,0x66,0x66,
	0x66,0x66,0x3C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x7E,0xDB,0xDB,0xDB,0xDB,0x7E,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x03,0x06,
	0x7E,0xCF,0xDB,0xDB,0xF3,0x7E,0x60,0xC0,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x1C,0x30,0x60,0x60,0x60,0x7C,0x60,
	0x60,0x60,0x60,0x30,0x1C,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x7C,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,
	0xC6,0xC6,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0xFE,0x00,0x00,0x00,0xFE,0x00,0x00,0x00,0xFE,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x18,0x18,0x18,
	0x7E,0x18,0x18,0x18,0x00,0x00,0x7E,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x60,0x30,0x18,0x0C,0x06,0x0C,0x18,
	0x30,0x60,0x00,0x7E,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x06,0x0C,0x18,0x30,0x60,0x30,0x18,0x0C,0x06,0x00,
	0x7E,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0E,0x1B,0x1B,
	0x1B,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,
	0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,
	0x18,0x18,0xD8,0xD8,0xD8,0x70,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x18,0x18,0x00,0x00,0x7E,0x00,0x00,
	0x18,0x18,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x76,0xDC,0x00,0x00,0x76,0xDC,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x38,0x6C,0x6C,0x6C,0x38,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x18,0x18,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x18,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0F,0x0C,0x0C,
	0x0C,0x0C,0x0C,0x0C,0x0C,0xEC,0x6C,0x6C,0x3C,0x1C,0x00,
	0x00,0x00,0x00,0x00,0x00,0xD8,0x6C,0x6C,0x6C,0x6C,0x6C,
	0x6C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x38,0x6C,0x0C,0x18,0x30,0x64,0x7C,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x7C,0x7C,0x7C,0x7C,0x7C,0x7C,0x7C,0x7C,0x7C,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,
};

uint8_t int10_font_14_alternate[20 * 15 + 1] = {
  0x1d,
  0x00, 0x00, 0x00, 0x00, 0x24, 0x66, 0xff,
  0x66, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x22,
  0x00, 0x63, 0x63, 0x63, 0x22, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x2b,
  0x00, 0x00, 0x00, 0x18, 0x18, 0x18, 0xff,
  0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x2d,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x4d,
  0x00, 0x00, 0xc3, 0xe7, 0xff, 0xdb, 0xc3,
  0xc3, 0xc3, 0xc3, 0xc3, 0x00, 0x00, 0x00,
  0x54,
  0x00, 0x00, 0xff, 0xdb, 0x99, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00,
  0x56,
  0x00, 0x00, 0xc3, 0xc3, 0xc3, 0xc3, 0xc3,
  0xc3, 0x66, 0x3c, 0x18, 0x00, 0x00, 0x00,
  0x57,
  0x00, 0x00, 0xc3, 0xc3, 0xc3, 0xc3, 0xdb,
  0xdb, 0xff, 0x66, 0x66, 0x00, 0x00, 0x00,
  0x58,
  0x00, 0x00, 0xc3, 0xc3, 0x66, 0x3c, 0x18,
  0x3c, 0x66, 0xc3, 0xc3, 0x00, 0x00, 0x00,
  0x59,
  0x00, 0x00, 0xc3, 0xc3, 0xc3, 0x66, 0x3c,
  0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00,
  0x5a,
  0x00, 0x00, 0xff, 0xc3, 0x86, 0x0c, 0x18,
  0x30, 0x61, 0xc3, 0xff, 0x00, 0x00, 0x00,
  0x6d,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xe6, 0xff,
  0xdb, 0xdb, 0xdb, 0xdb, 0x00, 0x00, 0x00,
  0x76,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xc3, 0xc3,
  0xc3, 0x66, 0x3c, 0x18, 0x00, 0x00, 0x00,
  0x77,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xc3, 0xc3,
  0xdb, 0xdb, 0xff, 0x66, 0x00, 0x00, 0x00,
  0x91,
  0x00, 0x00, 0x00, 0x00, 0x6e, 0x3b, 0x1b,
  0x7e, 0xd8, 0xdc, 0x77, 0x00, 0x00, 0x00,
  0x9b,
  0x00, 0x18, 0x18, 0x7e, 0xc3, 0xc0, 0xc0,
  0xc3, 0x7e, 0x18, 0x18, 0x00, 0x00, 0x00,
  0x9d,
  0x00, 0x00, 0xc3, 0x66, 0x3c, 0x18, 0xff,
  0x18, 0xff, 0x18, 0x18, 0x00, 0x00, 0x00,
  0x9e,
  0x00, 0xfc, 0x66, 0x66, 0x7c, 0x62, 0x66,
  0x6f, 0x66, 0x66, 0xf3, 0x00, 0x00, 0x00,
  0xf1,
  0x00, 0x00, 0x18, 0x18, 0x18, 0xff, 0x18,
  0x18, 0x18, 0x00, 0xff, 0x00, 0x00, 0x00,
  0xf6,
  0x00, 0x00, 0x18, 0x18, 0x00, 0x00, 0xff,
  0x00, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00,
  0x00
};

uint8_t int10_font_16_alternate[19 * 17 + 1] = {
  0x1d,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x24, 0x66, 0xff,
  0x66, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x30,
  0x00, 0x00, 0x3c, 0x66, 0xc3, 0xc3, 0xdb, 0xdb,
  0xc3, 0xc3, 0x66, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x4d,
  0x00, 0x00, 0xc3, 0xe7, 0xff, 0xff, 0xdb, 0xc3,
  0xc3, 0xc3, 0xc3, 0xc3, 0x00, 0x00, 0x00, 0x00,
  0x54,
  0x00, 0x00, 0xff, 0xdb, 0x99, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x56,
  0x00, 0x00, 0xc3, 0xc3, 0xc3, 0xc3, 0xc3, 0xc3,
  0xc3, 0x66, 0x3c, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x57,
  0x00, 0x00, 0xc3, 0xc3, 0xc3, 0xc3, 0xc3, 0xdb,
  0xdb, 0xff, 0x66, 0x66, 0x00, 0x00, 0x00, 0x00,
  0x58,
  0x00, 0x00, 0xc3, 0xc3, 0x66, 0x3c, 0x18, 0x18,
  0x3c, 0x66, 0xc3, 0xc3, 0x00, 0x00, 0x00, 0x00,
  0x59,
  0x00, 0x00, 0xc3, 0xc3, 0xc3, 0x66, 0x3c, 0x18,
  0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x5a,
  0x00, 0x00, 0xff, 0xc3, 0x86, 0x0c, 0x18, 0x30,
  0x60, 0xc1, 0xc3, 0xff, 0x00, 0x00, 0x00, 0x00,
  0x6d,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xe6, 0xff, 0xdb,
  0xdb, 0xdb, 0xdb, 0xdb, 0x00, 0x00, 0x00, 0x00,
  0x76,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xc3, 0xc3, 0xc3,
  0xc3, 0x66, 0x3c, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x77,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xc3, 0xc3, 0xc3,
  0xdb, 0xdb, 0xff, 0x66, 0x00, 0x00, 0x00, 0x00,
  0x78,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xc3, 0x66, 0x3c,
  0x18, 0x3c, 0x66, 0xc3, 0x00, 0x00, 0x00, 0x00,
  0x91,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x6e, 0x3b, 0x1b,
  0x7e, 0xd8, 0xdc, 0x77, 0x00, 0x00, 0x00, 0x00,
  0x9b,
  0x00, 0x18, 0x18, 0x7e, 0xc3, 0xc0, 0xc0, 0xc0,
  0xc3, 0x7e, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x9d,
  0x00, 0x00, 0xc3, 0x66, 0x3c, 0x18, 0xff, 0x18,
  0xff, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x9e,
  0x00, 0xfc, 0x66, 0x66, 0x7c, 0x62, 0x66, 0x6f,
  0x66, 0x66, 0x66, 0xf3, 0x00, 0x00, 0x00, 0x00,
  0xab,
  0x00, 0xc0, 0xc0, 0xc2, 0xc6, 0xcc, 0x18, 0x30,
  0x60, 0xce, 0x9b, 0x06, 0x0c, 0x1f, 0x00, 0x00,
  0xac,
  0x00, 0xc0, 0xc0, 0xc2, 0xc6, 0xcc, 0x18, 0x30,
  0x66, 0xce, 0x96, 0x3e, 0x06, 0x06, 0x00, 0x00,
  0x00
};
