"""Shared fixtures for unit tests involving dolfin."""

# Copyright (C) 2014-2014 Martin Sandve Alnaes and Aslak Wigdahl Bergersen
#
# This file is part of DOLFIN.
#
# DOLFIN is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# DOLFIN is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with DOLFIN. If not, see <http://www.gnu.org/licenses/>.

from __future__ import print_function
import pytest
import os
import shutil
import tempfile
import gc
import platform
from instant import get_status_output
from dolfin import MPI, mpi_comm_world, parameters
from dolfin import  *

# --- Test fixtures (use as is or as examples): ---

def gc_barrier():
    """Internal utility to easily switch on and off calls to
    gc.collect() and MPI.barrier(world) in all fixtures here.
    Helps make the tests deterministic when debugging.
    """
    gc.collect()
    if MPI.size(mpi_comm_world()) > 1:
        MPI.barrier(mpi_comm_world())

@pytest.yield_fixture(scope="function")
def gc_barrier_fixture():
    """Function decorator to call gc.collect() and MPI.barrier(world) before and after a test.

    Helps make the tests deterministic when debugging.
    """
    gc_barrier()
    yield
    gc_barrier()

use_gc_barrier = pytest.mark.usefixtures("gc_barrier_fixture")


@pytest.fixture(params=[False, True])
def true_false_fixture(request):
    "A fixture setting the values true and false."
    gc_barrier()
    return request.param


@pytest.fixture(scope="module")
def filedir(request):
    "Return the directory of the test module."
    gc_barrier()
    d = os.path.dirname(os.path.abspath(request.module.__file__))
    return d

@pytest.fixture(scope="module")
def rootdir(request):
    "Return the root directory of the repository. Assumes run from within repository filetree."
    gc_barrier()
    d = os.path.dirname(os.path.abspath(request.module.__file__))
    t = ''
    while t != "test":
        d, t = os.path.split(d)
    return d

@pytest.fixture(scope="module")
def datadir(request):
    "Return the directory of the shared test data. Assumes run from within repository filetree."
    d = os.path.dirname(os.path.abspath(request.module.__file__))
    t = ''
    while t != "test":
        d, t = os.path.split(d)
    return os.path.join(d, "test", "data")

def _create_tempdir(request):
    # Get directory name of test_foo.py file
    testfile = request.module.__file__
    testfiledir = os.path.dirname(os.path.abspath(testfile))

    # Construct name test_foo_tempdir from name test_foo.py
    testfilename = os.path.basename(testfile)
    outputname = testfilename.replace(".py", "_tempdir")

    # Get function name test_something from test_foo.py
    function = request.function.__name__

    # Join all of these to make a unique path for this test function
    basepath = os.path.join(testfiledir, outputname)
    path = os.path.join(basepath, function)

    # Add a sequence number to avoid collisions when tests are otherwise parameterized
    if MPI.rank(mpi_comm_world()) == 0:
        _create_tempdir._sequencenumber[path] += 1
        sequencenumber = _create_tempdir._sequencenumber[path]
        sequencenumber = MPI.sum(mpi_comm_world(), sequencenumber)
    else:
        sequencenumber = MPI.sum(mpi_comm_world(), 0)
    path += "__" + str(sequencenumber)

    # Delete and re-create directory on root node
    if MPI.rank(mpi_comm_world()) == 0:
        # First time visiting this basepath, delete the old and create a new
        if basepath not in _create_tempdir._basepaths:
            _create_tempdir._basepaths.add(basepath)
            #if os.path.exists(basepath):
            #    shutil.rmtree(basepath)
            # Make sure we have the base path test_foo_tempdir for this test_foo.py file
            if not os.path.exists(basepath):
                os.mkdir(basepath)

        # Delete path from old test run
        #if os.path.exists(path):
        #    shutil.rmtree(path)
        # Make sure we have the path for this test execution: e.g. test_foo_tempdir/test_something__3
        if not os.path.exists(path):
            os.mkdir(path)
    MPI.barrier(mpi_comm_world())

    return path
from collections import defaultdict
_create_tempdir._sequencenumber = defaultdict(int)
_create_tempdir._basepaths = set()

@pytest.fixture(scope="function")
def tempdir(request):
    """Return a unique directory name for this test function instance.

    Deletes and re-creates directory from previous test runs but lets
    the directory stay after the test run for eventual inspection.

    Returns the directory name, derived from the test file and function
    plus a sequence number to work with parameterized tests.

    Does NOT change the current directory.

    MPI safe (assuming mpi_comm_world() context).
    """
    gc_barrier()
    return _create_tempdir(request)

@pytest.yield_fixture(scope="function")
def cd_tempdir(request):
    """Return a unique directory name for this test function instance.

    Deletes and re-creates directory from previous test runs but lets
    the directory stay after the test run for eventual inspection.

    Returns the directory name, derived from the test file and function
    plus a sequence number to work with parameterized tests.

    Changes the current directory to the tempdir and resets cwd afterwards.

    MPI safe (assuming mpi_comm_world() context).
    """
    gc_barrier()
    cwd = os.getcwd()
    path = _create_tempdir(request)
    os.chdir(path)
    yield path
    os.chdir(cwd)

@pytest.yield_fixture
def pushpop_parameters():
    global parameters
    gc_barrier()
    prev = parameters.copy()
    yield parameters.copy()
    parameters.assign(prev)

def set_parameters_fixture(paramname, values, key=lambda x: x):
    """Return a fixture that sets and resets a global parameter
    to each of a list of values before and after each test run.

    Usage:
        my_fixture1 = set_parameters_fixture("linear_algebra_backend", ["PETSc", "STL"])
        my_fixture2 = set_parameters_fixture("linear_algebra_backend", [("uBLAS", "Sparse"), ("STL", "")], key=lambda x: x[0])

        def test_something1(my_fixture1):
            assert my_fixture1 in ("PETSc", "STL")
            assert parameters.linear_algebra_backend == my_fixture1

        def test_something2(my_fixture2):
            assert my_fixture2[0] in ("uBLAS", "STL")
            assert parameters.linear_algebra_backend == my_fixture2[0]

    Try it and see.
    """
    global parameters
    def _pushpop(request):
        gc_barrier()
        #TODO: This should be done more robust and
        #      assumes that parameters[][][] is not used.
        #TODO: Check if the backend exists?
        if '.' in paramname:
            first, second = paramname.split('.')
            prev = parameters[first][second]
            yield request.param
            parameters[first][second] = prev
        else:
            prev = parameters[paramname]               # Remember original value
            parameters[paramname] = key(request.param) # Set value
            yield request.param                        # Let test run
            parameters[paramname] = prev               # Reset value

    return pytest.yield_fixture(scope="function", params=values)(_pushpop)
