"""This module provides functionality for form assembly in Python,
corresponding to the C++ assembly and PDE classes.

The C++ :py:class:`assemble <dolfin.cpp.assemble>` function
(renamed to cpp_assemble) is wrapped with an additional
preprocessing step where code is generated using the
FFC JIT compiler.

The C++ PDE classes are reimplemented in Python since the C++ classes
rely on the dolfin::Form class which is not used on the Python side."""

# Copyright (C) 2007-2014 Anders Logg
#
# This file is part of DOLFIN.
#
# DOLFIN is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# DOLFIN is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with DOLFIN. If not, see <http://www.gnu.org/licenses/>.
#
# Modified by Martin Sandve Alnaes, 2008-2014.
# Modified by Johan Hake, 2008-2009.
# Modified by Garth N. Wells, 2008-2013.
# Modified by Joachim B. Haga, 2012.

__all__ = ["assemble", "assemble_system", "SystemAssembler"]

import types

# UFL modules
import ufl

# Import SWIG-generated extension module (DOLFIN C++)
import dolfin.cpp as cpp

# Local imports
from dolfin.fem.form import *

def _create_dolfin_form(form,
                        form_compiler_parameters=None,
                        # TODO: Remove these after 1.5 release:
                        cell_domains=None,
                        exterior_facet_domains=None,
                        interior_facet_domains=None):

    # First check if we got a cpp.Form
    if isinstance(form, cpp.Form):
        # Then we just try to use that one
        dolfin_form = form

        if not hasattr(form, "_compiled_form"):
            raise TypeError("Expected a dolfin form to have a _compiled_form attribute.")

        # Warn that we don't use the arguments if we get any
        if form_compiler_parameters is not None:
            cpp.warning("Ignoring form_compiler_parameters with a dolfin Form!")
        if cell_domains is not None:
            cpp.warning("Ignoring cell_domains with a dolfin Form!")
        if exterior_facet_domains is not None:
            cpp.warning("Ignoring exterior_facet_domains with a dolfin Form!")
        if interior_facet_domains is not None:
            cpp.warning("Ignoring interior_facet_domains with a dolfin Form!")

    elif isinstance(form, ufl.Form):
        # Wrap form
        dolfin_form = Form(form,
                           form_compiler_parameters=form_compiler_parameters)
        # Attach domains to form
        if cell_domains is not None:
            if not isinstance(cell_domains, cpp.MeshFunctionSizet):
                raise TypeError("expected a 'MeshFunction' of 'Sizet' or 'None', for argument 'cell_domains'")
            dolfin_form.set_cell_domains(cell_domains)
        if interior_facet_domains is not None:
            if not isinstance(interior_facet_domains, cpp.MeshFunctionSizet):
                raise TypeError("expected a 'MeshFunction' of 'Sizet' or 'None', for argument 'interior_facet_domains'")
            dolfin_form.set_interior_facet_domains(interior_facet_domains)
        if exterior_facet_domains is not None:
            if not isinstance(exterior_facet_domains, cpp.MeshFunctionSizet):
                raise TypeError("expected a 'MeshFunction' of 'Sizet' or 'None', for argument 'exterior_facet_domains'")
            dolfin_form.set_exterior_facet_domains(exterior_facet_domains)
    else:
        raise TypeError("Invalid form type %s" % (type(form),))

    return dolfin_form

# JIT assembler
def assemble(form,
             tensor=None,
             bcs=None,
             form_compiler_parameters=None,
             add_values=False,
             finalize_tensor=True,
             keep_diagonal=False,
             backend=None,
             # TODO: Remove after 1.5 release:
             cell_domains=None,
             exterior_facet_domains=None,
             interior_facet_domains=None,
             ):
    """
    Assemble the given form and return the corresponding tensor.

    *Arguments*
        Depending on the input form, which may be a functional, linear
        form, bilinear form or higher rank form, a scalar value, a vector,
        a matrix or a higher rank tensor is returned.

    In the simplest case, no additional arguments are needed. However,
    additional arguments may and must in some cases be provided as
    outlined below.

    The ``form`` can be either a UFL Form or a DOLFIN Form.

    If the form defines integrals over different subdomains,
    :py:class:`MeshFunctions <dolfin.cpp.MeshFunction>` over the
    corresponding topological entities defining the subdomains can be
    provided.

    The implementation of the returned tensor is determined by the
    default linear algebra backend. This can be overridden by
    specifying a different backend.

    Each call to assemble() will create a new tensor. If the
    ``tensor`` argument is provided, this will be used instead.
    Sparsity pattern of ``tensor`` is reset iff ``tensor.empty()`` holds.

    If the ``keep_diagonal`` is set to True, assembler ensures that
    potential zeros on a matrix diagonal are kept in sparsity pattern
    so every diagonal entry can be changed in a future (for example
    by ident() or ident_zeros()).

    Specific form compiler parameters can be provided by the
    ``form_compiler_parameters`` argument. Form compiler parameters
    can also be controlled using the global parameters stored in
    parameters["form_compiler"].

    *Examples of usage*
        The standard stiffness matrix ``A`` and a load vector ``b``
        can be assembled as follows:

        .. code-block:: python

            A = assemble(inner(grad(u),grad(v))*dx)
            b = assemble(f*v*dx)

        It is possible to explicitly prescribe the domains over which
        integrals wll be evaluated using the arguments
        ``cell_domains``, ``exterior_facet_domains`` and
        ``interior_facet_domains``. For instance, using a mesh
        function marking parts of the boundary:

        .. code-block:: python

            # MeshFunction marking boundary parts
            boundary_parts = MeshFunction("size_t", mesh, mesh.topology().dim()-1)

            # Sample variational forms
            a = inner(grad(u), grad(v))*dx + p*u*v*ds(0)
            L = f*v*dx - g*v*ds(1) + p*q*v*ds(0)

            A = assemble(a, exterior_facet_domains=boundary_parts)
            b = assemble(L, exterior_facet_domains=boundary_parts)

        To ensure that the assembled matrix has the right type, one may use
        the ``tensor`` argument:

        .. code-block:: python

            A = PETScMatrix()
            assemble(a, tensor=A)

        The form ``a`` is now assembled into the PETScMatrix ``A``.

    """
    if (cell_domains is not None
        or exterior_facet_domains is not None
        or interior_facet_domains is not None):
        cpp.deprecation("Parameters *_domains of assemble",
                        "1.4.0", "1.6.0",
                        "Parameter *_domains of assemble"
                        " will be removed. Include this in the ufl form instead.")

    # Create dolfin Form object referencing all data needed by assembler
    dolfin_form = _create_dolfin_form(form, form_compiler_parameters,
                                      cell_domains, exterior_facet_domains, interior_facet_domains)

    # Create tensor
    tensor = _create_tensor(form, dolfin_form.rank(), backend, tensor)

    # Call C++ assemble function
    assembler = cpp.Assembler()
    assembler.add_values = add_values
    assembler.finalize_tensor = finalize_tensor
    assembler.keep_diagonal = keep_diagonal
    assembler.assemble(tensor, dolfin_form)

    # Convert to float for scalars
    if dolfin_form.rank() == 0:
        tensor = tensor.get_scalar_value()

    # Apply (possibly list of) boundary conditions
    bcs = _wrap_in_list(bcs, 'bcs', cpp.DirichletBC)
    if bcs:
        cpp.deprecation("Passing \"bcs\" to assemble", "1.3.0", "1.4.0",
                        "Apply DirichletBC manually after an assemble.")
    for bc in bcs:
        bc.apply(tensor)

    # Return value
    return tensor

# JIT system assembler
def assemble_system(A_form,
                    b_form,
                    bcs=None,
                    x0=None,
                    form_compiler_parameters=None,
                    add_values=False,
                    finalize_tensor=True,
                    keep_diagonal=False,
                    A_tensor=None,
                    b_tensor=None,
                    backend=None,
                    # TODO: Remove after 1.5 release:
                    cell_domains=None,
                    exterior_facet_domains=None,
                    interior_facet_domains=None):
    """
    Assemble form(s) and apply any given boundary conditions in a
    symmetric fashion and return tensor(s).

    The standard application of boundary conditions does not
    necessarily preserve the symmetry of the assembled matrix. In
    order to perserve symmetry in a system of equations with boundary
    conditions, one may use the alternative assemble_system instead of
    multiple calls to :py:func:`assemble
    <dolfin.fem.assembling.assemble>`.

    *Examples of usage*

       For instance, the statements

       .. code-block:: python

           A = assemble(a)
           b = assemble(L)
           bc.apply(A, b)

       can alternatively be carried out by

       .. code-block:: python

           A, b = assemble_system(a, L, bc)

       The statement above is valid even if ``bc`` is a list of
       :py:class:`DirichletBC <dolfin.fem.bcs.DirichletBC>`
       instances. For more info and options, see :py:func:`assemble
       <dolfin.fem.assembling.assemble>`.

    """
    if (cell_domains is not None
        or exterior_facet_domains is not None
        or interior_facet_domains is not None):
        cpp.deprecation("Parameters *_domains of assemble_system",
                        "1.4.0", "1.6.0",
                        "Parameters *_domains of assemble_system"
                        " will be removed. Include this in the ufl form instead.")

    # Create dolfin Form objects referencing all data needed by assembler
    A_dolfin_form = _create_dolfin_form(A_form, form_compiler_parameters,
                                        cell_domains, exterior_facet_domains, interior_facet_domains)
    b_dolfin_form = _create_dolfin_form(b_form, form_compiler_parameters,
                                        cell_domains, exterior_facet_domains, interior_facet_domains)

    # Create tensors
    A_tensor = _create_tensor(A_form, A_dolfin_form.rank(), backend, A_tensor)
    b_tensor = _create_tensor(b_form, b_dolfin_form.rank(), backend, b_tensor)

    # Check bcs
    bcs = _wrap_in_list(bcs, 'bcs', cpp.DirichletBC)

    # Call C++ assemble function
    assembler = cpp.SystemAssembler(A_dolfin_form, b_dolfin_form, bcs)
    assembler.add_values = add_values
    assembler.finalize_tensor = finalize_tensor
    assembler.keep_diagonal = keep_diagonal
    if x0 is not None:
        assembler.assemble(A_tensor, b_tensor, x0)
    else:
        assembler.assemble(A_tensor, b_tensor)

    return A_tensor, b_tensor

def _wrap_in_list(obj, name, types=type):
    if obj is None:
        lst = []
    elif hasattr(obj, '__iter__'):
        lst = list(obj)
    else:
        lst = [obj]
    for obj in lst:
        if not isinstance(obj, types):
            raise TypeError("expected a (list of) %s as '%s' argument" % (str(types),name))
    return lst

def _create_tensor(form, rank, backend, tensor):
    "Create tensor for form"

    # Check if tensor is supplied by user
    if tensor is not None:
        return tensor

    # Check backend argument
    if (not backend is None) and (not isinstance(backend, cpp.GenericLinearAlgebraFactory)):
        raise TypeError("Provide a GenericLinearAlgebraFactory as 'backend'")

    # Create tensor
    if rank == 0:
        tensor = cpp.Scalar()
    elif rank == 1:
        if backend: tensor = backend.create_vector()
        else:       tensor = cpp.Vector()
    elif rank == 2:
        if backend: tensor = backend.create_matrix()
        else:       tensor = cpp.Matrix()
    else:
        raise RuntimeError("Unable to create tensors of rank %d." % rank)

    return tensor

class SystemAssembler(cpp.SystemAssembler):
    __doc__ = cpp.SystemAssembler.__doc__
    def __init__(self, A_form, b_form,
                 bcs=None,
                 form_compiler_parameters=None,
                 # TODO: Remove after 1.5 release:
                 cell_domains=None,
                 exterior_facet_domains=None,
                 interior_facet_domains=None):
        """
        Create a SystemAssembler

        * Arguments *
           a (ufl.Form, _Form_)
              Bilinear form
           L (ufl.Form, _Form_)
              Linear form
           bcs (_DirichletBC_)
              A list or a single DirichletBC (optional)
        """
        if (cell_domains is not None
            or exterior_facet_domains is not None
            or interior_facet_domains is not None):
            cpp.deprecation("Parameters *_domains of SystemAssembler",
                            "1.4.0", "1.6.0",
                            "Parameters *_domains of SystemAssembler"
                            " will be removed. Include this in the ufl form instead.")
        # Create dolfin Form objects referencing all data needed by assembler
        A_dolfin_form = _create_dolfin_form(A_form, form_compiler_parameters,
                                            cell_domains, exterior_facet_domains, interior_facet_domains)
        b_dolfin_form = _create_dolfin_form(b_form, form_compiler_parameters,
                                            cell_domains, exterior_facet_domains, interior_facet_domains)

        # Check bcs
        bcs = _wrap_in_list(bcs, 'bcs', cpp.DirichletBC)

        # Call C++ assemble function
        cpp.SystemAssembler.__init__(self, A_dolfin_form, b_dolfin_form, bcs)
