/* $Id: session.c,v 1.1 2005/01/29 21:06:35 holger Exp $ */

/*
 * Copyright (c) 2004, 2005 Holger Weiss
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif				/* HAVE_CONFIG_H */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#if HAVE_STRINGS_H
#include <strings.h>
#endif				/* HAVE_STRINGS_H */

#include "imap.h"
#include "conf.h"
#include "report.h"
#include "system.h"

/* currently, we use this in free_folders() only */
#define NEXT_FOLDER                 \
  	do {                        \
  		mb_new = mb->next;  \
  		free(mb);           \
  		mb = mb_new;        \
  	} while (/* CONSTCOND */ 0)

extern uint32_t exists;
extern int      debug_level;
extern char    *deletemail;

static void     free_folders(mailbox *folder);

/* delete mail on IMAP server: flow control and error checking */
int
IMAPdelete(const conf *config)
{
	capabilities    capa;
	mailbox        *mb,
	               *mb_new;
	uint32_t        pre_del_exists;
	int             i,
	                num_deleted;
	bool            authenticated;
#if WITH_SSL
	bool            ssl_conn = false;
#endif				/* WITH_SSL */

	/* connect to server */
	report(LOG_DEBUG,
	       "connecting to %s, port %d",
	       config->host,
	       config->port);
#if WITH_SSL
	switch (i = imap_connect(config->host,
	                         config->port,
	                         (ssl_conn = (strncasecmp(config->ssl, "yes", 3) == 0) ? true : false)))
#else
	switch (i = imap_connect(config->host, config->port))
#endif				/* WITH_SSL */
	{
	    case IMAP_OK:
		authenticated = 0;
		report(LOG_DEBUG,
		       "connected to %s",
		       config->host);
		break;
	    case IMAP_PREAUTH:
		authenticated = true;
		report(LOG_DEBUG,
		       "preauthenticated at %s",
		       config->host);
		break;
	    case IMAP_TLS:
		/* FALLTHROUGH */
	    case IMAP_SEND:
		report(LOG_ERROR,
		       "error connecting to %s",
		       config->host);
		free_folders(config->folder);
		return -1;
		/* NOTREACHED */
	    case IMAP_PROTO:
		imap_logout();
		/* FALLTHROUGH */
	    default:
		report(LOG_ERROR,
		       "error connecting to %s [%s]",
		       config->host,
		       IMAP_ERROR(i));
		free_folders(config->folder);
		return -1;
	}

#if WITH_SSL
	for (;;) {
#endif				/* WITH_SSL */
		/* check server capabilities */
		report(LOG_DEBUG, "checking %s capabilities", config->host);
		switch (i = imap_capability(&capa)) {
		    case IMAP_OK:
			switch (capa.proto) {
			    case IMAP4rev1:
				report(LOG_DEBUG,
				       "%s is an IMAP4rev1 server",
				       config->host);
				break;
			    case IMAP4old:
				report(LOG_DEBUG,
				       "%s is an IMAP4 server",
				       config->host);
				break;
			    case IMAP4unknown:
				/* FALLTHROUGH */
			    default:
				report(LOG_ERROR,
				       "%s is not an IMAP4/IMAP4rev1 server",
				       config->host);
				free_folders(config->folder);
				imap_logout();
				return -1;
			}
			if ((!authenticated)
			    && capa.logindisabled
#if WITH_SSL
			    && (!((strncasecmp(config->ssl, "starttls", 8) == 0) && (!ssl_conn)))
#endif				/* WITH_SSL */
			) {
				/*
				 * RFC 3501, 6.2.3: "A client implementation
				 * MUST NOT send a LOGIN command if the
				 * LOGINDISABLED capability is advertised."
				 */
				report(LOG_ERROR,
				       "%s has disabled LOGIN using the LOGINDISABLED capability",
				       config->host);
				free_folders(config->folder);
				imap_logout();
				return -1;
			}
			break;
		    case IMAP_NO:
			report(LOG_ERROR,
			       "hecking %s capabilities failed",
			       config->host);
			free_folders(config->folder);
			imap_logout();
			return -1;
			/* NOTREACHED */
		    case IMAP_PROTO:
			imap_logout();
			/* FALLTHROUGH */
		    default:
			report(LOG_ERROR,
			       "error checking %s capabilities [%s]",
			       config->host,
			       IMAP_ERROR(i));
			free_folders(config->folder);
			return -1;
		}

#if WITH_SSL
		/*
		 * RFC 3501, 6.2.1: "Once [TLS] has been started, the client
		 * MUST discard cached information about server capabilities
		 * and SHOULD re-issue the CAPABILITY command.  This is
		 * necessary to protect against man-in-the-middle attacks
		 * which alter the capabilities list prior to STARTTLS.  The
		 * server MAY advertise different capabilities after
		 * STARTTLS."
		 */
		if (ssl_conn)	/* (re)checked CAPABILITY via SSL/TLS */
			break;

		if (strncasecmp(config->ssl, "starttls", 8) == 0) {
			if (!capa.starttls) {
				imap_logout();
				report(LOG_ERROR,
				       "%s does not support STARTTLS",
				       config->host);
				free_folders(config->folder);
				return -1;
			}
			report(LOG_DEBUG, "requesting STARTTLS at %s", config->host);
			switch (i = imap_starttls(config->host)) {
			    case IMAP_OK:
				report(LOG_DEBUG,
				       "%s accepted STARTTLS",
				       config->host);
				ssl_conn = true;
				break;
			    case IMAP_NO:
				report(LOG_ERROR,
				       "STARTTLS at %s failed",
				       config->host);
				free_folders(config->folder);
				imap_logout();
				return -1;
				/* NOTREACHED */
			    case IMAP_TLS:
				/* FALLTHROUGH */
			    case IMAP_PROTO:
				imap_logout();
				/* FALLTHROUGH */
			    default:
				report(LOG_ERROR,
				       "error requesting STARTTLS at %s [%s]",
				       config->host,
				       IMAP_ERROR(i));
				free_folders(config->folder);
				return -1;
			}
		} else
			break;
	}
#endif				/* WITH_SSL */

	/* authenticate if needed */
	if (!authenticated) {
		report(LOG_DEBUG,
		       "authenticating %s at %s",
		       config->user,
		       config->host);
		switch (i = imap_login(config->user, config->pass)) {
		    case IMAP_OK:
			report(LOG_DEBUG,
			       "%s at %s authenticated",
			       config->user,
			       config->host);
			break;
		    case IMAP_NO:
			report(LOG_ERROR,
			       "authentication for %s at %s failed",
			       config->user,
			       config->host);
			free_folders(config->folder);
			imap_logout();
			return -1;
			/* NOTREACHED */
		    case IMAP_PROTO:
			imap_logout();
			/* FALLTHROUGH */
		    default:
			report(LOG_ERROR,
			       "error authenticating %s at %s [%s]",
			       config->user,
			       config->host,
			       IMAP_ERROR(i));
			free_folders(config->folder);
			return -1;
		}
	}

	for (mb = config->folder;
	     mb != NULL;
	     mb_new = mb->next, free(mb), mb = mb_new) {
		uint32_t       *message_UIDs;
		char            search[32];
		char           *date_str,
		               *response = NULL;

		/* select mailbox */
		report(LOG_DEBUG,
		       "selecting folder %s on %s",
		       mb->box,
		       config->host);
		switch (i = imap_select(mb->box)) {
		    case IMAP_OK:
			report(LOG_DEBUG,
			       "selected folder %s on %s",
			       mb->box,
			       config->host);
			break;
		    case IMAP_RO:
			report(LOG_ERROR,
			       "folder %s on %s is marked READ-ONLY",
			       mb->box,
			       config->host);
			continue;
			/* NOTREACHED */
		    case IMAP_NO:
			report(LOG_ERROR,
			       "selecting folder %s on %s failed",
			       mb->box,
			       config->host);
			continue;
			/* NOTREACHED */
		    case IMAP_PROTO:
			imap_logout();
			/* FALLTHROUGH */
		    default:
			report(LOG_ERROR,
			       "error selecting folder %s on %s [%s]",
			       mb->box,
			       config->host,
			       IMAP_ERROR(i));
			free_folders(mb);
			return -1;
		}

		/* search for messages to delete */
		date_str = mk_date(config->days);

		if (config->unseen) {
			report(LOG_DEBUG,
			       "deleting all messages older than %s in %s on %s",
			       date_str,
			       mb->box,
			       config->host);
#if HAVE_STRLCPY
			strlcpy(search, "BEFORE ", sizeof(search));
#else
			strncpy(search, "BEFORE ", sizeof(search) - 1);
			search[sizeof(search) - 1] = '\0';
#endif				/* HAVE_STRLCPY */
		} else {
			report(LOG_DEBUG,
			       "deleting seen messages older than %s in %s on %s",
			       date_str,
			       mb->box,
			       config->host);
#if HAVE_STRLCPY
			strlcpy(search, "SEEN BEFORE ", sizeof(search));
#else
			strncpy(search, "SEEN BEFORE ", sizeof(search) - 1);
			search[sizeof(search) - 1] = '\0';
#endif				/* HAVE_STRLCPY */
		}

#if HAVE_STRLCAT
		strlcat(search, date_str, sizeof(search));
#else
		strncat(search, date_str, sizeof(search) - 1 - strlen(search));
#endif				/* HAVE_STRLCAT */
		report(LOG_DEBUG,
		       "searching for messages to delete in %s on %s",
		       mb->box,
		       config->host);
		switch (i = imap_search(&response, search, config->use_uid)) {
		    case IMAP_OK:
			report(LOG_DEBUG,
			       "searched for messages to delete in %s on %s",
			       mb->box,
			       config->host);
			break;
		    case IMAP_NO:
			/* a failed SEARCH is strange, bail out */
			report(LOG_ERROR,
			       "search for messages to delete in %s on %s failed",
			       mb->box,
			       config->host);
			free(response);
			free_folders(mb);
			imap_logout();
			return -1;
			/* NOTREACHED */
		    case IMAP_PROTO:
			imap_logout();
			/* FALLTHROUGH */
		    default:
			report(LOG_ERROR,
			       "error searching for messages to delete in %s on %s [%s]",
			       mb->box,
			       config->host,
			       IMAP_ERROR(i));
			if (response != NULL)
				free(response);
			free_folders(mb);
			return -1;
		}

		/* parse search response */
		report(LOG_DEBUG,
		       "parsing IMAP SEARCH response for folder %s on %s",
		       mb->box,
		       config->host);
		if ((message_UIDs = parse_search_response(response)) == NULL) {
			report(LOG_ERROR,
			       "error parsing IMAP SEARCH response for folder %s on %s",
			       mb->box,
			       config->host);
			free(response);
			free_folders(mb);
			imap_logout();
			return -1;
		}
		report(LOG_DEBUG,
		       "parsed IMAP SEARCH response for folder %s on %s",
		       mb->box,
		       config->host);
		free(response);

		pre_del_exists = exists;

		if (*message_UIDs == 0) {
			report(LOG_DEBUG,
			       "no messages to delete in %s on %s",
			       mb->box,
			       config->host);
			num_deleted = 0;
			free(message_UIDs);
		} else {
			/* set "\Deleted" flag on messages */
			report(LOG_DEBUG,
			       "storing \\Deleted flag for messages in %s on %s",
			       mb->box,
			       config->host);
			if ((i = imap_store(message_UIDs, "\\Deleted", config->use_uid)) >= 0) {
				report(LOG_DEBUG,
				       "stored \\Deleted flag for %d %s in %s on %s",
				       i,
				       (i == 1) ? "message" : "messages",
				       mb->box,
				       config->host);
				free(message_UIDs);
				num_deleted = i;
			} else if (i == IMAP_NO) {
				/* print error and try next folder */
				report(LOG_ERROR,
				       "storing \\Deleted flag for messages in %s on %s failed",
				       mb->box,
				       config->host);
				report(LOG_ERROR, "won't expunge messages in %s on %s",
				       mb->box,
				       config->host);
				free(message_UIDs);
				continue;
			} else {
				if (i == IMAP_PROTO)
					imap_logout();

				report(LOG_ERROR,
				       "error storing \\Deleted flag for messages in %s on %s [%s]",
				       mb->box,
				       config->host,
				       IMAP_ERROR(i));
				report(LOG_ERROR, "won't expunge messages in %s on %s",
				       mb->box,
				       config->host);
				free(message_UIDs);
				free_folders(mb);
				return -1;
			}

			if (config->expunge) {
				/* expunge messages */
				report(LOG_DEBUG, "closing %s on %s", mb->box, config->host);
				switch (i = imap_close()) {
				    case IMAP_OK:
					report(LOG_DEBUG,
					       "closed %s on %s, expunged deleted messages",
					       mb->box,
					       config->host);
					break;
				    case IMAP_NO:
					/* this is strange, bail out */
					report(LOG_ERROR,
					       "closing %s on %s failed, messages not expunged",
					       mb->box,
					       config->host);
					free_folders(mb);
					imap_logout();
					return -1;
					/* NOTREACHED */
				    case IMAP_PROTO:
					imap_logout();
					/* FALLTHROUGH */
				    default:
					report(LOG_ERROR,
					       "error closing %s on %s [%s]",
					       mb->box,
					       config->host,
					       IMAP_ERROR(i));
					free_folders(mb);
					return -1;
				}
			}
		}

		if (debug_level >= LOG_DEFAULT)
			printf("deleted %d of %d %s for %s in %s on %s\n",
			       num_deleted,
			       pre_del_exists,
			       (num_deleted == 1) ? "message" : "messages",
			       config->user,
			       mb->box,
			       config->host);
	}

	report(LOG_DEBUG, "logging out of %s", config->host);
	imap_logout();
	return 0;
}

/* free(3) linked conf->folder list */
static void
free_folders(mailbox *mb)
{
	mailbox        *mb_new;

	while (mb != NULL)
		NEXT_FOLDER;
}
