/* $Id: imap.c,v 1.77 2006/04/09 22:20:38 holger Exp $ */

/*
 * Copyright (c) 2004, 2005, 2006 Holger Weiss
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif				/* HAVE_CONFIG_H */

#if HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif				/* HAVE_SYS_TYPES_H */
#include <ctype.h>
#if HAVE_LOCALE_H
#include <locale.h>
#endif				/* HAVE_LOCALE_H */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#if HAVE_STRINGS_H
#include <strings.h>
#endif				/* HAVE_STRINGS_H */
#include <time.h>

#include "imap.h"
#include "conf.h"
#include "report.h"
#include "sock.h"
#include "system.h"
#include "text.h"

#define CMD_BUFSIZE	 	 256
#define SOCK_BUFSIZE		1024

/* IMAP states */
#define NOT_AUTHENTICATED	0
#define AUTHENTICATED		1
#define SELECTED		2
#define LOGOUT			3

/* IMAP command tag */
#define TAGGED			true
#define UNTAGGED		false

_imap_error_mapping _imap_error_table[] = {
	{ IMAP_TLS,     "IMAP_TLS"     },
	{ IMAP_SEND,    "IMAP_SEND"    },
	{ IMAP_FATAL,   "IMAP_FATAL"   },
	{ IMAP_PROTO,   "IMAP_PROTO"   },
	{ IMAP_NO,      "IMAP_NO"      },
	{ IMAP_RO,      "IMAP_RO"      },
	{ IMAP_PREAUTH, "IMAP_PREAUTH" },
	{ IMAP_OK,      "IMAP_OK"      },
	{ IMAP_UNKNOWN, "IMAP_UNKNOWN" },
	{ 0,             NULL          }
};

uint32_t        exists = 0;	/* total number of messages */

#if WITH_SSL
static bool     ssl_conn;	/* are we using SSL? */
#endif				/* WITH_SSL */
static short    server_proto;	/* server protocol */
static short    state;		/* IMAP state */
static char     tag[6];		/* IMAP command tag */

static int      write_imap(const char *command, const char *pass);
static int      read_imap(char **response, bool tagged);
static bool     tag_cmd(void);
static char    *quotify(const char *buf, bool quote_amper);

/* connect to IMAP server */
int
#if WITH_SSL
imap_connect(const char *server, int port, bool use_ssl)
#else
imap_connect(const char *server, int port)
#endif				/* WITH_SSL */
{
	int i;

	if (sock_open(server, port) == -1) {
		sock_close();
		return IMAP_SEND;
	}
#if WITH_SSL
	if ((ssl_conn = use_ssl) && (sock_ssl_open(server, SSLv23) != 0)) {
		sock_close();
		return IMAP_TLS;
	}
#endif				/* WITH_SSL */
	if ((i = read_imap(NULL, UNTAGGED)) == IMAP_PREAUTH)
		state = AUTHENTICATED;
	else
		state = NOT_AUTHENTICATED;

	return i;
}

/* check server protocol */
int
imap_capability(capabilities *capa)
{
	int             i;
	char            command[] = "CAPABILITY";
	char           *response = NULL;

	if (write_imap(command, NULL) <= 0)
		return IMAP_SEND;
	if ((i = read_imap(&response, TAGGED)) != IMAP_OK) {
		if (response != NULL)
			free(response);
		return i;
	}

	/* read_imap(&response) returned IMAP_OK, so response is not NULL */
	if (strcasestr(response, "IMAP4rev1") != NULL)
		server_proto = IMAP4rev1;
	else if (strcasestr(response, "IMAP4") != NULL)
		server_proto = IMAP4old;
	else
		server_proto = IMAP4unknown;

	if (strcasestr(response, "IMAP4rev1") != NULL)
		capa->proto = IMAP4rev1;
	else if (strcasestr(response, "IMAP4") != NULL)
		capa->proto = IMAP4old;
	else
		capa->proto = IMAP4unknown;

	if (strcasestr(response, "STARTTLS") != NULL)
		capa->starttls = true;
	else
		capa->starttls = false;

	if (strcasestr(response, "LOGINDISABLED") != NULL)
		capa->logindisabled = true;
	else
		capa->logindisabled = false;

	server_proto = capa->proto;	/* save proto static */
	free(response);

	return i;
} 

#if WITH_SSL
/* STARTTLS (see RFC 2595 and RFC 3501, 6.2.1) */
int
imap_starttls(const char *server)
{
	int             i;
	char            command[] = "STARTTLS";

	if (write_imap(command, NULL) <= 0)
		return IMAP_SEND;

	if ((i = read_imap(NULL, TAGGED)) != IMAP_OK)
		return i;

	if (sock_ssl_open(server, TLSv1) != 0)
		return IMAP_TLS;

	return IMAP_OK;
}
#endif				/* WITH_SSL */

/* login to IMAP server */
int
imap_login(const char *user, const char *pass)
{
	int             i;
	char            command[CMD_BUFSIZE];

#if HAVE_STRLCPY
	strlcpy(command, "LOGIN ", sizeof(command));
#else
	strncpy(command, "LOGIN ", sizeof(command) - 1);
	command[sizeof(command) - 1] = '\0';
#endif				/* HAVE_STRLCPY */
#if HAVE_STRLCAT
	strlcat(command, quotify(user, false), sizeof(command));
#else
	strncat(command, quotify(user, false), sizeof(command) - 1 - strlen(command));
#endif				/* HAVE_STRLCAT */

	if (write_imap(command, quotify(pass, false)) <= 0)
		return IMAP_SEND;

	if ((i = read_imap(NULL, TAGGED)) == IMAP_OK)
		state = AUTHENTICATED;

	return i;
}

/* select mailbox folder */
int
imap_select(const char *folder)
{
	int             i;
	char            command[CMD_BUFSIZE];

#if HAVE_STRLCPY
	strlcpy(command, "SELECT ", sizeof(command));
#else
	strncpy(command, "SELECT ", sizeof(command) - 1);
	command[sizeof(command) - 1] = '\0';
#endif				/* HAVE_STRLCPY */
#if HAVE_STRLCAT
	strlcat(command, quotify(folder, true), sizeof(command));
#else
	strncat(command, quotify(folder, true), sizeof(command) - 1 - strlen(command));
#endif				/* HAVE_STRLCAT */

	if (write_imap(command, NULL) <= 0)
		return IMAP_SEND;

	if ((i = read_imap(NULL, TAGGED)) == IMAP_OK)
		state = SELECTED;

	return i;
}

/* message [UID] SEARCH */
int
imap_search(char **response, const char *search, bool uid_search)
{
	char            command[CMD_BUFSIZE];

	if (uid_search) {
#if HAVE_STRLCPY
		strlcpy(command, "UID SEARCH ", sizeof(command));
#else
		strncpy(command, "UID SEARCH ", sizeof(command) - 1);
		command[sizeof(command) - 1] = '\0';
#endif				/* HAVE_STRLCPY */
	} else {
#if HAVE_STRLCPY
		strlcpy(command, "SEARCH ", sizeof(command));
#else
		strncpy(command, "SEARCH ", sizeof(command) - 1);
		command[sizeof(command) - 1] = '\0';
#endif				/* HAVE_STRLCPY */
	}
#if HAVE_STRLCAT
	strlcat(command, search, sizeof(command));
#else
	strncat(command, search, sizeof(command) - 1 - strlen(command));
#endif				/* HAVE_STRLCAT */

	if (write_imap(command, NULL) <= 0)
		return IMAP_SEND;

	return read_imap(response, true);
}

/* store flags for each message via [UID] STORE */
int
imap_store(const uint32_t *message, const char *flags, bool uid_store)
{
	int             i = 0;
	char            command[CMD_BUFSIZE];
	char            flags_cmd[14];

	if (server_proto == IMAP4rev1) {

		/*
		 * In order to minimize network throughput, we use ".SILENT"
		 * if we're talking to an IMAP4rev1 server.
		 */
#if HAVE_STRLCPY
		strlcpy(flags_cmd, "+FLAGS.SILENT", sizeof(flags_cmd));
#else
		strncpy(flags_cmd, "+FLAGS.SILENT", sizeof(flags_cmd) - 1);
		flags_cmd[sizeof(flags_cmd) - 1] = '\0';
#endif				/* HAVE_STRLCPY */
	} else {
#if HAVE_STRLCPY
		strlcpy(flags_cmd, "+FLAGS", sizeof(flags_cmd));
#else
		strncpy(flags_cmd, "+FLAGS", sizeof(flags_cmd) - 1);
		flags_cmd[sizeof(flags_cmd) - 1] = '\0';
#endif				/* HAVE_STRLCPY */
	}

	while (*message) {
		int             value,
		                range = 0;

		/*
		 * In order to minimize network throughput, we put contiguous
		 * UIDs together so that only one UID STORE command has to be
		 * sent to the server for each contiguous block.
		 */
		if ((*(message + 1) - *message) == 1) {
			do {
				range++;
			} while ((*(message + range + 1) - *(message + range)) == 1);
			if (uid_store)
				snprintf(command,
					 sizeof(command),
					 "UID STORE %d:%d %s (%s)",
					 *message,
					 *(message + range),
					 flags_cmd,
					 flags);
			else
				snprintf(command,
					 sizeof(command),
					 "STORE %d:%d %s (%s)",
					 *message,
					 *(message + range),
					 flags_cmd,
					 flags);
		} else {	/* next UID != current UID + 1 */
			if (uid_store)
				snprintf(command,
					 sizeof(command),
					 "UID STORE %d %s (%s)",
					 *(message++),
					 flags_cmd,
					 flags);
			else
				snprintf(command,
					 sizeof(command),
					 "STORE %d %s (%s)",
					 *(message++),
					 flags_cmd,
					 flags);
		}
		if (write_imap(command, NULL) <= 0)
			return IMAP_SEND;
		if ((value = read_imap(NULL, TAGGED)) != IMAP_OK)
			return value;
		range++;
		i += range;
		message += range;
	}

	return i;
}

/* close folder, this expunges deleted messages */
int
imap_close(void)
{
	char            command[] = "CLOSE";

	if (write_imap(command, NULL) <= 0)
		return IMAP_SEND;

	return read_imap(NULL, TAGGED);
}

/* logout from IMAP server */
void
imap_logout(void)
{
	char            command[] = "LOGOUT";

	state = LOGOUT;
	if (write_imap(command, NULL) > 0)
		if (read_imap(NULL, TAGGED) == IMAP_FATAL)
			return;	/* socket closed already */

	sock_close();
}

/* parse IMAP SEARCH response, return 0 terminated uint32_t array */
uint32_t       *
parse_search_response(const char *response)
{
#define FLAG_BUFSIZE		1024
	int             i = 0,
	                bufsize = sizeof(int) * FLAG_BUFSIZE;
	uint32_t       *buf,
	               *new_buf,
	               *ip;
	const char     *p;

	report(LOG_DEBUG, "allocating %d byte for search response buffer", bufsize);
	if ((ip = buf = malloc(bufsize)) == NULL) {
		report(LOG_PERROR, "can't allocate memory for search response buffer");
		return (uint32_t *) NULL;
	}
	p = response;
	while ((p = strcasestr(p, "* SEARCH")) != NULL) {
		p += 8;		/* first char after "* SEARCH" */
		while (*p) {
			if (i >= FLAG_BUFSIZE) {	/* buffer is full */
				int             pos = ip - buf;	/* save current position */

				bufsize += sizeof(int) * FLAG_BUFSIZE;
				report(LOG_DEBUG, "increasing search response buffer to %d byte", bufsize);
				if ((new_buf = realloc(buf, bufsize)) == NULL) {
					report(LOG_PERROR, "can't reallocate memory for search response buffer");
					free(buf);
					return (uint32_t *) NULL;
				}
				buf = new_buf;
				ip = buf + pos;
				i = 0;
			}
			while ((*p == ' ') || (*p == '\t'))
				p++;	/* skip whitespace */
			if ((*p >= '0') && (*p <= '9')) {
				*(ip++) = (uint32_t) strtol(p, (char **) NULL, 10);
				while ((*(++p) >= '0') && (*p <= '9'))
					 /* rest of message number */ ;
				i++;
			} else
				break;
		}
	}
	*ip = 0;

	/*
	 * RFC 2060, 6.4.4 states that an untagged SEARCH response is
	 * REQUIRED by the server if a SEARCH command was issued by the
	 * client.  However, at least imap.web.de doesn't send an untagged
	 * SEARCH response if no messages matching the criteria were found.
	 * Hence, we don't return NULL in that case.
	 */
	return buf;
}

/* buffer containing date in IMAP format, n days ago */
char           *
mk_date(int days)
{
	static char     buf[12];
	time_t          now = time(NULL);
	time_t          then = now - (days * 24 * 60 * 60);

#if HAVE_SETLOCALE
	setlocale(LC_TIME, "C");
#endif				/* HAVE_SETLOCALE */

	/*
	 * We don't want leading zeros (see the date_day format specification
	 * in RFC 2060, 9.).  On most systems, we could format the date using
	 * "%e-%b-%Y".  However, at least NeXTStep doesn't support "%e", so
	 * we'll use "%d" and strip a leading zero ourselves.
	 */
	strftime(buf, sizeof(buf), "%d-%b-%Y", localtime(&then));

	return (*buf == '0') ? buf + 1 : buf;
}

/* needed for IMAP_ERROR() macro */
unsigned short
_lookup_imap_error(int err_code)
{
	unsigned short i;

	for (i = 0; _imap_error_table[i].str != NULL; i++)
		if (_imap_error_table[i].code == err_code)
			break;

	return (_imap_error_table[i].str == NULL) ? i - 1 : i;
}

/*
 * All commands sent to the IMAP server are piped through write_imap().  Here
 * we add the command tag and the terminating CRLF; the new buffer is then
 * handed over to our send(3) wrapper.  We return the wrappers return value.
 */
static int
write_imap(const char *command, const char *pass)
{
	/* bufsize = tag + space + command + "\r\n\0" */
	int             i;
	size_t          bufsize = strlen(command) + 9;
	char           *buf,
	               *new_buf;

	if ((buf = malloc(bufsize)) == NULL) {
		report(LOG_PERROR, "can't allocate memory for IMAP write buffer");
		return -1;
	}
	if (!tag_cmd()) {
		report(LOG_DEBUG, "sent too many commands");
		free(buf);
		return -1;
	}
	snprintf(buf, bufsize, "%s %s", tag, command);

	if (pass != NULL) {
		int             passlen = strlen(pass);

#if WITH_SSL
		report(LOG_INFO, "%s> %s \"%s\"", (ssl_conn) ? "IMAP/SSL" : "IMAP", buf, stars(passlen - 2));
#else
		report(LOG_INFO, "%s> %s \"%s\"", "IMAP", buf, stars(passlen - 2));
#endif				/* WITH_SSL */
		if ((new_buf = realloc(buf, bufsize += passlen + 1)) == NULL) {
			report(LOG_PERROR, "can't reallocate memory for IMAP write buffer");
			free(buf);
			return -1;
		}
		buf = new_buf;
#if HAVE_STRLCAT
		strlcat(buf, " ", bufsize);
		strlcat(buf, pass, bufsize);
#else
		strncat(buf, " ", bufsize - 1 - strlen(buf));
		strncat(buf, pass, bufsize - 1 - strlen(buf));
#endif				/* HAVE_STRLCAT */
	} else
#if WITH_SSL
		report(LOG_INFO, "%s> %s", (ssl_conn) ? "IMAP/SSL" : "IMAP", buf);
#else
		report(LOG_INFO, "%s> %s", "IMAP", buf);
#endif				/* WITH_SSL */

#if HAVE_STRLCAT
	strlcat(buf, "\r\n", bufsize);
#else
	strncat(buf, "\r\n", bufsize - 1 - strlen(buf));
#endif				/* HAVE_STRLCAT */

	i = sock_write(buf, bufsize - 1);	/* don't send the '\0' */
	free(buf);
	if (i <= 0)
		sock_close();

	return i;
}

/*
 * All messages from the IMAP server are received via read_imap().  The
 * response is parsed in order to return a value indicating (the type of)
 * success or failure appropriately to the caller.  Apart from that, if
 * **response is not NULL, we'll point *response to the complete server
 * response, so that further parsing can be done by the caller.
 */
static int
read_imap(char **response, bool tagged)
{
#define FREE_BUFFERS_CLOSE_SOCKET                                                     \
  	do {                                                                          \
  		if (buf != NULL)                                                      \
  			free(buf);                                                    \
  			buf = NULL;                                                   \
  		if (out_buf != NULL)                                                  \
  			free(out_buf);                                                \
  		sock_close();                                                         \
  	} while (/* CONSTCOND */ 0)

	size_t          bufsize = SOCK_BUFSIZE,
	                received = 0;
	int             i,
	                value;
	char           *out_buf = NULL,
	               *new_buf,
	               *buf,
	               *tag_p,
	               *p;
	bool            return_fatal = false;

	report(LOG_DEBUG, "allocating %lu byte for IMAP read buffer", (unsigned long) bufsize);
	if ((buf = malloc(bufsize)) == NULL) {
		report(LOG_PERROR, "can't allocate memory for IMAP read buffer");
		sock_close();
		return IMAP_FATAL;
	}

	/* data receive loop */
	for (;;) {
		int j;

		if ((i = sock_read(buf + received, SOCK_BUFSIZE - 1)) <= 0)
			break;	/* socket error or server closed connection */

		/*
		 * We received data.  Now, the question is whether or not we
		 * received _all_ data.  We'll stop waiting for more data only
		 * if all of the following conditions apply: (1) Currently,
		 * there is no more data on the socket.  (2) The data is CRLF
		 * terminated.  (3) If we're waiting for a tagged response:
		 * The buffer contains our command tag.
		 */
		received += i;
		buf[received] = '\0';
		report(LOG_DEBUG, "%lu byte of data in IMAP read buffer", (unsigned long) received);

		/* check for more data */
		if ((j = sock_peek()) == -1) {
			i = j;
			break;
		} else if (j) {
#if WITH_SSL
			if (ssl_conn)
				report(LOG_DEBUG, "more data on socket and/or in SSL record buffer");
			else
#endif				/* WITH_SSL */
				report(LOG_DEBUG, "more data on socket");
		} else {	/* j == 0 */
#if WITH_SSL
			if (ssl_conn)
				report(LOG_DEBUG, "no more data on socket nor in SSL record buffer");
			else
#endif				/* WITH_SSL */
				report(LOG_DEBUG, "no more data on socket");
			if (!crlf_terminated(buf, received))
				report(LOG_DEBUG, "packet not CRLF terminated, waiting for more data");
			else if (tagged && (strstr(buf, tag) == NULL))
				report(LOG_DEBUG, "command tag %s not found, waiting for more data", tag);
			else {
				report(LOG_DEBUG, "packet seems to be complete");
				break;	/* done, yeah */
			}
		}

		/*
		 * In the next turn, we'll reveive up to SOCK_BUFSIZE - 1
		 * byte; OTOH, we still have bufsize - received - 1 bytes of
		 * memory available.
		 */
		if ((new_buf = realloc(buf,
				       bufsize
				       += SOCK_BUFSIZE - 1
				       - (bufsize - received - 1))) == NULL) {
			report(LOG_PERROR, "can't reallocate memory for IMAP read buffer");
			FREE_BUFFERS_CLOSE_SOCKET;
			return IMAP_FATAL;
		}
		/*
		 * We allow bad servers to eat up all our memory.  Of course,
		 * we could hardcode a maximum bufsize, but I suggest simply
		 * not to connect to bad servers.
		 */
		report(LOG_DEBUG, "increased IMAP read buffer to %lu byte", (unsigned long) bufsize);
		buf = new_buf;
	}

	/* okay, now let's see what we've got */
	if (!received) {
		/* we received nothing */
		if (i <= -1)	/* user was notified by sock.c already */
			report(LOG_DEBUG, "connection error");
		else if (!i) {	/* server closed connection */
			if (state == LOGOUT)
				report(LOG_DEBUG, "connection reset by peer");
			else
				report(LOG_ERROR, "connection reset by peer");
		}

		FREE_BUFFERS_CLOSE_SOCKET;
		return IMAP_FATAL;
	} else {
		/* we received something */
		if (i <= -1) {
			report(LOG_DEBUG, "connection error");
			return_fatal = true;
		} else if (!i) {
			if (state == LOGOUT) {
				/*
				 * Sent LOGOUT, got response and then EOF, ok.
				 * Well, if the response doesn't include our
				 * command tag, it's not ok, but we'll check
				 * that below.
				 */
				report(LOG_DEBUG, "connection reset by peer");
			} else {
				report(LOG_ERROR, "connection reset by peer");
				return_fatal = true;
			}
		}
	}

	/* set callers pointer to server response */
	if (response != NULL)
		*response = buf;

	bufsize = strlen(buf) + 1;
	report(LOG_DEBUG, "allocating %lu byte for IMAP debug output buffer", (unsigned long) bufsize);
	if ((out_buf = malloc(bufsize)) == NULL) {
		report(LOG_PERROR, "can't allocate memory for IMAP debug output buffer");
		FREE_BUFFERS_CLOSE_SOCKET;
		return IMAP_FATAL;
	}

	/* print buffer line by line */
	for (p = buf;
	     (i = extract_line(out_buf, p, bufsize));
	     /* skip previous line and CRLF */
	     p += i, p = (*p) ? p + 1 : p, p = (*p) ? p + 1 : p)
#if WITH_SSL
		report(LOG_INFO, "%s< %s", (ssl_conn) ? "IMAP/SSL" : "IMAP", out_buf);
#else
		report(LOG_INFO, "%s< %s", "IMAP", out_buf);
#endif				/* WITH_SSL */

	/* special cases: BYE (RFC 2060, 7.1.5) and ALERT (RFC 2060, 7.1) */
	if ((state != LOGOUT) && ((p = strcasestr(buf, "* BYE")) != NULL)) {
		if (extract_line(out_buf, p + 6, bufsize))
			report(LOG_ERROR, "server closed IMAP connection: %s", out_buf);
		else
			report(LOG_ERROR, "server closed IMAP connection");
		FREE_BUFFERS_CLOSE_SOCKET;
		return IMAP_FATAL;
	}
	p = buf;
	while ((p = strcasestr(p, "[ALERT]")) != NULL)	/* RFC 2060, 7.1 */
		if (extract_line(out_buf, p += 8, bufsize))
			report(LOG_ERROR, "server sent ALERT: %s", out_buf);

	/*
	 * RFC 3501, 7.3.1: "The update from the EXISTS response MUST be
	 * recorded by the client."
	 */
	if ((p = strcasestr(buf, "EXISTS")) != NULL) {
		while ((p > buf) && (*p != '*'))
			p--;
		if (*p == '*')
			exists = (uint32_t) strtol(p + 1, (char **) NULL, 10);
	}

	tag_p = (tagged) ? strstr(buf, tag) : NULL;

	if (tag_p == NULL) {
		/* response does not contain current command tag */
		if (strcasestr(buf, "* OK") != NULL) {
			report(LOG_DEBUG, "received untagged IMAP OK");
			value = (tagged) ? IMAP_PROTO : IMAP_OK;
		} else if (strcasestr(buf, "* PREAUTH") != NULL) {
			report(LOG_DEBUG, "reveived untagged IMAP PREAUTH");
			value = (tagged) ? IMAP_PROTO : IMAP_PREAUTH;
		} else if ((p = strcasestr(buf, "BAD")) != NULL) {
			if (extract_line(out_buf, p + 4, bufsize))
				report(LOG_ERROR, "IMAP protocol error, server said: %s", out_buf);
			else
				report(LOG_ERROR, "IMAP protocol error");
			value = IMAP_PROTO;
		} else {
			report(LOG_ERROR, "unknown IMAP protocol error");
			value = IMAP_PROTO;
		}
	} else {
		/* response does contain current command tag */
		tag_p += sizeof(tag);	/* skip the tag */
		if (strncasecmp(tag_p, "OK", 2) == 0) {
			report(LOG_DEBUG, "received tagged IMAP OK");
			/* special case: [READ-ONLY] (RFC 2060, 6.3.1) */
			if (strcasestr(tag_p, "[READ-ONLY]")) {
				report(LOG_DEBUG, "received [READ-ONLY]");
				value = IMAP_RO;
			} else
				value = IMAP_OK;
		} else if (strncasecmp(tag_p, "NO", 2) == 0) {
			if (extract_line(out_buf, tag_p + 3, bufsize))
				report(LOG_ERROR, "server said NO: %s", out_buf);
			else
				report(LOG_ERROR, "received tagged IMAP NO");
			value = IMAP_NO;
		} else if (strncasecmp(tag_p, "BAD", 3) == 0) {
			if (extract_line(out_buf, tag_p + 4, bufsize))
				report(LOG_ERROR, "IMAP protocol error, server said: %s", out_buf);
			else
				report(LOG_ERROR, "IMAP protocol error");
			value = IMAP_PROTO;
		} else {
			report(LOG_ERROR, "unknown IMAP protocol error");
			value = IMAP_PROTO;
		}
	}

	/* buf must be free(3)d by caller if response != NULL */
	if (response == NULL)
		free(buf);

	free(out_buf);
	return (return_fatal) ? IMAP_FATAL : value;
}

/* generate unique IMAP command tag */
static bool
tag_cmd(void)
{
	static int      i = 0;

	if (++i > 9999)
		return false;
	snprintf(tag, sizeof(tag), "A%.4d", i);

	return true;
}

/* quoted string according to RFC 2060 */
static char    *
quotify(const char *buf, bool quote_amper)
{
	int             i = 0;
	static char     quoted[CONF_BUFSIZE];
	char           *p = quoted;

	*(p++) = '"';

	/*
	 * We have CONF_BUFSIZE byte available, minus the two quote signs and
	 * the null-termination.  If we must escape a character, we need two
	 * byte.  Hence, to be sure, we check CONF_BUFSIZE - 4.
	 */
	while ((*buf != '\0') && (i++ < (CONF_BUFSIZE - 4))) {
		if ((*buf == '"') || (*buf == '\\')) {
			*(p++) = '\\';
			i++;
		} else if (quote_amper && (*buf == '&')) {
			*(p++) = '&';
			*(p++) = '-';
			buf++, i++;
			continue;
		}
		*(p++) = *(buf++);
	}
	*(p++) = '"';
	*p = '\0';

	return quoted;
}
