/* $Id: deletemail.c,v 1.26 2005/01/29 21:06:34 holger Exp $ */

/*
 * Copyright (c) 2004, 2005 Holger Weiss
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif				/* HAVE_CONFIG_H */

#if HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif				/* HAVE_SYS_STAT_H */
#if HAVE_LIBGEN_H
#include <libgen.h>
#endif				/* HAVE_LIBGEN_H */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#if HAVE_UNISTD_H
#include <unistd.h>
#endif				/* HAVE_UNISTD_H */

#include "deletemail.h"
#include "session.h"
#include "conf.h"
#include "report.h"
#include "system.h"

typedef struct {
	bool            no_action;
	bool            unseen;
	bool            not_use_uid;
} cmdline;

int             debug_level = LOG_DEFAULT;
char           *deletemail;	/* basename(argv[0]) */

static char    *homedir(void);
static void     deletemail_version(void);
static void     deletemail_usage(int exit_value);

int
main(int argc, char **argv)
{
	extern int      optind;
	extern char    *optarg;

#if !STAT_MACROS_BROKEN
	struct stat     st;
#endif				/* not STAT_MACROS_BROKEN */
	cmdline         switches = {0, 0, 0};
	conf           *cnf,
	               *cnf_new;
	int             i;
	char            rcfile[128];
	char           *opt_file = NULL,
	               *rcdir;

#if HAVE_BASENAME
	deletemail = basename(argv[0]);
#else
	deletemail = argv[0];
#endif				/* HAVE_BASENAME */

	while ((i = getopt(argc, argv, "f:hnqsuvV")) != EOF)
		switch (i) {
		    case 'f':
			opt_file = optarg;
			break;
		    case 'n':
			switches.no_action = 1;
			break;
		    case 'q':
			debug_level = -1;
			break;
		    case 's':
			switches.not_use_uid = 1;
			break;
		    case 'u':
			switches.unseen = 1;
			break;
		    case 'v':
			debug_level++;
			break;
		    case 'V':
			deletemail_version();
			/* NOTREACHED */
		    case 'h':
			deletemail_usage(EXIT_SUCCESS);
			/* NOTREACHED */
		    case '?':
			/* FALLTHROUGH */
		    default:
			deletemail_usage(EXIT_FAILURE);
		}

	if ((argc - optind) != 0)
		deletemail_usage(EXIT_FAILURE);

	if (opt_file != NULL) {
		if (strlen(opt_file) > (sizeof(rcfile) - 1)) {
			report(LOG_ERROR, "path to deletemailrc too long");
			return -1;
		}
#if HAVE_STRLCPY
		strlcpy(rcfile, opt_file, sizeof(rcfile));
#else
		strncpy(rcfile, opt_file, sizeof(rcfile) - 1);
		rcfile[sizeof(rcfile) - 1] = '\0';
#endif				/* HAVE_STRLCPY */
	} else {
		rcdir = homedir();
		if ((strlen(rcdir) + 14) > (sizeof(rcfile) - 1)) {
			report(LOG_ERROR, "path to deletemailrc too long");
			return -1;
		}
#if HAVE_STRLCPY
		strlcpy(rcfile, rcdir, sizeof(rcfile));
#else
		strncpy(rcfile, rcdir, sizeof(rcfile) - 1);
		rcfile[sizeof(rcfile) - 1] = '\0';
#endif				/* HAVE_STRLCPY */
#if HAVE_STRLCAT
		strlcat(rcfile, "/.deletemailrc", sizeof(rcfile));
#else
		strncat(rcfile, "/.deletemailrc", sizeof(rcfile) - 1 - strlen(rcfile));
#endif				/* HAVE_STRLCAT */
	}

#if !STAT_MACROS_BROKEN
	if (stat(rcfile, &st) == -1) {
		report(LOG_PERROR, "can't stat(2) %s", rcfile);
		return -1;
	}
	if (st.st_mode & S_IROTH)
		report(LOG_DEFAULT, "configuration is WORLD READABLE, try 'chmod 600 %s'", rcfile);
	else if (st.st_mode & S_IRGRP)
		report(LOG_DEFAULT, "configuration is GROUP READABLE, try 'chmod 600 %s'", rcfile);
#endif				/* not STAT_MACROS_BROKEN */

	if ((cnf = readconf(rcfile)) == NULL)
		return -1;

	i = 0;
	do {
		if (switches.unseen)
			cnf->unseen = 1;
		if (switches.no_action)
			cnf->expunge = 0;
		if (switches.not_use_uid)
			cnf->use_uid = 0;
		if (IMAPdelete(cnf) != 0)
			i++;
		cnf_new = cnf->next;
		free(cnf);
	} while ((cnf = cnf_new) != NULL);

	return i;
}

static char    *
homedir(void)
{
	char           *home = getenv("HOME");

	if ((!home) || (*home != '/')) {
		report(LOG_ERROR, "HOME environment variable isn't set correctly");
		return "/";
	}
	return home;
}

static void
deletemail_version(void)
{

	puts(DELETEMAIL_STRING), exit(EXIT_SUCCESS);
}

static void
deletemail_usage(int exit_value)
{
	FILE           *stream;

	if (exit_value == 0)
		stream = stdout;
	else
		stream = stderr;

	fprintf(stream,
		"usage: %s [-nqsuv[v]] [-f <file>]\n"
		"       %s -h | -V\n"
		"\n"
		"options:\n"
		"  -f <file>\tspecify an alternate deletemailrc file\n"
		"  -n\t\tmark messages as deleted, but don't expunge\n"
		"  -q\t\tquiet output (only errors are printed)\n"
		"  -s\t\tdon't use UIDs in IMAP commands\n"
		"  -u\t\tdelete messages marked as unseen too\n"
		"  -v[v]\t\t[more] verbose output\n"
		"  -h\t\tprint this help message and exit\n"
		"  -V\t\tprint deletemail version and exit\n",
		deletemail, deletemail);

	exit(exit_value);
}
