/* $Id: conf.c,v 1.1 2005/01/29 21:06:33 holger Exp $ */

/*
 * Copyright (c) 2004, 2005 Holger Weiss
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif				/* HAVE_CONFIG_H */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#if HAVE_STRINGS_H
#include <strings.h>
#endif				/* HAVE_STRINGS_H */

#include "conf.h"
#include "report.h"
#include "system.h"
#include "text.h"

#if WITH_SSL
#define INITIALIZE_CONFIG(config)                                                    \
  	do {                                                                         \
  		if ((config = malloc(sizeof(conf))) == NULL) {                       \
  			report(LOG_PERROR,                                           \
  			       "can't allocate memory for configuration structure"); \
  			return NULL;                                                 \
  		}                                                                    \
  		config->folder = NULL;                                               \
  		config->host[0]                                                      \
  		    = config->user[0]                                                \
  		    = config->pass[0]                                                \
  		    = config->ssl[0]                                                 \
  		    = '\0';                                                          \
  		config->days                                                         \
  		    = config->port                                                   \
  		    = config->expunge                                                \
  		    = config->unseen                                                 \
  		    = config->use_uid                                                \
  		    = -1;                                                            \
  	} while (/* CONSTCOND */ 0)
#else
#define INITIALIZE_CONFIG(config)                                                    \
  	do {                                                                         \
  		if ((config = malloc(sizeof(conf))) == NULL) {                       \
  			report(LOG_PERROR,                                           \
  			       "can't allocate memory for configuration structure"); \
  			return NULL;                                                 \
  		}                                                                    \
  		config->folder = NULL;                                               \
  		config->host[0]                                                      \
  		    = config->user[0]                                                \
  		    = config->pass[0]                                                \
  		    = '\0';                                                          \
  		config->days                                                         \
  		    = config->port                                                   \
  		    = config->expunge                                                \
  		    = config->unseen                                                 \
  		    = config->use_uid                                                \
  		    = -1;                                                            \
  	} while (/* CONSTCOND */ 0)
#endif				/* WITH_SSL */

#define PARSE_ERROR	report(LOG_ERROR, "parse error reading %s, line %d", file, line)

#define TYPE_STR	0
#define TYPE_UINT	1
#define TYPE_LIST	2

static char    *skip_whitespace(const char *p);
static int      check_conf(conf *config, const conf *global);
static int      parse_value(char *parsed, const char *buf, short type);
static int      save_bool(short *target,
                          const char *buf,
                          const char *name,
                          const char *file,
                          int line);
static int      save_int(int *target,
                         const char *buf,
                         const char *name,
                         const char *file,
                         int line);
static int      save_str(char *target,
                         const char *buf,
                         const char *name,
                         const char *file,
                         int line);

conf           *
readconf(const char *file)
{
#define BUFSIZE		256
	FILE           *fp;
	conf           *global = NULL,
	               *config = NULL,
	               *cnf = NULL;
	size_t          len = 0;
	unsigned int    hosts = 0,
	                line = 1;
	char            buf[BUFSIZE],
	                parsed[CONF_BUFSIZE];
	char           *p = buf;

	if ((fp = fopen(file, "r")) == NULL) {
		report(LOG_PERROR, "can't open %s", file);
		return NULL;
	}
	while ((!feof(fp)) && (fgets(buf + len, BUFSIZE - len, fp) != NULL) && (!ferror(fp))) {
		/*
		 * If this line ends with a backslash, read the next line and
		 * append it to buf.  Overwrite the backslash and the current
		 * newline char in the next turn ("len -= 2").
		 */
		if (((len = strlen(buf)) >= 2) && buf[len - 2] == '\\') {
			len -= 2, line++;
			continue;
		} else
			len = 0;

		/* leading whitespace */
		p = skip_whitespace(buf);
		/* line is empty or a comment */
		if ((*p == '\0') || (*p == '#')) {
			line++;
			continue;
		}
		/* GLOBAL settings */
		if (strncasecmp(p, "GLOBAL", 6) == 0) {
			p = skip_whitespace(p + 6);
			if (!((*p == '\0') || (*p == '#'))) {
				PARSE_ERROR;
				return NULL;
			}
			if (global != NULL) {
				report(LOG_ERROR,
				       "parse error: second GLOBAL declaration in %s, line %d",
				       file,
				       line);
				return NULL;
			} else if (cnf != NULL) {
				report(LOG_ERROR,
				       "parse error: GLOBAL declaration not at top in %s, line %d",
				       file,
				       line);
				return NULL;
			}
			report(LOG_DEBUG, "GLOBAL decleration in %s, line %d", file, line);
			INITIALIZE_CONFIG(global);
			cnf = global;

			/* ACCOUNT settings */
		} else if (strncasecmp(p, "ACCOUNT", 7) == 0) {
			conf           *account;

			p = skip_whitespace(p + 7);
			if (!((*p == '\0') || (*p == '#'))) {
				PARSE_ERROR;
				return NULL;
			}
			report(LOG_DEBUG, "ACCOUNT decleration in %s, line %d", file, line);
			INITIALIZE_CONFIG(account);
			account->next = NULL;

			if (!hosts)
				config = account;
			else
				cnf->next = account;
			cnf = account;
			hosts++;
		} else if (cnf == NULL) {
			report(LOG_ERROR,
			       "need GLOBAL or ACCOUNT decleration first in %s, line %d",
			       file,
			       line);
			return NULL;
		} else if (strncasecmp(p, "host:", 5) == 0) {
			if (save_str(cnf->host, p + 5, "host:", file, line) != 0)
				return NULL;
		} else if (strncasecmp(p, "user:", 5) == 0) {
			if (save_str(cnf->user, p + 5, "user:", file, line) != 0)
				return NULL;
		} else if (strncasecmp(p, "pass:", 5) == 0) {
			if (save_str(cnf->pass, p + 5, "pass:", file, line) != 0)
				return NULL;
		} else if (strncasecmp(p, "port:", 5) == 0) {
			if (save_int(&(cnf->port), p + 5, "port:", file, line) != 0)
				return NULL;
		} else if (strncasecmp(p, "days:", 5) == 0) {
			if (save_int(&(cnf->days), p + 5, "days:", file, line) != 0)
				return NULL;
		} else if (strncasecmp(p, "expunge:", 8) == 0) {
			if (save_bool(&(cnf->expunge), p + 8, "expunge:", file, line) != 0)
				return NULL;
		} else if (strncasecmp(p, "unseen:", 7) == 0) {
			if (save_bool(&(cnf->unseen), p + 7, "unseen:", file, line) != 0)
				return NULL;
		} else if (strncasecmp(p, "use_uid:", 8) == 0) {
			if (save_bool(&(cnf->use_uid), p + 8, "use_uid:", file, line) != 0)
				return NULL;
		} else if (strncasecmp(p, "ssl:", 4) == 0) {
#if WITH_SSL
			if (save_str(cnf->ssl, p + 4, "ssl:", file, line) != 0)
#else
			report(LOG_ERROR,
			       "not built with SSL support in %s, line %d",
			       file,
			       line);
#endif				/* WITH_SSL */
			return NULL;
#if WITH_SSL
			/* special case */
			if (!((strncasecmp(cnf->ssl, "yes", CONF_BUFSIZE) == 0)
			    || (strncasecmp(cnf->ssl, "no", CONF_BUFSIZE) == 0) 
			    || (strncasecmp(cnf->ssl, "starttls", CONF_BUFSIZE) == 0))) {
				report(LOG_ERROR,
				       "parse error: ssl: not set to <yes|no|starttls> in %s, line %d",
				       file,
				       line);
				return NULL;
			}
#endif				/* WITH_SSL */
		} else if (strncasecmp(p, "folder:", 7) == 0) {
			/* we use a linked list for multiple folders */
			mailbox        *mb = NULL;
			int             i,
			                folders = 0;

			p += 7;
			while (((i = parse_value(parsed, p, TYPE_LIST)) > 0)
			       && strlen(parsed)) {
				mailbox        *new_mb = NULL;

				if ((new_mb = malloc(sizeof(mailbox))) == NULL) {
					report(LOG_PERROR, "can't allocate memory for folder name");
					return NULL;
				}
				new_mb->next = NULL;
				if (mb == NULL)
					cnf->folder = new_mb;
				else
					mb->next = new_mb;
				mb = new_mb;
#if HAVE_STRLCPY
				strlcpy(mb->box, parsed, CONF_BUFSIZE);
#else
				strncpy(mb->box, parsed, CONF_BUFSIZE - 1);
				mb->box[CONF_BUFSIZE - 1] = '\0';
#endif				/* HAVE_STRLCPY */
				report(LOG_DEBUG,
				       "saved folder: %s from %s, line %d",
				       mb->box,
				       file,
				       line);
				folders++;
				p += i;	/* look for more folders here */
			}
			if (i == -1 || folders == 0) {
				PARSE_ERROR;
				return NULL;
			}
		} else {
			PARSE_ERROR;
			return NULL;
		}
		line++;
	}
	fclose(fp);

	if (check_conf(config, global) != 0)
		return NULL;

	free(global);
	return config;
}

static char    *
skip_whitespace(const char *p)
{

	while ((*p == ' ') ||
	       (*p == '\t') ||
	       (*p == '\r') ||
	       (*p == '\n'))
		p++;

	return (char *) p;
}

static int
check_conf(conf *config, const conf *global)
{
#define CHECK_USE_GLOBAL_PNT(var)                                             \
  	do {                                                                  \
  		if ((account->var == NULL) && (global->var != NULL))          \
  			account->var = global->var;                           \
  	} while (/* CONSTCOND */ 0)
#define CHECK_USE_GLOBAL_STR(var)                                             \
  	do {                                                                  \
  		if ((!strlen(account->var)) && strlen(global->var)) {         \
  			strncpy(account->var, global->var, CONF_BUFSIZE - 1); \
  			account->var[CONF_BUFSIZE - 1] = '\0';                \
  		}                                                             \
  	} while (/* CONSTCOND */ 0)
#define CHECK_USE_GLOBAL_INT(var)                                             \
  	do {                                                                  \
  		if ((account->var == -1) && (global->var >= 0))               \
  			account->var = global->var;                           \
  	} while (/* CONSTCOND */ 0)

	conf           *account,
	               *account_new;

	if (global != NULL)
		for (account = config;
		     account != NULL;
		     account_new = account->next, account = account_new) {
			CHECK_USE_GLOBAL_PNT(folder);
			CHECK_USE_GLOBAL_STR(host);
			CHECK_USE_GLOBAL_STR(user);
			CHECK_USE_GLOBAL_STR(pass);
			CHECK_USE_GLOBAL_INT(port);
			CHECK_USE_GLOBAL_INT(days);
			CHECK_USE_GLOBAL_INT(expunge);
			CHECK_USE_GLOBAL_INT(unseen);
			CHECK_USE_GLOBAL_INT(use_uid);
#if WITH_SSL
			CHECK_USE_GLOBAL_STR(ssl);
#endif				/* WITH_SSL */
		}

	for (account = config;
	     account != NULL;
	     account_new = account->next, account = account_new) {

		/*
		 * The following values must be specified by the user.
		 */
		if (!strlen(account->host)) {
			report(LOG_ERROR,
			       "'host:' not specified");
			return -1;
		}
		if (!strlen(account->user)) {
			report(LOG_ERROR,
			       "'user:' not specified for %s",
			       account->host);
			return -1;
		}
		if (!strlen(account->pass)) {
			report(LOG_ERROR,
			       "'pass:' not specified for %s",
			       account->host);
			return -1;
		}
		if (account->days == -1) {
			report(LOG_ERROR,
			       "'days:' not specified for %s",
			       account->host);
			return -1;
		}
		/*
		 * The following values may optionally be specified by the
		 * user.  If they were not specified, use default values.
		 */
		if (account->expunge == -1) {
			account->expunge = 1;
			report(LOG_DEBUG,
			       "expunge not specified for %s, setting expunge to %s",
			       account->host,
			       (account->expunge) ? "YES" : "NO");
		}
#if WITH_SSL
		if (!strlen(account->ssl)) {
#if HAVE_STRLCPY
			strlcpy(account->ssl, "no", CONF_BUFSIZE);
#else
			strncpy(account->ssl, "no", CONF_BUFSIZE - 1);
			account->ssl[CONF_BUFSIZE - 1] = '\0';
#endif				/* HAVE_STRLCPY */
			report(LOG_DEBUG,
			       "ssl not specified for %s, setting ssl to %s",
			       account->host, "NO");
		}
#endif				/* WITH_SSL */
		if (account->unseen == -1) {
			account->unseen = 0;
			report(LOG_DEBUG,
			       "unseen not specified for %s, setting unseen to %s",
			       account->host,
			       (account->unseen) ? "YES" : "NO");
		}
		if (account->use_uid == -1) {
			account->use_uid = 1;
			report(LOG_DEBUG,
			       "use_uid not specified for %s, setting use_uid to %s",
			       account->host,
			       (account->use_uid) ? "YES" : "NO");
		}
		if (account->port == -1) {
#if WITH_SSL
			if (strncasecmp(account->ssl, "yes", 3) == 0)
				account->port = 993;
			else
#endif				/* WITH_SSL */
				account->port = 143;
			report(LOG_DEBUG,
			       "port not specified for %s, using %d",
			       account->host,
			       account->port);
		}
		if (account->folder == NULL) {
			if ((account->folder = malloc(sizeof(mailbox))) == NULL) {
				report(LOG_PERROR, "can't allocate memory for default folder name");
				return -1;
			}
			account->folder->next = NULL;
#if HAVE_STRLCPY
			strlcpy(account->folder->box, "INBOX", CONF_BUFSIZE);
#else
			strncpy(account->folder->box, "INBOX", CONF_BUFSIZE - 1);
			account->folder->box[CONF_BUFSIZE - 1] = '\0';
#endif				/* HAVE_STRLCPY */
			report(LOG_DEBUG,
			       "no folder specified for %s, using %s",
			       account->host,
			       account->folder->box);
		}
	}
	return 0;
}

static int
parse_value(char *parsed, const char *buf, short type)
{
	int             quoted;
	char            quote_sign = '"';
	char           *new_p = parsed;
	const char     *old_p = buf;

	old_p = skip_whitespace(old_p);

	if ((*old_p == '"') || (*old_p == '\''))
		quoted = 1, quote_sign = *old_p, old_p++;
	else
		quoted = 0;

	while (*old_p != '\0') {
		if ((*old_p == '\\') && (*(old_p + 1) != '\0'))
			old_p++;
		else if ((*old_p == '\r') || (*old_p == '\n'))
			break;
		else if ((!quoted) && ((*old_p == ' ') || (*old_p == '\t')))
			break;
		else if ((!quoted) && (*old_p == '#'))
			break;
		else if (quoted && (*old_p == quote_sign))
			break;
		else if ((type == TYPE_UINT) && !((*old_p >= '0') && (*old_p <= '9')))
			return -1;
		*(new_p++) = *(old_p++);
	}
	*new_p = '\0';

	if (quoted)
		if (*(old_p++) != quote_sign)
			return -1;

	old_p = skip_whitespace(old_p);

	if (type != TYPE_LIST) {
		/*
		 * If our value is within a TYPE_LIST of values, further
		 * characters don't indicate an error.  On the other hand,
		 * !strlen(parsed) doesn't indicate an error either, since
		 * maybe we're simply at the end of the list.  Hence, when
		 * parsing a list, the caller must handle the case that no
		 * list item was parsed at all.
		 */
		if ((*old_p != '\0') && (*old_p != '#'))
			return -1;
		if (!strlen(parsed))
			return -1;
	}
	return old_p - buf;	/* number of parsed bytes */
}

static int
save_bool(short *target,
          const char *buf,
          const char *name,
          const char *file,
          int line)
{
	char            parsed[CONF_BUFSIZE];

	if (parse_value(parsed, buf, TYPE_STR) > 0) {
		if ((strncasecmp(parsed, "yes", 3) == 0) || ((*parsed == '1') && (*(parsed + 1) == '\0'))) {
			*target = 1;
			report(LOG_DEBUG,
			       "saved %s %s from %s, line %d",
			       name,
			       (*target) ? "YES" : "NO",
			       file,
			       line);
		} else if ((strncasecmp(parsed, "no", 2) == 0) || ((*parsed == '0') && (*(parsed + 1) == '\0'))) {
			*target = 0;
			report(LOG_DEBUG,
			       "saved %s %s from %s, line %d",
			       name,
			       (*target) ? "YES" : "NO",
			       file,
			       line);
		} else {
			report(LOG_ERROR,
			       "parse error (not a boolean value) reading %s, line %d",
			       file,
			       line);
			return -1;
		}
	} else {
		PARSE_ERROR;
		return -1;
	}

	return 0;
}

static int
save_int(int *target,
         const char *buf,
         const char *name,
         const char *file,
         int line)
{
	char            parsed[CONF_BUFSIZE];

	if (parse_value(parsed, buf, TYPE_UINT) > 0) {
		*target = atoi(parsed);
		report(LOG_DEBUG,
		       "saved %s %d from %s, line %d",
		       name,
		       *target,
		       file,
		       line);
	} else {
		PARSE_ERROR;
		return -1;
	}

	return 0;
}

static int
save_str(char *target,
         const char *buf,
         const char *name,
         const char *file,
         int line)
{
	char            parsed[CONF_BUFSIZE];

	if (parse_value(parsed, buf, TYPE_STR) > 0) {
#if HAVE_STRLCPY
		strlcpy(target, parsed, CONF_BUFSIZE);
#else
		strncpy(target, parsed, CONF_BUFSIZE - 1);
		target[CONF_BUFSIZE - 1] = '\0';
#endif				/* HAVE_STRLCPY */
		if (strncmp(name, "pass", 4) == 0)	/* don't print passwords */
			report(LOG_DEBUG,
			       "saved %s %s from %s, line %d",
			       name,
			       stars(strlen(target)),
			       file,
			       line);
		else
			report(LOG_DEBUG,
			       "saved %s %s from %s, line %d",
			       name,
			       target,
			       file,
			       line);
	} else {
		PARSE_ERROR;
		return -1;
	}

	return 0;
}
