/*
 * Copyright (c) DbVis Software AB. All Rights Reserved.
 */

package com.onseven.dbvis.db.h2.sakila;

import java.io.Reader;
import java.sql.*;
import java.util.StringJoiner;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.h2.tools.TriggerAdapter;

/**
 * A H2 implementation of the triggers in the MySQL Sakila Sample Database.
 */
@SuppressWarnings("unused")
public abstract class SakilaTrigger extends TriggerAdapter {

   private static final Logger LOG = Logger.getLogger(SakilaTrigger.class.getName());

   @Override
   public void init(Connection connection, String schemaName, String triggerName, String tableName, boolean before, int type) throws SQLException {
      super.init(connection, schemaName, triggerName, tableName, before, type);
   }

   protected void log(ResultSet oldRow, ResultSet newRow) {
      if (LOG.isLoggable(Level.FINE)) {
         StringJoiner sj = new StringJoiner("\n");
         sj.add(getClass().getSimpleName());
         sj.add("\tOLD ROW: " + toString(oldRow));
         sj.add("\tNEW ROW: " + toString(newRow));
         LOG.fine(sj.toString());
      }
   }

   private String toString(ResultSet row) {
      StringJoiner sj = new StringJoiner("|");
      sj.add(String.valueOf(row));
      if (row !=null) {
         try {
            ResultSetMetaData md = row.getMetaData();
            int columnCount = md.getColumnCount();
            sj.add(columnCount + " columns");
            for (int i = 1; i <= columnCount; i++) {
               sj.add(String.format("%s=[%s]", md.getColumnLabel(i), row.getObject(i)));
            }
         }
         catch (SQLException e) {
            sj.add("No metadata: " + e);
         }
      }
      return sj.toString();
   }

   private static String getTitle(ResultSet row) throws SQLException {
      return row.getString("title");
   }

   private static int getFilmId(ResultSet row) throws SQLException {
      return row.getInt("film_id");
   }

   private static Reader getDescription(ResultSet row) throws SQLException {
      // return row.getClob("description") // throws a ClassCastException
      return row.getCharacterStream("description");
   }

   /**
    * CREATE TRIGGER ins_film AFTER INSERT ON film FOR EACH ROW BEGIN
    *     INSERT INTO film_text (film_id, title, description)
    *         VALUES (new.film_id, new.title, new.description);
    *   END;;
    */
   public static class ins_film extends SakilaTrigger {
      @Override
      public void fire(Connection connection, ResultSet oldRow, ResultSet newRow) throws SQLException {
         log(oldRow, newRow);
         try (PreparedStatement stmt = connection.prepareStatement("INSERT INTO sakila.film_text (film_id, title, description) VALUES (?, ?, ?)")) {
            stmt.setInt(1, getFilmId(newRow));
            stmt.setString(2, getTitle(newRow));
            stmt.setClob(3, getDescription(newRow));
            stmt.executeUpdate();
         }
      }
   }

   /**
    * CREATE TRIGGER upd_film AFTER UPDATE ON film FOR EACH ROW BEGIN
    *     IF (old.title != new.title) OR (old.description != new.description) OR (old.film_id != new.film_id)
    *     THEN
    *         UPDATE film_text
    *             SET title=new.title,
    *                 description=new.description,
    *                 film_id=new.film_id
    *         WHERE film_id=old.film_id;
    *     END IF;
    *   END;;
    */
   public static class upd_film extends SakilaTrigger {
      @Override
      public void fire(Connection connection, ResultSet oldRow, ResultSet newRow) throws SQLException {
         log(oldRow, newRow);
         boolean equal = getFilmId(oldRow) == getFilmId(newRow)
                     && getTitle(oldRow).equals(getTitle(newRow))
                     && getDescription(oldRow).equals(SakilaTrigger.getDescription(newRow));
         if (!equal) {
            try (PreparedStatement stmt = connection.prepareStatement("UPDATE sakila.film_text SET title = ?, description = ?, film_id = ? WHERE film_id = ?")) {
               stmt.setString(1, getTitle(newRow));
               stmt.setClob(2, getDescription(newRow));
               stmt.setInt(3, getFilmId(newRow));
               stmt.setInt(4, getFilmId(oldRow));
               stmt.executeUpdate();
            }
         }
      }
   }

   /**
    * CREATE TRIGGER del_film AFTER DELETE ON film FOR EACH ROW BEGIN
    *     DELETE FROM film_text WHERE film_id = old.film_id;
    *   END;;
    *   
    *   NOTE: handles the FILM_TEXT table which is not defined as a Foreign Key (unlike the other references). 
    */
   public static class del_film extends SakilaTrigger {
      @Override
      public void fire(Connection connection, ResultSet oldRow, ResultSet newRow) throws SQLException {
         log(oldRow, newRow);
         try (PreparedStatement stmt = connection.prepareStatement("DELETE FROM sakila.film_text WHERE film_id = ?")) {
            stmt.setInt(1, getFilmId(oldRow));
            stmt.executeUpdate();
         }
      }
   }

// NOT (YET) USED //  

//   private static void setTimestamp(ResultSet row, String columnName) throws SQLException {
//      row.updateTimestamp(columnName, new Timestamp(System.currentTimeMillis()));
//   }
//
//   /**
//    * CREATE TRIGGER customer_create_date BEFORE INSERT ON customer
//    * 	FOR EACH ROW SET NEW.create_date = NOW();
//    */
//   public static class customer_create_date extends SakilaTrigger {
//      @Override
//      public void fire(Connection connection, ResultSet oldRow, ResultSet newRow) throws SQLException {
//         log(oldRow, newRow);
//         setTimestamp(newRow, "create_date");
//      }
//   }
//
//   /**
//    * CREATE TRIGGER payment_date BEFORE INSERT ON payment
//    * 	FOR EACH ROW SET NEW.payment_date = NOW();
//    */
//   public static class payment_date extends SakilaTrigger {
//      @Override
//      public void fire(Connection connection, ResultSet oldRow, ResultSet newRow) throws SQLException {
//         log(oldRow, newRow);
//         setTimestamp(newRow, "payment_date");
//      }
//   }
//
//   /**
//    * CREATE TRIGGER rental_date BEFORE INSERT ON rental
//    * 	FOR EACH ROW SET NEW.rental_date = NOW();
//    */
//   public static class rental_date extends SakilaTrigger {
//      @Override
//      public void fire(Connection connection, ResultSet oldRow, ResultSet newRow) throws SQLException {
//         log(oldRow, newRow);
//         setTimestamp(newRow, "rental_date");
//      }
//   }
}
