/*
 * Copyright (c) DbVis Software AB. All Rights Reserved.
 */

package com.onseven.dbvis.db.h2.sakila;

import java.math.BigDecimal;
import java.sql.*;
import java.util.logging.Logger;

/**
 * A H2 implementation of the functions in the Sakila Sample Database.
 * See http://www.h2database.com/html/features.html#user_defined_functions
 */
@SuppressWarnings("unused")
public class SakilaFunction {

   private static final Logger LOG = Logger.getLogger(SakilaFunction.class.getName());

   /**
    * DELIMITER $$
    *
    * CREATE FUNCTION get_customer_balance(p_customer_id INT, p_effective_date DATETIME) RETURNS DECIMAL(5,2)
    *     DETERMINISTIC
    *     READS SQL DATA
    * BEGIN
    *
    *        #OK, WE NEED TO CALCULATE THE CURRENT BALANCE GIVEN A CUSTOMER_ID AND A DATE
    *        #THAT WE WANT THE BALANCE TO BE EFFECTIVE FOR. THE BALANCE IS:
    *        #   1) RENTAL FEES FOR ALL PREVIOUS RENTALS
    *        #   2) ONE DOLLAR FOR EVERY DAY THE PREVIOUS RENTALS ARE OVERDUE
    *        #   3) IF A FILM IS MORE THAN RENTAL_DURATION * 2 OVERDUE, CHARGE THE REPLACEMENT_COST
    *        #   4) SUBTRACT ALL PAYMENTS MADE BEFORE THE DATE SPECIFIED
    *
    *   DECLARE v_rentfees DECIMAL(5,2); #FEES PAID TO RENT THE VIDEOS INITIALLY
    *   DECLARE v_overfees INTEGER;      #LATE FEES FOR PRIOR RENTALS
    *   DECLARE v_payments DECIMAL(5,2); #SUM OF PAYMENTS MADE PREVIOUSLY
    *
    *   SELECT IFNULL(SUM(film.rental_rate),0) INTO v_rentfees
    *     FROM film, inventory, rental
    *     WHERE film.film_id = inventory.film_id
    *       AND inventory.inventory_id = rental.inventory_id
    *       AND rental.rental_date <= p_effective_date
    *       AND rental.customer_id = p_customer_id;
    *
    *   SELECT IFNULL(SUM(IF((TO_DAYS(rental.return_date) - TO_DAYS(rental.rental_date)) > film.rental_duration,
    *         ((TO_DAYS(rental.return_date) - TO_DAYS(rental.rental_date)) - film.rental_duration),0)),0) INTO v_overfees
    *     FROM rental, inventory, film
    *     WHERE film.film_id = inventory.film_id
    *       AND inventory.inventory_id = rental.inventory_id
    *       AND rental.rental_date <= p_effective_date
    *       AND rental.customer_id = p_customer_id;
    *
    *
    *   SELECT IFNULL(SUM(payment.amount),0) INTO v_payments
    *     FROM payment
    *
    *     WHERE payment.payment_date <= p_effective_date
    *     AND payment.customer_id = p_customer_id;
    *
    *   RETURN v_rentfees + v_overfees - v_payments;
    * END $$
    *
    * DELIMITER ;
    *
    * DELIMITER $$
    */
   public static BigDecimal get_customer_balance(Connection connection, int customer_id, Timestamp timestamp) throws SQLException {

      BigDecimal balance = new BigDecimal(0);

      {
         // rental fees
         PreparedStatement ps = connection.prepareStatement(
            "SELECT IFNULL(SUM(film.rental_rate),0) FROM film, inventory, rental " +
            " WHERE film.film_id = inventory.film_id" +
            " AND inventory.inventory_id = rental.inventory_id" +
            " AND rental.rental_date <= ?" +
            " AND rental.customer_id = ?"
         );
         ps.setTimestamp(1, timestamp);
         ps.setInt(2, customer_id);
         balance.add(getDecimal(ps.executeQuery()));
      }

      {
         // overdue fees
         PreparedStatement ps = connection.prepareStatement(
            "SELECT IFNULL( " +
            "    CASE WHEN DATEDIFF(DAYOFYEAR, RENTAL.RENTAL_DATE, RENTAL.RETURN_DATE) > FILM.RENTAL_DURATION " +
            "        THEN DATEDIFF(DAYOFYEAR, RENTAL.RENTAL_DATE, RENTAL.RETURN_DATE) - FILM.RENTAL_DURATION " +
            "        ELSE 0 " +
            "        END" +
            "    ,0)" +
            "FROM  RENTAL, INVENTORY, FILM " +
            "    WHERE " +
            "    FILM.FILM_ID = INVENTORY.FILM_ID" +
            "    AND INVENTORY.INVENTORY_ID = RENTAL.INVENTORY_ID" +
            "    AND RENTAL.RENTAL_DATE <= ?" +
            "    AND RENTAL.CUSTOMER_ID = ?;"
         );
         ps.setTimestamp(1, timestamp);
         ps.setInt(2, customer_id);
         balance.add(getDecimal(ps.executeQuery()));
      }

      {
         // payments
         PreparedStatement ps = connection.prepareStatement(
            "SELECT IFNULL(SUM(payment.amount),0) FROM payment WHERE payment.payment_date <= ? AND payment.customer_id = ?"
         );
         ps.setTimestamp(1, timestamp);
         ps.setInt(2, customer_id);
         balance.subtract(getDecimal(ps.executeQuery()));
      }

      return balance;
   }

   /**
    * DELIMITER $$
    *
    * CREATE FUNCTION inventory_held_by_customer(p_inventory_id INT) RETURNS INT
    * READS SQL DATA
    * BEGIN
    *   DECLARE v_customer_id INT;
    *   DECLARE EXIT HANDLER FOR NOT FOUND RETURN NULL;
    *
    *   SELECT customer_id INTO v_customer_id
    *   FROM rental
    *   WHERE return_date IS NULL
    *   AND inventory_id = p_inventory_id;
    *
    *   RETURN v_customer_id;
    * END $$
    *
    * DELIMITER ;
    *
    * DELIMITER $$
    */
   public static Integer inventory_held_by_customer(Connection connection, int inventory_id) throws SQLException {
      PreparedStatement ps = connection.prepareStatement(
         "SELECT customer_id FROM rental WHERE return_date IS NULL AND inventory_id = ?"
      );
      ps.setInt(1, inventory_id);
      ResultSet rs = ps.executeQuery();
      return rs.next() ? rs.getInt(1) : null;
   }

   /**
    * DELIMITER $$
    *
    * CREATE FUNCTION inventory_in_stock(p_inventory_id INT) RETURNS BOOLEAN
    * READS SQL DATA
    * BEGIN
    *     DECLARE v_rentals INT;
    *     DECLARE v_out     INT;
    *
    *     #AN ITEM IS IN-STOCK IF THERE ARE EITHER NO ROWS IN THE rental TABLE
    *     #FOR THE ITEM OR ALL ROWS HAVE return_date POPULATED
    *
    *     SELECT COUNT(*) INTO v_rentals
    *     FROM rental
    *     WHERE inventory_id = p_inventory_id;
    *
    *     IF v_rentals = 0 THEN
    *       RETURN TRUE;
    *     END IF;
    *
    *     SELECT COUNT(rental_id) INTO v_out
    *     FROM inventory LEFT JOIN rental USING(inventory_id)
    *     WHERE inventory.inventory_id = p_inventory_id
    *     AND rental.return_date IS NULL;
    *
    *     IF v_out > 0 THEN
    *       RETURN FALSE;
    *     ELSE
    *       RETURN TRUE;
    *     END IF;
    * END $$
    *
    * DELIMITER ;
    */
   public static boolean inventory_in_stock(Connection connection, int inventory_id) throws SQLException {
      {
         // never rented?
         PreparedStatement ps = connection.prepareStatement(
            "SELECT COUNT(*) FROM rental WHERE inventory_id = ?"
         );
         ps.setInt(1, inventory_id);
         if (isZero(ps)) {
            return true;
         }
      }

      {
         // not returned?
         PreparedStatement ps = connection.prepareStatement(
            "SELECT COUNT(rental_id) FROM rental WHERE inventory_id = ? AND return_date IS NULL"
         );
         ps.setInt(1, inventory_id);
         return isZero(ps);
      }
   }

   private static boolean isZero(PreparedStatement ps) throws SQLException {
      return getDecimal(ps.executeQuery()).compareTo(BigDecimal.ZERO) == 0;
   }

   private static BigDecimal getDecimal(ResultSet rs) throws SQLException {
      return rs != null && rs.next() ? rs.getBigDecimal(1) : BigDecimal.ZERO;
   }

}