/*
** Copyright (c) 2002  Hughes Technologies Pty Ltd.  All rights
** reserved.
**
** Terms under which this software may be used or copied are
** provided in the  specific license associated with this product.
**
** Hughes Technologies disclaims all warranties with regard to this
** software, including all implied warranties of merchantability and
** fitness, in no event shall Hughes Technologies be liable for any
** special, indirect or consequential damages or any damages whatsoever
** resulting from loss of use, data or profits, whether in an action of
** contract, negligence or other tortious action, arising out of or in
** connection with the use or performance of this software.
**
**
** $Id: test_httpd.c,v 1.10 2002/11/25 02:26:58 bambi Exp $
**
*/

#include "config.h"

#include <stdio.h>
#include <unistd.h>

#ifdef _WIN32
#  include <getopt.h>
#else
#  include <sys/time.h>
#endif

#include "httpd.h"

/*
** This is a static page of HTML.  It is loaded into the content
** tree using httpdAddStaticContent( ).
*/
#define test1_html "<HTML><BODY>This is just a test</BODY>"


/*
** Below are 2 dynamic pages, each generated by a C function.  The first
** is a simple page that offers a little dynamic info (the process ID)
** and the setups up a test link and a simple form.
** 
** The second page processes the form.  As you can see, you can access
** the form data from within your C code by accessing the symbol table
** using httpdGetVariableByName() (and other similar functions).  You
** can also include variables in the string passed to httpdOutput( ) and
** they will be expanded automatically.
*/
void index_html(server)
	httpd	*server;
{
	httpdPrintf(server,
	    "Welcome to the httpd server running in process number %d<P>\n",
	    getpid());
	httpdPrintf(server,
	    "Click <A HREF=/test1.html>here</A> to view a test page<P>\n");
	httpdPrintf(server,
	    "Click <A HREF=/login.html>here</A> to authenticate<P>\n");
	httpdPrintf(server,
	    "Or <A HREF=/wildcard/foo>here</A> for a test wildcard page<P>\n");
	httpdPrintf(server, "<P><FORM ACTION=test2.html METHOD=POST>\n");
	httpdPrintf(server, "Enter your name <INPUT NAME=name SIZE=10>\n");
	httpdPrintf(server, "<INPUT TYPE=SUBMIT VALUE=Click!><P></FORM>\n");
	return;
}

void test2_html(server)
	httpd	*server;
{
	httpVar	*variable;

	/*
	** Grab the symbol table entry to see if the variable exists
	*/
	variable = httpdGetVariableByName(server, "name");
	if (variable == NULL)
	{
		httpdPrintf(server,"Missing form data!");
		return;
	}

	/*
	** Use httpdOutput() rather than httpdPrintf() so that the variable
	** embedded in the text is expanded automatically
	*/
	httpdOutput(server,"Hello $name");
}

void test3_html(server)
	httpd	*server;
{
	char	*path;

	path = httpdRequestPath(server);
	httpdOutput(server,"Wilcard dynamic request received<P>");
	httpdPrintf(server,"The requested path was %s<P>", path);
}



void login_html(server)
	httpd	*server;
{
	if (httpdAuthenticate(server, "LibHTTPD Test") == 0)
		return;
	httpdPrintf(server, "Your username is '%s'<P>\n",
		server->request.authUser);
	httpdPrintf(server, "Your password is '%s'<P>\n",
 		server->request.authPassword);
	httpdOutput(server, 
		"Click <A HREF=login2.html>here</A> to force reauthentication");
	httpdOutput(server, ".  Use a username = test password = 123");
}



void login2_html(server)
	httpd	*server;
{
	if (httpdAuthenticate(server, "LibHTTPD Test") == 0)
	{
		httpdOutput(server, "Authentication failure(1).");
		return;
	}
	if (strcmp(server->request.authUser, "test") != 0 ||
	    strcmp(server->request.authPassword, "123") != 0)
	{
		httpdForceAuthenticate(server, "LibHTTPD Test");
		httpdOutput(server, "Authentication failure (2).");
		return;
	}
	httpdOutput(server, "Your login was accepted.");
}


int main(argc, argv)
	int	argc;
	char	*argv[];
{
	httpd	*server;
	char 	*host;
	int 	port,
		errFlag,
		result;
	extern char *optarg;
	extern int optind, opterr, optopt;
	int c;
	struct	timeval timeout;

	host = NULL;
	port = 80;
	errFlag = 0;
	while ( (c=getopt(argc,argv,"h:p:")) != -1 )
	{
		switch ( c ) 
		{
			case 'h':
				host=optarg;
				break;

			case 'p':
				port = atoi(optarg);
				break;
			default:
				errFlag++;
		}
	}
	if (errFlag)
	{
		fprintf(stderr,"usage: [-h <host IP>] [ -p <port >]\n");
		fprintf(stderr,"\nLibHTTPD version %s\n\n",LIBHTTPD_VERSION);
		exit(1);
	}


	/*
	** Create a server and setup our logging
	*/
	server = httpdCreate(host,port);
	if (server == NULL)
	{
		perror("Can't create server");
		exit(1);
	}
	httpdSetAccessLog(server, stdout);
	httpdSetErrorLog(server, stdout);

	/*
	** Setup some content for the server
	*/
	httpdAddCContent(server,"/", "index.html", HTTP_TRUE, 
		NULL, index_html);
	httpdAddCContent(server,"/", "test2.html", HTTP_FALSE, 
		NULL, test2_html);
	httpdAddCContent(server,"/", "login.html", HTTP_FALSE,
		NULL, login_html);
	httpdAddCContent(server,"/", "login2.html", HTTP_FALSE,
		NULL, login2_html);
	httpdAddCWildcardContent(server,"/wildcard", NULL, test3_html);
	httpdAddStaticContent(server, "/", "test1.html", HTTP_FALSE,
		NULL, test1_html);

	/*
	** Go into our service loop
	*/
	timeout.tv_sec = 5;
	timeout.tv_usec = 0;

	while(1 == 1)
	{
		result = httpdGetConnection(server, &timeout);
		if (result == 0)
		{
			printf("Timeout ... \n");
			continue;
		}
		if (result < 0)
		{
			printf("Error ... \n");
			continue;
		}
		if(httpdReadRequest(server) < 0)
		{
			httpdEndRequest(server);
			continue;
		}
		httpdProcessRequest(server);
		httpdEndRequest(server);
	}
}
