# Copyright (C) 2004 Scott W. Dunlop <sdunlop at users.sourceforge.net>
# 
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

#!/usr/bin/env python

import sys
from traceback import format_exception, print_exception
from BaseHTTPServer import HTTPServer, BaseHTTPRequestHandler
from urllib import quote_plus as uriquote
from urllib import unquote_plus as uriunquote
from exceptions import AttributeError
from cStringIO import StringIO
from types import StringType

from html import normalize
from errors import *

from Cookie import SimpleCookie

default_key = "Front Page"

class Request( BaseHTTPRequestHandler ):
    server_version = "CloudWiki/1.0"
    __slots__ = 'query', 'action', 'args', 'user', 'cookies'
    
    def __init__( self, *args, **kwargs ):
        self.query = None
        self.action = None
        self.args = {}
        self.user = None
        self.cookies = SimpleCookie()
        
        BaseHTTPRequestHandler.__init__( self, *args, **kwargs )
    
    def getQuery( self ):
        return self.query
    def setQuery( self, query ):
        self.query = query
    def getAction( self ):
        return self.action
    def setAction( self, action ):
        self.action = action
    def getArgs( self ):
        return self.args
    def getArg( self, key, default=None ):
        return self.args.get(key, default)
    def setArgs( self, content ):
        self.args = args
    def setArg( self, key, value ):
        self.args[key] = value
    def getUser( self ):
        return self.user
    def setUser( self, user ):
        self.user = user
    def normalizeQuery( self ):
        self.setQuery( normalize( self.getQuery() ) )
    def getCookies( self ):
        return self.cookies
    def getCookie( self, key, default=None ):
        try:
            return self.getCookies().get(key).value
        except AttributeError:
            return default

    def log_message( self, format, *args ):
        self.getServer().logLine(          
           "%s -- [%s] %s", 
            
            self.address_string(), 
            self.log_date_time_string(),
            format%args 
        )

    def do_HEAD( self ):
        self.parseUri( )
        self.parseCookies( )
        
        response = self.getServer().respondTo( self )

        self.sendResponseHeaders( response )

    def do_GET( self ):
        self.parseUri( )
        self.parseCookies( )

        response = self.getServer().respondTo( self )
        
        self.sendResponseHeaders( response )
        self.sendResponseContent( response )

    def do_POST( self ):
        self.parseUri( )
        self.parseContent( )
        self.parseCookies( )
        
        if not self.getAction():
            self.setAction( self.getArgs().get( 'action', '' ) )
        
        response = self.getServer().respondTo( self )
        
        self.sendResponseHeaders( response )
        self.sendResponseContent( response )

    def parseContent( self ):
        content_length = self.headers.get("Content-Length")

        if content_length is None:
            raise PostWithoutLengthError( request=self.path )
        else:
            content = ""            
            content_left = int( content_length )
            
            if content_left == 0:
                raise PostWithoutLengthError( request=self.path )
            else:
                while content_left > 0:
                    next = self.rfile.read( content_left )
                    content_left -= len( next )
                    content += next

            self.parseUriArgs( content )

    def getServer( self ):
        return self.server
        
    def parseUriPath( self, uri ):
        steps = uri.split( "/" )
        
        if len( steps ) < 2:
            query, action = '', ''
        elif len( steps ) == 2:
            query, action = (steps[1] or ''), ''
        elif len( steps ) == 3:
            query, action = (steps[1] or ''), (steps[2] or '')
        else:
            raise InvalidRequestError( request = uri )
            
        if query is not None:
            query = uriunquote( query )
        
        self.query = query
        self.action = action

    def parseUriArgs( self, uriArgs ):
        for arg_pair in uriArgs.split('&'):
            key_and_value = arg_pair.split( "=", 1 )
            
            arg_key = key_and_value[0]
            
            if( len( key_and_value ) == 2 ):
                self.args[ arg_key ] = uriunquote( key_and_value[1] )
            else:
                self.args[ arg_key ] = ''
        
    def parseUri( self  ):
        path_and_args = self.path.split( "?", 1 )
                
        self.parseUriPath( path_and_args[ 0 ] )

        if len( path_and_args ) == 2 :
            self.parseUriArgs( path_and_args[1] )

    def parseCookies( self ):
        for header in self.headers.getallmatchingheaders( 'Cookie' ):
            self.cookies.load( header )
        
    def sendResponseHeaders( self, response ):
        self.send_response( response.getCode() )

        for key, value in response.getHeaders():
            self.send_header( key, value )

        self.end_headers()

    def sendResponseContent( self, response ):
        self.sendData( response.getContent() )
                
    def sendLines( self, lines ):
        for line in lines:
            self.sendDine( line )
    
    def sendLine( self, line ):
        self.sendData( line )
        self.sendData( "\n" )
    
    def sendData( self, data ):
        self.wfile.write( data )

class Response( object ):
    __slots__ = ( "code", "cookies", "content", "contentType" )

    def __init__( self, code = 200):
        self.code = code
        self.cookies = []
        self.content = StringIO()
        self.contentType = "text/html" 

    def setCode( self, code ):
        self.code = code

    def getCode( self ):
        return self.code
                    
    def getContentType( self ):
        return self.contentType

    def setContentType( self, contentType ):
        self.contentType = contentType

    def addCookie( self, key, value ):
        self.cookies.append( ( key, value ) )

    def addContent( self, content ):
        self.content.write( content )
        
    def getCookies( self ):
        for cookie in self.cookies:
            yield cookie
            
    def getContent( self ):
        if not isinstance( self.content, StringType ):
            self.content = self.content.getvalue()
        
        return self.content

    def setContent( self, content ):
        self.content = StringIO( content )

    def getHeaders( self ):
        yield "Content-type", self.getContentType()
        yield "Response-length", len( self.getContent() )
        
        for key, value in self.getCookies():
            yield "Set-cookie", "%s=%s;Max-Age=3600" % ( key, value )

    def write( self, data ):
        if self.content is None: 
            self.setContent( data )
        else:
            self.content.write( data )

class Server( HTTPServer ):
    __slots__ = (
        'port', 'wiki', 'wikiUiTable', 'nodeUiTable'
    )
    
    def __init__( self, wiki, port = None ):
        self.port = port or wiki.getHttpPort() or 8080
        self.wiki = wiki

        HTTPServer.__init__( self, ('', self.port), self.getHandlerClass() )

    def getWiki( self ):
        return self.wiki

    def logData( self, format, *args ):
        self.getWiki().logData( format, *args )
        
    def logLine( self, format, *args ):
        self.getWiki().logLine( format, *args )
        
    def getHandlerClass( self ):
        return Request
        
    def getPort( self ):
        return self.port
        
    def runForever( self ):
        self.serve_forever()
        
    def getRedirectResponse( self ):
        return self.redirectResponse
    
    def getStylesheetResponse( self ):
        return self.stylesheetResponse
    
    def getError404( self ):
        return self.error404
        
    def authenticate( self, req, res ):
        auth = self.getWiki().getAuthenticator()
        
        if req.getAction() in ['logout', 'Logout']:
            res.addCookie( 'tk', None )
            req.setAction( '' )
            return
            
        tk = req.getCookie( 'tk', None )
        if tk is not None:
            un = auth.getUserForToken( tk )
            if un:
                req.setUser( un )
                return
        
        args = req.getArgs()

        un = args.get('un')        
        if un is None: return
            
        pw = args.get('pw')
        if pw is None: return
            
        results  = auth.authenticate( un, pw )
        if results:
            un, tk = results
            res.addCookie( 'tk', tk )
            req.setUser( un )
                    
    def respondTo( self, req ):
        wk = self.getWiki()
        res = Response()
        
        try:
            self.authenticate( req, res )
        
            fg = wk.getGatewayFor( req.getQuery() )
            if fg: 
                wk.getLexicon().execFragment( 
                    fg, res.write, { 'req': req, 'res': res } 
                )
                return res
            
            fg = wk.getMethodFor( req.getAction() )
            if fg:
                req.normalizeQuery()
                wk.getLexicon().execFragment( 
                    fg, res.write, { 'req': req, 'res': res } 
                )
                return res

            raise InvalidRequestError( request=req.getAction() )
        except IHateFavicons, exc:
            res.setContent( exc.asHtml() )
            res.setCode( exc.httpCode )
            return res
        except Exception, exc:
            wk = self.getWiki()
            for line in format_exception( *sys.exc_info() ):
                wk.logLine( line )

            try:
                res.setContent( exc.asHtml() )
                res.setCode( exc.httpCode )
            except:
                res.setCode( 500 )
                res.setContent( "<HTML><BODY>Error 500 -- Internal Server Error</BODY></HTML>" )
            return res

