(in-package :png-read)

(defgeneric decode-data (colour-type data png-state))

(defmethod decode-data ((colour-type (eql :greyscale)) data png-state)
  (let* ((h (height png-state))
         (w (width png-state))
         (bd (bit-depth png-state))
         (bda (max 1 (/ bd 8)))
         (scanlines (get-scanlines data h (1+ (ceiling (* bd w) 8)))))
    (declare (type (unsigned-byte 32) w h bd)
             (type (unsigned-byte 3) bda)
             (simple-vector scanlines))
    (setf (image-data png-state)
          (make-array (list w h) :element-type `(unsigned-byte ,bd)))
    (unfilter-scanlines scanlines bda)
    (let ((image-data (image-data png-state)))
      (iter (for scanline in-vector scanlines with-index k)
        (iter (for x in-vector scanline from 1 with-index xi by bda)
          (declare (fixnum x)
                   (type (signed-byte 32) xi))
          (ecase bd
            (1 (iter (for l from 0 below 8)
                 (declare (fixnum l))
                 (for ix = (+ (* (1- xi) 8) l))
                 (declare (fixnum ix))
                 (until (>= ix w))
                 (setf (aref image-data ix k)
                       (ldb (byte 1 (- 7 l)) x))))
            (2 (iter (for l from 0 below 4)
                 (declare (fixnum l))
                 (for ix = (+ (* (1- xi) 4) l))
                 (declare (fixnum ix))
                 (until (>= ix w))
                 (setf (aref image-data ix k)
                       (ldb (byte 2 (* 2 (- 3 l))) x))))
            (4 (iter (for l from 0 below 2)
                 (declare (fixnum l))
                 (for ix = (+ (* (1- xi) 2) l))
                 (declare (fixnum ix))
                 (until (>= ix w))
                 (setf (aref image-data ix k)
                       (ldb (byte 4 (* 4 (- 1 l))) x))))
            (8 (setf (aref image-data (1- xi) k) x))
            (16 (setf (aref image-data (floor (1- xi) 2) k)
                      (big-endian-vector-to-integer (subseq scanline xi (+ xi 2)))))))))
    png-state))

(defmethod decode-data ((colour-type (eql :truecolor)) data png-state)
  (let* ((h (height png-state))
         (w (width png-state))
         (bd (bit-depth png-state))
         (bda (/ bd 8))
         (scanlines (get-scanlines data h (1+ (* bda w 3)))))
    (declare (type (unsigned-byte 32) w h bd)
             (type (unsigned-byte 3) bda)
             (simple-vector scanlines))
    (setf (image-data png-state)
          (make-array (list w h 3) :element-type `(unsigned-byte ,bd)))
    (unfilter-scanlines scanlines (* bda 3))
    (case bd
      (8 (loop with image-data of-type (simple-array (unsigned-byte 8) (* * *)) = (image-data png-state)
               for scanline of-type (simple-array (unsigned-byte 8) (*)) across scanlines
               for k of-type (unsigned-byte 32) from 0
               do (loop for xi of-type (unsigned-byte 32) from 1 below (length scanline) by bda
                        for y of-type (unsigned-byte 32) from 0
                        do (setf (aref image-data (floor y 3) k (mod y 3))
                                 (loop for i from (1- bda) downto 0
                                       for j from xi below (expt 2 20)
                                       sum (ash (aref scanline j) (* 8 i))
                                         into s of-type (unsigned-byte 32)
                                       finally (return s))))))
      (16 (loop with image-data of-type (simple-array (unsigned-byte 16) (* * *)) = (image-data png-state)
                for scanline of-type (simple-array (unsigned-byte 8) (*)) across scanlines
                for k of-type (unsigned-byte 32) from 0
                do (loop for xi of-type (unsigned-byte 32) from 1 below (length scanline) by bda
                         for y of-type (unsigned-byte 32) from 0
                         do (setf (aref image-data (floor y 3) k (mod y 3))
                                  (loop for i from (1- bda) downto 0
                                        for j from xi below (expt 2 20)
                                        sum (ash (aref scanline j) (* 8 i))
                                          into s of-type (unsigned-byte 32)
                                        finally (return s)))))))
    png-state))

(defun set-image-slice-to-index (x y idx palette image-data)
  (iter (for k from 0 below 3)
    (setf (aref image-data x y k)
          (aref palette idx k))))

(defmethod decode-data ((colour-type (eql :indexed-colour)) data png-state)
  (let* ((h (height png-state))
         (w (width png-state))
         (pal (palette png-state))
         (bd (bit-depth png-state))
         (bda (max 1 (/ bd 8)))
         (scanlines (get-scanlines data h (1+ (ceiling (* bd w) 8)))))
    (setf (image-data png-state)
          (make-array (list w h 3) :element-type '(unsigned-byte 8)))
    (setf (index-data png-state)
          (make-array (list w h) :element-type `(unsigned-byte ,bd)))
    (unfilter-scanlines scanlines bda)
    (let ((image-data (image-data png-state))
          (index-data (index-data png-state)))
      (declare (type (simple-array (unsigned-byte 8) (* * *)) image-data))
      (iter (for scanline in-vector scanlines with-index k)
        (iter (for x in-vector scanline from 1 with-index xi by bda)
          (declare (fixnum x)
                   (type (signed-byte 32) xi))
          (ecase bd
            (1 (iter (for l from 0 below 8)
                 (declare (fixnum l))
                 (for xii = (+ (* (1- xi) 8) l))
                 (declare (fixnum xii))
                 (until (>= xii w))
                 (let ((idx (ldb (byte 1 (- 7 l)) x)))
                   (declare (type bit))
                   (setf (aref index-data xii k) idx)
                   (set-image-slice-to-index xii k idx pal image-data))))
            (2 (iter (for l from 0 below 4)
                 (declare (fixnum l))
                 (for xii = (+ (* (1- xi) 4) l))
                 (declare (fixnum xii))
                 (until (>= xii w))
                 (let ((idx (ldb (byte 2 (* 2 (- 3 l))) x)))
                   (declare (type (unsigned-byte 2)))
                   (setf (aref index-data xii k) idx)
                   (set-image-slice-to-index xii k idx pal image-data))))
            (4 (iter (for l from 0 below 2)
                 (declare (fixnum l))
                 (for xii = (+ (* (1- xi) 2) l))
                 (declare (fixnum xii))
                 (until (>= xii w))
                 (let ((idx (ldb (byte 4 (* 4 (- 1 l))) x)))
                   (declare (type (unsigned-byte 4) idx))
                   (setf (aref index-data xii k) idx)
                   (set-image-slice-to-index xii k idx pal image-data))))
            (8 (setf (aref index-data (1- xi) k) x)
             (set-image-slice-to-index (1- xi) k x pal image-data))))))
    png-state))

(defmethod decode-data ((colour-type (eql :greyscale-alpha)) data png-state)
  (let* ((h (height png-state))
         (w (width png-state))
         (bd (bit-depth png-state))
         (bda (/ bd 8))
         (scanlines (get-scanlines data h (1+ (ceiling (* 2 bd w) 8)))))
    (declare (type (unsigned-byte 32) w h bd)
             (type (unsigned-byte 3) bda)
             (simple-vector scanlines))
    (setf (image-data png-state)
          (make-array (list w h 2) :element-type `(unsigned-byte ,bd)))
    (unfilter-scanlines scanlines (* 2 bda))
    (let ((image-data (image-data png-state)))
      (iter (for scanline in-vector scanlines with-index k)
        (iter (for x in-vector scanline from 1 with-index xi by bda)
          (for y from 0)
          (declare (fixnum x y)
                   (type (signed-byte 32) xi))
          (ecase bd
            (8 (setf (aref image-data (floor y 2) k (mod y 2)) x))
            (16 (setf (aref image-data (floor y 2) k (mod y 2))
                      (big-endian-vector-to-integer (subseq scanline xi (+ xi 2)))))))))
    png-state))

(defmethod decode-data ((colour-type (eql :truecolor-alpha)) data png-state)
  (let* ((h (height png-state))
         (w (width png-state))
         (bd (bit-depth png-state))
         (bda (/ bd 8))
         (scanlines (get-scanlines data h (1+ (* bda w 4)))))
    (declare (type (unsigned-byte 32) w h bd)
             (type (unsigned-byte 3) bda)
             (simple-vector scanlines))
    (setf (image-data png-state)
          (make-array (list w h 4) :element-type `(unsigned-byte ,bd)))
    (unfilter-scanlines scanlines (* bda 4))
    (case bd
      (8 (loop with image-data of-type (simple-array (unsigned-byte 8) (* * *)) = (image-data png-state)
               for scanline of-type (simple-array (unsigned-byte 8) (*)) across scanlines
               for k of-type (unsigned-byte 32) from 0
               do (loop for xi of-type (unsigned-byte 32) from 1 below (length scanline) by bda
                        for y of-type (unsigned-byte 32) from 0
                        do (setf (aref image-data (floor y 4) k (mod y 4))
                                 (loop for i from (1- bda) downto 0
                                       for j from xi below (expt 2 20)
                                       sum (ash (aref scanline j) (* 8 i))
                                         into s of-type (unsigned-byte 32)
                                       finally (return s))))))
      (16 (loop with image-data of-type (simple-array (unsigned-byte 16) (* * *)) = (image-data png-state)
                for scanline of-type (simple-array (unsigned-byte 8) (*)) across scanlines
                for k of-type (unsigned-byte 32) from 0
                do (loop for xi of-type (unsigned-byte 32) from 1 below (length scanline) by bda
                         for y of-type (unsigned-byte 32) from 0
                         do (setf (aref image-data (floor y 4) k (mod y 4))
                                  (loop for i from (1- bda) downto 0
                                        for j from xi below (expt 2 20)
                                        sum (ash (aref scanline j) (* 8 i))
                                          into s of-type (unsigned-byte 32)
                                        finally (return s)))))))
    png-state))

(defun get-scanlines (data h filtered-scanline-length)
  (let ((scanlines (make-array h)))
    (iter (for i from 0 below h)
      (until (>= (* i filtered-scanline-length) (length data)))
      (setf (aref scanlines i) (subseq data
                                       (* i filtered-scanline-length)
                                       (min (length data) (* (1+ i) filtered-scanline-length)))))
    scanlines))

(declaim (inline sub-byte up-byte subup-byte paeth-predictor))
(defun sub-byte (xi scanline pixel-length)
  (if (> xi pixel-length)
      (aref scanline (- xi pixel-length))
      0))

(defun up-byte (xi k scanlines)
  (if (zerop k)
      0
      (the (unsigned-byte 8) (aref (aref scanlines (1- k)) xi))))

(defun subup-byte (xi k scanlines pixel-length)
  (if (or (zerop k)
          (<= xi pixel-length))
      0
      (the (unsigned-byte 8) (aref (aref scanlines (1- k)) (- xi pixel-length)))))

(defun paeth-predictor (xi k scanlines pixel-length)
  (let ((scanline (aref scanlines k)))
    (let ((a (sub-byte xi scanline pixel-length))
          (b (up-byte xi k scanlines))
          (c (subup-byte xi k scanlines pixel-length)))
      (declare (type (unsigned-byte 8) a b c))
      (let ((p (- (+ a
                     b)
                  c)))
        (let ((pa (abs (- p a)))
              (pb (abs (- p b)))
              (pc (abs (- p c))))
          (cond ((and (<= pa pb)
                      (<= pa pc))
                 a)
                ((<= pb pc)
                 b)
                (t c)))))))

(defun unfilter-scanlines (scanlines pixel-length)
  (declare (type simple-vector scanlines)
           (fixnum pixel-length))
  (iter (for scanline in-vector scanlines with-index k)
    (declare (type (simple-array (unsigned-byte 8) (*)) scanline)
             (fixnum k))
    (unless (zerop (length scanline))
      (ecase (aref scanline 0)
        (0 nil)
        (1 (iter (for x in-vector scanline from 1 with-index xi)
             (declare (fixnum x xi))
             (setf (aref scanline xi)
                   (ldb (byte 8 0) (+ x (sub-byte xi scanline pixel-length))))))
        (2 (iter (for x in-vector scanline from 1 with-index xi)
             (declare (fixnum x xi))
             (setf (aref scanline xi)
                   (ldb (byte 8 0) (+ x (up-byte xi k scanlines))))))
        (3 (iter (for x in-vector scanline from 1 with-index xi)
             (declare (fixnum x xi))
             (setf (aref scanline xi)
                   (ldb (byte 8 0)
                        (+ x (floor (+ (sub-byte xi scanline pixel-length)
                                       (up-byte xi k scanlines))
                                    2))))))
        (4 (iter (for x in-vector scanline from 1 with-index xi)
             (declare (fixnum x xi))
             (setf (aref scanline xi)
                   (ldb (byte 8 0)  (+ x (paeth-predictor xi k scanlines pixel-length)))))))))
  scanlines)

(defun finish-decoding (png-state)
  (unless (datastream png-state)
    (if (png-file png-state)
        (error "No data in image file ~a." (png-file png-state))
        (error "No data in stream.")))
  (let ((decompressed-data (decompress nil :zlib (datastream png-state))))
    (ecase (interlace-method png-state)
      (:no-interlace (decode-data (colour-type png-state) decompressed-data png-state))
      (:adam7-interlace (decode-interlaced decompressed-data png-state)))
    (values (image-data png-state)
            png-state)))
