//
//  Copyright(C) 2012 Taro Watanabe <taro.watanabe@nict.go.jp>
//

#include <iostream>

#include <cicada/operation.hpp>
#include <cicada/parameter.hpp>
#include <cicada/sort_topologically.hpp>

#include <cicada/operation/sort_topologically.hpp>
#include <cicada/operation/functional.hpp>

#include <utils/lexical_cast.hpp>
#include <utils/resource.hpp>
#include <utils/piece.hpp>

namespace cicada
{
  namespace operation
  {
    SortTopologically::SortTopologically(const std::string& parameter, const int __debug)
      : base_type("sort-topologically"), debug(__debug)
    {
      typedef cicada::Parameter param_type;
    
      param_type param(parameter);
      if (utils::ipiece(param.name()) != "sort-topologically")
	throw std::runtime_error("this is not a topologically sorter");

      for (param_type::const_iterator piter = param.begin(); piter != param.end(); ++ piter)
	std::cerr << "WARNING: unsupported parameter for topologically sorter: " << piter->first << "=" << piter->second << std::endl;
    }
    
    void SortTopologically::operator()(data_type& data) const
    {
      if (! data.hypergraph.is_valid()) return;
      
      hypergraph_type& hypergraph = data.hypergraph;
      hypergraph_type sorted;

      if (debug)
	std::cerr << name << ": " << data.id << std::endl;
      
      utils::resource start;

      cicada::topologically_sort(hypergraph, sorted);
	
      utils::resource end;
    
      if (debug)
	std::cerr << name << ": " << data.id
		  << " cpu time: " << (end.cpu_time() - start.cpu_time())
		  << " user time: " << (end.user_time() - start.user_time())
		  << " thread time: " << (end.thread_time() - start.thread_time())
		  << std::endl;
    
      if (debug)
	std::cerr << name << ": " << data.id
		  << " # of nodes: " << sorted.nodes.size()
		  << " # of edges: " << sorted.edges.size()
		  << " valid? " << utils::lexical_cast<std::string>(sorted.is_valid())
		  << std::endl;

      statistics_type::statistic_type& stat = data.statistics[name];
	
	++ stat.count;
	stat.node += sorted.nodes.size();
	stat.edge += sorted.edges.size();
	stat.user_time += (end.user_time() - start.user_time());
	stat.cpu_time  += (end.cpu_time() - start.cpu_time());
	stat.thread_time  += (end.thread_time() - start.thread_time());
    
      hypergraph.swap(sorted);
    }
  };
};
