#!/usr/bin/env python
"""Run Mercurial test suite with cHg"""
import errno, os, signal, sys, time

CHGROOT = os.path.abspath(os.path.join(os.path.dirname(__file__), '..'))
HGROOT = os.path.join(os.getcwd(), '..')

sys.path.insert(0, HGROOT)
sys.path.insert(0, os.path.join(HGROOT, 'tests'))

runtests = __import__('run-tests')

_extraconfigs = {
    # preload 'purge' so that mq.uisetup() can add --mq option to it
    'test-mq.t': ['extensions.purge='],
    }

# switch background server by environment variables (SHOULD BE FIXED LATER)
_defaultenvsuminclude = []
_envsuminclude = {
    'test-convert-svn-branches.t': _defaultenvsuminclude + ['HG'],
    'test-hook.t': _defaultenvsuminclude + ['PYTHONPATH'],
    }

_chgcallcnt = {}  # call-count by envsum (socket name)
_chgalive = set()  # envsum (socket name) of alive servers

def updatechghist():
    try:
        f = open(CHGHIST)
        try:
            for l in f:
                k = l.rstrip('\n')
                try:
                    _chgcallcnt[k] += 1
                except KeyError:
                    _chgcallcnt[k] = 1
                _chgalive.add(k)
        finally:
            f.close()
        os.unlink(CHGHIST)
    except IOError:
        return

MAXCHGSERVERS = 8

def _killchg(keys):
    pids = {}
    for k in keys:
        if k not in _chgalive:
            continue

        try:
            f = open(os.path.join(CHGTMP, '%s.pid' % k))
            try:
                pids[k] = int(f.read())
            except ValueError:
                continue
            finally:
                f.close()
        except IOError, err:
            if err.errno != errno.ENOENT:
                raise

    if not pids:
        return

    for k, pid in pids.iteritems():
        try:
            os.kill(pid, 0)
            runtests.vlog('# Killing cHg server %s (pid=%d, callcnt=%d)'
                          % (k, pid, _chgcallcnt[k]))
            os.kill(pid, signal.SIGTERM)
        except OSError, err:
            if err.errno != errno.ESRCH:
                raise

    time.sleep(0.25)
    for k, pid in pids.iteritems():
        try:
            os.kill(pid, 0)
            runtests.vlog('# Daemon process %d is stuck - really killing it'
                          % pid)
            os.kill(pid, signal.SIGKILL)
        except OSError, err:
            if err.errno != errno.ESRCH:
                raise
            _chgalive.remove(k)

def killchgbylfu():
    """Kill least-frequenty-used chg servers"""
    keys = [k for k, v in sorted(_chgcallcnt.iteritems(),
                                 key=lambda e: e[1])]
    _killchg(keys[:-MAXCHGSERVERS])

def killallchg():
    _killchg(_chgalive)

def _wraptestcls(cls):
    class ChgTest(cls):
        def __init__(self, *args, **kwargs):
            super(ChgTest, self).__init__(*args, **kwargs)
            excfg = _extraconfigs.get(self.name)
            if excfg:
                self._extraconfigopts.extend(excfg)

        def setUp(self):
            super(ChgTest, self).setUp()
            # pattern for printenv | grep
            inc = _envsuminclude.get(self.name, _defaultenvsuminclude)
            os.environ['CHGENVSUMINCLUDEPAT'] = '^\\(%s\\)=' % '\\|'.join(inc)

        def tearDown(self):
            super(ChgTest, self).tearDown()
            updatechghist()
            killchgbylfu()
    return ChgTest

class ChgTestRunner(runtests.TestRunner):
    TESTTYPES = [(ext, _wraptestcls(cls))
                 for ext, cls in runtests.TestRunner.TESTTYPES]

    def _runtests(self, tests):
        global CHGTMP, CHGHIST
        os.environ['CHGTMP'] = CHGTMP = os.path.join(self._hgtmp, 'chg')
        os.mkdir(CHGTMP)
        os.environ['CHGHIST'] = CHGHIST = os.path.join(CHGTMP, 'history')
        _chgcallcnt.clear()
        _chgalive.clear()
        try:
            super(ChgTestRunner, self)._runtests(tests)
        finally:
            killallchg()

def main(args):
    os.environ['CHGHG'] = os.path.join(HGROOT, 'hg')
    os.environ['PYTHONPATH'] = os.pathsep.join([
        os.path.join(CHGROOT, 'hgext'),
        HGROOT,
        os.environ.get('PYTHONPATH', '')])
    args.extend(['--with-hg', os.path.join(CHGROOT, 'tests', 'hg')])

    if sys.platform == 'linux2':
        os.environ['CHGENVSUMCMD'] = 'md5sum'
    else:
        os.environ['CHGENVSUMCMD'] = 'md5'

    runner = ChgTestRunner()
    return runner.run(args)

if __name__ == '__main__':
    sys.exit(main(sys.argv[1:]))
