// Copyright (c) 1999
// Utrecht University (The Netherlands),
// ETH Zurich (Switzerland),
// INRIA Sophia-Antipolis (France),
// Max-Planck-Institute Saarbruecken (Germany),
// and Tel-Aviv University (Israel).  All rights reserved.
//
// This file is part of CGAL (www.cgal.org)
//
// $URL: https://github.com/CGAL/cgal/blob/v5.5/Kernel_23/include/CGAL/kernel_to_kernel.h $
// $Id: kernel_to_kernel.h 0779373 2020-03-26T13:31:46+01:00 Sébastien Loriot
// SPDX-License-Identifier: LGPL-3.0-or-later OR LicenseRef-Commercial
//
//
// Author(s)     : Stefan Schirra

#ifndef CGAL_KERNEL_TO_KERNEL_H
#define CGAL_KERNEL_TO_KERNEL_H

#include <CGAL/config.h>

#ifdef CGAL_USE_LEDA
#include <CGAL/LEDA_basic.h>
#include <CGAL/leda_integer.h>
#include <LEDA/geo/rat_point.h>
#endif

#include <CGAL/Point_2.h>
#include <CGAL/Segment_2.h>
#include <CGAL/Cartesian.h>
#include <CGAL/Homogeneous.h>

namespace CGAL {

template <class NumberType>
struct Cartesian_double_to_Homogeneous
{
  typedef Point_2< Homogeneous< NumberType> >    Point2;
  typedef Segment_2< Homogeneous< NumberType> >  Segment;

  Cartesian_double_to_Homogeneous() {}

  Point2
  operator()(  const Point_2<Cartesian<double> >& p) const
  { return Point2( NumberType(p.x()), NumberType(p.y()) ); }

  Segment
  operator()(  const Segment_2<Cartesian<double> >& s) const
  {
    return Segment( Point2( NumberType(s.source().x()),
                            NumberType(s.source().y()) ),
                    Point2( NumberType(s.target().x()),
                            NumberType(s.target().y()) ) );
  }
};

#ifdef CGAL_USE_LEDA
struct Cartesian_double_to_H_double_int
{
  typedef Point_2< Homogeneous< double> >    Point2;
  typedef Segment_2< Homogeneous< double> >  Segment;

  Segment
  operator()(  const Segment_2< Cartesian< double> >& s) const
  {
    leda_rat_point rs =  leda_point(s.source().x(), s.source().y());
    leda_rat_point rt =  leda_point(s.target().x(), s.target().y());

    return Segment(
      Point2(CGAL_LEDA_SCOPE::to_double(rs.X()),
             CGAL_LEDA_SCOPE::to_double(rs.Y()),
             CGAL_LEDA_SCOPE::to_double(rs.W())),
      Point2(CGAL_LEDA_SCOPE::to_double(rt.X()),
             CGAL_LEDA_SCOPE::to_double(rt.Y()),
             CGAL_LEDA_SCOPE::to_double(rt.W())) );
  }
};

struct Cartesian_float_to_H_double_int
{
  typedef Point_2< Homogeneous< double> >    Point2;
  typedef Segment_2< Homogeneous< double> >  Segment;

  Segment
  operator()(  const Segment_2< Cartesian< float> >& s) const
  {
    leda_rat_point rs =  leda_point(s.source().x(), s.source().y());
    leda_rat_point rt =  leda_point(s.target().x(), s.target().y());

    return Segment(
      Point2(CGAL_LEDA_SCOPE::to_double(rs.X()),
             CGAL_LEDA_SCOPE::to_double(rs.Y()),
             CGAL_LEDA_SCOPE::to_double(rs.W())),
      Point2(CGAL_LEDA_SCOPE::to_double(rt.X()),
             CGAL_LEDA_SCOPE::to_double(rt.Y()),
             CGAL_LEDA_SCOPE::to_double(rt.W())) );
  }
};
#endif // CGAL_USE_LEDA

} //namespace CGAL

#endif // CGAL_KERNEL_TO_KERNEL_H
