#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <sys/types.h>
#include <sys/socket.h>
#if HAVE_SYS_UTSNAME_H
#include <sys/utsname.h>
#endif
#if HAVE_SYS_PARAM_H
#include <sys/param.h>
#endif
#include <netinet/in.h>
#include <arpa/inet.h>
#if HAVE_NET_ETHERNET_H
#include <net/ethernet.h>
#endif
#if HAVE_NETINET_ETHER_H
#include <netinet/ether.h>
#endif
#include <net/if.h>
#if HAVE_NET_IF_DL_H
#include <net/if_dl.h>
#endif
#include <sys/ioctl.h>
#if HAVE_NETINET_ETHER_H
#include <netinet/if_ether.h>
#endif
#if HAVE_SYS_SYSCTL_H
#include <sys/sysctl.h>
#endif

#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#if HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <pcap.h>

extern char* optarg;

#ifndef MAX_IFS
#define MAX_IFS 128
#endif

#define CDP_CAP_ROUTER 0x01
#define CDP_CAP_TBRIDG 0x02
#define CDP_CAP_SBRIDG 0x04
#define CDP_CAP_SWITCH 0x08
#define CDP_CAP_HOST   0x10

struct __attribute__ ((__packed__)) cdp_header { 
/* ethernet 802.3 header */
	unsigned char dst_addr[6];
	unsigned char src_addr[6];
	u_int16_t length;
/* LLC */
	u_int8_t dsap;
	u_int8_t ssap;
/* llc control */
	u_int8_t control;
	u_int8_t orgcode[3];
	u_int16_t protocolId;
};

static struct utsname myuname;
static char mysysname[512];
static int debug=0;
static unsigned char capabilities[4]={0x00,0x10,0x00,0x00};

#if !HAVE_VSNPRINTF
#if HAVE___VSNPRINTF
#include <stdarg.h>
/* Solaris 2.5.1 implementation of vsnprintf.. */
int __vsnprintf(char* str, size_t size, const char* format, va_list ap);
int vsnprintf(char* str, size_t size, const char* format, va_list ap)
{ 
	return __vsnprintf(str,size,format,ap);
}; 
#else
#error "Don't know how to handle vsnprintf(3) calls."
#endif
#endif

#if !HAVE_SNPRINTF
#if HAVE___VSNPRINTF
#include <stdarg.h>
int snprintf(char* str, size_t size, const char* format, ...)
{ 
	va_list ap;
	int retval;
	va_start(ap, format);
	retval=vsnprintf(str,size,format,ap);
	va_end(ap);
	return retval;
}; 
#else
#error "Don't know how to handle snprintf(3) calls."
#endif
#endif

#if !HAVE_DAEMON
/* prototype of daemon(3), defined in daemon.c */
int daemon(int,int);
#endif

#ifndef HAVE_PCAP_INJECT
#if HAVE_PCAP_SENDPACKET
int 
pcap_inject(pcap_t* p, const void* buffer, size_t size)
{ 
	if(pcap_sendpacket(p, buffer, size) == 0) 
		return size;
	return -1;
};
#else
#error unable to emulate pcap_inject
#endif
#endif


#ifndef HAVE_STRLCPY
size_t strlcpy(char* dst, const char* src, size_t size);
#endif

#if !HAVE_ETHER_NTOA
char* 
my_ether_ntoa(unsigned char* mac)
{
	static char a[32];
	snprintf(a, sizeof(a), "%2.2x:%2.2x:%2.2x:%2.2x:%2.2x:%2.2x",
		mac[0], mac[1], mac[2], mac[3], mac[4], mac[5]);
	return a;
};
#endif

int
sx_write_long(unsigned char* buffer, u_int32_t data)
{ 
#if WORDS_BIGENDIAN
	buffer[0]=(data>>24)&0xff;
	buffer[1]=(data>>16)&0xff;
	buffer[2]=(data>>8)&0xff;
	buffer[3]=data&0xff;
#else 
	buffer[3]=(data>>24)&0xff;
	buffer[2]=(data>>16)&0xff;
	buffer[1]=(data>>8)&0xff;
	buffer[0]=data&0xff;
#endif
	return 1;
};

int
sx_write_short(unsigned char* buffer, u_int16_t data)
{ 
#if WORDS_BIGENDIAN
	buffer[0]=(data>>8)&0xff;
	buffer[1]=data&0xff;
#else
	buffer[1]=(data>>8)&0xff;
	buffer[0]=data&0xff;
#endif
	return 1;
};

int
cdp_buffer_init(unsigned char* buffer, int len, unsigned char* myether)
{ 
	memset(buffer,0,len);

	buffer[0]=0x01;
	buffer[1]=0x00;
	buffer[2]=0x0c;
	buffer[3]=buffer[4]=buffer[5]=0xcc; 

	memcpy(buffer+6,myether,6);

	((struct cdp_header*)buffer)->dsap=0xaa;
	((struct cdp_header*)buffer)->ssap=0xaa;
	((struct cdp_header*)buffer)->control=0x03;
	((struct cdp_header*)buffer)->orgcode[2]=0x0c;
	sx_write_short((unsigned char*)&(((struct cdp_header*)buffer)->protocolId),
		htons(0x2000));

	buffer+=sizeof(struct cdp_header);

	buffer[0]=0x1; /* cdp version */
	buffer[1]=0xb4; /* cdp holdtime, 180 sec by default */
	buffer[2]=buffer[3]=0; /* checksum - will calculate later */

	return 4+sizeof(struct cdp_header);
};

unsigned 
lldp_encode_pdu(unsigned char* buffer, unsigned tlv, unsigned tlen, 
	unsigned char* data)
{ 
	buffer[0]=(tlv<<1)|(tlen>>8);
	buffer[1]=(tlen&0xff);
	if(tlen) { 
		memcpy(buffer+2,data,tlen);
	};
	return tlen+2;
};

int
lldp_buffer_init(unsigned char* buffer, int len, unsigned char* myether)
{ 
	unsigned char macpdu[7];

	if(len<14+7) { 
		return 0;
	};
	memset(buffer,0,len);
	buffer[0]=0x01;
	buffer[1]=0x80;
	buffer[2]=0xc2;
	buffer[3]=buffer[4]=0x00;
	buffer[5]=0x0e;

	memcpy(buffer+6,myether,6);

	buffer[12]=0x88;
	buffer[13]=0xcc;

	macpdu[0]=0x04;
	memcpy(macpdu+1,myether,6);

	return 14+lldp_encode_pdu(buffer+14,1,7,macpdu);
};

int
lldp_add_interface(unsigned char* buffer, unsigned size, char* iface)
{ 
	unsigned char hname[128];
	hname[0]=7;
	strlcpy((char*)hname+1,iface,sizeof(hname)-1);
	if(strlen(iface)+3>size) 
		return 0;
	return lldp_encode_pdu(buffer,2,strlen(iface)+1,hname);
};

int
lldp_add_ttl(unsigned char* buffer, unsigned size, unsigned ttl)
{ 
	uint16_t tt=htons(ttl);
	if(size<4) 
		return 0;
	return lldp_encode_pdu(buffer,3,2,(unsigned char*)&tt);
};

int
lldp_add_ifname(unsigned char* buffer, unsigned size, unsigned char* iface)
{ 
	if(size<strlen((char*)iface+2))
		return 0;
	return lldp_encode_pdu(buffer,4,strlen((char*)iface),iface);
};

int
lldp_add_hostname(unsigned char* buffer, unsigned size)
{ 
	unsigned char hostname[128];
	
	if(gethostname((char*)hostname,sizeof(hostname))==-1) { 
		strlcpy((char*)hostname,"Amnesiac",sizeof(hostname));
	};

	if(size<strlen((char*)hostname)+2) 
		return 0;

	return lldp_encode_pdu(buffer,5,strlen((char*)hostname),hostname);
};

int
lldp_add_sysdescr(unsigned char* buffer, unsigned size)
{ 
#if HAVE_SYS_UTSNAME_H
	struct utsname uts;
	unsigned char description[256];
	if(uname(&uts)!=0)
		return 0;
	snprintf((char*)description, sizeof(description), "%s %s %s %s",
		uts.sysname, uts.release, uts.version, uts.machine);
	if(size<strlen((char*)description)+2)
		return 0;
	return lldp_encode_pdu(buffer,6,strlen((char*)description),
		description);
#else
	return 0;
#endif
};


int
lldp_add_capabilities(unsigned char* buffer, unsigned size)
{ 
	if(size<6) 
		return 0;
	return lldp_encode_pdu(buffer,7,4,capabilities);
};

int
lldp_add_address(unsigned char* buffer, unsigned size, uint32_t address)
{ 
	unsigned char imgmt[2+4+5+1];
	memset(imgmt,0,sizeof(imgmt));
	imgmt[0]=5;
	imgmt[1]=1;
	memcpy(imgmt+2,&address,4);
	imgmt[6]=1;
	if(size<sizeof(imgmt)+2) 
		return 0;
	return lldp_encode_pdu(buffer,8,sizeof(imgmt),imgmt);
};

#ifdef AF_INET6
int
lldp_add_v6address(unsigned char* buffer, unsigned size, struct sockaddr_in6
	address)
{ 
	if(!IN6_IS_ADDR_UNSPECIFIED(&address.sin6_addr)) { 
		unsigned char i6mgmt[2+16+5+1];
		memset(i6mgmt,0,sizeof(i6mgmt));
		i6mgmt[0]=17;
		i6mgmt[1]=2;
		memcpy(i6mgmt+2,&address.sin6_addr,16);
		i6mgmt[18]=1;
		return lldp_encode_pdu(buffer,8,sizeof(i6mgmt),i6mgmt);
	};
	return 0;
};
#endif

int
lldp_add_eolldp(unsigned char* buffer, unsigned size)
{ 
	if(size<2) 
		return 0;
	return lldp_encode_pdu(buffer,0,0,NULL);
};

int
cdp_add_device_id(unsigned char* buffer, int len)
{ 
	char hostname[128];
	gethostname(hostname,128);

	if((strlen(hostname)+4)>len) return 0;

	*(u_int16_t*)buffer=htons(0x0001); /* type=deviceId */
	*((u_int16_t*)(buffer+2))=htons(strlen(hostname)+4); /* total length */
	memcpy(buffer+4,hostname,strlen(hostname));

	return strlen(hostname)+4;
};

int
cdp_add_address(unsigned char* buffer, int len, u_int32_t addr)
{ 
	if(!addr) return 0;
	if(len<17) return 0;

	sx_write_short(buffer,htons(0x02)); 
	sx_write_short(buffer+2,htons(17)); 
	sx_write_long(buffer+4,htonl(1));
	buffer[8]=1; /* nlpid */
	buffer[9]=1; /* proto length */
	buffer[10]=0xcc; /* proto id: cc==IP */
	sx_write_short(buffer+11,htons(4));
	sx_write_long(buffer+13,addr); /* XXXX! */

	return 17;
};

int
cdp_add_interface(unsigned char* buffer, int len, char* interface)
{ 
	if(!interface) return 0;
	if(len<(strlen(interface)+4)) return 0;

	sx_write_short(buffer,htons(0x0003)); /* type=PortId */
	sx_write_short(buffer+2,htons(strlen(interface)+4)); /* totallength*/
	memcpy(buffer+4,interface,strlen(interface));

	return strlen(interface)+4;
};

int
cdp_add_capabilities(unsigned char* buffer, int len)
{ 
	if(len<8) return 0;

	sx_write_short(buffer,htons(0x0004)); /* type=Capabilities */
	sx_write_short(buffer+2,htons(8)); /* totallength*/
	sx_write_long(buffer+4,htonl(CDP_CAP_HOST)); /* no capabilities */

	return 8;
};

int
cdp_add_software_version(unsigned char* buffer, int len)
{ 
	if((strlen(mysysname)+4)>len) return 0;

	sx_write_short(buffer,htons(0x0005)); /* type=software version */
	sx_write_short(buffer+2,htons(strlen(mysysname)+4)); 
		/* totallength*/
	memcpy(buffer+4,mysysname,strlen(mysysname));

	return strlen(mysysname)+4;
};

int 
cdp_add_platform(unsigned char* buffer, int len)
{ 
	if((strlen(myuname.machine)+4)>len) return 0;
	sx_write_short(buffer,htons(0x0006)); /* type=platform */
	sx_write_short(buffer+2,htons(strlen(myuname.machine)+4)); /* totallength*/
	memcpy(buffer+4,myuname.machine,strlen(myuname.machine));

	return strlen(myuname.machine)+4;
};

static uint16_t 
checksum(uint16_t *buffer, int size)
{
	unsigned long cksum=0;

	while (size > 1) {
		cksum += *buffer++;
		size -= sizeof(uint16_t);
	}

	if (size)
		cksum += *(uint8_t*)buffer;

	cksum = (cksum >> 16) + (cksum & 0xffff);
	cksum += (cksum >>16);

	return (uint16_t)(~cksum);
} 

unsigned short
cdp_checksum(unsigned char *ptr, int length) {
  if (length % 2 == 0) {
    /* The doc says 'standard IP checksum', so this is what we do. */
    return checksum((u_short *)ptr, length);
  } else {
    /* An IP checksum is not defined for an odd number of bytes... */
    /* Tricky. */
    /* Treat the last byte as an unsigned short in network order. */

    int c = ptr[length-1];
    unsigned short *sp = (unsigned short *)(&ptr[length-1]);
    unsigned short ret;

    *sp = htons(c);
    ret = checksum((u_short *)ptr, length+1);
    ptr[length-1] = c;
    return ret;
  };
}

int
usage()
{ 
	printf("Usage: cdpd [-i iface] [-t time] [-dhor] [-c|-l]\n");
	printf("-a      : send announces on all ethernet interfaces with IP"
		" addresses configured\n");
	printf("-c      : do NOT send CDP packets (enabled by default)\n");
	printf("-d      : increase debug level and do not daemonise\n");
	printf("-i name : interface to send cdp packets on (only ethernet ones"
		" supported)\n");
	printf("-h      : this help message\n");
	printf("-o      : run once - send only one packet over each interface "
		"and exit\n");
	printf("-l      : do NOT send LLDP packets (enabled by default)\n");
	printf("-r      : announce host as a Router-capable device (Station by"
		" default)\n");
	printf("-t time : period between announces (60 sec by default)\n");
	printf("\nCopyright (c) 2001-2008, Alexandre Snarskii\n");
	return 0;
};

struct cdp_interface {
	struct cdp_interface* next;
	char* name;
	struct sockaddr_in address;
#ifdef AF_INET6
	struct sockaddr_in6 ipv6address;
#endif
	unsigned char eaddr[6];
	pcap_t* pcap;
	char errbuf[PCAP_ERRBUF_SIZE];
};

struct cdp_interface*
cdp_interface_always(struct cdp_interface* list, const char* name)
{
	while(list) {
		if(list->name && !strcmp(list->name,name)) { 
			return list;
		};
		list=list->next;
	};
	return NULL;
};

struct cdp_interface*
cdp_interface_add(struct cdp_interface** head, const char* name)
{
	struct cdp_interface* cdp;

	if(!name || !head) return NULL;

	if((cdp=cdp_interface_always(*head,name))) return cdp;

	cdp=malloc(sizeof(struct cdp_interface));
	if(!cdp) { 
		fprintf(stderr,"malloc error: %s\n", strerror(errno));
		exit(1);
	};
	memset(cdp,0,sizeof(struct cdp_interface));
	cdp->name=strdup(name);

	cdp->pcap=pcap_open_live(name,0,0,0,cdp->errbuf);
	if(!cdp->pcap) { 
		fprintf(stderr,"Unable to open PCAP on %s: %s\n", name, cdp->errbuf);
		free(cdp);
		return NULL;
	};

	if(pcap_datalink(cdp->pcap)!=DLT_EN10MB) { 
		if(debug)
			fprintf(stderr,"Datalink: %s is not ethernet: %s\n", name,
				pcap_datalink_val_to_description(pcap_datalink(cdp->pcap)));
		pcap_close(cdp->pcap);
		free(cdp);
		return NULL;
	};
	
	if(!*head) { 
		*head=cdp;
	} else { 
		struct cdp_interface* b=*head;
		while(b->next) b=b->next;
		b->next=cdp;
	};
	return cdp;
};

static int
cdp_debug_packet(struct cdp_interface* cifa, int offset, unsigned char* buffer)
{ 
	int i, j;
	printf("Sent over: %s, total length: %i\n", cifa->name, offset);
	for(i=0;i<offset/16;i++) { 
		printf("%4.4x ",i);
		for(j=0;j<16;j++)
			printf("%2.2x ",buffer[16*i+j]);
		for(j=0;j<8;j++) 
			if(isprint(buffer[16*i+j])) 
				printf("%c",buffer[16*i+j]);
			else 
				printf(".");
		printf(" ");
		for(j=8;j<16;j++) 
			if(isprint(buffer[16*i+j])) 
				printf("%c",buffer[16*i+j]);
			else 
				printf(".");

		printf("\n");
	};
	if(offset%16) { 
		i=offset/16;
		printf("%4.4x ",i);
		for(j=0;j<offset%16;j++)
			printf("%2.2x ",buffer[16*i+j]);
		for(j=offset%16; j<16; j++) 
			printf("   ");
		for(j=0;j<(offset%16>8?8:offset%16);j++) 
			if(isprint(buffer[16*i+j])) 
				printf("%c",buffer[16*i+j]);
			else 
				printf(".");
		printf(" ");
		for(j=8;j<offset%16;j++) 
			if(isprint(buffer[16*i+j])) 
				printf("%c",buffer[16*i+j]);
			else 
				printf(".");

		printf("\n");
	};
	return 0;
};
	

int
main(int argc, char* argv[])
{ 
	char c;
	int timeout=60, ret=0;
	unsigned char buffer[1600];
	int offset;
	int once=0, ininited=0, sendlldp=1, sendcdp=1, router=0;
	int	allfaces=0;
	struct cdp_interface *ifaces=NULL;

	while((c=getopt(argc,argv,"i:dt:hoalcr"))!=EOF) { 
	switch(c) { 
		case 'c': sendcdp=0;
			break;
		case 'd': debug++;
			break;
		case 'i': 
			cdp_interface_add(&ifaces,optarg); 
			ininited++;
			break;
		case 't': timeout=atoi(optarg);
			if(timeout<=0) { 
				printf("wrong value to timeout - reverting to default 60 sec\n");
				timeout=60;
			};
			break;
		case 'o': once=1;
			break;
		case 'a': 
			allfaces=1;
			break;
		case 'l': 
			sendlldp=0;
			break;
		case 'r': router=1;
			break;
		default: usage();
			exit(1);
	};
	};

	if(!sendlldp && !sendcdp) { 
		printf("Sorry, -l and -c is mutually exclusive flags\n");
		exit(1);
	};
	if(debug) { 
		printf("starting cdpd compiled with %s\n", pcap_lib_version());
	};

	if(router) { 
#if HAVE_SYSCTLBYNAME
		int fwea;
		size_t fwealen=sizeof(fwea);
#else 
 #if HAVE_PROC_SYS_NET_IPV4_IP_FORWARD
        FILE* f;
 #endif
#endif
		/* we can do routing */
		capabilities[1]=0x10;
#if HAVE_SYSCTLBYNAME
		/* FreeBSD version */
		if(sysctlbyname("net.inet.ip.forwarding",&fwea,&fwealen,NULL,0)==-1) {
			fprintf(stderr,"Unable to get value of net.inet.ip.forwarding: "
				"%s\n", strerror(errno));
		} else { 
			if(fwea) capabilities[3]=0x10; /* routing enabled */
			else { 
#ifdef AF_INET6
				if(sysctlbyname("net.inet6.ip6.forwarding",&fwea,&fwealen,
					NULL,0)==-1) { 
					fprintf(stderr,"Unable to get value of " 
						"net.inet6.ip6.forwarding: %s\n", strerror(errno));
				} else { 
					if(fwea) capabilities[3]=0x10; /* ipv6 routing enabled */
				};
#endif
			};
		};
#else 
#if HAVE_PROC_SYS_NET_IPV4_IP_FORWARD
	/* do not have SYSCTLBYNAME, and at least Linux manpage recommends not
	 * to use sysctl at all... Well, will try linux /proc file 
	 */
	f=fopen("/proc/sys/net/ipv4/ip_forward", "r");
	if(!f) { 
      /* ok, have no way to check actually, declaring itself router */
  	  capabilities[3]=0x10;
	} else { 
      char buffer[10];
      if(fgets(buffer,sizeof(buffer),f)!=NULL) { 
        if(buffer[0]=='1') { 
           capabilities[3]=0x10;
        };
      } else { 
        /* unable to read /proc file ?? well, user thinks we're router */
        capabilities[3]=0x10;
      };
    };
#else /* no /proc file, unable to check */
    capabilities[3]=0x10;
#endif
#endif
	} else { 
		/* this is a station, and station capability is always enabled */
		capabilities[1]=0x80;
		capabilities[3]=0x80;
	};

	if((!ifaces && !ininited) || allfaces) { 
		/* no interfaces given at commandline, so, trying to initialise
			all interfaces.. */
		char errbuf[PCAP_ERRBUF_SIZE];
		pcap_if_t* alldevs=NULL, *dev;
		pcap_findalldevs(&alldevs,errbuf);
		for(dev=alldevs;dev;dev=dev->next) { 
			struct cdp_interface* cdp;
			pcap_addr_t* addr;
#if __linux__ || __CYGWIN__
			int macfound=0;
			if(!strcmp(dev->name,"all")) continue;
#endif
			if(dev->flags&PCAP_IF_LOOPBACK) continue;
			if(!dev->addresses) continue;

			cdp=cdp_interface_add(&ifaces,dev->name);
			if(!cdp) continue; 

			for(addr=dev->addresses; addr; addr=addr->next) { 
				if(addr->addr && addr->addr->sa_family==AF_INET) { 
					if(!cdp->address.sin_addr.s_addr) { 
						/* structure copy */
						cdp->address=*(struct sockaddr_in*)addr->addr;
					};
#ifdef AF_INET6
				} else if(addr->addr && addr->addr->sa_family==AF_INET6) { 
					memcpy(&cdp->ipv6address, addr->addr, 
						sizeof(struct sockaddr_in6));
#endif
#ifdef AF_LINK
				} else if(addr->addr && addr->addr->sa_family==AF_LINK) { 
					struct sockaddr_dl* sdl=(struct sockaddr_dl*)addr->addr;
					if(sdl->sdl_alen!=sizeof(cdp->eaddr)) { 
						continue;
					};
					memcpy(cdp->eaddr,LLADDR(sdl),sdl->sdl_alen);
					if(debug) { 
						printf("%s: %s (BSD method)\n", dev->name, 
#if HAVE_ETHER_NTOA
							ether_ntoa((struct ether_addr*)cdp->eaddr)); 
#else
							my_ether_ntoa(cdp->eaddr));
#endif
					};
#else
#ifdef SIOCGIFHWADDR		
				};
				if(!macfound) { 
					struct ifreq ifr;
					int sockfd=socket(AF_INET,SOCK_DGRAM,0);
					if(sockfd<0) { 
						fprintf(stderr,"socket: %s\n",strerror(errno));
						continue;
					};
					memset(&ifr, 0, sizeof(struct ifreq));
					strlcpy(ifr.ifr_name,dev->name,sizeof(ifr.ifr_name));
					if(!strncmp(dev->name, "\\Device\\NPF_", 12)) { 
						/* hack around windows names */
						strlcpy(ifr.ifr_name, dev->name+12, 
							sizeof(ifr.ifr_name));
					};
					if(ioctl(sockfd,SIOCGIFHWADDR,&ifr)<0) { 
						fprintf(stderr,"ioctl(SIOCGIFHWADDR): %s (name: %s,"
							"oname: %s\n", strerror(errno), ifr.ifr_name,
							dev->name);
						continue;
					};
					memcpy(cdp->eaddr,ifr.ifr_ifru.ifru_hwaddr.sa_data,6);
					close(sockfd);
					macfound=1;
					if(debug) { 
						printf("%s: %s (Linux method)\n", dev->name, 
#if HAVE_ETHER_NTOA
							ether_ntoa((struct ether_addr*)cdp->eaddr)); 
#else
							my_ether_ntoa(cdp->eaddr));
#endif
					};
#else
#error "Don't known how to detect mac address"
#endif
#endif
				};
			};
		};

		pcap_freealldevs(alldevs);
	};

	if(!ifaces) { 
		printf("No valid interfaces found, exiting..\n");
		exit(1);
	};

	uname(&myuname);
	snprintf(mysysname,sizeof(mysysname),"%s %s %s",
		myuname.sysname, myuname.release, myuname.version);

#ifndef __CYGWIN__
	if(!debug && !once) 
		daemon(0,0);
#endif

	while(1) { 
		struct cdp_interface* cifa=ifaces;
		while(cifa) { 
			if(sendcdp) { 
				offset=0;
				offset=cdp_buffer_init(buffer,sizeof(buffer),cifa->eaddr);
		
				offset+=cdp_add_device_id(buffer+offset,sizeof(buffer)-offset);
				offset+=cdp_add_address(buffer+offset,sizeof(buffer)-offset,
					cifa->address.sin_addr.s_addr);
				offset+=cdp_add_interface(buffer+offset,sizeof(buffer)-offset,
					cifa->name);
				offset+=cdp_add_capabilities(buffer+offset,
					sizeof(buffer)-offset);
				offset+=cdp_add_software_version(buffer+offset,
					sizeof(buffer)-offset);
	
				offset+=cdp_add_platform(buffer+offset,sizeof(buffer)-offset);
		
				((struct cdp_header*)buffer)->length=htons(offset-14);
			
				*(u_short*)(buffer+sizeof(struct cdp_header)+2)=cdp_checksum(
					buffer+sizeof(struct cdp_header),
					offset-sizeof(struct cdp_header));
				if(pcap_inject(cifa->pcap,buffer,offset)!=offset){
					printf("%s: wrote only %i bytes: %s\n", cifa->name, ret, 
						strerror(errno));
				};
				if(debug>1) { 
					cdp_debug_packet(cifa,offset,buffer);
				};
			};
			if(sendlldp) { 
				offset=0;
				offset=lldp_buffer_init(buffer,sizeof(buffer),cifa->eaddr);
				offset+=lldp_add_interface(buffer+offset,sizeof(buffer)-offset,
					cifa->name);
				offset+=lldp_add_ttl(buffer+offset,sizeof(buffer)-offset,
					timeout*3);
				offset+=lldp_add_ifname(buffer+offset,sizeof(buffer)-offset,
					(unsigned char*)cifa->name);
				offset+=lldp_add_hostname(buffer+offset,sizeof(buffer)-offset);
				offset+=lldp_add_sysdescr(buffer+offset,sizeof(buffer)-offset);
				offset+=lldp_add_capabilities(buffer+offset,
					sizeof(buffer)-offset);
				offset+=lldp_add_address(buffer+offset,sizeof(buffer)-offset,
					cifa->address.sin_addr.s_addr);
#ifdef AF_INET6
				offset+=lldp_add_v6address(buffer+offset,sizeof(buffer)-offset,
					cifa->ipv6address);
#endif
				offset+=lldp_add_eolldp(buffer+offset,sizeof(buffer)-offset);
				if(pcap_inject(cifa->pcap,buffer,offset)!=offset){
					printf("error writing to %s: %s\n", cifa->name, 
						strerror(errno));
				};
				if(debug>1) { 
					cdp_debug_packet(cifa,offset,buffer);
				};
			};
			cifa=cifa->next;
		};  /* all interfaces done */
		if(once) 
			return 0;
		sleep(timeout);
	};
	return 0;
};
