/*
  This file is part of CDO. CDO is a collection of Operators to manipulate and analyse Climate model Data.

  Author: Uwe Schulzweida

*/

/*
   This module contains the following operators:

      Derivepar     gheight           Geopotential height
      Derivepar     sealevelpressure  Sea level pressure
*/

#include <cdi.h>

#include "process_int.h"
#include "cdo_vlist.h"
#include "vertical_interp.h"
#include "stdnametable.h"
#include "util_string.h"
#include "const.h"
#include "cdo_zaxis.h"
#include "cdo_options.h"

void MakeGeopotHeight(double *geop, double *gt, double *gq, double *ph, int nhor, int nlev);

void vlist_hybrid_vct(int vlistID, int &rzaxisIDh, int &rnvct, Varray<double> &vct, int &rnhlevf);

void *
Derivepar(void *process)
{
  ModelMode mode(ModelMode::UNDEF);
  int varID, levelID;
  int surfaceID = -1;
  int sgeopotID = -1, geopotID = -1, tempID = -1, humID = -1, psID = -1, lnpsID = -1, presID = -1, gheightID = -1;
  int lnpsID2 = -1;
  // int clwcID = -1, ciwcID = -1;
  char paramstr[32];
  char varname[CDI_MAX_NAME], stdname[CDI_MAX_NAME];
  // double *lwater = nullptr, *iwater = nullptr;
  size_t nmiss, nmissout = 0;
  gribcode_t gribcodes;

  cdo_initialize(process);

  // clang-format off
  const auto GHEIGHT          = cdo_operator_add("gheight",            0, 0, nullptr);
  const auto SEALEVELPRESSURE = cdo_operator_add("sealevelpressure",   0, 0, nullptr);
  // clang-format on

  const auto operatorID = cdo_operator_id();

  operator_check_argc(0);

  const auto streamID1 = cdo_open_read(0);
  const auto vlistID1 = cdo_stream_inq_vlist(streamID1);

  auto gridID0 = vlistGrid(vlistID1, 0);
  if (gridInqType(gridID0) == GRID_SPECTRAL) cdo_abort("Spectral data unsupported!");

  auto gridsize = vlist_check_gridsize(vlistID1);

  int zaxisIDh = -1;
  int nvct = 0;
  int nhlevf = 0;
  Varray<double> vct;
  vlist_hybrid_vct(vlistID1, zaxisIDh, nvct, vct, nhlevf);

  if (Options::cdoVerbose)
    for (int i = 0; i < nvct / 2; ++i) cdo_print("vct: %5d %25.17f %25.17f", i, vct[i], vct[nvct / 2 + i]);

  if (zaxisIDh == -1) cdo_abort("No 3D variable with hybrid sigma pressure coordinate found!");

  VarList varList1;
  varListInit(varList1, vlistID1);
  varListSetUniqueMemtype(varList1);

  const auto nvars = vlistNvars(vlistID1);

  auto useTable = false;
  for (varID = 0; varID < nvars; varID++)
    {
      const auto tableNum = tableInqNum(vlistInqVarTable(vlistID1, varID));
      if (tableNum > 0 && tableNum != 255)
        {
          useTable = true;
          break;
        }
    }

  if (Options::cdoVerbose && useTable) cdo_print("Using code tables!");

  for (varID = 0; varID < nvars; varID++)
    {
      const auto gridID = varList1[varID].gridID;
      const auto zaxisID = varList1[varID].zaxisID;
      const auto nlevels = varList1[varID].nlevels;
      const auto instNum = institutInqCenter(vlistInqVarInstitut(vlistID1, varID));
      const auto tableNum = tableInqNum(vlistInqVarTable(vlistID1, varID));

      auto code = varList1[varID].code;

      const auto param = varList1[varID].param;
      cdiParamToString(param, paramstr, sizeof(paramstr));
      int pnum, pcat, pdis;
      cdiDecodeParam(param, &pnum, &pcat, &pdis);
      if (pdis >= 0 && pdis < 255) code = -1;

      if (useTable)
        {
          if (tableNum == 2)
            {
              mode = ModelMode::WMO;
              wmo_gribcodes(&gribcodes);
            }
          else if (tableNum == 128 || tableNum == 0)
            {
              mode = ModelMode::ECHAM;
              echam_gribcodes(&gribcodes);
            }
          //  KNMI: HIRLAM model version 7.2 uses tableNum=1    (LAMH_D11*)
          //  KNMI: HARMONIE model version 36 uses tableNum=1   (grib*)
          //  (opreational NWP version) KNMI: HARMONIE model version 38 uses tableNum=253 (grib,grib_md)
          //   and tableNum=1 (grib_sfx) (research version)
          else if (tableNum == 1 || tableNum == 253)
            {
              mode = ModelMode::HIRLAM;
              hirlam_harmonie_gribcodes(&gribcodes);
            }
        }
      else
        {
          mode = ModelMode::ECHAM;
          echam_gribcodes(&gribcodes);
        }

      if (Options::cdoVerbose) cdo_print("Mode = %d  Center = %d  Param = %s", static_cast<int>(mode), instNum, paramstr);

      if (code <= 0 || code == 255)
        {
          vlistInqVarName(vlistID1, varID, varname);
          cstr_to_lower_case(varname);

          int length = CDI_MAX_NAME;
          cdiInqKeyString(vlistID1, varID, CDI_KEY_STDNAME, stdname, &length);
          cstr_to_lower_case(stdname);

          code = echamcode_from_stdname(stdname);
          if (code < 0)
            {
              // clang-format off
              if      (-1 == sgeopotID && cdo_cmpstr(varname, "geosp")) code = gribcodes.geopot;
              else if (-1 == psID      && cdo_cmpstr(varname, "aps")) code = gribcodes.ps;
              else if (-1 == psID      && cdo_cmpstr(varname, "ps")) code = gribcodes.ps;
              else if (-1 == lnpsID    && cdo_cmpstr(varname, "lsp")) code = gribcodes.lsp;
              else if (-1 == lnpsID2   && cdo_cmpstr(varname, "lnps")) code = 777;
              else if (-1 == tempID    && cdo_cmpstr(varname, "t")) code = gribcodes.temp;
              else if (-1 == humID     && cdo_cmpstr(varname, "q")) code = gribcodes.hum;
              // else if ( geopotID  == -1 && cdo_cmpstr(stdname, "geopotential_full")) code = gribcodes.geopot;
              // else if (cdo_cmpstr(varname, "clwc")) code = 246;
              // else if (cdo_cmpstr(varname, "ciwc")) code = 247;
              // clang-format on
            }
        }

      // clang-format off
      if      (code == gribcodes.geopot  && nlevels == 1) sgeopotID = varID;
      else if (code == gribcodes.geopot  && nlevels == nhlevf) geopotID = varID;
      else if (code == gribcodes.temp    && nlevels == nhlevf) tempID = varID;
      else if (code == gribcodes.hum     && nlevels == nhlevf) humID = varID;
      else if (code == gribcodes.ps      && nlevels == 1) psID = varID;
      else if (code == gribcodes.lsp     && nlevels == 1) lnpsID = varID;
      else if (code == 777               && nlevels == 1) lnpsID2 = varID;
      else if (code == gribcodes.gheight && nlevels == nhlevf) gheightID = varID;
      // else if (code == 246) clwcID    = varID;
      // else if (code == 247) ciwcID    = varID;
      // clang-format on

      if (operatorID == SEALEVELPRESSURE) humID = -1;

      if (gridInqType(gridID) == GRID_SPECTRAL && zaxisInqType(zaxisID) == ZAXIS_HYBRID)
        cdo_abort("Spectral data on model level unsupported!");

      if (gridInqType(gridID) == GRID_SPECTRAL) cdo_abort("Spectral data unsupported!");
    }

  if (Options::cdoVerbose)
    {
      cdo_print("Found:");
      // clang-format off
      if (-1 != humID)     cdo_print("  %s -> %s", var_stdname(specific_humidity), varList1[humID].name);
      if (-1 != tempID)    cdo_print("  %s -> %s", var_stdname(air_temperature), varList1[tempID].name);
      if (-1 != psID)      cdo_print("  %s -> %s", var_stdname(surface_air_pressure), varList1[psID].name);
      if (-1 != lnpsID)    cdo_print("  LOG(%s) -> %s", var_stdname(surface_air_pressure), varList1[lnpsID].name);
      if (-1 != sgeopotID) cdo_print("  %s -> %s", var_stdname(surface_geopotential), varList1[sgeopotID].name);
      if (-1 != geopotID)  cdo_print("  %s -> %s", var_stdname(geopotential), varList1[geopotID].name);
      if (-1 != gheightID) cdo_print("  %s -> %s", var_stdname(geopotential_height), varList1[gheightID].name);
      // clang-format on
    }

  if (lnpsID != -1 && lnpsID2 != -1) cdo_abort("Found LOG(%s) twice: lsp and lnps!", var_stdname(surface_air_pressure));

  if (tempID == -1) cdo_abort("%s not found!", var_stdname(air_temperature));

  Varray<double> array(gridsize);
  Varray<double> sgeopot(gridsize), ps(gridsize);
  Varray<double> temp(gridsize * nhlevf);
  Varray<double> halfPress(gridsize * (nhlevf + 1));
  Varray<double> hum, gheight;
  if (operatorID == GHEIGHT)
    {
      if (humID == -1)
        cdo_warning("%s not found - using algorithm without %s!", var_stdname(specific_humidity), var_stdname(specific_humidity));
      else
        hum.resize(gridsize * nhlevf);

      gheight.resize(gridsize * (nhlevf + 1));
    }

  Varray<double> fullPress, sealevelpressure;
  if (operatorID == SEALEVELPRESSURE)
    {
      fullPress.resize(gridsize * nhlevf);

      surfaceID = zaxis_from_name("surface");
      sealevelpressure.resize(gridsize);
    }

  if (zaxisIDh != -1 && sgeopotID == -1)
    {
      if (geopotID == -1)
        cdo_warning("%s not found - set to zero!", var_stdname(surface_geopotential));
      else
        cdo_print("%s not found - using bottom layer of %s!", var_stdname(surface_geopotential), var_stdname(geopotential));

      varray_fill(sgeopot, 0.0);
    }

  presID = lnpsID;
  if (zaxisIDh != -1 && lnpsID == -1)
    {
      if (psID == -1)
        cdo_abort("%s not found!", var_stdname(surface_air_pressure));
      else
        presID = psID;
    }

  if (Options::cdoVerbose)
    {
      if (presID == lnpsID)
        cdo_print("using LOG(%s)", var_stdname(surface_air_pressure));
      else
        cdo_print("using %s", var_stdname(surface_air_pressure));
    }

  const auto vlistID2 = vlistCreate();
  vlistDefNtsteps(vlistID2, vlistNtsteps(vlistID1));

  int var_id = -1;

  if (operatorID == GHEIGHT)
    {
      var_id = geopotential_height;
      varID = vlistDefVar(vlistID2, gridID0, zaxisIDh, TIME_VARYING);
    }
  else if (operatorID == SEALEVELPRESSURE)
    {
      var_id = air_pressure_at_sea_level;
      varID = vlistDefVar(vlistID2, gridID0, surfaceID, TIME_VARYING);
    }
  else
    cdo_abort("Internal problem, invalid operatorID: %d!", operatorID);

  vlistDefVarParam(vlistID2, varID, cdiEncodeParam(var_echamcode(var_id), 128, 255));
  cdiDefKeyString(vlistID2, varID, CDI_KEY_NAME, var_name(var_id));
  cdiDefKeyString(vlistID2, varID, CDI_KEY_STDNAME, var_stdname(var_id));
  cdiDefKeyString(vlistID2, varID, CDI_KEY_UNITS, var_units(var_id));

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  const auto streamID2 = cdo_open_write(1);
  cdo_def_vlist(streamID2, vlistID2);

  int tsID = 0;
  while (true)
    {
      const auto nrecs = cdo_stream_inq_timestep(streamID1, tsID);
      if (nrecs == 0) break;

      cdo_taxis_copy_timestep(taxisID2, taxisID1);
      cdo_def_timestep(streamID2, tsID);

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdo_inq_record(streamID1, &varID, &levelID);
          cdo_read_record(streamID1, array.data(), &nmiss);

          const auto offset = gridsize * levelID;

          if (zaxisIDh != -1)
            {
              if (varID == sgeopotID)
                {
                  varray_copy(gridsize, array, sgeopot);
                }
              else if (varID == geopotID && sgeopotID == -1 && (levelID + 1) == nhlevf)
                {
                  varray_copy(gridsize, array, sgeopot);
                }
              else if (varID == presID)
                {
                  if (lnpsID != -1)
                    for (size_t i = 0; i < gridsize; ++i) ps[i] = std::exp(array[i]);
                  else if (psID != -1)
                    varray_copy(gridsize, array, ps);
                }
              else if (varID == tempID)
                array_copy(gridsize, array.data(), &temp[offset]);
              else if (varID == humID)
                array_copy(gridsize, array.data(), &hum[offset]);
            }
        }

      if (zaxisIDh != -1)
        {
          // check range of psProg
          auto mm = varray_min_max(ps);
          if (mm.min < MIN_PS || mm.max > MAX_PS) cdo_warning("Surface pressure out of range (min=%g max=%g)!", mm.min, mm.max);

          // check range of surface geopot
          mm = varray_min_max(sgeopot);
          if (mm.min < MIN_FIS || mm.max > MAX_FIS) cdo_warning("Orography out of range (min=%g max=%g)!", mm.min, mm.max);
        }

      {
        static auto lwarn = true;
        if (lwarn)
          {
            lwarn = false;
            double minVal = 1.e33, maxVal = -1.e33;
            for (levelID = 0; levelID < varList1[tempID].nlevels; levelID++)
              {
                const auto mm = varray_min_max(gridsize, &temp[gridsize * levelID]);
                minVal = std::min(minVal, mm.min);
                maxVal = std::max(maxVal, mm.max);
              }
            if (minVal < MIN_T || maxVal > MAX_T) cdo_warning("Temperature out of range (min=%g max=%g)!", minVal, maxVal);
          }
      }

      if (humID != -1)
        {
          static auto lwarn = true;
          if (lwarn)
            {
              lwarn = false;
              double minVal = 1.e33, maxVal = -1.e33;
              for (levelID = 0; levelID < varList1[humID].nlevels; levelID++)
                {
                  const auto mm = varray_min_max(gridsize, &hum[gridsize * levelID]);
                  minVal = std::min(minVal, mm.min);
                  maxVal = std::max(maxVal, mm.max);
                }
              if (minVal < -0.1 || maxVal > MAX_Q) cdo_warning("Humidity out of range (min=%g max=%g)!", minVal, maxVal);
            }
        }

      if (operatorID == GHEIGHT)
        {
          vct_to_hybrid_pressure((double *) nullptr, halfPress.data(), vct.data(), ps.data(), nhlevf, gridsize);
          array_copy(gridsize, sgeopot.data(), gheight.data() + gridsize * nhlevf);
          MakeGeopotHeight(gheight.data(), temp.data(), hum.data(), halfPress.data(), gridsize, nhlevf);

          nmissout = 0;
          varID = 0;
          const auto nlevels = nhlevf;
          for (levelID = 0; levelID < nlevels; levelID++)
            {
              cdo_def_record(streamID2, varID, levelID);
              cdo_write_record(streamID2, gheight.data() + levelID * gridsize, nmissout);
            }
        }
      else if (operatorID == SEALEVELPRESSURE)
        {
          vct_to_hybrid_pressure(fullPress.data(), halfPress.data(), vct.data(), ps.data(), nhlevf, gridsize);

          extrapolate_P(sealevelpressure.data(), &halfPress[gridsize * (nhlevf)], &fullPress[gridsize * (nhlevf - 1)],
                        sgeopot.data(), &temp[gridsize * (nhlevf - 1)], gridsize);

          cdo_def_record(streamID2, 0, 0);
          cdo_write_record(streamID2, sealevelpressure.data(), 0);
        }
      else
        cdo_abort("Internal error");

      tsID++;
    }

  cdo_stream_close(streamID2);
  cdo_stream_close(streamID1);

  vlistDestroy(vlistID2);

  cdo_finish();

  return nullptr;
}
