/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

#ifdef HAVE_CONFIG_H
#include "config.h" // restrict
#endif

#include <float.h>
#include <fenv.h>
#include <assert.h>

#include "compare.h"
#include "array.h"

//#pragma STDC FENV_ACCESS ON

const char *
fpe_errstr(int fpeRaised)
{
  const char *errstr = nullptr;

  // clang-format off
  if      (fpeRaised & FE_DIVBYZERO) errstr = "division by zero";
  else if (fpeRaised & FE_INEXACT)   errstr = "inexact result";
  else if (fpeRaised & FE_INVALID)   errstr = "invalid result";
  else if (fpeRaised & FE_OVERFLOW)  errstr = "overflow";
  else if (fpeRaised & FE_UNDERFLOW) errstr = "underflow";
  // clang-format on

  return errstr;
}

std::pair<double, double>
varrayMinMaxMV(const size_t len, const double *array, const double missval, size_t &nvals)
{
  double vmin = DBL_MAX;
  double vmax = -DBL_MAX;

  nvals = 0;
  for (size_t i = 0; i < len; ++i)
    {
      if (!DBL_IS_EQUAL(array[i], missval))
        {
          if (array[i] < vmin) vmin = array[i];
          if (array[i] > vmax) vmax = array[i];
          nvals++;
        }
    }

  return std::pair<double, double>(vmin, vmax);
}

std::pair<double, double>
varrayMinMaxMV(const size_t len, const double *array, const double missval)
{
  size_t nvals;
  return varrayMinMaxMV(len, array, missval, nvals);
}

std::pair<double, double>
varrayMinMaxMV(const size_t len, const Varray<double> &array, const double missval)
{
  size_t nvals;
  return varrayMinMaxMV(len, array.data(), missval, nvals);
}

void
varrayMinMaxSum(const size_t len, const Varray<double> &array, double &rmin, double &rmax, double &rsum)
{
  // rmin, rmax and rsum will be initialized in Info

  // #pragma omp parallel for default(none) shared(min, max, array, gridsize)
  // reduction(+:mean) #pragma omp simd reduction(+:mean) reduction(min:min)
  // reduction(max:max) aligned(array:16)
  for (size_t i = 0; i < len; ++i)
    {
      if (array[i] < rmin) rmin = array[i];
      if (array[i] > rmax) rmax = array[i];
      rsum += array[i];
    }
}

void
varrayMinMaxSum_f(const size_t len, const Varray<float> &array, double &rmin, double &rmax, double &rsum)
{
  // rmin, rmax and rsum will be initialized in Info

  // #pragma omp parallel for default(none) shared(min, max, array, gridsize)
  // reduction(+:mean) #pragma omp simd reduction(+:mean) reduction(min:min)
  // reduction(max:max) aligned(array:16)
  for (size_t i = 0; i < len; ++i)
    {
      if (array[i] < rmin) rmin = array[i];
      if (array[i] > rmax) rmax = array[i];
      rsum += array[i];
    }
}

size_t
varrayMinMaxSumMV(const size_t len, const Varray<double> &array, const double missval, double &rmin, double &rmax, double &rsum)
{
  // rmin, rmax and rsum will be initialized in Info

  size_t nvals = 0;
  for (size_t i = 0; i < len; ++i)
    {
      if (!DBL_IS_EQUAL(array[i], missval))
        {
          if (array[i] < rmin) rmin = array[i];
          if (array[i] > rmax) rmax = array[i];
          rsum += array[i];
          nvals++;
        }
    }

  if (nvals == 0 && IS_EQUAL(rmin, DBL_MAX)) rmin = missval;
  if (nvals == 0 && IS_EQUAL(rmax, -DBL_MAX)) rmax = missval;

  return nvals;
}

size_t
varrayMinMaxSumMV_f(const size_t len, const Varray<float> &array, const double missval, double &rmin, double &rmax, double &rsum)
{
  // rmin, rmax and rsum will be initialized in Info

  size_t nvals = 0;
  for (size_t i = 0; i < len; ++i)
    {
      if (!DBL_IS_EQUAL(array[i], missval))
        {
          if (array[i] < rmin) rmin = array[i];
          if (array[i] > rmax) rmax = array[i];
          rsum += array[i];
          nvals++;
        }
    }

  if (nvals == 0 && IS_EQUAL(rmin, DBL_MAX)) rmin = missval;
  if (nvals == 0 && IS_EQUAL(rmax, -DBL_MAX)) rmax = missval;

  return nvals;
}

void
varrayMinMaxMean(const size_t len, const Varray <double> &array, double &rmin, double &rmax, double &rmean)
{
  rmin = DBL_MAX;
  rmax = -DBL_MAX;
  double sum = 0.0;
  varrayMinMaxSum(len, array, rmin, rmax, sum);

  rmean = len ? sum / (double) len : 0;
}

size_t
varrayMinMaxMeanMV(const size_t len, const Varray <double> &array, const double missval, double &rmin, double &rmax, double &rmean)
{
  rmin = DBL_MAX;
  rmax = -DBL_MAX;
  double sum = 0.0;
  size_t nvals = varrayMinMaxSumMV(len, array, missval, rmin, rmax, sum);

  rmean = nvals ? sum / (double) nvals : missval;

  return nvals;
}

void
arrayMinMaxMask(const size_t len, const double *array, const Varray<int> &mask, double &rmin, double &rmax)
{
  double zmin = DBL_MAX;
  double zmax = -DBL_MAX;

  if (!mask.empty())
    {
      for (size_t i = 0; i < len; ++i)
        {
          if (!mask[i])
            {
              if (array[i] < zmin) zmin = array[i];
              if (array[i] > zmax) zmax = array[i];
            }
        }
    }
  else
    {
      for (size_t i = 0; i < len; ++i)
        {
          if (array[i] < zmin) zmin = array[i];
          if (array[i] > zmax) zmax = array[i];
        }
    }

  rmin = zmin;
  rmax = zmax;
}

void
arrayAddArray(const size_t len, double *restrict array1, const double *restrict array2)
{
  //#ifdef  _OPENMP
  //#pragma omp parallel for default(none) shared(array1,array2)
  //#endif
  for (size_t i = 0; i < len; ++i) array1[i] += array2[i];
}

void
arrayAddArrayMV(const size_t len, double *restrict array1, const double *restrict array2, const double missval)
{
  if (std::isnan(missval))
    {
      for (size_t i = 0; i < len; i++)
        if (!DBL_IS_EQUAL(array2[i], missval))
          {
            if (!DBL_IS_EQUAL(array1[i], missval))
              array1[i] += array2[i];
            else
              array1[i] = array2[i];
          }
    }
  else
    {
      for (size_t i = 0; i < len; i++)
        if (IS_NOT_EQUAL(array2[i], missval))
          {
            if (IS_NOT_EQUAL(array1[i], missval))
              array1[i] += array2[i];
            else
              array1[i] = array2[i];
          }
    }
}

size_t
arrayNumMV(const size_t len, const double *restrict array, const double missval)
{
  size_t nmiss = 0;

  if (std::isnan(missval))
    {
      for (size_t i = 0; i < len; ++i)
        if (DBL_IS_EQUAL(array[i], missval)) nmiss++;
    }
  else
    {
      for (size_t i = 0; i < len; ++i)
        if (IS_EQUAL(array[i], missval)) nmiss++;
    }

  return nmiss;
}

size_t
varrayNumMV(const size_t len, const Varray<double> &array, const double missval)
{
  return arrayNumMV(len, array.data(), missval);
}

size_t
varrayNumMV_f(const size_t len, const Varray<float> &array, const double missval)
{
  size_t nmiss = 0;

  if (std::isnan(missval))
    {
      for (size_t i = 0; i < len; ++i)
        if (DBL_IS_EQUAL(array[i], missval)) nmiss++;
    }
  else
    {
      for (size_t i = 0; i < len; ++i)
        if (IS_EQUAL(array[i], missval)) nmiss++;
    }

  return nmiss;
}

std::pair<double, double>
varrayMinMax(const size_t len, const double *restrict array)
{
  double vmin = DBL_MAX;
  double vmax = -DBL_MAX;

  // #pragma omp parallel for default(none) shared(min, max, array, gridsize)
  // reduction(+:mean) #pragma omp simd reduction(+:mean) reduction(min:min)
  // reduction(max:max) aligned(array:16)
  for (size_t i = 0; i < len; ++i)
    {
      if (array[i] < vmin) vmin = array[i];
      if (array[i] > vmax) vmax = array[i];
    }

  return std::pair<double, double>(vmin, vmax);
}

std::pair<double, double>
varrayMinMax(const size_t len, const Varray<double> &v)
{
  return varrayMinMax(len, v.data());
}

std::pair<double, double>
varrayMinMax(const Varray<double> &v)
{ 
  return varrayMinMax(v.size(), v.data());
}

double
varrayMin(const size_t len, const Varray<double> &v)
{
  assert(v.size() >= len);

  double min = v[0];

  for (size_t i = 0; i < len; ++i)
    if (v[i] < min) min = v[i];

  return min;
}

double
varrayMax(const size_t len, const Varray<double> &v)
{
  assert(v.size() >= len);

  double max = v[0];

  for (size_t i = 0; i < len; ++i)
    if (v[i] > max) max = v[i];

  return max;
}

double
varrayRange(const size_t len, const Varray<double> &v)
{
  assert(v.size() >= len);

  double min = v[0];
  double max = v[0];

  for (size_t i = 0; i < len; ++i)
    {
      if (v[i] < min) min = v[i];
      if (v[i] > max) max = v[i];
    }

  return (max - min);
}

double
varrayMinMV(const size_t len, const Varray<double> &v, const double missval)
{
  assert(v.size() >= len);

  double min = DBL_MAX;

  for (size_t i = 0; i < len; ++i)
    if (!DBL_IS_EQUAL(v[i], missval))
      if (v[i] < min) min = v[i];

  if (IS_EQUAL(min, DBL_MAX)) min = missval;

  return min;
}

double
varrayMaxMV(const size_t len, const Varray<double> &v, const double missval)
{
  assert(v.size() >= len);

  double max = -DBL_MAX;

  for (size_t i = 0; i < len; ++i)
    if (!DBL_IS_EQUAL(v[i], missval))
      if (v[i] > max) max = v[i];

  if (IS_EQUAL(max, -DBL_MAX)) max = missval;

  return max;
}

double
varrayRangeMV(const size_t len, const Varray<double> &v, const double missval)
{
  assert(v.size() >= len);

  double min = DBL_MAX;
  double max = -DBL_MAX;

  for (size_t i = 0; i < len; ++i)
    {
      if (!DBL_IS_EQUAL(v[i], missval))
        {
          if (v[i] < min) min = v[i];
          if (v[i] > max) max = v[i];
        }
    }

  double range;
  if (IS_EQUAL(min, DBL_MAX) && IS_EQUAL(max, -DBL_MAX))
    range = missval;
  else
    range = max - min;

  return range;
}

double
varraySum(const size_t len, const Varray<double> &v)
{
  assert(v.size() >= len);

  double sum = 0;

  for (size_t i = 0; i < len; ++i) sum += v[i];

  return sum;
}

double
varraySumMV(const size_t len, const Varray<double> &v, const double missval)
{
  assert(v.size() >= len);

  double sum = 0;
  size_t nvals = 0;

  if (std::isnan(missval))
    {
      for (size_t i = 0; i < len; ++i)
        if (!DBL_IS_EQUAL(v[i], missval))
          {
            sum += v[i];
            nvals++;
          }
    }
  else
    {
      for (size_t i = 0; i < len; ++i)
        if (IS_NOT_EQUAL(v[i], missval))
          {
            sum += v[i];
            nvals++;
          }
    }

  if (!nvals) sum = missval;

  return sum;
}

double
varrayMean(const size_t len, const Varray<double> &v)
{
  assert(v.size() >= len);

  const double sum = varraySum(len, v);

  return sum / len;
}

double
varrayMeanMV(const size_t len, const Varray<double> &v, const double missval)
{
  assert(v.size() >= len);

  double sum = 0, sumw = 0;

  for (size_t i = 0; i < len; ++i)
    if (!DBL_IS_EQUAL(v[i], missval))
      {
        sum += v[i];
        sumw += 1;
      }

  double missval1 = missval, missval2 = missval;
  return DIVMN(sum, sumw);
}

double
varrayWeightedMean(const size_t len, const Varray<double> &v, const Varray<double> &w, const double missval)
{
  assert(v.size() >= len);
  assert(w.size() >= len);

  double sum = 0, sumw = 0;

  for (size_t i = 0; i < len; ++i)
    {
      sum += w[i] * v[i];
      sumw += w[i];
    }

  return IS_EQUAL(sumw, 0.) ? missval : sum / sumw;
}

double
varrayWeightedMeanMV(const size_t len, const Varray<double> &v, const Varray<double> &w, const double missval)
{
  assert(v.size() >= len);
  assert(w.size() >= len);

  const double missval1 = missval, missval2 = missval;
  double sum = 0, sumw = 0;

  for (size_t i = 0; i < len; ++i)
    if (!DBL_IS_EQUAL(v[i], missval1) && !DBL_IS_EQUAL(w[i], missval1))
      {
        sum += w[i] * v[i];
        sumw += w[i];
      }

  return DIVMN(sum, sumw);
}

double
varrayAvgMV(const size_t len, const Varray<double> &v, const double missval)
{
  assert(v.size() >= len);

  const double missval1 = missval, missval2 = missval;
  double sum = 0, sumw = 0;

  for (size_t i = 0; i < len; ++i)
    {
      sum = ADDMN(sum, v[i]);
      sumw += 1;
    }

  return DIVMN(sum, sumw);
}

double
varrayWeightedAvgMV(const size_t len, const Varray<double> &v, const Varray<double> &w, const double missval)
{
  assert(v.size() >= len);
  assert(w.size() >= len);

  const double missval1 = missval, missval2 = missval;
  double sum = 0, sumw = 0;

  for (size_t i = 0; i < len; ++i)
    if (!DBL_IS_EQUAL(w[i], missval))
      {
        sum = ADDMN(sum, MULMN(w[i], v[i]));
        sumw = ADDMN(sumw, w[i]);
      }

  return DIVMN(sum, sumw);
}
