/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

*/

#include <cdi.h>

#include "process_int.h"
#include <mpim_grid.h>
#include "cdi_lockedIO.h"


static void
varrms(const Varray<double> w, const FieldVector &field1, const FieldVector &field2, Field &field3)
{
  auto grid1 = field1[0].grid;
  auto grid2 = field2[0].grid;
  const auto missval1 = field1[0].missval;
  const auto missval2 = field2[0].missval;
  double rsum = 0, rsumw = 0;

  const auto nlev = field1.size();
  const auto len = gridInqSize(grid1);
  if (len != gridInqSize(grid2)) cdoAbort("fields have different size!");

  /*
  if ( nmiss1 )
  */
  {
    for (size_t k = 0; k < nlev; k++)
      {
        const auto array1 = field1[k].vec;
        const auto array2 = field2[k].vec;
        for (size_t i = 0; i < len; i++) /*	  if ( !DBL_IS_EQUAL(w[i], missval1) ) */
          {
            rsum = ADDMN(rsum, MULMN(w[i], MULMN(SUBMN(array2[i], array1[i]), SUBMN(array2[i], array1[i]))));
            rsumw = ADDMN(rsumw, w[i]);
          }
      }
  }
  /*
else
  {
    for ( i = 0; i < len; i++ )
      {
        rsum  += w[i] * array1[i];
        rsumw += w[i];
      }
  }
  */

  const double ravg = SQRTMN(DIVMN(rsum, rsumw));

  size_t rnmiss = 0;
  if (DBL_IS_EQUAL(ravg, missval1)) rnmiss++;

  field3.vec[0] = ravg;
  field3.nmiss = rnmiss;
}

void *
Varrms(void *process)
{
  int lastgrid = -1;
  int oldcode = 0;
  int nrecs;
  size_t nmiss;
  int varID, levelID;

  cdoInitialize(process);

  const auto needWeights = true;

  const auto streamID1 = cdoOpenRead(0);
  const auto streamID2 = cdoOpenRead(1);

  const auto vlistID1 = cdoStreamInqVlist(streamID1);
  const auto vlistID2 = cdoStreamInqVlist(streamID2);

  double slon = 0;
  double slat = 0;
  const auto gridID3 = gridCreate(GRID_LONLAT, 1);
  gridDefXsize(gridID3, 1);
  gridDefYsize(gridID3, 1);
  gridDefXvals(gridID3, &slon);
  gridDefYvals(gridID3, &slat);

  vlistClearFlag(vlistID1);
  const auto nvars = vlistNvars(vlistID1);
  for (varID = 0; varID < nvars; varID++) vlistDefFlag(vlistID1, varID, 0, true);

  int vlistID3 = vlistCreate();
  cdoVlistCopyFlag(vlistID3, vlistID1);

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID3 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID3, taxisID3);

  const auto ngrids = vlistNgrids(vlistID1);
  int index = 0;
  const auto gridID1 = vlistGrid(vlistID1, index);

  if (needWeights && gridInqType(gridID1) != GRID_LONLAT && gridInqType(gridID1) != GRID_GAUSSIAN)
    cdoAbort("Unsupported gridtype: %s", gridNamePtr(gridInqType(gridID1)));

  vlistChangeGridIndex(vlistID3, index, gridID3);
  if (ngrids > 1) cdoAbort("Too many different grids!");

  const auto streamID3 = cdoOpenWrite(2);
  cdoDefVlist(streamID3, vlistID3);

  FieldVector2D vars1, vars2;
  fieldsFromVlist(vlistID1, vars1, FIELD_VEC);
  fieldsFromVlist(vlistID2, vars2, FIELD_VEC);

  const auto gridsizemax = vlistGridsizeMax(vlistID1);
  Varray<double> weights;
  if (needWeights) weights.resize(gridsizemax);

  Field field3;
  field3.resize(1);
  field3.grid = gridID3;

  int tsID = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID1, tsID)))
    {
      nrecs = cdoStreamInqTimestep(streamID2, tsID);

      taxisCopyTimestep(taxisID3, taxisID1);
      cdoDefTimestep(streamID3, tsID);

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID, &levelID);
          cdoReadRecord(streamID1, vars1[varID][levelID].vec.data(), &nmiss);
          if (nmiss) cdoAbort("Missing values unsupported for this operator!");

          cdoInqRecord(streamID2, &varID, &levelID);
          cdoReadRecord(streamID2, vars2[varID][levelID].vec.data(), &nmiss);
          if (nmiss) cdoAbort("Missing values unsupported for this operator!");
        }

      for (varID = 0; varID < nvars; varID++)
        {
          auto wstatus = false;
          auto gridID = vars1[varID][0].grid;
          if (needWeights && gridID != lastgrid)
            {
              lastgrid = gridID;
              wstatus = gridWeights(gridID, weights.data());
            }
          const auto code = vlistInqVarCode(vlistID1, varID);
          if (wstatus != 0 && tsID == 0 && code != oldcode) cdoWarning("Using constant area weights for code %d!", oldcode = code);

          field3.missval = vars1[varID][0].missval;
          varrms(weights, vars1[varID], vars2[varID], field3);

          cdoDefRecord(streamID3, varID, 0);
          cdoWriteRecord(streamID3, field3.vec.data(), field3.nmiss);
        }

      tsID++;
    }

  cdoStreamClose(streamID3);
  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  vlistDestroy(vlistID3);

  cdoFinish();

  return nullptr;
}
