/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Rotuv      rotuvb          Backward rotation
*/

#include <cdi.h>

#include "cdo_options.h"
#include "dmemory.h"
#include "process_int.h"
#include "param_conversion.h"
#include <mpim_grid.h>


static void
rot_uv_back(int gridID, Varray<double> &us, Varray<double> &vs)
{
  double xpole = 0, ypole = 0, angle = 0;
  if (gridInqType(gridID) == GRID_PROJECTION && gridInqProjType(gridID) == CDI_PROJ_RLL)
    gridInqParamRLL(gridID, &xpole, &ypole, &angle);

  const auto nlon = gridInqXsize(gridID);
  const auto nlat = gridInqYsize(gridID);

  Varray<double> xvals(nlon);
  Varray<double> yvals(nlat);
  gridInqXvals(gridID, xvals.data());
  gridInqYvals(gridID, yvals.data());

  /* Convert lat/lon units if required */
  char units[CDI_MAX_NAME];
  gridInqXunits(gridID, units);
  grid_to_degree(units, 1, &angle, "angle");
  grid_to_degree(units, 1, &xpole, "xpole");
  grid_to_degree(units, nlon, xvals.data(), "grid center lon");
  gridInqYunits(gridID, units);
  grid_to_degree(units, 1, &ypole, "ypole");
  grid_to_degree(units, nlat, yvals.data(), "grid center lat");

  if (xpole > 180) xpole -= 360;
  if (angle > 180) angle -= 360;

  for (size_t ilat = 0; ilat < nlat; ilat++)
    for (size_t ilon = 0; ilon < nlon; ilon++)
      {
        const auto i = ilat * nlon + ilon;
        const auto xval = lamrot_to_lam(yvals[ilat], xvals[ilon], ypole, xpole, angle);
        const auto yval = phirot_to_phi(yvals[ilat], xvals[ilon], ypole, angle);
        usvs_to_uv(us[i], vs[i], yval, xval, ypole, xpole, &us[i], &vs[i]);
      }
}

#define MAXARG 16384

void *
Rotuv(void *process)
{
  int varID, levelID;
  int nrecs;
  int chcodes[MAXARG];
  const char *chvars[MAXARG];
  char varname[CDI_MAX_NAME];
  char varname2[CDI_MAX_NAME];

  cdoInitialize(process);

  operatorInputArg("pairs of u and v in the rotated system");

  int nch = operatorArgc();
  if (nch % 2) cdoAbort("Odd number of input arguments!");

  bool lvar = false;  // We have a list of codes
  int len = (int) cdoOperatorArgv(0).size();
  int ix = (cdoOperatorArgv(0)[0] == '-') ? 1 : 0;
  for (int i = ix; i < len; ++i)
    if (!isdigit(cdoOperatorArgv(0)[i]))
      {
        lvar = true;  // We have a list of variables
        break;
      }

  if (lvar)
    {
      for (int i = 0; i < nch; i++) chvars[i] = cdoOperatorArgv(i).c_str();
    }
  else
    {
      for (int i = 0; i < nch; i++) chcodes[i] = parameter2int(cdoOperatorArgv(i));
    }

  const auto streamID1 = cdoOpenRead(0);

  const auto vlistID1 = cdoStreamInqVlist(streamID1);
  const auto vlistID2 = vlistDuplicate(vlistID1);

  const auto nvars = vlistNvars(vlistID1);

  const auto maxrecs = vlistNrecs(vlistID1);
  std::vector<RecordInfo> recList(maxrecs);

  std::vector<std::vector<size_t>> varnmiss(nvars);
  Varray3D<double> vardata(nvars);

  bool lfound[MAXARG];
  for (int i = 0; i < nch; i++) lfound[i] = false;

  if (lvar)
    {
      for (varID = 0; varID < nvars; varID++)
        {
          vlistInqVarName(vlistID2, varID, varname);
          for (int i = 0; i < nch; i++)
            if (strcmp(varname, chvars[i]) == 0) lfound[i] = true;
        }
      for (int i = 0; i < nch; i++)
        if (!lfound[i]) cdoAbort("Variable %s not found!", chvars[i]);
    }
  else
    {
      for (varID = 0; varID < nvars; varID++)
        {
          const auto code = vlistInqVarCode(vlistID2, varID);
          for (int i = 0; i < nch; i++)
            if (code == chcodes[i]) lfound[i] = true;
        }
      for (int i = 0; i < nch; i++)
        if (!lfound[i]) cdoAbort("Code %d not found!", chcodes[i]);
    }

  for (varID = 0; varID < nvars; varID++)
    {
      const auto gridID = vlistInqVarGrid(vlistID1, varID);
      if (!(gridInqType(gridID) == GRID_PROJECTION && gridInqProjType(gridID) == CDI_PROJ_RLL))
        cdoAbort("Only rotated lon/lat grids supported!");

      const auto gridsize = gridInqSize(gridID);
      const auto nlevel = zaxisInqSize(vlistInqVarZaxis(vlistID1, varID));
      varnmiss[varID].resize(nlevel);
      vardata[varID].resize(nlevel);
      for (levelID = 0; levelID < nlevel; levelID++) vardata[varID][levelID].resize(gridsize);
    }

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  const auto streamID2 = cdoOpenWrite(1);
  cdoDefVlist(streamID2, vlistID2);

  int tsID = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID1, tsID)))
    {
      taxisCopyTimestep(taxisID2, taxisID1);
      cdoDefTimestep(streamID2, tsID);

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID, &levelID);

          recList[recID].varID = varID;
          recList[recID].levelID = levelID;
          recList[recID].lconst = vlistInqVarTimetype(vlistID1, varID) == TIME_CONSTANT;

          cdoReadRecord(streamID1, vardata[varID][levelID].data(), &varnmiss[varID][levelID]);
          if (varnmiss[varID][levelID]) cdoAbort("Missing values unsupported for this operator!");
        }

      for (int i = 0; i < nch; i += 2)
        {
          for (varID = 0; varID < nvars; varID++)
            {
              if (lvar)
                {
                  vlistInqVarName(vlistID2, varID, varname);
                  if (strcmp(varname, chvars[i]) == 0) break;
                }
              else
                {
                  const auto code = vlistInqVarCode(vlistID2, varID);
                  if (code == chcodes[i]) break;
                }
            }

          if (varID == nvars) cdoAbort("u-wind not found!");

          const auto usvarID = varID;

          for (varID = 0; varID < nvars; varID++)
            {
              if (lvar)
                {
                  vlistInqVarName(vlistID2, varID, varname);
                  if (strcmp(varname, chvars[i + 1]) == 0) break;
                }
              else
                {
                  const auto code = vlistInqVarCode(vlistID2, varID);
                  if (code == chcodes[i + 1]) break;
                }
            }

          if (varID == nvars) cdoAbort("v-wind not found!");

          const auto vsvarID = varID;

          if (Options::cdoVerbose)
            {
              if (lvar)
                {
                  vlistInqVarName(vlistID2, usvarID, varname);
                  vlistInqVarName(vlistID2, vsvarID, varname2);
                  cdoPrint("Using var %s [%s](u) and var %s [%s](v)", varname, chvars[i], varname2, chvars[i + 1]);
                }
              else
                cdoPrint("Using code %d [%d](u) and code %d [%d](v)", vlistInqVarCode(vlistID1, usvarID), chcodes[i],
                         vlistInqVarCode(vlistID1, vsvarID), chcodes[i + 1]);
            }

          const auto gridID = vlistInqVarGrid(vlistID1, varID);
          const auto nlevel1 = zaxisInqSize(vlistInqVarZaxis(vlistID1, usvarID));
          const auto nlevel2 = zaxisInqSize(vlistInqVarZaxis(vlistID1, vsvarID));
          if (nlevel1 != nlevel2) cdoAbort("u-wind and v-wind have different number of levels!");

          for (levelID = 0; levelID < nlevel1; levelID++)
            rot_uv_back(gridID, vardata[usvarID][levelID], vardata[vsvarID][levelID]);
        }

      for (int recID = 0; recID < nrecs; recID++)
        {
          varID = recList[recID].varID;
          levelID = recList[recID].levelID;
          cdoDefRecord(streamID2, varID, levelID);
          cdoWriteRecord(streamID2, vardata[varID][levelID].data(), varnmiss[varID][levelID]);
        }

      tsID++;
    }

  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  cdoFinish();

  return nullptr;
}
