/*
**      cdecl -- C gibberish translator
**      src/lexer.l
**
**      Copyright (C) 2017  Paul J. Lucas, et al.
**
**      This program is free software: you can redistribute it and/or modify
**      it under the terms of the GNU General Public License as published by
**      the Free Software Foundation, either version 3 of the License, or
**      (at your option) any later version.
**
**      This program is distributed in the hope that it will be useful,
**      but WITHOUT ANY WARRANTY; without even the implied warranty of
**      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**      GNU General Public License for more details.
**
**      You should have received a copy of the GNU General Public License
**      along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * @file
 * Defines helper macros, data structures, variables, functions, and the
 * tokenizer for C/C++ declarations.
 */

/** @cond DOXYGEN_IGNORE */

%option nounput
%option yylineno

%{
/** @endcond */

// local
#include "config.h"                     /* must go first */
#include "c_ast.h"
#include "c_ast_util.h"
#include "c_keyword.h"
#include "c_typedef.h"
#include "lexer.h"
#include "literals.h"
#include "misc.h"
#include "options.h"
#include "util.h"
#include "parser.h"                     /* must go last */

/// @cond DOXYGEN_IGNORE

// standard
#include <assert.h>
#include <stdio.h>
#include <stdlib.h>                     /* for strtol(3) */
#include <string.h>

/// @endcond

/**
 * Asserts that we're in \a MODE of operation.
 * If not, returns the error token.
 *
 * @param MODE The `c_mode` we're asserting.
 */
#define ASSERT_MODE(MODE) \
  BLOCK( if ( c_mode != MODE_ ## MODE ) return Y_ERROR; )

/**
 * Sets <code>\ref lexer_token</code> to the current token.
 */
#define SET_TOKEN                 lexer_token = yytext

/**
 * Overrides flex's fatal error message to print the message in our format and
 * also exit with the status code we want.
 *
 * @param MSG The error message to print.
 * @hideinitializer
 */
#define YY_FATAL_ERROR(MSG)       lexer_fatal( (MSG) )

/**
 * Overrides flex's input.
 *
 * @param BUF The buffer to use.
 * @param BYTES_READ Set to the number of bytes read.
 * @param BYTES_MAX The maximum number of bytes to read.
 * @hideinitializer
 */
#define YY_INPUT(BUF,BYTES_READ,BYTES_MAX) \
  (BYTES_READ) = lexer_get_input( (BUF), (BYTES_MAX) )

/**
 * This code is inserted by flex at the beginning of each rule to set the
 * current token location information.
 *
 * @hideinitializer
 */
#define YY_USER_ACTION                                \
  yylloc.first_line   = yylloc.last_line = yylineno;  \
  yylloc.first_column = column;                       \
  yylloc.last_column  = column + yyleng - 1;          \
  column += yyleng;

///////////////////////////////////////////////////////////////////////////////

/**
 * cdecl keyword info.
 */
struct keyword {
  char const *literal;                  ///< C string literal of the keyword.
  int         y_token;                  ///< Bison token number.
  char const *c_keyword_synonym;        ///< Synonym of C/C++ keyword, if any.
  bool        allow_explain;            ///< Allow when explaining C/C++?
};
typedef struct keyword keyword_t;

/// @cond DOXYGEN_IGNORE
#define C_SYN(C_KEYWORD,ALLOW)  0, C_KEYWORD, ALLOW
#define TOKEN(T)                T, NULL,      false
/// @endcond

/**
 * All cdecl keywords (that are not C/C++ keywords).
 *
 * @hideinitializer
 */
static keyword_t const CDECL_KEYWORDS[] = {
  { L_ALL,            TOKEN( Y_ALL )                  },
  { L_ARRAY,          TOKEN( Y_ARRAY )                },
  { L_AS,             TOKEN( Y_AS )                   },
  { L_ATOMIC,         C_SYN( L__ATOMIC,       false ) },
  { L_AUTOMATIC,      C_SYN( L_AUTO,          false ) },
  { L_BLOCK,          TOKEN( Y_BLOCK )                },  // Apple extension
  { L__BOOL,          C_SYN( L_BOOL,          true  ) },
  { L_CAST,           TOKEN( Y_CAST )                 },
  { L_CHARACTER,      C_SYN( L_CHAR,          false ) },
  { L_COMPLEX,        C_SYN( L__COMPLEX,      true  ) },
  { L_CONSTANT,       C_SYN( L_CONST,         false ) },
  { L_DECLARE,        TOKEN( Y_DECLARE )              },
  { L_DEFINE,         TOKEN( Y_DEFINE )               },
  { L_DYNAMIC,        TOKEN( Y_DYNAMIC )              },
  { L_ENUMERATION,    C_SYN( L_ENUM,          false ) },
  { L_EXIT,           TOKEN( Y_QUIT )                 },
  { L_EXPLAIN,        TOKEN( Y_EXPLAIN )              },
  { L_EXTERNAL,       C_SYN( L_EXTERN,        false ) },
  { L_FUNC,           TOKEN( Y_FUNCTION )             },
  { L_FUNCTION,       TOKEN( Y_FUNCTION )             },
  { L_HELP,           TOKEN( Y_HELP )                 },
  { L_IMAGINARY,      C_SYN( L__IMAGINARY,    true  ) },
  { L_INTEGER,        C_SYN( L_INT,           false ) },
  { L_INTO,           TOKEN( Y_INTO )                 },
  { L_LEN,            TOKEN( Y_LENGTH )               },
  { L_LENGTH,         TOKEN( Y_LENGTH )               },
  { L_MBR,            TOKEN( Y_MEMBER ),              },
  { L_MEMBER,         TOKEN( Y_MEMBER ),              },
  { L_NORETURN,       C_SYN( L__NORETURN,     true  ) },
  { L_OF,             TOKEN( Y_OF ),                  },
  { L_OVERRIDDEN,     C_SYN( L_OVERRIDE,      false ) },
  { L_POINTER,        TOKEN( Y_POINTER )              },
  { L_PREDEF,         TOKEN( Y_PREDEFINED )           },
  { L_PREDEFINED,     TOKEN( Y_PREDEFINED )           },
  { L_PTR,            TOKEN( Y_POINTER )              },
  { L_PURE,           TOKEN( Y_PURE )                 },
  { L_Q,              TOKEN( Y_QUIT )                 },
  { L_QUIT,           TOKEN( Y_QUIT )                 },
  { L_REF,            TOKEN( Y_REFERENCE )            },
  { L_REFERENCE,      TOKEN( Y_REFERENCE )            },
  { L_REINTERPRET,    TOKEN( Y_REINTERPRET )          },
  { L_RESTRICTED,     C_SYN( L_RESTRICT,      false ) },
  { L_RET,            TOKEN( Y_RETURNING )            },
  { L_RETURNING,      TOKEN( Y_RETURNING )            },
  { L_RVALUE,         TOKEN( Y_RVALUE )               },
  { L_SET,            TOKEN( Y_SET )                  },
  { L_SHOW,           TOKEN( Y_SHOW )                 },
  { L_STRUCTURE,      C_SYN( L_STRUCT,        false ) },
  { L__THREAD_LOCAL,  C_SYN( L_THREAD_LOCAL,  true  ) },
  { L_TO,             TOKEN( Y_TO )                   },
  { L_TYPE,           C_SYN( L_TYPEDEF,       false ) },
  { L_TYPES,          TOKEN( Y_TYPES )                },
  { L_USER,           TOKEN( Y_USER )                 },
  { L_VAR,            TOKEN( Y_VARIABLE )             },
  { L_VARARGS,        C_SYN( L_ELLIPSIS,      false ) },
  { L_VARIABLE,       TOKEN( Y_VARIABLE )             },
  { L_VARIADIC,       C_SYN( L_ELLIPSIS,      false ) },
  { L_VECTOR,         TOKEN( Y_ARRAY )                },
  { NULL,             TOKEN( 0 )                      }
};

/**
 * Hyphenated English keyword-to-literal mapping.
 */
struct hyphenated {
  char const *token;                    ///< Hyphenated English literal.
  char const *literal;                  ///< C language literal it maps to.
};
typedef struct hyphenated hyphenated_t;

/**
 * All hyphenated English keywords.  These are handled seperately because
 * hyphenated words are not valid identifiers (since `-` is not valid in an
 * identifier).
 *
 * @hideinitializer
 */
static hyphenated_t const HYPHENATED_KEYWORDS[] = {
  { "carries-dependency", L_CARRIES_DEPENDENCY  },
  { "maybe-unused",       L_MAYBE_UNUSED        },
  { "no-exception",       L_NOEXCEPT            },
  { "non-discardable",    L_NODISCARD           },
  { "non-returning",      L__NORETURN           },
  { "non-throwing",       L_THROW               },
};

// extern variables
char const     *lexer_token;            ///< Guaranteed pointer of `yytext`.

// local variables
static int      column;                 ///< Column position of current token.
static int      column_newline;         ///< Column position of newline.
static char    *input_buf;              ///< Entire current input line.
static size_t   input_len;              ///< Length of line in `input_buf`.
static size_t   input_sent;             ///< How many bytes returned to flex.
static bool     is_eof;                 ///< Encountered EOF?

// local functions
static void     lexer_fatal( char const* );

////////// local functions ////////////////////////////////////////////////////

/**
 * Finds a cdecl keyword matching \a s, if any.
 *
 * @param s The string to find.
 * @return Returns a pointer to the corresponding keyword or null if not found.
 */
static keyword_t const* cdecl_keyword_find( char const *s ) {
  for ( keyword_t const *k = CDECL_KEYWORDS; k->literal; ++k ) {
    if ( strcmp( s, k->literal ) != 0 )
      continue;
    if ( c_mode == MODE_ENGLISH_TO_GIBBERISH ||
         (k->c_keyword_synonym && k->allow_explain) )
      return k;
    break;
  } // for
  return NULL;
}

/**
 * Finds a hyphenated English literal.
 *
 * @param s The string to find.
 * @return Returns the corresponding C language literal or null if none.
 */
static char const* hyphenated_find( char const *s ) {
  char const *literal = NULL;
  for ( hyphenated_t const *h = HYPHENATED_KEYWORDS; h->token; ++h ) {
    if ( strcmp( s, h->token ) == 0 ) {
      literal = h->literal;
      if ( literal == L__NORETURN && opt_lang >= LANG_CPP_MIN ) {
        //
        // _Noreturn is a special case. In C11, it's "_Noreturn"; in C++11,
        // it's "noreturn".
        //
        // Depending on the current language, we map "non-returning" either to
        // "_Noreturn" or "noreturn".
        //
        literal = L_NORETURN;
      }
      break;
    }
  } // for
  return literal;
}

/**
 * Gets a line of input for flex and keeps a copy for use later if printing an
 * error message.
 *
 * @param buf A pointer to the buffer to write into.
 * @param buf_capacity The capacity of \a buf.
 * @return Returns the number of bytes read.
 */
static size_t lexer_get_input( char *buf, size_t buf_capacity ) {
  size_t input_avail = input_len - input_sent;
  if ( input_avail == 0 ) {
    static size_t input_capacity;
    ssize_t bytes_read = getline( &input_buf, &input_capacity, yyin );
    if ( bytes_read == -1 ) {
      FERROR( yyin );
      bytes_read = 0;
    }
    input_avail = input_len = (size_t)bytes_read;
    input_sent = 0;
  }
  //
  // Given that the default buffer capacity (YY_READ_BUF_SIZE) for flex is
  // 8192, it's unlikely that this will ever be true and that we'll have to
  // return the input line in chunks; but might as well code for the case.
  //
  if ( input_avail > buf_capacity )
    input_avail = buf_capacity;
  memcpy( buf, input_buf + input_sent, input_avail );
  input_sent += input_avail;
  return input_avail;
}

/**
 * Resets the column position upon encountering a newline.
 */
static inline void newline( void ) {
  column_newline = column;
  column = 0;
}

/**
 * Parses an integer from <code>\ref lexer_token</code>.
 *
 * @param base The integer base to use.
 * @return Returns the integer value.
 */
static inline int parse_int( int base ) {
  //
  // We don't need to check for errors from strtol(3) here since the lexer
  // guarantees that we'll be passed only a valid number token.
  //
  // Standard strtol(3) doesn't understand a "0b" (binary) prefix, so skip over
  // it if the base is 2.
  //
  return (int)strtol( lexer_token + (base == 2 ? 2 : 0), NULL, base );
}

/**
 * Standard lex function to know whether to continue parsing upon reaching EOF.
 *
 * @return Returns 1 if done or 0 to continue parsing (a new file set via
 * \c yyin).
 */
static int yywrap( void ) {
  return 1 /* done */;
}

////////// extern functions ///////////////////////////////////////////////////

char const* lexer_input_line( size_t *plen ) {
  if ( plen != NULL )
    *plen = input_len;
  return input_buf;
}

void lexer_loc( int *pline, int *pcolumn ) {
  assert( pline != NULL );
  assert( pcolumn != NULL );

  *pline = yylineno;

  if ( column == 0 )
    *pcolumn = column_newline;
  else if ( lexer_token[0] == '\n' )
    *pcolumn = column;
  else
    *pcolumn = column - strlen( lexer_token );
}

void lexer_reset( bool hard_reset ) {
  if ( hard_reset ) {
    is_eof = false;
    yylineno = 0;
  }
  c_mode = MODE_ENGLISH_TO_GIBBERISH;
  input_len = input_sent = 0;
  newline();
}

///////////////////////////////////////////////////////////////////////////////

/// @cond DOXYGEN_IGNORE

%}

A             [A-Za-z_]
az            [a-z]
B             [01]
O             [0-7]
D             [0-9]
H             [0-9A-Fa-f]
S             [ \f\r\t\v]

cknr_name     ([cC][kK]&[rR])|([kK]&[rR][cC]?)
cxx_name      [cC]"++"({D}{D})?
hyphenated    {az}+-{az}+
identifier    {A}({A}|{D})*
int_suffix    ([lL][lL]?[uU]?)|([uU][lL]?[lL]?)
lang_name     {cknr_name}|{cxx_name}

binary        0[bB]{B}+{int_suffix}?
octal         0{O}*{int_suffix}?
decimal       [1-9]{D}*{int_suffix}?
hexadecimal   0[xX]{H}+{int_suffix}?

%%

              /*
               * Special case: If the _Atomic keyword is immediately followed
               * by a left parenthesis, it is interpreted as a type specifier
               * ..., not as a type qualifier.
               */
_Atomic/{S}*\(  { SET_TOKEN; return Y_ATOMIC_SPEC; }

{identifier}  {
                SET_TOKEN;

                keyword_t const *const ck = cdecl_keyword_find( lexer_token );
                if ( ck != NULL ) {
                  if ( ck->c_keyword_synonym == NULL )
                    return ck->y_token;
                  if ( ck->c_keyword_synonym == L__NORETURN &&
                       opt_lang >= LANG_CPP_MIN ) {
                    //
                    // _Noreturn is a special case. In C11, it's "_Noreturn"
                    // and is a reserved word. In C++11, it's "noreturn", but
                    // is NOT a reserved word.
                    //
                    // When parsing C11, we also want to allow "noreturn" as a
                    // synonym (since that's provided via <stdnoreturn.h>).
                    //
                    // When parsing C++11, we want to recognize "noreturn" only
                    // when it's an attribute as [[noreturn]].
                    //
                    // When parsing English:
                    //
                    //  + We want to allow "_Noreturn" when we're translating
                    //    to C only and not C++.
                    //
                    //  + We want to allow either "noreturn" or "non-returning"
                    //    in all cases.
                    //
                    // However, when either parsing or translating to C++, we
                    // do NOT want to allow "noreturn" to be a synonym for
                    // "_Noreturn" since that's illegal in C++.  Hence, we do
                    // nothing.
                    //
                  }
                  else {
                    lexer_token = ck->c_keyword_synonym;
                  }
                }

                c_typedef_t const *const t = c_typedef_find( lexer_token );
                if ( t != NULL ) {
                  yylval.c_typedef = t;
                  return Y_TYPEDEF_TYPE;
                }

        find_c_keyword:
                NO_OP;
                c_keyword_t const *k = c_keyword_find( lexer_token );
                if ( k == NULL && c_mode == MODE_ENGLISH_TO_GIBBERISH ) {
                  //
                  // We have to look-up attribute names when parsing English
                  // because there can't be any special syntax for them, e.g.:
                  //
                  //      declare x as deprecated int
                  //
                  k = c_attribute_find( lexer_token );
                }
                else {
                  //
                  // We do NOT look-up attribute names when parsing C++ because
                  // they are not reserved words.  For example:
                  //
                  //      [[noreturn]] void noreturn();
                  //
                  // is legal.
                  //
                }
                if ( k != NULL ) {
                  yylval.type = k->type;
                  return k->y_token;
                }

                yylval.name = check_strdup( lexer_token );
                return Y_NAME;
              }

{hyphenated}  {
                ASSERT_MODE( ENGLISH_TO_GIBBERISH );
                char const *const literal = hyphenated_find( yytext );
                if ( literal == NULL )
                  return Y_ERROR;
                lexer_token = literal;
                goto find_c_keyword;
              }

{lang_name}   {
                SET_TOKEN;
                yylval.name = check_strdup( lexer_token );
                return Y_LANG_NAME;
              }

{binary}      { SET_TOKEN; yylval.number = parse_int(  2 ); return Y_NUMBER; }
{octal}       { SET_TOKEN; yylval.number = parse_int(  8 ); return Y_NUMBER; }
{decimal}     { SET_TOKEN; yylval.number = parse_int( 10 ); return Y_NUMBER; }
{hexadecimal} { SET_TOKEN; yylval.number = parse_int( 16 ); return Y_NUMBER; }

"&&"          { SET_TOKEN; return Y_2AMPERSAND; }
"->"          { SET_TOKEN; return Y_ARROW; }
"..."         { SET_TOKEN; return Y_ELLIPSIS; }
"::"          { SET_TOKEN; return Y_2COLON; }
"?"           { SET_TOKEN; return Y_HELP; }
"[["          { SET_TOKEN; return Y_2LBRACKET; }
"]]"          { SET_TOKEN; return Y_2RBRACKET; }

"^"           |                         /* Apple extension: block indicator */
"&"           |
"("           |
")"           |
"*"           |
","           |
"<"           |
"="           |
">"           |
"["           |
"]"           { SET_TOKEN; return lexer_token[0]; }
";"           { SET_TOKEN; return Y_END; }

"/*"          {                         /* ignore C-style comments */
                for ( int c = input(), prev_c = '\0'; c != EOF;
                      prev_c = c, c = input() ) {
                  ++column;
                  switch ( c ) {
                    case '\n':
                      newline();
                      break;
                    case '/':
                      if ( prev_c == '*' )
                        goto c_comment_done;
                      break;
                  } // switch
                } // for

        c_comment_done:
                NO_OP;
              }

"//".*        ;                         /* ignore C++-style comments */
#.*           ;                         /* ignore preprocessor directives */
{S}+          ;                         /* ignore all other whitespace */
\\\n          ;                         /* eat escaped newlines */

\n            {
                SET_TOKEN;
                newline();
                return Y_END;
              }

<<EOF>>       {
                SET_TOKEN;
                newline();
                //
                // The first time we encounter EOF, we want to treat it as if
                // it were a newline by returning Y_END so commands in the
                // parser always end in Y_END.
                //
                // Requesting more characters after encountering EOF will
                // simply continue to return EOF, so the second time we
                // encounter EOF, treat it as EOF by returning no token.
                //
                return (is_eof = !is_eof) ? Y_END : YY_NULL;
              }

.             {
                SET_TOKEN;
                return Y_ERROR;
              }

%%

/// @endcond

///////////////////////////////////////////////////////////////////////////////

/**
 * Called by flex only when there's a fatal error.
 *
 * @param msg The error message to print.
 */
static void lexer_fatal( char const *msg ) {
  //
  // This is defined down here to avoid having to declare yy_fatal_error
  // ourselves and having to get it right being subject to possible changes in
  // its signature in different flex versions.
  //
  INTERNAL_ERR( "%s\n", msg );

  // Not reached -- "called" just to silence the "unused function" warning.
  yy_fatal_error( msg );
}

///////////////////////////////////////////////////////////////////////////////
/* vim:set et sw=2 ts=2: */
