"""
======================COPYRIGHT/LICENSE START==========================

Minor.py: Data compatibility handling

Copyright (C) 2007 Rasmus Fogh (CCPN project)
 
=======================================================================

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.
 
A copy of this license can be found in ../../../../license/LGPL.license.
 
This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.
 
You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

======================COPYRIGHT/LICENSE END============================

To obtain more information about this code:

- CCPN website (http://www.ccpn.ac.uk)

- contact Rasmus Fogh (ccpn@bioc.cam.ac.uk)

=======================================================================

If you are using this software for academic purposes, we suggest
quoting the following reference:

===========================REFERENCE START=============================
Rasmus H. Fogh, Wayne Boucher, Wim F. Vranken, Anne
Pajon, Tim J. Stevens, T.N. Bhat, John Westbrook, John M.C. Ionides and
Ernest D. Laue (2005). A framework for scientific data modeling and 
automated software development. Bioinformatics 21, 1678-1684.
===========================REFERENCE END===============================
 
"""

# NB this file will only be used as part of Minor upgrades

from memops.general.Implementation import ApiError

def correctData(topObj, delayDataDict, toNewObjDict, mapping=None,
                topObjElem=None):
  """ update topObj object tree using information in delayDataDict
  May be used either to postprocess a file load (minor upgrade)
  or as part of an in-memory data transfer (major upgrade)
  
  topObj is the MemopsRoot in the new tree
  toNewObjDict is _ID:newObj for minor 
    and oldObj/oldObjId:newObj for major upgrades
    
    NB this function does nothing without further additions
  """
  
  
  emptyDict = {}
  emptyList = []
  doGet = delayDataDict.get
  
  pName = topObj.packageName
  
  #
  if pName == 'ccp.molecule.MolStructure':
    # Fix MolStructure
    fixMolStructure(topObj, delayDataDict, toNewObjDict, topObjElem)

  elif pName == 'ccp.nmr.NmrCalc':
    # Fix Nmr
    fixNmrCalc(topObj, delayDataDict)

  elif pName == 'ccp.molecule.ChemComp':
    # Fix Nmr
    fixChemComp(topObj, delayDataDict)

def fixChemComp(topObj, delayDataDict):
  """ Fix StereoChemistry.stereoClass / .refStereoChemistry
  """
  
  emptyDict = {}
  emptyList = []
  doGet = delayDataDict.get
  
  pName = topObj.packageName
  
  store = topObj.root.currentStereochemistryStore
  for stereo in doGet(topObj, emptyDict).get('stereochemistries', emptyList):
    stereoClass = doGet(stereo, emptyDict).get('stereoClass')
    refStereo = store.findFirstRefStereochemistry(stereoClass=stereoClass)
    stereo.refStereochemistry = refStereo

def fixMolStructure(topObj, delayDataDict, toNewObjDict, topObjElem):
  """ 
  """
  
  if len(toNewObjDict) == 1:
    # Must be a partial load - only TopObjetc is present. Do nothing now.
    return
  
  #print '### Fixing mol structure'
  #print '### toNewObjDict', len(toNewObjDict)
  
  emptyDict = {}
  emptyList = []
  doGet = delayDataDict.get
  
  memopsRoot = topObj.parent
  topObjByGuid = memopsRoot.__dict__.get('topObjects')
  
  # handle atoms differing only by altLocationCode
  atomIds = doGet(topObj, emptyDict).get('orderedAtoms', emptyList)
  #print '### found atomIds:', len(atomIds)
  if atomIds:
    atomMap = {}  # for duplicates only
    dataDict = {}
    atoms = [toNewObjDict[x] for x in atomIds]
    nAtoms = len(atoms)
  
    dd = {}
    altLocationCodes = dataDict['altLocationCode'] = [' ']*nAtoms
    for ii,atom in enumerate(atoms):
      key = (atom.residue,atom.name)
      ll = dd.get(key)
      if ll is None:
        dd[key] = [atom]
      else:
        ll.append(atom)
      ll2 = doGet(atom, emptyDict).get('altLocationCode')
      if ll2:
        altLocationCodes[ii] = ll2[0]
        
        
    duplicates = [tt[1] for tt in sorted(dd.items()) if len(tt[1]) > 1]
    if duplicates:
      # We have atoms that differ only by altLocationCode
      for ats in duplicates:
        at0 = ats[0]
        res = at0.residue
        atList = doGet(at0.residue, emptyDict).get('atoms', emptyList)
        for at in ats[1:]:
          atomMap[at] = at0
          # Now remove extra copies from delayDataDict tree
          atList.remove(at)

    # get models in order
    models = doGet(topObj, emptyDict).get('models', emptyList)
    if models:
      models = [tt[1] for tt in sorted((x.serial,x) for x in models)]
      nModels = len(models)
      
      # Amplify altLocationCodes to correct length, or remove as is the case
      ll = dataDict['altLocationCode']
      if [x for x in ll if x != ' ']:
        dataDict['altLocationCode'] = ll*nModels
      else:
        del dataDict['altLocationCode']
    
      # Now read actual coordinates. Data are in an ElementTree
      dataMatrices = topObjElem.findall('COOR.StructureEnsemble.dataMatrices')[0]
      for dataMatrix in dataMatrices.findall('COOR.DataMatrix'):
        name = (dataMatrix.get('name') or
                dataMatrix.findtext('COOR.DataMatrix.name'))
        if name in ('coordinates', 'occupancies', 'bFactors'):
          ss = dataMatrix.findtext('COOR.DataMatrix.data')
          if ss:
            floats = [float(x) for x in ss.split()]
            if floats:
                dataDict[name] = floats
                ss = dataMatrix.findtext('COOR.DataMatrix.shape')
                if ss:
                  shape = [int(x) for x in ss.split()]
                  shape = shape[:2]
                  if shape != [nModels,nAtoms]:
                    raise Exception(
                     "Data shape %s does not match nModels %s, nAtoms %s"
                     % (shape, nModels, nAtoms))
 
      # reset coordinates array
      occupancies = dataDict.get('occupancies')
      if occupancies:
        dataDict['occupancy'] = occupancies
        del dataDict['occupancies']
      bFactors = dataDict.get('bFactors')
      if bFactors:
        dataDict['bFactor'] = bFactors
        del dataDict['bFactors']
      coordinates = dataDict.get('coordinates')
      if coordinates:
        del dataDict['coordinates']
        dataDict['x'], dataDict['y'],  dataDict['z'] = zip(*(coordinates[x:x+3]
         for x in range(0,len(coordinates),3)))
    
    #print '### dataDict keys', dataDict.keys()
    #print '### dataDict items', [(tt[0],len(tt[1])) for tt in sorted(dataDict.items())]
    
    offset = 0
    items = dataDict.items()
    for ii,model in enumerate(models):
      for jj,atom in enumerate(atoms):
        at = atomMap.get(atom, atom)
        coord = at.newCoord(model=model)
        for tag, ll in items:
          setattr(coord, tag, ll[offset])
        atom.__dict__['coords'][(model,coord.altLocationCode)] = coord
 
        offset += 1
      

def fixNmrCalc(topObj, delayDataDict):
  # 

  emptyDict = {}
  emptyList = []
  doGet = delayDataDict.get
  
  memopsRoot = topObj.parent
  topObjByGuid = memopsRoot.__dict__.get('topObjects')
  
  for run in doGet(topObj, emptyDict).get('runs', emptyList):
    for data in doGet(run, emptyDict).get('data', emptyList):
 
      if data.__class__.__name__ == 'MolResidueData':
        chainCodes = doGet(data, emptyDict).get('chainCodes', emptyList)
        if len(chainCodes) == 1:
          # Only case where new data are writable to old
          data.chainCode = chainCodes[0]

