LICENSE = """
======================COPYRIGHT/LICENSE START==========================

StructureBasic.py: Part of the CcpNmr Analysis program

Copyright (C) 2003-2010 Wayne Boucher and Tim Stevens (University of Cambridge)

=======================================================================

The CCPN license can be found in ../../../../license/CCPN.license.

======================COPYRIGHT/LICENSE END============================

for further information, please contact :

- CCPN website (http://www.ccpn.ac.uk/)

- email: ccpn@bioc.cam.ac.uk

- contact the authors: wb104@bioc.cam.ac.uk, tjs23@cam.ac.uk
=======================================================================

If you are using this software for academic purposes, we suggest
quoting the following references:

===========================REFERENCE START=============================
R. Fogh, J. Ionides, E. Ulrich, W. Boucher, W. Vranken, J.P. Linge, M.
Habeck, W. Rieping, T.N. Bhat, J. Westbrook, K. Henrick, G. Gilliland,
H. Berman, J. Thornton, M. Nilges, J. Markley and E. Laue (2002). The
CCPN project: An interim report on a data model for the NMR community
(Progress report). Nature Struct. Biol. 9, 416-418.

Wim F. Vranken, Wayne Boucher, Tim J. Stevens, Rasmus
H. Fogh, Anne Pajon, Miguel Llinas, Eldon L. Ulrich, John L. Markley, John
Ionides and Ernest D. Laue (2005). The CCPN Data Model for NMR Spectroscopy:
Development of a Software Pipeline. Proteins 59, 687 - 696.

===========================REFERENCE END===============================

"""

from math import sqrt, cos, sin, atan2
from os import path, mkdir

from ccp.general.Geometry import calcTorsionAngleRadians, calcTorsionAngleDegrees
from ccp.general.Io import getStdChemComps
from ccp.general.Io import getChemComp
from ccp.util.Molecule import makeChain, addMolResidues, makeMolecule

from ccpnmr.analysis.core.MoleculeBasic import findMatchingChains, getLinkedResidue

from memops.gui.MessageReporter import showOkCancel,showWarning,showYesNo
from memops.universal.Util import returnInt, returnFloat
from memops.universal.Geometry import matrixMultiply
                  
BACKBONE_ATOMS = {'protein':('N','C','CA'),
                  'RNA':("OP1","P","O3'","O5'","C3'","C4'","C5'"),
                  'DNA':("OP1","P","O3'","O5'","C3'","C4'","C5'"),
                  'carbohydrate':("C1","C2","C3","C4","C5","C6","C7","C8"),}

TWOPI = 6.2831853071795864

def alignCoordinates(coords1, coords0, allCoords, W):
  """Align on two sets of coordinates (may be sub sets of whole)
             and then update all coordinates given this new position.
             Coordinates are weighted for alignment.
  .. describe:: Input
  
  List of List of Floats (x,y,z), List of List of Floats (reference x,y,z),
             List of List of Floats (x,y,z), List of Floats (weights)

  .. describe:: Output

  List of List of Floats (aligned x,y,z),
             List of List of Floats (all updated x,y,z),
             Float (fitting score)
  """
  from ccp.c.StructUtil import alignCoordinates as cAlignCoordinates
  
  rMat, error = cAlignCoordinates(coords1, coords0)
  # coords1 are modified in-place

  for n in range(len(allCoords)):
    [allCoords[n][0],allCoords[n][1],allCoords[n][2]] = matrixMultiply(rMat,allCoords[n])
    
  return coords1, allCoords, error


def alignStructures(structures):
  """Align structures by minimising weighted atomic RMSD.
             All members of any entered ensembles are aligned.

  .. describe:: Input
  
  List of MolStructure.StructureEnsembles

  .. describe:: Output

  List of MolStructure.StructureEnsembles, Float (Fit error), 
             Float (overall RMSD), List of Floats (Atom RMSDs)
  """

  # TBD: Arbitrary atom selection
  # TBD: Non-equivalent atom mappings
  
  from ccp.util.Validation import storeResidueValidations, storeModelValidations
  from ccpnmr.analysis.core.ValidationBasic import getEnsembleValidationStore
  from ccpnmr.analysis.core.ValidationBasic import ANALYSIS_RMSD_CONTEXT, RMSD_KEYWORDS
  from ccp.c.StructUtil import alignEnsemble

  structures = list(structures)
  
 
  null = structures[0].root.findFirstChemElementStore()

  nModels = sum([len(s.models) for s in structures])

  if nModels < 2:
    return structures, 0.0, [], {}

  numAtoms  = None
  ensemble  = []
  weights   = []
  allCoords = []
  
  k = 0;
  for structure in structures:
    coordLists = getStructureCoordinates(structure, model=None)
    coordList = coordLists[0]
    
    if numAtoms is None:
      numAtoms = len(coordList)
      
    elif numAtoms != len(coordList):
      raise Exception('Attempt to align structures with different atoms')

    if not weights:
      for coordObj in coordList:
        chemAtom = coordObj.atom.chemAtom
        mass = chemAtom.chemElement.mass
        weights.append(min(14.0,mass))

    for coordObjs in coordLists:
      coords = [[0.0,0.0,0.0]] * numAtoms
      
      for i in xrange(numAtoms):
        coordObj = coordObjs[i]
        coords[i] = [coordObj.x, coordObj.y, coordObj.z]
      
      allCoords.append(coordObjs)
      ensemble.append(coords)
  
  # Call to the C code to do the alignment
  error, atomRmsds, structureRmsds = alignEnsemble(ensemble, weights)
  weights = [x**-2.0 for x in atomRmsds]
  error, atomRmsds, structureRmsds = alignEnsemble(ensemble, weights)
  
  models = []
  j = 0
  for structure in structures:
    models = structure.sortedModels()
    k = j+len(models)
    scores = structureRmsds[j:k]
    validStore = getEnsembleValidationStore(structure,
                                            ANALYSIS_RMSD_CONTEXT,
                                            RMSD_KEYWORDS)
    storeModelValidations(validStore, ANALYSIS_RMSD_CONTEXT,
                          RMSD_KEYWORDS[1], models, scores)  
    j = k
      
  # Fill in the data model coords
  for i, alignVals in enumerate(ensemble):
    coordObjs = allCoords[i]
    
    for j in xrange(numAtoms):
      coordObj = coordObjs[j]
      coordObj.x, coordObj.y, coordObj.z = alignVals[j]
  
  atomRmsdDict = {}
  for i, coord in enumerate(allCoords[0]):
    atomRmsdDict[coord.atom] = atomRmsds[i]
  
  for structure in structures:
    
    # Reset cache
    structure.coordDict = {}

    residues = []
    caRmsds  = []
    cbRmsds  = []
    coRmsds  = []
    hnRmsds  = []
    bbRmsds  = []
    #scRmsds  = []
    aaRmsds  = []
    for chain in structure.coordChains:
      for residue in chain.residues:
        molType = residue.residue.molResidue.molType
        bbAtoms = BACKBONE_ATOMS.get(molType, [])
        
        caRmsd = None
        cbRmsd = None
        coRmsd = None
        hnRmsd = None
        bbRmsd = []
        #scRmsd = []
        aaRmsd = []
        for atom in residue.atoms:
          atomName = atom.name
          rmsd = atomRmsdDict[atom]
          
          if atomName == 'CA':
            caRmsd = rmsd
          elif atomName in 'O':
            coRmsd = rmsd
          elif atomName == 'CB':
            cbRmsd = rmsd
          elif atomName == 'H':
            hnRmsd = rmsd
            
          if atomName in bbAtoms:
            bbRmsd.append(rmsd)
            
          aaRmsd.append(rmsd)
            
        cAlpha = residue.findFirstAtom(name='CA')
        
        if aaRmsd:
          residues.append(residue)
          caRmsds.append(caRmsd)
          cbRmsds.append(cbRmsd)
          coRmsds.append(coRmsd)
          hnRmsds.append(hnRmsd)
          bbRmsds.append(sum(bbRmsd)/(len(bbRmsd) or 1.0))
          aaRmsds.append(sum(aaRmsd)/(len(aaRmsd) or 1.0))
        
    validStore = getEnsembleValidationStore(structure,
                                            ANALYSIS_RMSD_CONTEXT,
                                            RMSD_KEYWORDS)
    
      
     
    storeResidueValidations(validStore, ANALYSIS_RMSD_CONTEXT, 'CA', residues, caRmsds)
    storeResidueValidations(validStore, ANALYSIS_RMSD_CONTEXT, 'O', residues, coRmsds)
    storeResidueValidations(validStore, ANALYSIS_RMSD_CONTEXT, 'CB', residues, cbRmsds)
    storeResidueValidations(validStore, ANALYSIS_RMSD_CONTEXT, 'H', residues, hnRmsds)
    storeResidueValidations(validStore, ANALYSIS_RMSD_CONTEXT, 'backbone', residues, bbRmsds)
    #storeResidueValidations(validStore, ANALYSIS_RMSD_CONTEXT, 'sidechain', residues, scRmsds)
    storeResidueValidations(validStore, ANALYSIS_RMSD_CONTEXT, 'all', residues, aaRmsds)
       
  return structures, error, structureRmsds, atomRmsdDict


def compareEnsembles(structure1, structure2, compareBackboneOnly=False):
  """Compare models in two ensembles by minimising weighted atomic RMSD.
     The alignment is only done in the C world, not in the Python world.
     If compareBackboneOnly then only backbone atoms are used.

  .. describe:: Input
  
  MolStructure.StructureEnsemble, MolStructure.StructureEnsemble, Boolean

  .. describe:: Output

  Float (overall RMSD), Dict: MolSystem.Residue --> RMSD
  """

  from ccp.c.StructUtil import alignEnsemble

  # make sure ChemElements are loaded??
  null = structure1.root.findFirstChemElementStore()

  atomNameDict = compareBackboneOnly and BACKBONE_ATOMS

  numAtoms  = None
  weights   = []
  coordsDict = {}
  coordListDict = {}
  
  for structure in (structure1, structure2):
    for model in structure.models:
      coordLists = getStructureCoordinates(structure, model, atomNameDict)
      coordList = coordLists[0]
 
      if numAtoms is None:
        numAtoms = len(coordList)
      elif numAtoms != len(coordList):
        raise Exception('Attempt to compare structures with different atoms')

      if not weights:
        for coordObj in coordList:
          chemAtom = coordObj.atom.chemAtom
          mass = chemAtom.chemElement.mass
          weights.append(min(14.0,mass))

      coords = [[0.0,0.0,0.0]] * numAtoms
      for i in xrange(numAtoms):
        coordObj = coordList[i]
        coords[i] = [coordObj.x, coordObj.y, coordObj.z]
      coordsDict[model] = coords
      coordListDict[model] = coordList

  residueRmsdDict = {}
  residueCountDict = {}
  ensemble  = 2*[0]
  for model1 in structure1.sortedModels():
    ensemble[0] = coordsDict[model1]
    allCoord = coordListDict[model1]
    for model2 in structure2.sortedModels():
      ensemble[1] = coordsDict[model2]

      # Call to the C code to do the alignment
      error, atomRmsds, structureRmsds = alignEnsemble(ensemble, weights)
      weights = [x**-2.0 for x in atomRmsds]
      error, atomRmsds, structureRmsds = alignEnsemble(ensemble, weights)
      
      # atomRmsds are out by sqrt(2) because C code
      # divides by nensembles not nensembles-1
      for i, coord in enumerate(allCoord):
        rmsd = atomRmsds[i]
        residue = coord.atom.residue.residue
        residueRmsdDict[residue] = residueRmsdDict.get(residue, 0) + 2 * rmsd * rmsd
        residueCountDict[residue] = residueCountDict.get(residue, 0) + 1

  rmsd2Tot = 0
  countTot = 0
  for residue in residueRmsdDict:
    rmsd2 = residueRmsdDict[residue]
    rmsd2Tot += rmsd2
    count = residueCountDict[residue]
    countTot += count
    residueRmsdDict[residue] = sqrt(rmsd2 / count)

  if countTot > 0:
    rmsd = sqrt(rmsd2Tot / countTot)

  return rmsd, residueRmsdDict

def getMeanStrucCoords(structureCoords):
  """Find the mean position of input coordinates of multiple structures.
  .. describe:: Input
  
  List of List of List of Floats (x,y,z per atom per structure)

  .. describe:: Output

  List of List of Floats (mean x,y,z per atom)
  """

  meanCoords = []
  N = len(structureCoords)
  
  for coord in range(len(structureCoords[0])):
    sx = 0
    sy = 0
    sz = 0
    for s in range(N):
      sx  += structureCoords[s][coord][0] 
      sy  += structureCoords[s][coord][1] 
      sz  += structureCoords[s][coord][2]

    meanCoords.append([sx/N,sy/N,sz/N])

  return meanCoords

def getRmsd(structureCoords):
  """Find the root mean square devatation for a list of structures
             (their coordinates in lists)
  .. describe:: Input
  
  List of List of List of Floats (x,y,z per atom per structure)

  .. describe:: Output

  Float
  """

  N  = 0
  d2 = 0
  for coord in range(len(structureCoords[0])):
    d2i = 0
    Ni = 0
    for s1 in range(len(structureCoords)-1):
      for s2 in range(s1+1,len(structureCoords)):
        dx  = structureCoords[s1][coord][0] - structureCoords[s2][coord][0]
        dy  = structureCoords[s1][coord][1] - structureCoords[s2][coord][1]
        dz  = structureCoords[s1][coord][2] - structureCoords[s2][coord][2]
        d2  += (dx*dx) + (dy*dy) + (dz*dz)
        d2i += (dx*dx) + (dy*dy) + (dz*dz)
        Ni += 1
        N  += 1
    d2i = sqrt(d2i/Ni)
    #print "RMSD>", coord, d2i

  return sqrt(d2/N)
  
def getStructureCoordinates(structure, model=None, atomNameDict=None):
  """Get a list of coordinate objects from a structure in a consistent order.
             Option to select coords from a given model if required.
             Option to extract only atoms of certain names
  .. describe:: Input
  
  MolStructure.StructureEnsemble, MolStructure.Model, List of Strings

  .. describe:: Output

  List of List of MolStructure.Coords (1st list per atom 2nd per model)
  """

  if model:
    models = [model,]
  else:  
    models = structure.sortedModels()

  coordList = [[] for n in models]

  for chain in structure.sortedCoordChains():
    for residue in chain.sortedResidues():
      atomNames = atomNameDict and atomNameDict.get(residue.residue.molType, [])
      for atom in residue.sortedAtoms():
        if atomNameDict and atom.name not in atomNames:
          continue
        for i, model in enumerate(models):
          coord = atom.findFirstCoord(model=model)
          coordList[i].append(coord)

  return coordList

def centerCoordinates(coords):
  """More the centre of mass of input coordinates to the origin.
  .. describe:: Input
  
  List of List of Floats (x,z,y coords)

  .. describe:: Output

  List of List of Floats (x,z,y coords)
  """

  if coords:
    centreOfMass = getCentreOfMass(coords)
    vector = [-c for c in centreOfMass]
    moveCoords(coords, vector)
    return coords


def centerStructures(structures):
  """Move the input structures' centre of mass  to the origin.
  .. describe:: Input
  
  List of MolStructure.StructureEnsembles

  .. describe:: Output

  None
  """

  for structure in structures:
    coords = getStructureCoordinates(structure)
    centerCoordinates(coords)
 
def moveCoords(coords, vector):
  """Move data model coordinates according to a vactor.
  .. describe:: Input
  
  List of MolStructure.Coords, List of Floats (translation vector)

  .. describe:: Output

  List of MolStructure.Coords
  """

  (dx,dy,dz) = vector
  for coord in coords:
    coord.x += dx
    coord.y += dy
    coord.z += dz

  return coords

def getMeanCoords(coordList):
  """Find the mean position of data model coordinates.
  .. describe:: Input
  
  List of MolStructure.Coords

  .. describe:: Output

  List of Floats (x,y,z coords of mean)
  """

  N = len(coordList)
  x = 0
  y = 0
  z = 0
  for coord in coordList:
    x += coord.x
    y += coord.y
    z += coord.z
    
  if N > 0:
    x /= N
    y /= N
    z /= N
   
  return (x,y,z) 

def getCentreOfMass(coords):
  """Find the center of mass of data model coordinates. (Uss atomic weights)
  .. describe:: Input
  
  List of MolStructure.Coords

  .. describe:: Output

  List of Floats (x,y,z centre of mass)
  """

  (cx,cy,cz) = [0,0,0]

  M = 0
  for coord in coords:
    atom = coord.atom
    mass = atom.atom.chemAtom.chemElement.mass
    coord = atom.findFirstCoord()
    cx += mass * coord.x
    cy += mass * coord.y
    cz += mass * coord.z
    M  += mass 
    
  cx = cz/M
  cy = cz/M
  cz = cz/M

  return (cx,cy,cz)

def getResiduePhiPsi(residue, inDegrees=True, model=None):
  """Find the Phi and Psi backbone dihedral angles for a residue in a structure.
             Option inDegrees can be set to False to get an angle in radians.
             Option to specify which model of an ensemble to use, otherwise
             all models are considered and the angles are an average.
  .. describe:: Input
  
  MolStructure.Residue, Boolean, MolStructure.Model

  .. describe:: Output

  2-List of Floats (Phi, Psi)
  """

  chain = residue.chain
  sysResidue = residue.residue
  
  if not sysResidue:
    return None, None

  if model:
    models = [model,]
    structure = model.structureEnsemble
  else:
    structure = chain.structureEnsemble
    models = structure.sortedModels() 

  N = float(len(models))

  sysResiduePrev = getLinkedResidue(sysResidue, linkCode='prev')
  sysResidueNext = getLinkedResidue(sysResidue, linkCode='next')

  if not (sysResiduePrev and sysResidueNext):
    return None, None
    
  meanPhi = None
  meanPsi = None
  residuePrev = chain.findFirstResidue(seqId=sysResiduePrev.seqId)
  residueNext = chain.findFirstResidue(seqId=sysResidueNext.seqId)
  
  if residuePrev and residueNext:
    atomC0 = residuePrev.findFirstAtom(name='C')
    atomN  = residue.findFirstAtom(name='N')
    atomCa = residue.findFirstAtom(name='CA')
    atomC  = residue.findFirstAtom(name='C')
    atomN2 = residueNext.findFirstAtom(name='N')
 
    if atomC0 and atomN and atomCa and atomC and atomN2:
      angles = []
      
      for model in models:
        coords = []
        for atom in (atomC0, atomN, atomCa, atomC, atomN2):
          coordObj = atom.findFirstCoord(model=model)
          coords.append( [coordObj.x,coordObj.y,coordObj.z] )
 
        if inDegrees:
          phi = calcTorsionAngleDegrees(coords[0],coords[1],coords[2],coords[3])
          psi = calcTorsionAngleDegrees(coords[1],coords[2],coords[3],coords[4])
 
        else:
          phi = calcTorsionAngleRadians(coords[0],coords[1],coords[2],coords[3])
          psi = calcTorsionAngleRadians(coords[1],coords[2],coords[3],coords[4])
 
        angles.append((phi,psi))
 
      if N == 1.0:
        meanPhi, meanPsi = angles[0]
        
      else:
        sumCosPhi = 0.0
        sumSinPhi = 0.0
        sumCosPsi = 0.0
        sumSinPsi = 0.0
        
        for phi, psi in angles:
        
          if inDegrees:
            phi = TWOPI * phi/360.0
            psi = TWOPI * psi/360.0

          sumCosPhi += cos(phi)
          sumSinPhi += sin(phi) 
          sumCosPsi += cos(psi)
          sumSinPsi += sin(psi) 

        meanPhi = atan2(sumSinPhi/N,sumCosPhi/N)
        meanPsi = atan2(sumSinPsi/N,sumCosPsi/N)
        
        if inDegrees:
          meanPhi *= 360/TWOPI
          meanPsi *= 360/TWOPI
 
  return meanPhi, meanPsi
  

def getAtomSetCoords(atomSet, structure, model=None):
  """Find the coordinates corresponding to an NMR atom set in a given model
  of a given structure structure. The model defaults to an arbitrary one
  if none is specified.
  
  .. describe:: Input
  
  Nmr.AtomSet, MolStructure.StructureEnsemble, MolStructure.Model

  .. describe:: Output

  List of List of Floats (x,y,z for each atom)
  """

  if not model:
    model = structure.findFirstModel()

  key = '%s:%s:%d:%d' % (atomSet, # Could be real AtomSet or a ConstraintSet one!
                         structure.molSystem.code,
                         structure.ensembleId,
                         model.serial)

  if not hasattr(structure,'coordDict'):
    structure.coordDict = {}
    
  else:
    coordList = structure.coordDict.get(key)
 
    if coordList:
      return coordList

  atom    = atomSet.findFirstAtom()
  residue = atom.residue
  chain   = residue.chain
  coordChain = structure.findFirstCoordChain(code=chain.code)
  if not coordChain:
    #showWarning('Warning', 'Couldn\'t find coordinate chain')
    return []
  
  coordResidue = coordChain.findFirstResidue(seqId=residue.seqId)
  if not coordResidue:
    data = (residue.ccpCode,residue.seqCode)
    msg  = 'Couldn\'t find coordinate residue %s %d' % data
    print msg
    #showWarning('Warning', msg)
    return []

  coordList = []
  findAtom = coordResidue.findFirstAtom

  for atom in atomSet.atoms:
    coordAtom = findAtom(name=atom.name)
      
    if coordAtom:
      coord = coordAtom.findFirstCoord(model=model)
      
      if coord:
        coordList.append(coord)

  if not coordList:
    data = (chain.code, residue.ccpCode, residue.seqCode, atomSet.name)
    print 'Couldn\'t find coordinate atoms %s %s %d %s' % data
    return []
  
  structure.coordDict[key] = coordList
  
  return coordList

def getAtomSetsDihedral(atomSets, structure, model=None, inDegrees=True):
  """Measure the dihedral angle in a structure between four groups
             of atoms (lists of atom sets). Input atom sets are a list of lists
             to allow for ambigous assignments (e.g. Ser Hb*) here the mean position is used.
             If mo nodel is specified the values are averates over the whole ensemble.
             Option to specify degrees or radians. 
  .. describe:: Input
  
  4 List of List of Nmr.AtomSets or 4 List of Nmr.AtomSets,
             MolStructure.StructureEnsemble, Int, Boolean

  .. describe:: Output

  Float 
  """

  assert len(atomSets) == 4

  if model:
    models = [model,]
  else:
    models = structure.sortedModels()

  for i, atomSet in enumerate(atomSets):
    if hasattr(atomSet, 'className') and (atomSet.className == 'AtomSet'):
      atomSets[i] = [atomSet,]
 
  angles = []
  
  for model in models:
    coords = []
    
    for atomSetList in atomSets:
      coord = [0.0,0.0,0.0]
      n = 0.0
      # average over ambiguous atom sets e.g. Val Hg1* & Hg2*
      for atomSet in atomSetList:
        # average over atoms in atom set e.g. methyl
        for coord0 in getAtomSetCoords(atomSet, structure, model):
          coord[0] += coord0.x
          coord[1] += coord0.y
          coord[2] += coord0.z
          n += 1.0
      
      if n:
        coords.append([v/n for v in coord])

    if coords and len(coords) == 4:
      angle = calcTorsionAngleRadians(coords[0],coords[1],coords[2],coords[3])
      angles.append(angle)

  meanVal = None
  sumCos  = 0.0
  sumSin  = 0.0
  
  N = float(len(angles))
  for a in angles:
    sumCos += cos(a)
    sumSin += sin(a) 

  if N:
    sumCos /= N
    sumSin /= N
    meanVal = atan2(sumSin/N,sumCos/N)

    if inDegrees:
      meanVal *= 360/TWOPI

  return meanVal

def getAtomSetsDistance(atomSets1, atomSets2, structure, model=None, method='noe'):
  """
  Find the distance between two atom sets in a specified structure or ensemble
  of structures. Distances for multi atom atom sets are calculated using the
  NOE sum method by default. A model may be specified if the structure
  ensemble has many models, otherwise all models in the ensemble will be
  considered and an avergate distance is returned. The method can be either
  "noe" for NOE sum, "min" for minimum distance or "max" for maximum distance.
  
  .. describe:: Input
  
  Nmr.AtomSet, Nmr.AtomSet, MolStructure.StructureEnsemble,
  MolStructure.Model or None, Word

  .. describe:: Output

  Float (distance)
  """
  
  if model:
    models = [model,]
  else:
    models = structure.models 

  atomSets1 = set(atomSets1)
  atomSets2 = set(atomSets2)

  assert atomSets1 and atomSets2 and structure
  
  if atomSets1 == atomSets2:
    if len(atomSets1) == 2:
      atomSets0 = list(atomSets1)
      for resonanceSet in atomSets0[0].resonanceSets:
        if resonanceSet in atomSets0[1].resonanceSets: # Prochiral
          atomSets1 = [atomSets0[0],]
          atomSets2 = [atomSets0[1],]
          break
    
    if atomSets1 == atomSets2: # Not prochiral
      return 0.0

  ensembleCoords = []
  ensembleCoordsAppend = ensembleCoords.append
  
  for model in models:
    coordList1 = []
    coordList2 = []
    coordList1Append = coordList1.append
    coordList2Append = coordList2.append
    
    for atomSet in atomSets1:
      for coord in getAtomSetCoords(atomSet, structure, model):
        coordList1Append(coord)
      
    for atomSet in atomSets2:
      for coord in getAtomSetCoords(atomSet, structure, model):
        coordList2Append(coord)

    if coordList1 and coordList2:
      ensembleCoordsAppend((coordList1,coordList2))
 
  if not ensembleCoords:
    return
  
  numPairs = float(len(atomSets1)*len(atomSets2))  
  noeEnsemble = 0.0
  minDist2 = None
  maxDist2 = None

  n = 0.0
  for coords1, coords2 in ensembleCoords:
    noeSum = 0.0

    for coord1 in coords1:
      x = coord1.x
      y = coord1.y
      z = coord1.z
      
      for coord2 in coords2:
        dx = x - coord2.x
        dy = y - coord2.y
        dz = z - coord2.z
        dist2 = (dx*dx) + (dy*dy) + (dz*dz)
        
        if dist2 > 0:
          
          if method == 'noe':
            noeSum += dist2 ** -3.0
          
          else:
            if (minDist2 is None) or (dist2 < minDist2):
              minDist2 = dist2

            if (maxDist2 is None) or (dist2 > maxDist2):
              maxDist2 = dist2

    noeEnsemble += sqrt(noeSum/numPairs)
    n += 1.0

  if method == 'min':
    return sqrt(minDist2)
  elif method == 'max':
    return sqrt(maxDist2)
  elif (noeEnsemble > 0.0) and n:
    noeEnsemble /= n
    return noeEnsemble ** (-1/3.0)


def getBestNamingSystem(residues, atomNamesList):
  """Determine the best naming system for a list of list of atom names
             which correspond to the input residues.
  .. describe:: Input
  
  List of MolSystem.Residues, List of List of Words (imported atom names)

  .. describe:: Output

  Word (ChemAtomSysName.namingSystem)
  """
  
  scoreDict = {}
  chemComps = set([])
  for i, residue in enumerate(residues):
    if not residue: # Could be mismatched position from seq alignment
      continue
      
    atomNames = atomNamesList[i]
    chemComp  = residue.molResidue.chemComp
    
    if chemComp in chemComps:
      continue
      
    chemComps.add(chemComp)
    for namingSystem in chemComp.namingSystems:
      name = namingSystem.name
      found = set([])
      
      for atomName in atomNames:
        namingSystemR = namingSystem
        done = False
        
        while namingSystemR:
          # First choice plain mapping
          for atomSysName in namingSystemR.findAllAtomSysNames(sysName=atomName):
            found.add(atomSysName.sysName)
            done = True            
            break
 
          # Otherwise try alternative sys names
          else:
            for atomSysName0 in namingSystemR.atomSysNames:
              if atomName in atomSysName0.altSysNames:
                found.add(atomSysName0.sysName)
                done = True
          
          if done:
            break
          
          namingSystemR = namingSystemR.atomReference
 
      scoreDict[name] = len(found)


  namingSystems = ['IUPAC','PDB','PDB_REMED','BMRB','XPLOR','CYANA2.1','DIANA',
                   'GROMOS','MSD','SYBYL','UCSF','AQUA','DISGEO','DISMAN',
                   'MOLMOL','MSI']
   
  bestSc = 0
  bestNs = 'PDB'
  for ns in namingSystems:
    if scoreDict.get(ns, 0) > bestSc:
      bestSc = scoreDict[ns]
      bestNs = ns
  
  return bestNs

def getBestChemComp(project, resName, atomNames, molType=None):
  """Find the best matching ccpCode (often 3 letters) for the input
             residue name that has the input atomNames 
  .. describe:: Input
  
  Implementation.Project, Word (imported residue name),
             List of Words (imported atom names)

  .. describe:: Output

  Word (Molecule.MolResidue.ccpCodes)
  """
  
  chemComp = None
  if not molType:
    molType  = getBestMolType(atomNames)
  
  if molType in ('DNA','RNA','DNA/RNA'):
    if len(resName) == 1:
      if 'PD' in atomNames:
        resName = resName + '11'
        
  if molType == 'protein':
    resName = resName[0] + resName[1:].lower()
  
  chemComp = project.findFirstChemComp(ccpCode=resName.upper(), molType=molType) or \
             project.findFirstChemComp(ccpCode=resName, molType=molType)
  
  if chemComp:
    return chemComp
    
  ccpCodeDict = {}
  chemComps = getStdChemComps(project, molTypes=[molType,])
  
  for chemComp0 in chemComps:
    ccpCodeDict[chemComp0.ccpCode] = chemComp0
    ccpCodeDict[chemComp0.code3Letter] = chemComp0
  
  chemComp = ccpCodeDict.get(resName)
  
  if chemComp is None:
    for chemCompTest in chemComps:
      for namingSystem in chemCompTest.namingSystems:
        for sysName in namingSystem.chemCompSysNames:
          if sysName.sysName == resName:
            ccpCodeDict[resName] = chemCompTest
            chemComp = chemCompTest
            break
 
      else:
        continue
      break 
    
    if not chemComp:
      chemComp = getChemComp(project, molType, resName)
      
    if not chemComp and molType != 'other':
      chemComp = getChemComp(project, 'other', resName)
      if not chemComp:
        resName = resName[0] + resName[1:].lower()
        chemComp = getChemComp(project, 'other', resName)
    
  return chemComp

def getBestMolType(atomNames, ccpCodes=None):
  """Determine the best molecule type (protein, DNA, RNA, carbohydrate or
             nonpolymer) given the input atom names and residue ccpCodes
  .. describe:: Input
  
  List of Words (imported atom names),
             List of Words (Molecule.MolResidue.ccpCodes)

  .. describe:: Output

  Word (Molecule.Molecule.molType)
  """

  molType = 'other'
  
  if ("C3'" in atomNames) and ("C5'" in atomNames) and ("C2" in atomNames):
    molType = 'DNA'
    if "O2'" in atomNames:
      molType = 'RNA'
      
  elif ("C3*" in atomNames) and ("C5*" in atomNames) and ("C2" in atomNames):
    # PDB Naming system different from others
    molType = 'DNA'
    if "O2*" in atomNames:
      molType = 'RNA'
 
  elif 'CA' in atomNames:
    molType = 'protein'

  elif ("C1" in atomNames) and ("C2" in atomNames) and ("C3" in atomNames) and ("C4" in atomNames) \
       and ( ("O2" in atomNames) or ("O3" in atomNames) or ("04" in atomNames)):
    molType = 'carbohydrate'
  
  return molType


def getStructureFromFile(molSystem, filePath, fileType='rough', doWarnings=True):
  """Creates a structure ensemble belonging to a molecular system 
             by loading a PDB style file.
             Option to specify the type of file loaded (proper PDB, rough PDB or CNS style)
  .. describe:: Input
  
  MolSystem.MolSystem, String (File Name), String (file type)

  .. describe:: Output

  MolStructure.StructureEmsemble
  """
  
  if fileType == 'true':  
    coordDict = makeStructureDictFromPdb(filePath, fileType='pdb')
  elif fileType == 'cns':
    coordDict = makeStructureDictFromPdb(filePath, fileType='cns')
  else:
    coordDict = makeStructureDictFromRoughPdb(filePath)

  ensemble = None

  if coordDict:
    ensemble = makeStructures(coordDict, molSystem, doWarnings=doWarnings)
    models = ensemble.sortedModels()
    fileRoot, fileExt = path.splitext(path.split(filePath)[1])
    
    for model in models:
      if len(models) > 1:
        details = 'file:%s (MODEL %d)' % (filePath, model.serial)
        name = '%s %d' % (fileRoot, model.serial)
      else:
        details = 'file:%s' % (filePath)
        name = fileRoot
      
      model.details = details
      model.name = name
    
  return ensemble

def makeEnsemble(structures, replaceStructures=True, modelSerials=None):
  """Combine the input structure ensembles (or selected models from them)
             into a single structure ensemble. Option to replace the input
             structures entirely (i.e. delete them), Option to use only certain
             models: If set uses only model numbers in input list.
  .. describe:: Input
  
  List of MolStructure.StructureEmsembles, Boolean,
             List of Ints (MolStructure.Model.serial) or None

  .. describe:: Output

  MolStructure.StructureEmsemble
  """
  
  structures = list(structures)
  structure0 = structures[0] # Abitrary but it is checked to be consistent
  molSystem  = structure0.molSystem
  coordDict  = {}
  
  keys = []
  modelDict = {}
  modelDetails = []
  
  # Check input, collate models and coordinates 
  
  m = 0
  for structure in structures:
    for model in structure.sortedModels():
      if modelSerials and (model.serial not in modelSerials):
        continue
    
      modelDict[model] = m
      modelDetails.append( (model.name, model.details) )
      m += 1
  
    if structure.molSystem is not molSystem:
      showWarning('Ensemble Creation Failed','Molecular System mismatch')
  
    i = 0
    for chain in structure.sortedCoordChains():
      chainCode = chain.code
      
      for residue in chain.sortedResidues():
        resId = str(residue.seqId) 
        
        for atom in residue.sortedAtoms():
          key = '%s.%s.%s' % (chainCode, resId, atom.name)
        
          if i >= len(keys):
            keys.append(key)
            
          else:
            if keys[i] != key:
              msg = 'Structures do not match at atom %s' % key
              showWarning('Ensemble Creation Failed', msg)
              return 
          
          if not coordDict.has_key(key):
            coordDict[key]= []
  
          coordList = list(atom.coords)
          if modelSerials:
            coordList = [c for c in coordList if c.model.serial in modelSerials]
  
          coordDict[key].extend(coordList)
          i += 1
  
  # Get the object that will carry all the coords
  
  if replaceStructures: 
    ensemble = structure0
  
  else:
    project = molSystem.root
    num = 1
    
    while project.findFirstStructureEnsemble(molSystem=molSystem, ensembleId=num):
      num += 1
 
    ensemble = project.newStructureEnsemble(molSystem=molSystem, ensembleId=num)
 
    for chain0 in structure0.coordChains:
      chain = ensemble.newChain(code=chain0.code)
 
      for residue0 in chain0.residues:
        residue = chain.newResidue(seqCode=residue0.seqCode,
                                   seqId=residue0.seqId,
                                   seqInsertCode=residue0.seqInsertCode)
        
        for atom0 in residue0.atoms:
          atom = residue.newAtom(name=atom0.name)
  
  # Make the required models in target ensemble
  
  while len(ensemble.models) < m:
    ensemble.newModel()
  
  models = ensemble.sortedModels()
  
  # Transfer any original details etc
  
  for i, (name, details) in enumerate(modelDetails):
    model = models[i]
    model.name = name
    model.details = details
  
  # Add coordinates to target ensemble
  
  for chain in ensemble.coordChains:
    chainCode = chain.code
 
    for residue in chain.residues:
      resId = str(residue.seqId)
 
      for atom in residue.atoms:
        key = '%s.%s.%s' % (chainCode, resId, atom.name)
        
        for coord in coordDict[key]:
          model = coord.model
          
          if model.structureEnsemble is not ensemble:
            i = modelDict[model]
            
            atom.newCoord(model=models[i],
                          altLocationCode=coord.altLocationCode,
                          bFactor=coord.bFactor,
                          occupancy=coord.occupancy,
                          x=coord.x,
                          y=coord.y,
                          z=coord.z)
  
  # Cleanup old structures if required
              
  if replaceStructures:
    for structure in structures:
      if structure is not ensemble:
        structure.delete()

  return ensemble

def makeStructures(strucDict, molSystem, doWarnings=False):
  """Makes structure ensemble from a structure dictionary
             [model][chainCode][resNum][atomName] = (x,y,z coords)
             in a mol system. Options to supress warnings and thus
             automatically link non-identical, but similar chains
             or make a new chain if none is found,
  .. describe:: Input
  
  Dictionary, MolSystem.MolSystem, Boolean

  .. describe:: Output

  MolStructures.StructureEnsemble
  """

  from ccp.util.Molecule import nextChainCode

  project = molSystem.root
  
  structures = molSystem.sortedStructureEnsembles()
  if structures:
    eId = structures[-1].ensembleId + 1
  else:
    eId = 1

  structure = project.newStructureEnsemble(molSystem=molSystem, ensembleId=eId)
  
  models = []
  for m in strucDict.keys():
    models.append((m, structure.newModel()))

  usedChains = []
  failedAtoms = []
  msAtomDict = {}
  chainsDict = strucDict[0]
  
  chCodes = list(chainsDict.keys())
  chCodes.sort()
  nOrigChCodes = len(chCodes)
  iChCode = 0
  for chCode in chCodes:
    iChCode += 1
    resDict = chainsDict[chCode]
    seqIds  = resDict.keys()
    seqIds.sort()

    chemComps  = []
    ccpCodes   = []
    seqIdsGood = []
    
    molTypes = set()
    resNames = []
    for seqId in seqIds:
      ll = resDict[seqId].keys()
      resName   = resDict[seqId][ll[0]]['resName']
      atomNames = [tt[0] for tt in ll]
      chemComp  = getBestChemComp(project, resName, atomNames)
      resNames.append(resName)
      
      if chemComp:
        chemComps.append(chemComp)
        ccpCodes.append(chemComp.ccpCode)
        seqIdsGood.append(seqId)
        molTypes.add(chemComp.molType)

    if not seqIdsGood:
      msg  = 'Could not find any matching CCPN ChemComps for sequence: '
      msg += ' '.join(resNames)
      showWarning('Structure Creation Failed',msg)
      continue
      
    seqIds = seqIdsGood
    msChains, mappings = findMatchingChains(molSystem, ccpCodes,
                                            excludeChains=usedChains,
                                            doWarning=doWarnings,
                                            molTypes=molTypes)
    
    nChains = len(msChains)
    if nChains == 1:
      msChain = msChains[0]
      mapping = mappings[0]
    
    elif nChains:
      from memops.gui.DataEntry import askString
       
      codes = [c.code for c in msChains]
      msg = 'Structure chain %s matches %d \nCCPN chains' % (chCode, nChains)
      msg += ' equally well: %s\n' % ( ' '.join(codes),)
      msg += 'Which CCPN chain should be linked to?'
      
      if chCode in codes:
        default = chCode
      else:
        default = codes[0]
      
      choice = None
      while choice not in codes:
        choice = askString('Query', msg, default) or ''
        choice = choice.strip()
      
      index = codes.index(choice)                        
      msChain = msChains[index]
      mapping = mappings[index]
      
    else:
      msChain = None
      
    if nChains and doWarnings:
      nRes = len(mapping)
      startPair = 0
      endPair = nRes-1
      
      for i in range(nRes):
        residueA, residueB = mapping[i]
        if residueA and residueB:
          startPair = i
          break

      for i in range(nRes):
        j = nRes-i-1
        residueA, residueB = mapping[j]
        if residueA and residueB:
          endPair = j
          break
      
      if (i > 10) or (startPair > 10):
        msg  = 'Structure appears to be truncated by more than 10 residues, '
        msg += 'compared to existing chain. Really link to chain %s?' % msChain.code
        msg += ' (Otherwise a new chain will be made)' 
        if not showYesNo('Query', msg):
          msChain = None

      else:
        for i in range(startPair, endPair+1):
          if None in mapping[i]:
            msg  = 'Structure sequence matches an existing chain, '
            msg += 'but not exactly. Really link to chain %s?' % msChain.code
            msg += ' (Otherwise a new chain will be made)'
            if not showYesNo('Query', msg):
              msChain = None
            break
        
    if not msChain:
      # no matching chain - make new one
      # use existing chain code from PDB file if possible
      sysChainCode = chCode.strip()
      if not sysChainCode or molSystem.findFirstChain(code=sysChainCode):
        sysChainCode = nextChainCode(molSystem)
      
      if molTypes == set(('DNA',)) and iChCode <= nOrigChCodes:
        # check for special case: 
        # double stranded DNA with a single chain code
        oneLetterCodes = [cc.code1Letter for cc in chemComps]
        if None not in oneLetterCodes:
          # All ChemComps are (variants of) Std bases. 
          # Not sure certain, but this should mostly work.
          nCodes = len(oneLetterCodes)
          halfway, remainder = divmod(nCodes, 2)
          if not remainder:
            # even number of codes
            resMap = { 'A': 'T', 'T': 'A', 'G': 'C', 'C': 'G' }
            for ii in range(halfway):
              if oneLetterCodes[ii] != resMap[oneLetterCodes[-1-ii]]:
                break
            else:
              # the second half is the reverse complement of the first half.
              # treat as two separate DNA chains
              # Move second half to a new chain, and remove from this chain.
              newChCode = chCode
              while newChCode in chCodes:
                newChCode=chr(ord(newChCode)+1)
              newResDict = chainsDict[newChCode] = {}
              for ii in range(halfway, nCodes):
                iix = seqIds[ii]
                newResDict[iix] = resDict[iix]
                del resDict[iix]
              
              # put both chains on end of list for later (re)processing
              chCodes.append(chCode)
              chCodes.append(newChCode)
              continue
              
      codes = (molSystem.code, sysChainCode)
      msg  = 'Structure residue sequence (chain %s) not in molecular system. ' % chCode
      msg += 'Make new molecular system %s chain (%s) for this sequence?' % codes
      if not doWarnings or (not molSystem.chains) or (doWarnings and showOkCancel('Confirm',msg)):
                
        atomNames   = resDict[seqIds[0]].keys()
 
        molType   = chemComps[0].molType
        ccpCodes0 = []
        startNum  = resDict[seqIds[0]][atomNames[0]]['seqCode']
 
        project = molSystem.root
        molecule = makeMolecule(project, molType, [])

        for i in range(len(seqIds)):
          chemComp = chemComps[i]
 
          if (chemComp.molType != molType) or (i and (seqIds[i] != seqIds[i-1]+1)):
            newMolResidues = addMolResidues(molecule, molType, ccpCodes0, 
                                            startNum=startNum)
            
            # set seqCodes and seqInsertCodes
            xSeqIds = seqIds[i-len(newMolResidues):i]
            for j,x in enumerate(xSeqIds):
              rr = newMolResidues[j]
              for dummy in resDict[x]:
                # future-safe of getting a random atomDict for seqId x
                dd = resDict[x][dummy]
                break
              rr.seqCode = dd['seqCode']
              rr.seqInsertCode = dd['insertCode']
            
            ccpCodes0 = [chemComp.ccpCode,]
            molType   =  chemComp.molType
            startNum  = seqIds[i]

          else:
            ccpCodes0.append(chemComp.ccpCode)
 
        if ccpCodes0:
          addMolResidues(molecule, molType, ccpCodes0, startNum=startNum)
 
        msChain = makeChain(molSystem,molecule,code=sysChainCode)
        
        resMapping = {}
        for i, residue in enumerate(msChain.sortedResidues()):
          resMapping[i] = residue
 
        # TBD deal with HETATMs, proper Molecule name,
        # store CCPN xml as non-standard naming system?

      else:
        continue
        
    else:
      sysChainCode = msChain.code
      resMapping = {}
      for i, residue in mapping:
        resMapping[i] = residue
    
    usedChains.append(msChain)
      
    atomNamesList = []
    msResidues = []
    for j, seqId in enumerate(seqIds):
      atomNames = [tt[0] for tt in resDict[seqId].keys()]
      atomNamesList.append(atomNames)
      msResidues.append(resMapping.get(j))
 
    namingSystem = getBestNamingSystem(msResidues, atomNamesList)
    
    coordChain = structure.newChain(code=sysChainCode)
    structure.override = True
 
    for j, seqId in enumerate(seqIds):
      msResidue = msResidues[j]
 
      if not msResidue: # Structure is bigger
        continue

      resName = ccpCodes[j]
      ccpCode = msResidue.ccpCode
 
      if doWarnings:
        if resName != ccpCode:
          msg  = 'Residue names [%s,%s] don\'t match\nin'
          msg += ' loaded molecular system\nchain %s position %d'
          data = (resName,ccpCode,chCode,seqId)
          showWarning('Warning', msg % data)
          continue
 
        if msResidue is None:
          msg  = 'No equivalent molecular system residue'
          msg += '\nfor PDB chain %s residue %d'
          showWarning('Warning', msg % (chCode,seqId))
          continue
 
      coordResidue = coordChain.newResidue(seqCode=msResidue.seqCode,
                                           seqInsertCode=msResidue.seqInsertCode,
                                           seqId=msResidue.seqId)

      atomCheckDict = {}
      systemAtoms = set()
      for atomKey in resDict[seqId].keys():
        
        atomName, altLoc = atomKey
      
        key = '%s:%s:%s' % (atomName, msResidue, namingSystem)
        systemAtom = msAtomDict.get(key)
        
        if not systemAtom:
          systemAtom = findMatchingMolSystemAtom(atomName,
                                                 msResidue,
                                                 namingSystem,
                                                 systemAtoms,)
          msAtomDict[key] = systemAtom
 
          if (systemAtom is None) or atomCheckDict.get((systemAtom, altLoc)):
            failedAtoms.append('%s %d %s' % (chCode,seqId,atomName) )
            continue
            
        systemAtoms.add(systemAtom)
        atomCheckDict[(systemAtom, altLoc)] = True
        coordAtom = coordResidue.findFirstAtom(name=systemAtom.name)
        if coordAtom is None:
          # we must have altLocation alternatives here
          coordAtom = coordResidue.newAtom(name=systemAtom.name)
 
        for m, model in models:
          coordDict = strucDict[m][chCode][seqId][atomKey]
          occupancy = coordDict.get('occupancy')
          bFactor   = coordDict.get('bFactor')

          x = coordDict['x']
          y = coordDict['y']
          z = coordDict['z']
          c = coordAtom.newCoord(x=x, y=y, z=z, altLocationCode=altLoc, 
                                 model=model)

          if occupancy is not None:
            c.setOccupancy(occupancy)
          if bFactor is not None:
            c.setBFactor(bFactor)


    # final validity check
    try:
      structure.checkAllValid()
    except:
      try:
        structure.delete()
      except:
        pass 
      raise
    # reset switches
    structure.override = False
 
  if failedAtoms and doWarnings:
    msg = 'No equivalent molecular system atoms for PDB atoms: %s'
    showWarning('Warning', msg % ( ' '.join(failedAtoms) ))

  if not structure.coordChains:
    structure.delete()
    structure = None

  return structure


def findMatchingMolSystemAtom(atomName, residue, namingSystem, excludeAtoms):
  """
  Find the best matching CCPN atom name in a residue for the input
  atom name in the input naming system Will try other naming systems
  if the input one doesn't work
  
  .. describe:: Input
  
  Word (imported atom name), MolSystem.Residue,
  Word (ChemComp.NamingSystem.name), List of MolSystem.Atoms

  .. describe:: Output

  Word (MolSystem.Atom.name)
  """
  #nulciec = ('DNA','RNA','DNA/RNA')
  #weirdos = {'O1P':'OP1','O2P':'OP2','C5A':'C7'}
  #if weirdos.get(atomName) and residue.molResidue.molType in nulciec:
  #  atomName = weirdos[atomName]

  atom = None
  chemComp = residue.chemCompVar.chemComp
  namingSystem0 = chemComp.findFirstNamingSystem(name=namingSystem)
  atomSysNames = []
  atomSysNamesAlt = []
  
  if namingSystem0:
    namingSystemR = namingSystem0
    
    while namingSystemR:
      # First choice plain mapping
      for atomSysName in namingSystemR.findAllAtomSysNames(sysName=atomName):
        atomSysNames.append(atomSysName)
          
      # Otherwise try alternative sys names
      for atomSysName0 in namingSystemR.atomSysNames:
        if atomName in atomSysName0.altSysNames:
          atomSysNamesAlt.append(atomSysName0)
          #break
      
      namingSystemR = namingSystemR.atomReference
  
  atomSysNames.extend(atomSysNamesAlt)
  # Next chance any naming system plain mapping 
  for namingSystem0 in chemComp.namingSystems:
    for atomSysName in namingSystem0.findAllAtomSysNames(sysName=atomName):
      atomSysNames.append(atomSysName)

  # Final chance any naming system alt name
  for namingSystem0 in chemComp.namingSystems:
    for atomSysName in namingSystem0.atomSysNames:
      if atomName in atomSysName.altSysNames:
        atomSysNames.append(atomSysName)
 
  # Find the molSystem atom
  for atomSysName in atomSysNames:
    if atomSysName.sysName == atomName:
      atom = residue.findFirstAtom(name=atomSysName.atomName)
      if atom:
        if atom in excludeAtoms:
          continue
          
        return atom
 
  # last resort: try any naming system alternative sys names
  for atomSysName in atomSysNames:
    if atomName in atomSysName.altSysNames:
      atom = residue.findFirstAtom(name=atomSysName.atomName)
      if atom:
        if atom in excludeAtoms:
          continue
          
        return atom
  
  if not atom:
    atom = residue.findFirstAtom(name=atomName)

  if not atom and atomName == 'HN':
    atom = residue.findFirstAtom(name='H')

  if not atom and atomName == 'O':
    atom = residue.findFirstAtom(name="O'")
    
  return atom

def makeStructureDictFromPdb(fileName, fileType):
  """Make a structure dictionary [model][chainCode][resNum][atomName] =
             (x,y,z coords) from a CNS or true PDB format PDB file.
  .. describe:: Input
  
  String (file name), fileType ('cns' or 'pdb')

  .. describe:: Output

  Dictionary
  """
  
  if fileType == 'cns':
    from ccp.format.cns import coordinatesIO
    coordFile = coordinatesIO.CnsCoordinateFile(fileName)
  elif fileType == 'pdb':
    from ccp.format.pdb import coordinatesIO
    coordFile = coordinatesIO.PdbCoordinateFile(fileName)
  else:
    raise Exception("Illegal fileType: %s" % fileType)
    
  coordFile.read(maxNum = 999)
  
  modelCoords = coordFile.modelCoordinates
  
  dict = {}
  for modelNum in modelCoords:
    # tracking dictionary
    trackDict = {}
    dict[modelNum] = {}
    lastSeqId = 0
    for coord in modelCoords[modelNum]:
      atomName  = coord.atomName
      chainId   = coord.chainId.rstrip()
      seqCode   = coord.seqCode
      insertCode   = coord.insertionCode
      segId     = coord.segId
      altLoc    = coord.altLoc
        
      if chainId == '':
        if segId:
          chainId = segId
        else:
          chainId = 'A'
      
      if dict[modelNum].get(chainId) is None:
        dict[modelNum][chainId] = {}
      
      tt1 = (chainId,seqCode,insertCode)
      seqId = trackDict.get(tt1)
      if seqId is None:
        seqId = trackDict[tt1] = lastSeqId + 1
        lastSeqId = seqId
        dict[modelNum][chainId][seqId] = {}
      
      tt2 = (atomName, altLoc)
      if dict[modelNum][chainId][seqId].get(tt2) is None:
        coordDict = {}
        dict[modelNum][chainId][seqId][tt2] = coordDict
      
      else:
        raise Exception("Duplicate record: %s %s %s %s %s" % 
                        (chainId, seqCode, insertCode, atomName, altLoc))
      
      coordDict['seqCode'] = seqCode
      coordDict['insertCode'] = insertCode
      coordDict['segId'] = segId
      
      coordDict['resName'] = coord.resName
      coordDict['x'] = coord.x
      coordDict['y'] = coord.y
      coordDict['z'] = coord.z
      
      if fileType == 'pdb':
        coordDict['occupancy'] = coord.occupancy
        coordDict['bFactor']   = coord.bFactor
        coordDict['atomType']  = coord.atomType
        coordDict['hetFlag']   = coord.hetFlag
       
  return dict


def makePdbFromStructure(fileName,structure,model=None,useOxt=False):
  """Make a PDB file from a structure or ensemble of structures.
             If a model is passed in the PDB style file will contain only
             that model's coordinates, otherwise all models will be used. 
  .. describe:: Input
  
  String (file name), MolStructure.StructureEnsemble,
             MolStructure.Model

  .. describe:: Output

  None
  """
  """
  COLUMNS        DATA  TYPE    FIELD        DEFINITION
  -------------------------------------------------------------------------------------
   1 -  6        Record name   "ATOM  "
   7 - 11        Integer       serial       Atom  serial number.
  13 - 16        Atom          name         Atom name.
  17             Character     altLoc       Alternate location indicator.
  18 - 20        Residue name  resName      Residue name.
  22             Character     chainID      Chain identifier.
  23 - 26        Integer       resSeq       Residue sequence number.
  27             AChar         iCode        Code for insertion of residues.
  31 - 38        Real(8.3)     x            Orthogonal coordinates for X in Angstroms.
  39 - 46        Real(8.3)     y            Orthogonal coordinates for Y in Angstroms.
  47 - 54        Real(8.3)     z            Orthogonal coordinates for Z in Angstroms.
  55 - 60        Real(6.2)     occupancy    Occupancy.
  61 - 66        Real(6.2)     tempFactor   Temperature  factor.
  77 - 78        LString(2)    element      Element symbol, right-justified.
  79 - 80        LString(2)    charge       Charge  on the atom.
  """

  if model:
    models = [model,]
  else:
    models = structure.sortedModels()
 
  fileHandle = open(fileName, 'w')
  lFormat = '%-80.80s\n'
  pdbFormat = '%-6.6s%5.1d %4.4s%s%3.3s %s%4.1d%s   %8.3f%8.3f%8.3f%6.2f%6.2f          %2.2s  \n'
  terFormat = '%-6.6s%5.1d      %s %s%4.1d%s                                                     \n'                            
  
  fileHandle.write('REMARK 210 REMARK:\n')
  fileHandle.write('REMARK 210\n')
  fileHandle.write('REMARK 210 Generated by CcpNmr Analysis\n')
  for model in models:
    remark = ('REMARK 210 CCPN ccp.molecule.MolStructure.Model %s' 
              % model.getExpandedKey())
    fileHandle.write(lFormat  % remark)
  
  #ins = ' '
  for m, model in enumerate(models):
    i = 0
    
    if len(models) > 1:
      line = 'MODEL     %4d' % (m+1)
      fileHandle.write(lFormat  % line)
  
    for chain in structure.sortedCoordChains():
      a = 'ATOM'
      if chain.chain.molecule.molType == 'nonpolymer':
        a = 'HETATM'
        
      ch = chain.chain.pdbOneLetterCode
      if not ch.strip():
        ch = chain.chain.code[0]
 
      for residue in chain.sortedResidues():
        chemComp = residue.residue.chemCompVar.chemComp
        tlc = chemComp.code3Letter
        s = residue.residue.seqCode
        ins = residue.residue.seqInsertCode

        for atom in residue.sortedAtoms():
          i    += 1
          n     = '%-3s' % atom.name
	  
	  if useOxt and (n=="O''"):
	    n = 'OXT'
	  
          coord = atom.findFirstCoord(model=model)
          alc   = coord.altLocationCode
          x     = coord.x
          y     = coord.y
          z     = coord.z
          o     = coord.occupancy
          b     = coord.bFactor
          e     = atom.atom.chemAtom.chemElement.symbol
          line  = pdbFormat % (a,i,n,alc,tlc,ch,s,ins,x,y,z,o,b,e)
          fileHandle.write(line)
          
      i += 1
      line  = terFormat % ('TER',i,tlc,ch,s,ins)
      fileHandle.write(line)

    if len(models) > 1:
      fileHandle.write(lFormat  % 'ENDMDL')
  
  fileHandle.write(lFormat  % 'END')
        
  fileHandle.close()
  

def makeStructureDictFromRoughPdb(fileName):
  """Make a structure dictionary [model][chainCode][resNum][atomName] =
             (x,y,z coords) from a non-standard PDB file.
  .. describe:: Input
  
  String (file name)

  .. describe:: Output

  Dictionary
  """
  
  residueKeys = set([])
  insertOffset = 0

  dict = {}
  modelNum = 0
  fileHandle = open(fileName)
  for line in fileHandle.readlines():
    
    key = line[0:6].strip()
    if key == 'ENDMDL':
      modelNum +=1
      insertOffset = 0
      residueKeys = set([])
      
    elif key == 'ATOM':
      #serial    = returnInt(line[6:11])
      atomName  = line[12:16].strip()
      altLoc    = line[16:17]
      resName   = line[17:20].strip()
      chainId   = line[21:22].strip()
      seqCode   = returnInt(line[22:26])
      insertCode = line[26:27]
      x         = returnFloat(line[30:38])
      y         = returnFloat(line[38:46])
      z         = returnFloat(line[46:54])
      occupancy = returnFloat(line[54:60])
      bFactor   = returnFloat(line[60:66])
      segId     = line[72:76].strip()
      atomType  = line[76:78].strip()
      seqId = seqCode + insertOffset
      #charge    = line[78:80].strip()
 
      if chainId == '':
        if segId:
          chainId = segId
        else:
          chainId = 'A'
 
      if dict.get(modelNum) is None:
        dict[modelNum] = {}
      if dict[modelNum].get(chainId) is None:
        dict[modelNum][chainId] = {}
        
      resKey = (chainId, seqCode, insertCode)
      if resKey not in residueKeys:
        # Not seen this residue key before
        residueKeys.add(resKey)
        
        if dict[modelNum][chainId].get(seqId) is not None:
          # SeqId is known -> seq insert
          # Residue number is one higher
          insertOffset += 1
          seqId += 1
        
        dict[modelNum][chainId][seqId] = {}
      
      atomKey = (atomName, altLoc)
      if dict[modelNum][chainId][seqId].get(atomKey) is None:
        coordDict = {}
        dict[modelNum][chainId][seqId][atomKey] = coordDict
      
      else:
        raise Exception("Duplicate record: %s %s %s %s %s" % 
                        (chainId, seqCode, insertCode, atomName, altLoc))
      
      coordDict['resName'] = resName
      coordDict['seqCode'] = seqCode
      coordDict['insertCode'] = insertCode
      coordDict['segId'] = segId
      coordDict['x'] = x
      coordDict['y'] = y
      coordDict['z'] = z
      coordDict['occupancy'] = occupancy
      coordDict['bFactor']   = bFactor
      coordDict['atomType']  = atomType
 
  return dict
  
'''
def makeStructureDictFromTruePdb(fileName):
  """Make a structure dictionary [model][chainCode][resNum][atomName] = 
             (x,y,z coords) from a well formatted PDB file.
  .. describe:: Input
  
  String (file name)

  .. describe:: Output

  Dictionary
  """
  from ccp.format.pdb import coordinatesIO

  coordFile = coordinatesIO.PdbCoordinateFile(fileName)
  coordFile.read(maxNum = 999)
  
  modelCoords = coordFile.modelCoordinates
  
  dict = {}
  for modelNum in modelCoords:
    dict[modelNum] = {}
    for pdbCoord in modelCoords[modelNum]:
      serial    = pdbCoord.serial
      atomName  = pdbCoord.atomName
      resName   = pdbCoord.resName
      chainId   = pdbCoord.chainId
      seqCode   = pdbCoord.seqCode
      insertCode = pdbCoord.insertionCode
      altLoc    = pdbCoord.altLoc
      x         = pdbCoord.x
      y         = pdbCoord.y
      z         = pdbCoord.z
      occupancy = pdbCoord.occupancy
      bFactor   = pdbCoord.bFactor
      atomType  = pdbCoord.atomType
      hetFlag   = pdbCoord.hetFlag

      if chainId == '':
        if segId:
          chainId = segId
        else:
          chainId = 'A'
      
      if dict[modelNum].get(chainId) is None:
        dict[modelNum][chainId] = {}
      if dict[modelNum][chainId].get(seqCode) is None:
        dict[modelNum][chainId][seqCode] = {}
      if dict[modelNum][chainId][seqCode].get(atomName) is None:
        coordDict = {}
        dict[modelNum][chainId][seqCode][atomName] = coordDict
      
      coordDict['resName'] = resName
      coordDict['seqCode'] = seqCode
      coordDict['insertCode'] = insertCode
      coordDict['altLoc'] = altLoc
      coordDict['x'] = x
      coordDict['y'] = y
      coordDict['z'] = z
      coordDict['occupancy'] = occupancy
      coordDict['bFactor']   = bFactor
      coordDict['atomType']  = atomType
      coordDict['hetFlag']   = hetFlag
       
  return dict
'''
