LICENSE = """
======================COPYRIGHT/LICENSE START==========================

MoleculeBasic.py: Part of the CcpNmr Analysis program

Copyright (C) 2003-2010 Wayne Boucher and Tim Stevens (University of Cambridge)

=======================================================================

The CCPN license can be found in ../../../../license/CCPN.license.

======================COPYRIGHT/LICENSE END============================

for further information, please contact :

- CCPN website (http://www.ccpn.ac.uk/)

- email: ccpn@bioc.cam.ac.uk

- contact the authors: wb104@bioc.cam.ac.uk, tjs23@cam.ac.uk
=======================================================================

If you are using this software for academic purposes, we suggest
quoting the following references:

===========================REFERENCE START=============================
R. Fogh, J. Ionides, E. Ulrich, W. Boucher, W. Vranken, J.P. Linge, M.
Habeck, W. Rieping, T.N. Bhat, J. Westbrook, K. Henrick, G. Gilliland,
H. Berman, J. Thornton, M. Nilges, J. Markley and E. Laue (2002). The
CCPN project: An interim report on a data model for the NMR community
(Progress report). Nature Struct. Biol. 9, 416-418.

Wim F. Vranken, Wayne Boucher, Tim J. Stevens, Rasmus
H. Fogh, Anne Pajon, Miguel Llinas, Eldon L. Ulrich, John L. Markley, John
Ionides and Ernest D. Laue (2005). The CCPN Data Model for NMR Spectroscopy:
Development of a Software Pipeline. Proteins 59, 687 - 696.

===========================REFERENCE END===============================

"""
import re

from memops.general.Util import copySubTree

try:
  from memops.gui.MessageReporter import showWarning, showOkCancel, showYesNo
except ImportError:
  from memops.universal.MessageReporter import showWarning, showOkCancel, showYesNo

from ccp.general.ChemCompOverview import chemCompStdDict

from ccpnmr.analysis.core.AssignmentBasic import assignAtomsToRes, assignResonanceResidue, getResidueResonances

from ccp.util.Molecule import makeChain
from ccp.util.NmrExpPrototype import longRangeTransfers
from ccp.util.LabeledMolecule import getIsotopomerSingleAtomFractions, getIsotopomerAtomPairFractions
from ccp.util.LabeledMolecule import singleAtomFractions, atomPairFractions

STANDARD_ISOTOPES = ['1H','13C','15N','31P','2H','29Si','19F','17O', '79Br']
# 1H, 2H, 3He, 13C, 15N, 14N, 19F, 31P, 17O, 10B, 11B, 35Cl,
# 37Cl, 43Ca, 195Pt 6Li, 7Li, 9Be, 21Ne, 23Na, 25Mg, 27Al,
# 29Si, 33S, 39K, 40K,  41K, 45Sc, 47Ti, 49Ti, 50V, 51V,
# 53Cr, 55Mn, 57Fe, 59Co,  61Ni, 63Cu, 65Cu, 67Zn, 69Ga,
# 71Ga, 73Ge, 77Se, 81Br, 87Rb, 87Sr, 95Mo, 109Ag, 113Cd,
# 125Te, 127I, 133Cs, 135Ba, 137Ba, 139La, 183W, 199Hg

DEFAULT_ISOTOPES = {'H':'1H','C':'13C','N':'15N','P':'31P','Si':'29Si',
                    'F':'19F','O':'16O', 'Br':'79Br',}

BLOSUM62={'A':{'A': 4,'R':-1,'N':-2,'D':-2,'C': 0,'Q':-1,'E':-1,'G': 0,'H':-2,'I':-1,'L':-1,
               'K':-1,'M':-1,'F':-2,'P':-1,'S': 1,'T': 0,'W':-3,'Y':-2,'V': 0,'X':0},
          'R':{'A':-1,'R': 5,'N': 0,'D':-2,'C':-3,'Q': 1,'E': 0,'G':-2,'H': 0,'I':-3,'L':-2,
               'K': 2,'M':-1,'F':-3,'P':-2,'S':-1,'T':-1,'W':-3,'Y':-2,'V':-3,'X':0},
          'N':{'A':-2,'R': 0,'N': 6,'D': 1,'C':-3,'Q': 0,'E': 0,'G': 0,'H': 1,'I':-3,'L':-3,
               'K': 0,'M':-2,'F':-3,'P':-2,'S': 1,'T': 0,'W':-4,'Y':-2,'V':-3,'X':0},
          'D':{'A':-2,'R':-2,'N': 1,'D': 6,'C':-3,'Q': 0,'E': 2,'G':-1,'H':-1,'I':-3,'L':-4,
               'K':-1,'M':-3,'F':-3,'P':-1,'S': 0,'T':-1,'W':-4,'Y':-3,'V':-3,'X':0},
          'C':{'A': 0,'R':-3,'N':-3,'D':-3,'C': 9,'Q':-3,'E':-4,'G':-3,'H':-3,'I':-1,'L':-1,
               'K':-3,'M':-1,'F':-2,'P':-3,'S':-1,'T':-1,'W':-2,'Y':-2,'V':-1,'X':0},
          'Q':{'A':-1,'R': 1,'N': 0,'D': 0,'C':-3,'Q': 5,'E': 2,'G':-2,'H': 0,'I':-3,'L':-2,
               'K': 1,'M': 0,'F':-3,'P':-1,'S': 0,'T':-1,'W':-2,'Y':-1,'V':-2,'X':0},
          'E':{'A':-1,'R': 0,'N': 0,'D': 2,'C':-4,'Q': 2,'E': 5,'G':-2,'H': 0,'I':-3,'L':-3,
               'K': 1,'M':-2,'F':-3,'P':-1,'S': 0,'T':-1,'W':-3,'Y':-2,'V':-2,'X':0},
          'G':{'A': 0,'R':-2,'N': 0,'D':-1,'C':-3,'Q':-2,'E':-2,'G': 6,'H':-2,'I':-4,'L':-4,
               'K':-2,'M':-3,'F':-3,'P':-2,'S': 0,'T':-2,'W':-2,'Y':-3,'V':-3,'X':0},
          'H':{'A':-2,'R': 0,'N': 1,'D':-1,'C':-3,'Q': 0,'E': 0,'G':-2,'H': 8,'I':-3,'L':-3,
               'K':-1,'M':-2,'F':-1,'P':-2,'S':-1,'T':-2,'W':-2,'Y': 2,'V':-3,'X':0},
          'I':{'A':-1,'R':-3,'N':-3,'D':-3,'C':-1,'Q':-3,'E':-3,'G':-4,'H':-3,'I': 4,'L': 2,
               'K':-3,'M': 1,'F': 0,'P':-3,'S':-2,'T':-1,'W':-3,'Y':-1,'V': 3,'X':0},
          'L':{'A':-1,'R':-2,'N':-3,'D':-4,'C':-1,'Q':-2,'E':-3,'G':-4,'H':-3,'I': 2,'L': 4,
               'K':-2,'M': 2,'F': 0,'P':-3,'S':-2,'T':-1,'W':-2,'Y':-1,'V': 1,'X':0},
          'K':{'A':-1,'R': 2,'N': 0,'D':-1,'C':-3,'Q': 1,'E': 1,'G':-2,'H':-1,'I':-3,'L':-2,
               'K': 5,'M':-1,'F':-3,'P':-1,'S': 0,'T':-1,'W':-3,'Y':-2,'V':-2,'X':0},
          'M':{'A':-1,'R':-1,'N':-2,'D':-3,'C':-1,'Q': 0,'E':-2,'G':-3,'H':-2,'I': 1,'L': 2,
               'K':-1,'M': 5,'F': 0,'P':-2,'S':-1,'T':-1,'W':-1,'Y':-1,'V': 1,'X':0},
          'F':{'A':-2,'R':-3,'N':-3,'D':-3,'C':-2,'Q':-3,'E':-3,'G':-3,'H':-1,'I': 0,'L': 0,
               'K':-3,'M': 0,'F': 6,'P':-4,'S':-2,'T':-2,'W': 1,'Y': 3,'V':-1,'X':0},
          'P':{'A':-1,'R':-2,'N':-2,'D':-1,'C':-3,'Q':-1,'E':-1,'G':-2,'H':-2,'I':-3,'L':-3,
               'K':-1,'M':-2,'F':-4,'P': 7,'S':-1,'T':-1,'W':-4,'Y':-3,'V':-2,'X':0},
          'S':{'A': 1,'R':-1,'N': 1,'D': 0,'C':-1,'Q': 0,'E': 0,'G': 0,'H':-1,'I':-2,'L':-2,
               'K': 0,'M':-1,'F':-2,'P':-1,'S': 4,'T': 1,'W':-3,'Y':-2,'V':-2,'X':0},
          'T':{'A': 0,'R':-1,'N': 0,'D':-1,'C':-1,'Q':-1,'E':-1,'G':-2,'H':-2,'I':-1,'L':-1,
               'K':-1,'M':-1,'F':-2,'P':-1,'S': 1,'T': 5,'W':-2,'Y':-2,'V': 0,'X':0},
          'W':{'A':-3,'R':-3,'N':-4,'D':-4,'C':-2,'Q':-2,'E':-3,'G':-2,'H':-2,'I':-3,'L':-2,
               'K':-3,'M':-1,'F': 1,'P':-4,'S':-3,'T':-2,'W':11,'Y': 2,'V':-3,'X':0},
          'Y':{'A':-2,'R':-2,'N':-2,'D':-3,'C':-2,'Q':-1,'E':-2,'G':-3,'H': 2,'I':-1,'L':-1,
               'K':-2,'M':-1,'F': 3,'P':-3,'S':-2,'T':-2,'W': 2,'Y': 7,'V':-1,'X':0},
          'V':{'A': 0,'R':-3,'N':-3,'D':-3,'C':-1,'Q':-2,'E':-2,'G':-3,'H':-3,'I': 3,'L': 1,
               'K':-2,'M': 1,'F':-1,'P':-2,'S':-2,'T': 0,'W':-3,'Y':-1,'V': 4,'X':0},
          'X':{'A': 0,'R': 0,'N': 0,'D': 0,'C': 0,'Q': 0,'E': 0,'G': 0,'H': 0,'I': 0,'L': 0,
               'K': 0,'M': 0,'F': 0,'P': 0,'S': 0,'T': 0,'W': 0,'Y': 0,'V': 0,'X':0}
          }
 
 
NAMAT={'A':{'A': 2,'C':-5,'G':-5,'T':-5,'U':-5,'R': 1,'Y':-5,'N':0,'X':0},
       'C':{'A':-5,'C': 2,'G':-5,'T':-5,'U':-5,'R':-5,'Y': 1,'N':0,'X':0},
       'G':{'A':-5,'C':-5,'G': 2,'T':-5,'U':-5,'R': 1,'Y':-5,'N':0,'X':0},
       'T':{'A':-5,'C':-5,'G':-5,'T': 2,'U': 2,'R':-5,'Y': 1,'N':0,'X':0},
       'U':{'A':-5,'C':-5,'G':-5,'T': 2,'U': 2,'R':-5,'Y': 1,'N':0,'X':0},
       'R':{'A': 1,'C':-5,'G': 1,'T':-5,'U':-5,'R': 1,'Y':-5,'N':0,'X':0},
       'Y':{'A':-5,'C': 1,'G':-5,'T': 1,'U': 1,'R':-5,'Y': 1,'N':0,'X':0},
       'N':{'A': 0,'C': 0,'G': 0,'T': 0,'U': 0,'R': 0,'Y': 0,'N':0,'X':0},
       'X':{'A': 0,'C': 0,'G': 0,'T': 0,'U': 0,'R': 0,'Y': 0,'N':0,'X':0}}

# Should really be derived or modelled attrs
PROTEIN_RESIDUE_CLASS_DICT = {'Acidic'       :['Asp','Glu'],
                              'Basic'        :['Arg','Lys','His'],
                              'Charged'      :['Asp','Glu','Arg','Lys','His'],
                              'Polar'        :['Asn','Gln','Asp','Glu','Arg','Lys','His','Ser','Thr','Tyr'],
                              'Non-polar'    :['Ala','Phe','Gly','Ile','Leu','Met','Pro','Val','Trp','Cys'],
                              'Hydrophilic'  :['Ser','Asp','Glu','Arg','Lys','His','Asp','Glu','Pro','Tyr'],
                              'Hydrophobic'  :['Phe','Met','Ile','Leu','Val','Cys','Trp','Ala','Thr','Gly'],
                              'Amide'        :['Asn','Gln'],
                              'Hydroxyl'     :['Ser','Thr','Tyr'],
                              'Aromatic'     :['Phe','Ptr','Tyr','Trp'],
                              'Beta-branched':['Thr','Val','Ile'],
                              'Small'        :['Cys','Ala','Ser','Asp','Thr','Gly','Asn'],
                              'Neutral'      :['Ala','Asn','Cys','Gln','Gly','Ile','Leu','Met',
                                               'Phe','Pro','Ser','Thr','Trp','Tyr','Val'],
                              'Methyl'       :['Ala','Met','Ile','Leu','Thr','Val'],
                             }

# X is an unusual base, not ambiguiuty

STEREO_PREFIX = 'stereo_'
CARBOHYDRATE_MOLTYPE = 'carbohydrate'
PROTEIN_MOLTYPE = 'protein'
OTHER_MOLTYPE = 'other'
DNA_MOLTYPE = 'DNA'
RNA_MOLTYPE = 'RNA'
DNARNA_MOLTYPE = 'DNA/RNA'

def getUnicodeGreek():
  """
   {'a':u'\u03B1','b':u'\u03B2','g':u'\u03B3','d':u'\u03B4','e':u'\u03B5',
    'z':u'\u03B6','h':u'\u03B7','q':u'\u03B8','i':u'\u03B9','k':u'\u03BA',
    'l':u'\u03BB','m':u'\u03BC','n':u'\u03BD','x':u'\u03BE','o':u'\u03BF',
    'p':u'\u03C0','r':u'\u03C1','j':u'\u03C2','s':u'\u03C3','t':u'\u03C4', # j : Other sigma
    'u':u'\u03C5','f':u'\u03C6','c':u'\u03C7','y':u'\u03C8','w':u'\u03C9'}
  """
  dict = {}
  romanLetterOrder = 'ABGDEZHQIKLMNXOPRJSTUFCYW'
  u = 913
  l = 945
  for a in romanLetterOrder:
    dict[a] = unichr(u)
    dict[a.lower()] = unichr(l)
    u += 1
    l += 1

  return dict   
   
unicodeGreek = getUnicodeGreek()

userResidueCodesDict = {DNA_MOLTYPE:{'A':'Ade','T':'Thy','G':'Gua','C':'Cyt','U':'Ura'},
                        RNA_MOLTYPE:{'A':'Ade','T':'Thy','G':'Gua','C':'Cyt','U':'Ura','I':'Ino'},
                        PROTEIN_MOLTYPE:{},
                        CARBOHYDRATE_MOLTYPE:{}
                        }

def getUnicodeAtomName(name, elementSymbol):
  """Return a unicode string for a protein atom name, i.e.
             including any greek characters 
  .. describe:: Input
  
  Word, Word (ChemElement.symbol)

  .. describe:: Output
  
  Unicode Word
  """ 
  
  l = len(elementSymbol)
  
  if l == len(name):
    return name
  
  else:
    letter = name[l]
    
    if letter.islower():
      return name[:l] + unicodeGreek.get(letter,letter) + name[l+1:]
    else:
      return name

def getResidueCode(obj):
  """Get a text code for a residue/molResidue/resonanceGroup,
             defaults to the ccpCode if a custom code is not present
             in userResidueCodesDict 
  .. describe:: Input
  
  MolSystem.Residue

  .. describe:: Output
  
  Word
  """ 
  
  ccpCode = obj.ccpCode
  molType = obj.molType
    
  if molType is None:
    molType = PROTEIN_MOLTYPE
  
  ccpCodeDict = userResidueCodesDict.get(molType, {})
  residueCode = ccpCodeDict.get(ccpCode, ccpCode)
  
  if molType == CARBOHYDRATE_MOLTYPE:
    descriptor = obj.descriptor
    n = len(STEREO_PREFIX)
    
    if descriptor and (len(descriptor)> n) and descriptor.startswith(STEREO_PREFIX):
      if descriptor[n] == '1':
        residueCode = 'a-' + residueCode
      elif descriptor[n] == '2':
        residueCode = 'b-' + residueCode
 
  return residueCode

def getLinkedResidue(residue, linkCode='prev'):
  """Find a residue, if it exists, that is linked to the
             input residue by a given type of molResidue link.
  .. describe:: Input
  
  MolSystem.Residue

  .. describe:: Output
  
  MolSystem.Residue
  """  

  if not hasattr(residue, 'linkedResidueDict'):
    residue.linkedResidueDict = {}
  else:
    if residue.linkedResidueDict.has_key(linkCode):
      return residue.linkedResidueDict[linkCode]
  
  residue2    = None
  chain       = residue.chain
  molResidue  = residue.molResidue
  linkEnd     = molResidue.findFirstMolResLinkEnd(linkCode=linkCode)
  
  if linkEnd:
    molResLink = linkEnd.molResLink
    if molResLink:
      for linkEnd2 in molResLink.molResLinkEnds:
        if linkEnd2 is not linkEnd:
          residue2 = chain.findFirstResidue(molResidue=linkEnd2.molResidue)
  
  residue.linkedResidueDict[linkCode] = residue2
  return residue2


def getLinkAtoms(residueA, linkCode='prev'):

  chain = residueA.chain
  molResidue = residueA.molResidue
  
  molResLinkEndA = molResidue.findFirstMolResLinkEnd(linkCode=linkCode)
  chemAtomA = molResLinkEndA.linkEnd.boundChemAtom

  atomA = residueA.findFirstAtom(chemAtom=chemAtomA)
  atomB = None
  
  if molResLinkEndA and molResLinkEndA.molResLink:
  
    for molResLinkEndB in molResLinkEndA.molResLink.molResLinkEnds:
    
      if molResLinkEndB is not molResLinkEndA:
      
        residueB = chain.findFirstResidue(molResidue=molResLinkEndB.molResidue)
        chemAtomB = molResLinkEndB.linkEnd.boundChemAtom
        atomB = residueB.findFirstAtom(chemAtom=chemAtomB)
        
        break
  
  return atomA, atomB


def getResidueObservableAtoms(residue, refExperiment=None, labelling=None,
                              minFraction=0.1, jCouplingBonds=(1,2,3),
                              usePermissiveShifts=False,
                              chemElements=('H','C','N','F','P')): 
  """
  Determine which atoms of a chem comp varient would give rise to
  observable resonances considering a given reference experiment
  and/or an isotope labelling scheme. Can specify minimum fraction of
  an isotope to consider something observable and the chemical elements which
  you are observing. Boolean option to match database min and max
  chemical shift bounds to atom sites, rather than randon coil shift
  values (default).
  
  .. describe:: Input
  
  MolSystem.Residue, NmrExpPrototype.RefExperiment,
  ChemCompLabel.LabelingScheme or LabeledMolecule.LabeledMixture,
  Float, Boolean, List of Words

  .. describe:: Output
  
  List of ChemComp.ChemAtoms
  """

  if not jCouplingBonds:
    jCouplingBonds = (0,)

  atomSiteDict   = {}
  isotopomerDict = {}
  atomSitesAll   = {}
  
  if refExperiment:
    for atomSite in refExperiment.nmrExpPrototype.atomSites:
      isotope = atomSite.isotopeCode
      if not atomSitesAll.has_key(isotope):
        atomSitesAll[isotope] = []
  
      atomSitesAll[isotope].append(atomSite)
 
  isotopeDict = {}
  
  if atomSitesAll:
    isotopes = atomSitesAll.keys()
  
  else:
    isotopes = []
    for element in chemElements:
      isotope = DEFAULT_ISOTOPES.get(element)
    
      if isotope:
        isotopes.append(isotope)
  
  for isotope in isotopes:
    element = isotope
 
    while element[0] in '0123456789':
      element = element[1:]
 
    isotopeDict[element] = isotope

 
  filteredAtoms = []
  prevResidue = getLinkedResidue(residue, linkCode='prev')
  nextResidue = getLinkedResidue(residue, linkCode='next')
  
  natAbundance = residue.root.findFirstLabelingScheme(name='NatAbun')
  
  #print residue.seqCode, residue.ccpCode, refExperiment.name
  for residue0 in (prevResidue,residue,nextResidue):
    isotopomers = None
    
    if residue0:
      resId = residue0.molResidue.serial
      atoms = residue0.atoms
      
      # Compile isotopomers for this residue
      if labelling and (labelling.className == 'LabelingScheme'):
        chemComp   = residue0.chemCompVar.chemComp
        ccpCode    = chemComp.ccpCode
        molType    = chemComp.molType
        chemCompLabel = labelling.findFirstChemCompLabel(ccpCode=ccpCode,
                                                         molType=molType)

        if not chemCompLabel:
          chemCompLabel = natAbundance.findFirstChemCompLabel(ccpCode=ccpCode,
                                                              molType=molType)


        if chemCompLabel:
          isotopomers  = chemCompLabel.isotopomers
          isotopomerDict[residue0] = isotopomers
            
          
    else:
      atoms = []  
   
    #atoms0 = [] # Those which make it through the filter
    for atom in atoms:
      chemAtom = atom.chemAtom
      isotope  = isotopeDict.get(chemAtom.elementSymbol)
    
      if not isotope:
        continue
    
      if chemAtom.waterExchangeable:
        continue

      if isotopomers:
        fractionDict = getIsotopomerSingleAtomFractions(isotopomers,atom.name,chemAtom.subType)
        # Exclude if no isotope incorporation above threshold
        fraction = fractionDict.get(isotope, minFraction)
        if fraction < minFraction:
          continue
          
      elif labelling:
        fractionDict = singleAtomFractions(labelling, resId, atom.name)
        if not fractionDict:
          continue
        
        fraction = fractionDict.get(isotope, minFraction)
        if fraction < minFraction:
          continue
        
 
      atomSitesIsotope = atomSitesAll.get(isotope)
      if atomSitesIsotope:
        setSize = None
        
        if usePermissiveShifts:
          shifts = getChemicalShiftBounds(chemAtom)
          if not shifts:
            shifts = [getRandomCoilShift(chemAtom),]

        else:
          shifts = [getRandomCoilShift(chemAtom),]
 
        for atomSite in atomSitesIsotope:
                
          maxShift = atomSite.maxShift
          if (maxShift is not None) and (shifts[0] > maxShift):
            continue
 
          minShift = atomSite.minShift
          if (minShift is not None) and (shifts[-1] < minShift):
            continue
        
          if setSize is None:
            setSize     = 1
            chemAtomSet = chemAtom.chemAtomSet
 
            if chemAtomSet:
              setSize = len(chemAtomSet.chemAtoms)
        
          minNumber =  atomSite.minNumber
          if setSize < minNumber:
            continue
 
          maxNumber = atomSite.maxNumber
          if maxNumber and (setSize>maxNumber):
            continue
 
          numberStep = atomSite.numberStep
          if (setSize-minNumber) % numberStep != 0:
            continue
             
          if atomSiteDict.get(atomSite) is None:
            atomSiteDict[atomSite] = []
          atomSiteDict[atomSite].append(atom)
          
          #print 'AS', atomSite.name, atom.name
        
      filteredAtoms.append(atom)
      
  
  if refExperiment:
    #print refExperiment.name
    
    # Atom sites which are possibly visible given dims
    observableAtomSites = {}
    for refExpDim in refExperiment.refExpDims:
      for refExpDimRef in refExpDim.refExpDimRefs:
        for atomSite in refExpDimRef.expMeasurement.atomSites:
          observableAtomSites[atomSite] = True
  
    # Get prototype graph atomSite routes
  
    graphRoutes = []
    for expGraph in refExperiment.nmrExpPrototype.expGraphs:
      expSteps = [(es.stepNumber, es) for es in expGraph.expSteps]
      expSteps.sort()
      routes = []
      stepNumber, expStep = expSteps[0]

      for atomSite in expStep.expMeasurement.atomSites:
        route = [(atomSite,None,stepNumber)]
        routes.append(route)
      
      while True:
        routes2 = []
      
        for route in routes:
          atomSiteA, null, stepA = route[-1]
          #print atomSiteA.name, step
      
          for expTransfer in atomSiteA.expTransfers:
            atomSites = list(expTransfer.atomSites)
            atomSites.remove(atomSiteA)
            atomSiteB = atomSites[0]
 
            if not expTransfer.transferToSelf:
              if atomSiteB is atomSiteA:
                continue
 
            for stepB, expStepB in expSteps:
              if stepA > stepB:
                continue
            
              if atomSiteB in expStepB.expMeasurement.atomSites:
                routes2.append( route[:] + [(atomSiteB,expTransfer,stepB)] )
                #print ['%s %d' % (a[0].name, a[2]) for a in routes2[-1]]
                break
        
        if routes2:
          routes = routes2
        else:
          break  
      
      for route in routes:
        atomRoutes = []
        lastAtomSite = route[-1][0]
      
        for i in range(len(route)-1):
          atomSiteA, null, stepA = route[i]
          atomSiteB, expTransfer, stepB = route[i+1] 
          transferType = expTransfer.transferType
          
          #print stepA, atomSiteA.name, stepB, atomSiteB.name, transferType

          if atomRoutes:
            atomsA = [r[-1][0] for r in atomRoutes]
          else:
            atomsA = atomSiteDict[atomSiteA]
            
          atomRoutes2 = []
          for atomA in atomsA:
            for atomB in atomSiteDict[atomSiteB]:
              if isotopomerDict:
                chemAtomA = atomA.chemAtom
                chemAtomB = atomB.chemAtom
                subTypeA  = chemAtomA.subType
                subTypeB  = chemAtomB.subType
                isotopeA  = isotopeDict[chemAtomA.elementSymbol]
                isotopeB  = isotopeDict[chemAtomB.elementSymbol]
                residueA  = atomA.residue
                residueB  = atomB.residue
 
                if residueA is residueB:
                  isotopomersA = isotopomerDict.get(residueA)
                  atomNames    = (atomA.name, atomB.name)
                  subTypes     = (subTypeA, subTypeB)
                  pairDict     = getIsotopomerAtomPairFractions(isotopomersA, atomNames, subTypes)
                  fraction     = pairDict.get((isotopeA, isotopeB), minFraction)
 
                  if fraction  < minFraction:
                    continue
 
                else:
                  isotopomersA = isotopomerDict.get(residueA)
                  isotopomersB = isotopomerDict.get(residueB)

                  if isotopomersB and isotopomersA:
                    fractionDictA = getIsotopomerSingleAtomFractions(isotopomersA, atomA.name, subTypeA)
                    fractionDictB = getIsotopomerSingleAtomFractions(isotopomersB, atomB.name, subTypeB)
                    fraction = fractionDictA.get(isotopeA, 1.0) * fractionDictB.get(isotopeB, 1.0)
 
                    if fraction < minFraction:
                      continue
                      
              elif labelling:
                chemAtomA = atomA.chemAtom
                chemAtomB = atomB.chemAtom
                isotopeA  = isotopeDict[chemAtomA.elementSymbol]
                isotopeB  = isotopeDict[chemAtomB.elementSymbol]
                residueA = atomA.residue
                residueB = atomB.residue
                molResidueA = residueA.molResidue
                molResidueB = residueB.molResidue
                resIds = (molResidueA.serial, molResidueB.serial)
                atomNames = (atomA.name, atomB.name)
               
                pairDict = atomPairFractions(labelling, resIds, atomNames)
                fraction = pairDict.get((isotopeA, isotopeB), minFraction)
 
                if fraction  < minFraction:
                  continue
                      
              addAtom = False
              if transferType in longRangeTransfers:
                addAtom = True
 
              elif transferType in ('onebond','CP') and areAtomsBound(atomA, atomB):
                addAtom = True
 
              elif transferType == 'TOCSY'and areAtomsTocsyLinked(atomA, atomB):
                addAtom = True
 
              elif transferType == 'Jcoupling':
                numBonds = getNumConnectingBonds(atomA, atomB, limit=max(jCouplingBonds))
                if numBonds in jCouplingBonds:
                  addAtom = True
 
              elif transferType == 'Jmultibond' and not areAtomsBound(atomA, atomB):
                numBonds = getNumConnectingBonds(atomA, atomB, limit=max(jCouplingBonds))
                if numBonds in jCouplingBonds:
                  addAtom = True
 
              if addAtom:
                grown = True
                #print 'AB', atomA.name, atomA.residue.seqCode,'+', atomB.name, atomB.residue.seqCode
                if not atomRoutes:
                  atomRoutes2.append( [(atomA,atomSiteA),(atomB,atomSiteB),] )
                  #print atomA.name, atomB.name
                
                else:
                  for atomRoute in atomRoutes:
                    atomRoutes2.append( atomRoute[:] + [(atomB,atomSiteB),] )
                  #print '+', atomB.name

               
          atomRoutes = []
          for atomRoute in atomRoutes2:
            if atomRoute[-1][1] is lastAtomSite:
              atomRoutes.append(atomRoute)
              
        graphRoutes.append(atomRoutes)

 
    observableAtoms = set()
    for routes in graphRoutes:
      for route in routes:

        for atomB, atomSiteB in route:
          if atomB.residue is residue: # Must have one atom from this residue
            for atomA, atomSiteA in route:
              if observableAtomSites.get(atomSiteA):
                observableAtoms.add(atomA)
            break
   
  else:
    observableAtoms = filteredAtoms
   
  return list(observableAtoms)

def getNumConnectingBonds(atom1, atom2, limit=5):
  """
  Get the minimum number of binds that connect two atoms.
  Stops at a specified limit (and returns None if not within it)
  
  .. describe:: Input
  
  MolSystem.Atom, MolSystem.atom, Int

  .. describe:: Output
  
  Int
  """
  
  num = 0
  atoms = set([atom1,])
  
  while atom2 not in atoms:
    if num > limit:
      return None
      
    atoms2 = atoms.copy()
    
    for atom in atoms2:
      atoms.update(getBoundAtoms(atom))

    num += 1

  return num

def areAtomsTocsyLinked(atom1, atom2):
  """
  Determine if two atoms have a connectivity that may be observable in a TOCSY experiment
  
  .. describe:: Input
  
  MolSystem.Atom, MolSystem.atom

  .. describe:: Output
  
  Boolean
  """
   
  if not hasattr(atom1, 'tocsyDict'):
    atom1.tocsyDict = {}
  elif atom1.tocsyDict.has_key(atom2):
    return atom1.tocsyDict[atom2]

  if not hasattr(atom2, 'tocsyDict'):
    atom2.tocsyDict = {}
  elif atom2.tocsyDict.has_key(atom1):
    return atom2.tocsyDict[atom1]
    
  chemAtom1 = atom1.chemAtom
  chemAtom2 = atom2.chemAtom
  element1  = chemAtom1.elementSymbol
  element2  = chemAtom2.elementSymbol
  
  if element1 != element2:
    boolean = False
    
  elif areAtomsBound(atom1, atom2):
    boolean = True
  
  else:
 
    residue1 = atom1.residue
    residue2 = atom2.residue
 
    if residue1 is not residue2:
      boolean = False
      
    else:
      atomsA = set([atom1,])
      boolean = True
      while atom2 not in atomsA:
        atomsB = atomsA.copy()
 
        for atomB in atomsB:
          for atom3 in getBoundAtoms(atomB):
            if atom3.residue is not residue1:
              continue
            
            if element1 == 'H':
              if atom3.chemAtom.elementSymbol != 'H':
                for atom4 in getBoundAtoms(atom3):
                  if atom4.chemAtom.elementSymbol == 'H':
                    break
                else:
                  continue
 
            if atom3.chemAtom.elementSymbol == element1:
              if not hasattr(atom3, 'tocsyDict'):
                atom3.tocsyDict = {}
 
              atom1.tocsyDict[atom3] = True
              atom3.tocsyDict[atom1] = True
 
            atomsA.add(atom3)
 
        if atomsA == atomsB: # Nothing more to add and atom2 not in linked set
          boolean = False
          break

  atom1.tocsyDict[atom2] = boolean
  atom2.tocsyDict[atom1] = boolean
  return boolean

def getChemicalShiftBounds(chemAtom, threshold=0.001, sourceName='BMRB'):
  """
  Return the min and max chemical shifts for a
  given atom type observed in the databases
  
  .. describe:: Input
  
  ChemComp.ChemAtom, Float, Word

  .. describe:: Output
  
  Float
  """

  key = '%s:%s' % (sourceName, threshold)
  if not hasattr(chemAtom, 'chemicalShiftBounds'):
    chemAtom.chemicalShiftBounds = {}

  else:
    region = chemAtom.chemicalShiftBounds.get(key)
    if region:
      return region
  
  chemComp = chemAtom.chemComp
  ccpCode  = chemComp.ccpCode
  molType  = chemComp.molType
  
  chemAtomNmrRef = getChemAtomNmrRef(chemAtom.root, chemAtom.name, ccpCode,
                                     molType=molType, sourceName=sourceName)

  if chemAtomNmrRef:
    distribution  = chemAtomNmrRef.distribution
    refPoint      = chemAtomNmrRef.refPoint
    refValue      = chemAtomNmrRef.refValue
    valuePerPoint = chemAtomNmrRef.valuePerPoint
    n = len(distribution)
 
    minPt  = 0
    maxPt  = n-1
 
    for v in distribution:
      if v < threshold:
        minPt += 1
      else:
        break
 
    for v in distribution[::-1]:
      if v < threshold:
        maxPt -= 1
      else:
        break
 
    maxPpm = ((maxPt - refPoint) * valuePerPoint) + refValue
    minPpm = ((minPt - refPoint) * valuePerPoint) + refValue
    region  = (minPpm, maxPpm)
    chemAtom.chemicalShiftBounds[key] = region

  return region
       
def getRandomCoilShift(chemAtom, context=None, sourceName='BMRB'):
  """
  Get the random coil chemical shift value of a chemAtom
 
  .. describe:: Input
  
  ChemComp.ChemAtom

  .. describe:: Output
  
  Float
  """
  
  value = None
  if not hasattr(chemAtom, 'randomCoilShiftDict'):
    chemAtom.randomCoilShiftDict = {}
    
  elif chemAtom.randomCoilShiftDict.has_key(sourceName):
    value = chemAtom.randomCoilShiftDict[sourceName]
 
    if not context:
      return value
 
  if value is None:
    chemComp = chemAtom.chemComp
    ccpCode  = chemComp.ccpCode
    molType  = chemComp.molType
    chemAtomNmrRef = getChemAtomNmrRef(chemAtom.root, chemAtom.name, ccpCode,
                                       molType=molType, sourceName=sourceName)
    
    if not chemAtomNmrRef and chemAtom.chemAtomSet:
      chemAtomNmrRef = getChemAtomNmrRef(chemAtom.root, chemAtom.chemAtomSet.name, ccpCode,
                                         molType=molType, sourceName=sourceName)

    if chemAtomNmrRef:
      value = chemAtomNmrRef.randomCoilValue
      if value is None:
        value = chemAtomNmrRef.meanValue
 
    chemAtom.randomCoilShiftDict[sourceName] = value

  if context and (value is not None):
    correctionDict = getRandomCoilShiftCorrectionsDict()
    
    atomName = chemAtom.name
    
    if atomName in ('HA2','HA3'):
      atomName = 'HA'
    
    offsets = [2,1,-1,-2]
    indices = [0,1,3,4]
    
    for i in range(4):
      residue = context[indices[i]]
      
      if residue is not None:
        ccpCode = residue.ccpCode
        atomDict = correctionDict[offsets[i]].get(ccpCode, {})
        value += atomDict.get(atomName, 0.0)
    
  return value


def getChemAtomNmrRef(project, atomName, ccpCode, molType=PROTEIN_MOLTYPE, sourceName='BMRB'):
  """
  Retrieve an NMR chemical shift reference atom record
  
  .. describe:: Input
  
  Implementation.Project, Word (ChemAtom.name), Word (ChemComp.ccpCode),
  Word, (chemComp.molType), Word

  .. describe:: Output
  
  Float
  """
  
  #key = '%s:%s:%s:%s' % (atomName, ccpCode, molType, sourceName)
  #if not hasattr(project, 'chemAtomNmrRefDict'):
  #  project.chemAtomNmrRefDict = {}
  #else:
    

  nmrRefStore = project.findFirstNmrReferenceStore(molType=molType, ccpCode=ccpCode)
  
  chemAtomNmrRef = None
  if nmrRefStore:
    chemCompNmrRef = nmrRefStore.findFirstChemCompNmrRef(sourceName=sourceName)
    
    if chemCompNmrRef:
      chemCompVarNmrRef = chemCompNmrRef.findFirstChemCompVarNmrRef(linking='any',
                                                                    descriptor='any')
      
      if chemCompVarNmrRef:
        for chemAtomNmrRef1 in chemCompVarNmrRef.chemAtomNmrRefs:
          if atomName == chemAtomNmrRef1.name:
            chemAtomNmrRef = chemAtomNmrRef1
            
            break
      else:
        data = (molType,ccpCode)
        msg  = 'Could not load reference NMR data for'
        msg += 'general chem comp variant %s:%s' % data
        showWarning('Warning', msg)
        return
        
    else:
      data = (molType,ccpCode,sourceName)
      showWarning('Warning',
                  'Could not load reference NMR data for %s:%s source=%s' % data)
      return
      
  else:
    data = (molType,ccpCode)
    showWarning('Warning',
                'Could not load reference NMR data for %s:%s' % data)
    return
 
  if not chemAtomNmrRef:
    atomName2 = atomName[:-1]
    for chemAtomNmrRef1 in chemCompVarNmrRef.chemAtomNmrRefs:
      if atomName2 == chemAtomNmrRef1.name:
        chemAtomNmrRef = chemAtomNmrRef1
        break

  
  return chemAtomNmrRef       


def getAtomsTorsion(atoms):
  """Get the chemical torsion object equivalent
             to an ordered input list of four atoms
  .. describe:: Input
  
  4-List of MolSystem.Atoms

  .. describe:: Output
  
  ChemComp.ChemTorsion
  """

  assert len(atoms) == 4
  
  residue = list(atoms)[0].residue
  for i in (1,2,3):
    if not areAtomsBound(atoms[i-1],atoms[i]):
      showWarning('Failure',
                  'Get atom torsion failed: Atoms %s and %s are not bound' % (atoms[i-1],atoms[i]))
      return
    
    residue1 = atoms[i].residue
    if residue1 is not residue:
      if residue1.seqCode > residue.seqCode:
        residue = residue1
    
  chemCompVar = residue.chemCompVar
  chemAtoms   = []
  linkAtoms   = []
  
  for atom in atoms:
    if atom.residue is residue:
      chemAtoms.append(atom.chemAtom)
    else:
      if linkAtoms:
        linkAtoms.append(chemCompVar.findFirstChemAtom(name='prev_2'))
      else:
        linkAtoms.append(chemCompVar.findFirstChemAtom(name='prev_1'))
  
  chemAtoms = chemAtoms + linkAtoms
  chemAtoms.sort()

  for chemTorsion in chemCompVar.chemTorsions:
    chemAtoms0 = list(chemTorsion.chemAtoms)
    chemAtoms0.sort()
    
    if chemAtoms0 == chemAtoms:
      return chemTorsion
 
def findMatchingChain(molSystem, ccpCodes, excludeChains=None, molTypes=None, doWarning=True):
  """* * * Deprecated in light of findMathcingChains() * * * 
             Find the mol system chain that best matches the input ccpCodes
             (like three letter codes). Useful for trying to match structures
             to existing molecular data. Optional argument to specify which
             chains cannot be matched.
  .. describe:: Input
  
  MolSystem.MolSystem, List of Strings (MolSystem.MolResidue.ccpCodes),
             List of MolSystem.Chains

  .. describe:: Output
  
  Tuple of (MolSystem.Chain, Int (index of first matching ccpCode),
             Int (first matching MolSystem.Residue.seqId))
  """

  chains = []
  for chain in molSystem.sortedChains():
    if excludeChains and (chain in excludeChains):
      continue
    
    sequence = []
    if not chain.residues:
      continue
    
    chains.append(chain)
    
    residues = []
    for residue in chain.sortedResidues():
      molType = residue.molType
      if molTypes and (molType not in molTypes): 
        break
      
      residues.append( (residue.seqId, residue) )
    
    else:  
      residues.sort()
 
      for residue in residues:
        sequence.append( residue[1].ccpCode )

      len0 = len(sequence)
      len1 = len(ccpCodes)
 
      if len0 < len1:
        continue
 
      elif len0 == len1:
        if sequence == ccpCodes:
          mapping = [(i, residues[i][1]) for i in range(len0)]
          return chain, mapping

        else:
          misMatch = 0
          for i in range(len0):
            if (ccpCodes[i] is not None) and (ccpCodes[i] != sequence[i]):
              misMatch = 1
              break
          if misMatch:
            continue
          else:
            mapping = [(i, residues[i][1]) for i in range(len0)]
            return chain, mapping
 
      else:
        d = len0 - len1
        for x in range(d+1):
          misMatch = 0
          for i in range(len1):
            if (ccpCodes[i] is not None) and (ccpCodes[i] != sequence[i+x]):
              misMatch = 1
              break
          if misMatch:
            continue
          else:
            mapping = [(i, residues[i+x][1]) for i in range(len1)]
            return chain, mapping
  
  scoreList = []
  
  bestMapping = None
  bestScore   = 0
  bestChain   = None

  for chain in chains:
    mapping, score = getSequenceResidueMapping(chain, ccpCodes)
    scoreList.append((score, chain, mapping))
    
  if scoreList:
    scoreList.sort()
    bestScore, bestChain, bestMapping = scoreList[-1]

  chain = None
  if bestScore and ( bestScore/float(len(bestChain.residues)) > 2.0 ):
    chain = bestChain
    
    if chain.molecule.molType in (CARBOHYDRATE_MOLTYPE): # DNA & RNA now hopefully fine to align
      chain = None
    
    if doWarning:
      msg  = 'Residue sequence matches an existing chain, but not exactly. '
      msg += 'Really link to chain %s? (Otherwise a new chain will be made)' % chain.code
      if not showYesNo('Query', msg):
        chain = None      

  return chain, bestMapping 

def findMatchingChains(molSystem, ccpCodes, excludeChains=None, molTypes=None, doWarning=True):
  """Find the mol system chains that best match the input ccpCodes
             (like three letter codes). Useful for trying to match structures
             to existing molecular data. Optional argument to specify which
             chains cannot be matched.
  .. describe:: Input
  
  MolSystem.MolSystem, List of Strings (MolSystem.MolResidue.ccpCodes),
             List of MolSystem.Chains

  .. describe:: Output
  
  Tuple of (MolSystem.Chain, Int (index of first matching ccpCode),
             Int (first matching MolSystem.Residue.seqId))
  """

  chains0 = []
  for chain in molSystem.sortedChains():
    if excludeChains and (chain in excludeChains):
      continue
    
    #sequence = []
    if not chain.residues:
      continue
    
    chains0.append(chain)

  
  bestMapping = None
  bestChain   = None

  scoreList = []
  for chain in chains0:
    mapping, score = getSequenceResidueMapping(chain, ccpCodes)
   
    if score:
      scoreList.append((score, chain, mapping))
  
  chains = []
  mappings = [] 
  if scoreList:
    scoreList.sort()
    scoreList.reverse()
    bestScore, bestChain, bestMapping = scoreList[0]
    lenSeq = len(ccpCodes)
    
    # Find perfect stretches
    stretch = set([])
    i = 0
    for pair in bestMapping:
      if None in pair:
        stretch.add(i)
        i = 0
      else:
        i += 1
    
    stretch.add(i)
    
    if (bestScore/float(len(bestChain.residues)) > 2.0) or \
       ((max(stretch) == lenSeq) and (lenSeq > 20)):
      # No really bad alignments
    
      chains = [bestChain, ]
      mappings = [bestMapping, ]
      for score, chain, mapping in scoreList[1:]:
        if score == bestScore:
          chains.append(chain)
          mappings.append(mapping)   

  return chains, mappings

def newMolSystem(project):
  """Get a new molSystem for a project with a unique code.
  .. describe:: Input
  
  Implementation.Project

  .. describe:: Output
  
  MolSystem.MolSystem
  """

  i = 1
  while project.findFirstMolSystem(code='MS%d' % (i)):
    i += 1
  molSystem = project.newMolSystem(code='MS%d' % (i))
  molSystem.name = molSystem.code
 
  return molSystem

def transferChainAssignments(chainA, chainB):
  """Transfer any atomic assignments from one chain to another where possible.
  .. describe:: Input
  
  MolSystem.Chain, MolSystem.Chain

  .. describe:: Output
  
  None
  """

  mapping = getChainResidueMapping(chainA, chainB)
  for residueA, residueB in mapping:
    if residueB:
      resonancesB = getResidueResonances(residueB)
      if resonancesB:
        msg  = 'Destination residue %s%d has assignments. Continue?.'
        data = (residueB.seqCode,residueB.ccpCode)
        if not showOkCancel('Warning', msg % data):
          return
 
  for residueA, residueB in mapping:
    if residueA:
      if residueB is None:
        msg = 'Residue %d%s has no equivalent in destination chain'
        data = (residueA.seqCode,residueA.ccpCode)
        showWarning('Warning', msg % data)
      else:
        transferResidueAssignments(residueA,residueB)

def copyMolecule(molecule, newName=None):
  """Make a new molecule based upon the sequence of an existing ome
  .. describe:: Input
  
  Molecule.Molecule

  .. describe:: Output
  
  Molecule.Molecule
  """
  
  project = molecule.root
  i       = len(project.molecules) + 1
  newName = newName or 'Molecule %d' % (i)
  newMolecule = copySubTree(molecule, project, topObjectParameters={'name':newName,}, maySkipCrosslinks=1 )
  
  return newMolecule

def transferResidueAssignments(residueA,residueB):
  """
  Move any atomic assignments from one residue to another where possible.
  
  .. describe:: Input
  
  MolSystem.Residue, MolSystem.Residue

  .. describe:: Output
  
  None
  """
  
  resonancesA = getResidueResonances(residueA)

  for resonance in resonancesA:
    assignResonanceResidue(resonance, residueB)

def duplicateResidueAssignments(residueA, residueB, experimentChains=None):
  """
  Assign residueB to an equivalent set of resonances as residueA
  Optional dictionary, keyed by experiment to specifiy which chain (should be parent of residueA
  or residueB or None - for both) an experiment's peak assignments should go with.
 
  .. describe:: Input
  
  MolSystem.Residue, MolSystem.Residue, Dict of Nmr.Experiment:MolSystem.Chain (or None)

  .. describe:: Output
  
  None
  """
  
  from ccpnmr.analysis.core.AssignmentBasic import assignResToDim
  
  project = residueA.root
  nmrProject = project.currentNmrProject
  
  atomSetDict = {}
  for atom in residueB.atoms:
    atomSet = atom.atomSet
    if atomSet:
      atomSetDict[atomSet.name] = atomSet
  
  if experimentChains is None:
    experimentChains = {}
  else:
    for experiment in experimentChains:
      chain = experimentChains[experiment]
      molSystems = experiment.molSystems
      if chain and chain.molSystem not in molSystems:
        experiment.addMolSystem(chain.molSystem)
      else:
        if residueA.chain.molSystem not in molSystems:
          experiment.addMolSystem(residueA.chain.molSystem)
        if residueB.chain.molSystem not in molSystems:
          experiment.addMolSystem(residueB.chain.molSystem)
  
  resonancesA = getResidueResonances(residueA)
  resonancesB = getResidueResonances(residueB)

  resonancesDictA = {}
  for resonance in resonancesA:
    atomSetNames = [ass.name for ass in resonance.resonanceSet.atomSets]
    atomSetNames.sort()
    key = tuple(atomSetNames)
    resonancesDictA[key] = resonancesDictA.get(key, []) + [resonance,]

  resonancesDictB = {}
  for resonance in resonancesB:
    atomSetNames = [ass.name for ass in resonance.resonanceSet.atomSets]
    atomSetNames.sort()
    key = tuple(atomSetNames)
    resonancesDictB[key] = resonancesDictB.get(key, []) + [resonance,]

 
  transfers = []
  pureTransfer = True
  for key in resonancesDictA.keys():
    resonances1 = resonancesDictA[key]
    resonances2 = resonancesDictB.get(key, [])
    
    atomSets = []
    for atomSetName in key:
      if atomSetDict.has_key(atomSetName):
        atomSets.append(atomSetDict[atomSetName])
    
    if len(atomSets) == len(key): # If we find all equiv atom sets in destination (not always true if different res type)

      # Duplications of resonances is only done if needed at a per atomSet level
      doDuplicate = False
      contribs = set()
      
      for resonance1 in resonances1:
        contribs.update(resonance1.peakDimContribs)
        
      if contribs:
        for contrib in contribs:
          experiment = contrib.peakDim.peak.peakList.dataSource.experiment
          if experimentChains.get(experiment, residueB.chain) is not residueB.chain: # Not a plain transfer
            doDuplicate = True
            pureTransfer = False
            break
      
      else: # All orphans
        for experiment in experimentChains:
          if experimentChains[experiment] is not residueB.chain:
            doDuplicate = True
            pureTransfer = False
            break
             
               
      if doDuplicate: # Not a plain transfer Need to make new resonances
        while len(resonances2) < len(resonances1):
          resonances2.append(None)
        

        for i, resonance1 in  enumerate(resonances1):
          resonance2  = resonances2[i]
          peakDimContribs = list(resonance1.peakDimContribs)
          
          if peakDimContribs:
            for peakDimContrib in peakDimContribs:
              peakDim    = peakDimContrib.peakDim
              experiment = peakDim.peak.peakList.dataSource.experiment
              chainB     = experimentChains.get(experiment)
 
              if chainB is None: # Duplication on this peak
                if resonance2 is None:
                  resonance2 = nmrProject.newResonance(isotopeCode=resonances1[0].isotopeCode)
                  assignAtomsToRes(atomSets, resonance2)
                  resonances2[i] = resonance2
 
                assignResToDim(peakDim, resonance2, doWarning=False)
 
              elif chainB is residueB.chain: # Transfer this peak
                if resonance2 is None:
                  resonance2 = nmrProject.newResonance(isotopeCode=resonances1[0].isotopeCode)
                  assignAtomsToRes(atomSets, resonance2)
                  resonances2[i] = resonance2
 
                assignResToDim(peakDim, resonance2, doWarning=False)
                for peakContrib in peakDimContrib.peakContribs:
                  if len(peakContrib.peakDimContribs) < 2:
                    peakContrib.delete()
                
                if not peakDimContrib.isDeleted:
                  peakDimContrib.delete()
           
          else:
            if resonance2 is None:
              resonance2 = nmrProject.newResonance(isotopeCode=resonances1[0].isotopeCode)
              assignAtomsToRes(atomSets, resonance2)
              resonances2[i] = resonance2
             
              
      else: 
        # Plain transfer, move existing resonances to different atoms
        transfers.append((resonances1, atomSets))

  # Transfer original spin system to destination residue
  # to keep existing links and attrs (pure transfers only)
  spinSystem = nmrProject.findFirstResonanceGroup(residue=residueA)
  if spinSystem and pureTransfer:
    spinSystem.residue = residueB
    spinSystem.ccpCode = residueB.ccpCode
    spinSystem.chains = [residueB.chain,]
  
  # Clear all resonance to spinSystem links first so
  # that assigniment can pickup the original spin system
  for resonances, atomSets in transfers:
    for resonance in resonances:
      resonance.resonanceGroup = None
    
  for resonances, atomSets in transfers:
    for resonance in resonances:
      assignAtomsToRes(atomSets, resonance)
   
   
def getSequenceResidueMapping(chain, sequence):
  """Match a sequence of residue ccpCodes to an existing chain and calculate the alignment score.
  .. describe:: Input
  
  MolSystem.Chain, List of Words (MolSystem.Residue.ccpCodes)

  .. describe:: Output
  
  List of 2-List of [Int or None, MolSystem.Residue or None], Float (score)
  """
  from ccp.general.Constants import ccpCodeToCode1LetterDict
  
  molType = chain.molecule.molType

  if ccpCodeToCode1LetterDict.get(molType) is None:
    letterDict = {}
  else:
    letterDict = ccpCodeToCode1LetterDict[molType]
  
  seq1 = chain.molecule.stdSeqString
  seq2 = ''.join([letterDict.get(ccpCode, 'X') for ccpCode in sequence])
  
  if molType in (DNA_MOLTYPE, RNA_MOLTYPE, DNARNA_MOLTYPE):
    seqA, seqB, score = sequenceAlign(seq1,seq2,NAMAT)
 
  else:
    seqA, seqB, score = sequenceAlign(seq1,seq2,BLOSUM62)
 
  sortedResidues = chain.sortedResidues()

  x = 0
  y = 0
  mapping = []
  for i in range(len(seqA)):
    mapping.append([None,None])
    if seqA[i] != '-':
      mapping[i][1] = sortedResidues[x] 
      x += 1
      
    if seqB[i] != '-':
      mapping[i][0] = y 
      y += 1

  return mapping, score

def getChainResidueMapping(chainA, chainB):
  """Find the corresponding pairs of residues in two sequence similar chains.
             Matches independent chain fragments, which can be of different molTypes
  .. describe:: Input
  
  MolSystem.Chain, MolSystem.Chain

  .. describe:: Output
  
  List of List of [MolSystem.Residue or None]
  """
  
  fragmentsA = []
  for fragment in chainA.chainFragments:
    molType  = fragment.findFirstResidue().molResidue.molType
    seq      = ''
    residues = fragment.residues
    for residue in residues:
      seq += residue.chemCompVar.chemComp.code1Letter or 'X'

    fragmentsA.append( (residues, molType, seq, fragment) )
  

  fragmentsB = []
  for fragment in chainB.chainFragments:
    molType  = fragment.findFirstResidue().molResidue.molType
    seq      = ''
    residues = fragment.residues
    for residue in residues:
      seq += residue.chemCompVar.chemComp.code1Letter or 'X'
  
    fragmentsB.append( (residues, molType, seq, fragment) )
    
  
  scores = []  
  for residuesA, molTypeA, seqA, fragmentA in fragmentsA:
    for residuesB, molTypeB, seqB, fragmentB in fragmentsB:
      if molTypeB != molTypeA:
        continue
      
      if molType == PROTEIN_MOLTYPE:
        seq1, seq2, score = sequenceAlign(seqA,seqB,BLOSUM62)

      elif molType in (DNA_MOLTYPE, RNA_MOLTYPE):
        seq1, seq2, score = sequenceAlign(seqA,seqB,NAMAT)
      
      else:
        seq1, seq2 = seqA, seqB
        
        while len(seq1) < len(seq2):
          seq1 += '-'
        while len(seq2) < len(seq1):
          seq2 += '-'
        
        n1 = float(len(seqA))
        n2 = float(len(seqB))
        score = min(n1,n2)/max(n1,n2)
        
        if residuesA[0].seqCode == residuesB[0].seqCode:
          score += 1.0
  
      scores.append((score, seq1, seq2, residuesA, residuesB, fragmentA, fragmentB))
  
  scores.sort()
  
  done = {}
  totalScore = 0.0  
  mapping = []
  while scores:
    score, seq1, seq2, residuesA, residuesB, fragmentA, fragmentB = scores.pop()
    
    if done.get(fragmentA):
      continue

    if done.get(fragmentB):
      continue

    done[fragmentA] = True
    done[fragmentB] = True
    totalScore += score
    
    x = 0
    y = 0
    j = len(mapping)
    for i in range(len(seq1)):
      k = i+j
      mapping.append([None,None])
      residueA = None
      residueB = None
      
      if seq1[i] != '-':
        residueA = residuesA[x]
        if done.get(residueA):
          residueA = None
        else:
          mapping[k][0] = residueA
          
        x += 1
 
      if seq2[i] != '-':
        residueB = residuesB[y]
        if done.get(residueB):
          residueB = None
        else:
          mapping[k][1] = residueB
 
        y += 1
      
      if residueA and residueB:
        done[residueA] = True
        done[residueB] = True

  return mapping, totalScore


def getBoundAtoms(atom):
  """Get a list of atoms bound to a given atom..
  .. describe:: Input
  
  MolSystem.Atom

  .. describe:: Output
  
  List of MolSystem.Atoms
  """
  
  if hasattr(atom, 'boundAtoms'):
    return atom.boundAtoms
  
  atoms    = []
  chemAtom = atom.chemAtom
  residue  = atom.residue
  
  chemAtomDict = {}
  for atom2 in residue.atoms:
    # Only atoms specific to ChemCompVar :-)
    chemAtomDict[atom2.chemAtom] = atom2
  
  for chemBond in chemAtom.chemBonds:
    for chemAtom2 in chemBond.chemAtoms:
      if chemAtom2 is not chemAtom:
        atom2 = chemAtomDict.get(chemAtom2)
        if atom2:
          atoms.append(atom2)
  
  linkEnd = residue.chemCompVar.findFirstLinkEnd(boundChemAtom=chemAtom)
  if linkEnd:
    molResLinkEnd = residue.molResidue.findFirstMolResLinkEnd(linkEnd=linkEnd)
    
    if molResLinkEnd:
      molResLink = molResLinkEnd.molResLink
    
      if molResLink:
        for molResLinkEnd2 in molResLink.molResLinkEnds:
          if molResLinkEnd2 is not molResLinkEnd:
            residue2 = residue.chain.findFirstResidue(molResidue=molResLinkEnd2.molResidue)
            
            if residue2:
              chemAtom2 = molResLinkEnd2.linkEnd.boundChemAtom
              atom2 = residue2.findFirstAtom(chemAtom=chemAtom2)
              
              if atom2:
                atoms.append(atom2)
              
            break
  
  atom.boundAtoms = atoms    
  return atoms
  
def areAtomsBound(atom1, atom2):
  """Dertemine whether two atoms are bonded together
  .. describe:: Input
  
  MolSystem.Atom, MolSystem.Atom

  .. describe:: Output
  
  Boolean
  """

  if not hasattr(atom1, 'isAtomBound'):
    atom1.isAtomBound = {}
  elif atom1.isAtomBound.has_key(atom2):
    return atom1.isAtomBound[atom2]

  if not hasattr(atom2, 'isAtomBound'):
    atom2.isAtomBound = {}
  elif atom2.isAtomBound.has_key(atom1):
    return atom2.isAtomBound[atom1]

  isBound = False
  
  if atom1 is not atom2:
    residue1 = atom1.residue
    residue2 = atom2.residue
    
    if residue2.chain is residue1.chain:
      if residue2 is not residue1:

        linkEnd1 = residue1.chemCompVar.findFirstLinkEnd(boundChemAtom=atom1.chemAtom)
        if not linkEnd1:
          isBound = False

        else:
          linkEnd2 = residue2.chemCompVar.findFirstLinkEnd(boundChemAtom=atom2.chemAtom)
          if not linkEnd2:
            isBound = False
 
          else:
            molResLinkEnd1 = residue1.molResidue.findFirstMolResLinkEnd(linkEnd=linkEnd1)
            if not molResLinkEnd1:
              isBound = False
 
            else:
              molResLinkEnd2 = residue2.molResidue.findFirstMolResLinkEnd(linkEnd=linkEnd2)
              if not molResLinkEnd2:
                isBound = False
 
              elif molResLinkEnd2 in molResLinkEnd1.molResLink.molResLinkEnds:
                isBound = True
 
              else:
                isBound = False

      else:
        for chemBond in atom1.chemAtom.chemBonds:
          if atom2.chemAtom in chemBond.chemAtoms:
            isBound = True
            break

  atom1.isAtomBound[atom2] = isBound
  atom2.isAtomBound[atom1] = isBound
  
  return isBound

def areResonancesBound(resonance1,resonance2):
  """
  Determine whether two resonances are assigned to directly bonded atoms
  
  .. describe:: Input
  
  Nmr.Resonance, Nmr.Resonance

  .. describe:: Output
  
  Boolean
  """

  if resonance1 is resonance2:
    return False
  
  
  resonanceSet1 = resonance1.resonanceSet
  resonanceSet2 = resonance2.resonanceSet
    
  if resonanceSet1 and resonanceSet2:
    
    
    atomSets1 = resonanceSet1.atomSets
    atomSets2 = resonanceSet2.atomSets
    
    bound1 = resonance1.covalentlyBound
    bound2 = resonance2.covalentlyBound
    
    # Have to look through everything to get the right equiv
    # & prochiral pair - Val CGa HGb: check both atomSets for each
    # Phe Ce* He*: check both atoms (only one atomSet each)
    for atomSet1 in atomSets1:
      for atom1 in atomSet1.atoms:
        for atomSet2 in atomSets2:
          for atom2 in atomSet2.atoms:
            if areAtomsBound(atom1, atom2):
              # Val Cgb - Hgb* can appear bound,
              # so check resonance links
              if (resonance1.isotopeCode == '1H') and ( len(atomSets2) > 1 ):
                if bound1 and resonance2 not in bound1:
                  continue

              elif (resonance2.isotopeCode == '1H') and ( len(atomSets1) > 1 ):
                if bound2 and resonance1 not in bound2:
                  continue
              
              return True

    return False

  from ccpnmr.analysis.core.AssignmentBasic import getBoundResonances
 
  resonances = getBoundResonances(resonance1)
  if resonance2 in resonances:
    return True
  
  else:
    return False  

def findBoundResonances(resonance):
  """Find any resonances which are assigned to atoms covalently bound
             to the assigned atoms of the input resonance
  .. describe:: Input
  
  Nmr.Resonance

  .. describe:: Output
  
  List of Nmr.Resonances
  """

  from ccpnmr.analysis.core.AssignmentBasic import getBoundResonances
  
  return getBoundResonances(resonance)

def greekSortAtomNames(dataList, molType=PROTEIN_MOLTYPE):
  """Sorts a list of atom names according to greek/sidechain order when letters are latinised
  .. describe:: Input
  
  List of Strings

  .. describe:: Output
  
  List of Strings (sorted)
  """
  
  sub = re.sub
  
  #order = 'ABGDEZHQIKLMNXOPRSTUFCYW'
  
  sortList = []
  for x in dataList:
    if type(x) in (tuple,list):
      sortName = x[0]
    else:
      sortName = x
    
    sortName = sortName.upper()  
    sortName = sub('(.+\')', 'zzz@\\1', sortName)
    sortName = sub('^(\d)','zz@\\1', sortName)
    sortName = sub('N(\S*)','\'\'@N\\1',sortName)
    sortName = sub('CO','\'\'@CO',sortName)
    sortName = sub('G', 'c', sortName)
    sortName = sub('Z', 'f', sortName)
    sortName = sub('Q', 'hzz', sortName)
    sortName = sub('X', 'nzz', sortName)
    sortName = sub('F', 'v', sortName)
    sortName = sub('C', 'w', sortName)
    sortName = sub('W', 'z', sortName)
    sortName = sortName.upper()  
    
    if molType == PROTEIN_MOLTYPE:
      sortName = sub('Hn', 'H1n', sortName)
      
    sortList.append( (sortName,x) )
    
  sortList.sort()
  dataList = [e[1] for e in sortList]
  
  return dataList

def getMolTypeCcpCodes(molType='all', project=None):
  """Gives ccpCodes for chemComps according to molecule type: e.g. DNA
             Project can be input to search for non-standard types.
  .. describe:: Input
  
  Implementation.Project, String (ChemComp.molType or 'all')

  .. describe:: Output
  
  List of Words (ChemComp.CcpCodes)
  """

  ccpCodes = []
  if molType == 'all':
    molTypes = [PROTEIN_MOLTYPE, DNA_MOLTYPE, RNA_MOLTYPE,
                CARBOHYDRATE_MOLTYPE, OTHER_MOLTYPE]
  else:
    molTypes = [molType,]
    
  for molType in molTypes:
    chemCompDict = getChemCompOverview(molType, project)
    
    if chemCompDict:
      ccpCodes.extend( chemCompDict.keys() )
        
  if ccpCodes:
    ccpCodes.sort()
  
  return ccpCodes

def getChemCompOverview(molType, project=None):
  """Get a dictionary containing details of all available chemical compounds
             for a given moleucle type. Project can be input to search for loaded,
             but non standard chem comps.
  .. describe:: Input
  
  Word (Molecule.MolResidue.MolType), Implementation.Project

  .. describe:: Output
  
  Dict of ChemComp.ccpCode:[Word, Word, Line, Word]
             (1-letter Code, 3-letter Code, name, mol formula)
  """
  
  if molType == OTHER_MOLTYPE:
    from ccp.general.ChemCompOverview import chemCompOverview
    chemCompDict = chemCompOverview.get(molType, {}) 
    
  else:
    chemCompDict = chemCompStdDict.get(molType, {}) 
   
  if project: 
    for chemComp in project.findAllChemComps(molType=molType):
      ccpCode  = chemComp.ccpCode
      
      if chemCompDict.get(ccpCode) is None:
        chemCompVar = chemComp.findFirstChemCompVar(linking=None, isDefaultVar=True) \
                      or chemComp.findFirstChemCompVar(isDefaultVar=True) \
                      or chemComp.findFirstChemCompVar()
        
        if chemCompVar:
          molFormula = chemCompVar.formula
        else:
          molFormula = ''  
      
        chemCompDict[ccpCode] = [chemComp.code1Letter,
                                 chemComp.code3Letter,
                                 chemComp.name,
                                 None]   # Added RHF 1/7/10 for bug fix.
      
  return  chemCompDict
        
def makeMolSystemLink(residueA,residueB,linkEndA,linkEndB):
  """Make a molSystemLink given two residues and the linkEnds to be joined
  .. describe:: Input
  
  MolSystem.Residue, MolSystem.Residue, ChemComp.LinkEnd, ChemComp.LinkEnd

  .. describe:: Output
  
  MolSystem.MolSystemLink
  """
  molSystem = residueA.chain.molSystem  
  molSysLinkEndA = residueA.newMolSystemLinkEnd(linkCode=linkEndA.linkCode)
  molSysLinkEndB = residueB.newMolSystemLinkEnd(linkCode=linkEndB.linkCode)
  molSysLinkEnds = (molSysLinkEndA,molSysLinkEndB)
  
  try:
    molSystemBond = molSystem.newMolSystemLink(molSystemLinkEnds=molSysLinkEnds) 
  except Exception:
    return

  removeAtoms = []
  for atom in removeAtoms:
    atomSet = atom.atomSet
    if atomSet:
      for resonanceSet in atomSet.resonanceSets:
        resonanceSet.delete()
      atomSet.delete()
    atom.delete()

  for residue in (residueA,residueB):
    residueMapping = getResidueMapping(residue)
    for atomSetMapping in residueMapping.atomSetMappings:
      atomSetMapping.delete()
      
    makeResidueAtomSets(residue)

  return molSystemBond



def makeAtomSet(guiName,atoms,chemAtomSet,mappingType):
  """
  Make atomSet and atomSetMapping for equivalent atoms
  
  .. describe:: Input
  
  Word (AtomSet.name), List of MolSystem.Atoms,
  ChemComp.ChemAtomSet, Word (AtomSetMapping.mappingType)
  
  .. describe:: Output
  
  Nmr.AtomSet
  """
  
  # RHFogh 3/12/09 - refactored to reduce getAtomSet calls
  
  atom0 = list(atoms)[0]
  project = atom0.root
  
  atomSets = [x.atomSet for x in atoms]
  atomSet0 = atomSets[0]
  aSet = set(atomSets)
  if len(aSet) != 1:
    for atomSet in aSet:
      if atomSet and not atomSet.resonanceSets:
        atomSet.delete()
  
  nmrProject = project.currentNmrProject
  
  if atomSet0 is None:
    atomSet = nmrProject.newAtomSet(atoms=atoms)
  else:
    atomSet = atomSet0
 
  residue = atom0.residue
    
  residueMapping = getResidueMapping(residue)
  if not residueMapping.findFirstAtomSetMapping(name=guiName):
    makeAtomSetMapping(residueMapping,guiName,(atomSet,),chemAtomSet,mappingType)

  atomSet.name = guiName
  return atomSet

def getResidueMapping(residue, aromaticsEquivalent=True):
  """Gives the Analysis.ResidueMapping for a residue
             Makes a new one with new AtomSetsMappings if not exists
             Makes a ChainMapping too if needed.
  .. describe:: Input
  
  MolSystem.Residue

  .. describe:: Output
  
  Analysis.ResidueMapping
  """
  
  if hasattr(residue, 'residueMapping'):
    return residue.residueMapping
  
  residueMapping = None
  analysisProject = residue.root.currentAnalysisProject
  chainMapping = analysisProject.findFirstChainMapping(chain=residue.chain)

  if not chainMapping:
    chainMapping = analysisProject.newChainMapping(molSystemCode=residue.chain.molSystem.code,
                                                   chainCode=residue.chain.code)
    chainMapping.residueMappingDict = {}
  else:
    if not hasattr(chainMapping, 'residueMappingDict'):
      chainMapping.residueMappingDict = {}
    
    residueMapping = chainMapping.residueMappingDict.get(residue.seqId)
    
    if not residueMapping:
      residueMapping = chainMapping.findFirstResidueMapping(seqId=residue.seqId)
      chainMapping.residueMappingDict[residue.seqId] = residueMapping
      
  if not residueMapping:
    residueMapping = chainMapping.newResidueMapping(seqId=residue.seqId)
    #makeResidueAtomSets(residue)

  residue.residueMapping = residueMapping 
  
  if not residueMapping.atomSetMappings:
    makeResidueAtomSets(residue, aromaticsEquivalent=aromaticsEquivalent)

  return residueMapping

def makeAtomSetMapping(residueMapping,name,atomSets,chemAtomSet,mappingType,resonances=None):
  """Make atomSetMapping given atomSets and mapping type
  .. describe:: Input
  
  Analysis.ResidueMapping, Word, MolSystem.Residue,
             List of Nmr.AtomSets, ChemComp.ChemAtomSet,
             Word, Word, List of Nmr.Resonances

  .. describe:: Output
  
  Analysis.AtomSetMapping
  """

  atom          = list(atomSets)[0].findFirstAtom()
  elementSymbol = atom.chemAtom.elementSymbol
  serials       = []
  for atomSet in atomSets:
    serials.append(atomSet.serial)

  molType = residueMapping.residue.molResidue.molType    
  guiName = makeGuiName(name, elementSymbol, molType)
      
  atomSetMapping = residueMapping.newAtomSetMapping(name=guiName,mappingType=mappingType,
                                                    atomSetSerials=serials,
                                                    chemAtomSet=chemAtomSet,
                                                    elementSymbol=elementSymbol)
                                                    
  if resonances is not None:
    resSerials = []
    for resonance in resonances:
      resSerials.append(resonance.serial)
    atomSetMapping.setResonanceSerials(resSerials)

  return atomSetMapping

def makeGuiName(name, elementSymbol, molType):
  """Convert atom or atomSet name into name for gui: e.g H becomes Hn
  .. describe:: Input
  
  Word (Nmr.AtomSet.name), Word, Word

  .. describe:: Output
  
  Word 
  """

  #if molType == PROTEIN_MOLTYPE:
  #  if name == 'H':
  #    name = name + 'n'
   
  guiName = elementSymbol + name[len(elementSymbol):].lower()  

  return guiName

def makeResidueLocTag(residue, chemAtoms):
  """Make unique identifier for a given residue type in a given chain location
  .. describe:: Input
  
  MolSystem.Residue, List of ChemCmp.chemAtoms

  .. describe:: Output
  
  Word
  """

  # check if any atoms have been deleted (to make a molSystemLink)
  tag = residue.ccpCode + residue.linking
  
  if len(chemAtoms) != len(residue.atoms):
    names = []
    for chemAtom in chemAtoms:
      names.append(chemAtom.name)
    for atom in residue.atoms:
      if atom.name in names:
        names.remove(atom.name)
    for name in names:
      tag = tag + '-' + name
      
  return tag

def makeResidueAtomSetsNonEquivalent(residue):
  """Remake a residue's atom sets if they are found to be non-equivalent: e.g.
             if an aromatic ring does not rotate quickly on the NMR timescale
  .. describe:: Input
  
  MolSystem.Residue

  .. describe:: Output
  
  None
  """
  
  residueMapping = getResidueMapping(residue)
  #chain   = residue.chain
  molType = residue.molResidue.molType
  nonequivalent     = {}
  elementSymbolDict = {}
  chemAtomSetDict   = {}
  for atom in residue.atoms:
    chemAtom = atom.chemAtom
    chemAtomSetDict[atom] = chemAtom
    elementSymbol = chemAtom.elementSymbol
    if chemAtom.chemAtomSet:
      chemAtomSet = chemAtom.chemAtomSet
      name = chemAtomSet.name
      if chemAtomSet.isEquivalent is None: # i.e. not False, aromatic rotation
        if nonequivalent.get(name) is None:
          nonequivalent[name] = []
        nonequivalent[name].append(atom)
        elementSymbolDict[name] = chemAtom.elementSymbol
        chemAtomSetDict[name] = chemAtomSet

  for groupName in nonequivalent.keys():
    atoms = nonequivalent[groupName]
    atomSet = atoms[0].atomSet
    if atomSet:
      for atom in atoms[1:]:
        if atom.atomSet is not atoms[0].atomSet:
          return # already seperate
    
    elementSymbol = elementSymbolDict[groupName]
    chemAtomSet   = chemAtomSetDict[groupName]
    resonances = []

    if atomSet:
      for resonanceSet in atomSet.resonanceSets:
        for resonance in resonanceSet.resonances:
          resonances.append(resonance)
    
      atomSet.delete()
      name = makeGuiName(groupName, elementSymbol, molType)
      atomSetMapping = residueMapping.findFirstAtomSetMapping(name=name)
      if atomSetMapping:
        atomSetMapping.delete()

    atomSets = []
    atomSetNames = []
    for atom in atoms:
      name = chemAtomSetDict[atom].name
      atomSet = makeAtomSet(name,(atom,),chemAtomSet,'stereo')
      atomSets.append(atomSet)
      atomSetNames.append(name)

    resonanceSet = None
    for resonance in resonances:
      resonanceSet = assignAtomsToRes(atomSets,resonance,resonanceSet)

    for n, atom in enumerate(atoms):
      name = chemAtomSetDict[atom].name
      name2 = makeNonStereoName(molType, name, n)
      makeGuiMultiAtomSet(residue, name2, atomSetNames,elementSymbol,'nonstereo',chemAtomSet)

    makeGuiMultiAtomSet(residue, groupName, atomSetNames,elementSymbol,'ambiguous',chemAtomSet)

def makeResidueAtomSetsEquivalent(residue):
  """Remake a residue's atom sets if they are found to be equivalent: e.g.
             if an aromatic ring rotates quickly on the NMR timescale
  .. describe:: Input
  
  MolSystem.Residue

  .. describe:: Output
  
  None
  """

  #chain   = residue.chain
  molType = residue.molResidue.molType
  residueMapping = getResidueMapping(residue)
  equivalent        = {}
  elementSymbolDict = {}
  chemAtomSetDict   = {}
  for atom in residue.atoms:
    chemAtom = atom.chemAtom
    chemAtomSetDict[atom] = chemAtom
    elementSymbol = chemAtom.elementSymbol
    if chemAtom.chemAtomSet:
      chemAtomSet = chemAtom.chemAtomSet
      name = chemAtomSet.name
      if chemAtomSet.isEquivalent is None: # i.e. not False, aromatic rotation
        if equivalent.get(name) is None:
          equivalent[name] = []
        equivalent[name].append(atom)
        elementSymbolDict[name] = chemAtom.elementSymbol
        chemAtomSetDict[name] = chemAtomSet
         
  for groupName in equivalent.keys():
    atoms = equivalent[groupName]
    if atoms[0].atomSet:
      for atom in atoms[1:]:
        if atom.atomSet is atoms[0].atomSet:
          return
    
    elementSymbol = elementSymbolDict[groupName]
    chemAtomSet   = chemAtomSetDict[groupName]
    resonances = []
    for atom in atoms:
      # TBD more nested layers?
      # delete ambiguous
      name = makeGuiName(chemAtomSet.name, elementSymbol, molType)
      atomSetMapping = residueMapping.findFirstAtomSetMapping(name=name)
      if atomSetMapping:
        atomSetMapping.delete()

      # delete stereospecific
      name = makeGuiName(atom.chemAtom.name, elementSymbol, molType)
      atomSetMapping = residueMapping.findFirstAtomSetMapping(name=name)
      if atomSetMapping:
        atomSetMapping.delete()
        
      # delete non-stereospecific
      name = makeGuiName(makeNonStereoName(molType, atom.chemAtom.name), elementSymbol, molType)
      atomSetMapping = residueMapping.findFirstAtomSetMapping(name=name)
      if atomSetMapping:
        atomSetMapping.delete()
        
      atomSet = atom.atomSet
      if atomSet:
        for resonanceSet in atomSet.resonanceSets:
          resonances.extend(resonanceSet.resonances)
        atomSet.delete()
   
    elementSymbol = elementSymbolDict[groupName]
    chemAtomSet = chemAtomSetDict[groupName]
    # make single equivalent group
    #makeAtomSet(,groupName,atoms,chemAtomSet,'simple')
    atomSet = makeAtomSet(groupName,atoms,chemAtomSet,'simple')
    for resonance in resonances:
      assignAtomsToRes([atomSet,],resonance)
    

def makeResidueAtomSets(residue, aromaticsEquivalent=True):
  """Make all atomSets and atomSetMappings for a given residue
             Aromatic Phe, Tyr (Hd1,Hd2), (He1,He2) can be made into 
             single equivalent atom sets due to rotation.
  .. describe:: Input
  
  MolSystem.Residue, Boolean

  .. describe:: Output
  
  None
  """
  
  getResidueMapping(residue)
  
  equivalent = {}
  elementSymbolDict = {}
  nonequivalent = {}
  multiSet = {}
  chemAtomSetDict = {}
  inMultiSet = {}
  molType = residue.molResidue.molType
  
  for atom in residue.atoms:  
    chemAtom = atom.chemAtom
    chemAtomSetDict[atom] = chemAtom
    elementSymbol = chemAtom.elementSymbol
    chemAtomSet = chemAtom.chemAtomSet

    if chemAtomSet is None:
      name = chemAtom.name
      makeAtomSet(name,(atom,),None,'simple')
      
    else:
      name = chemAtomSet.name
      elementSymbolDict[name] = elementSymbol
      chemAtomSetDict[name] = chemAtomSet
      if chemAtomSet.isEquivalent:
        if equivalent.get(name) is None:
          equivalent[name] = []
        equivalent[name].append(atom)
        
      elif (chemAtomSet.isEquivalent is None) and atom.atomSet and (len(atom.atomSet.atoms) > 1):
        # aromatic rotation prev set
        if equivalent.get(name) is None:
          equivalent[name] = []
        equivalent[name].append(atom)
           
      elif (chemAtomSet.isEquivalent is None) and (not atom.atomSet) and aromaticsEquivalent:
        # aromatic rotation to be set
        if equivalent.get(name) is None:
          equivalent[name] = []
        equivalent[name].append(atom)
          
      else:
        if nonequivalent.get(name) is None:
          nonequivalent[name] = []
        nonequivalent[name].append(atom)
   
      if chemAtomSet.chemAtomSet is not None:
        multiName = chemAtomSet.chemAtomSet.name
        chemAtomSetDict[multiName] = chemAtomSet.chemAtomSet
        elementSymbolDict[multiName] = elementSymbol
        if multiSet.get(multiName) is None:
          multiSet[multiName] = {}
        multiSet[multiName][name] = 1
        inMultiSet[name] = multiName

  for groupName in equivalent.keys():
    atoms = equivalent[groupName]
    elementSymbol = elementSymbolDict[groupName]
    chemAtomSet = chemAtomSetDict[groupName]
    if len(atoms)==2:
      # not enough atoms for multi sets!
      makeAtomSet(groupName,atoms,chemAtomSet,'simple')
    else:
      if inMultiSet.get(groupName):
        # e.g. for Val Hg1*
        makeAtomSet(groupName,atoms,chemAtomSet,'stereo')
 
      else:
        makeAtomSet(groupName,atoms,chemAtomSet,'simple')

  for groupName in nonequivalent.keys():
    atoms = nonequivalent[groupName]
    elementSymbol = elementSymbolDict[groupName]
    chemAtomSet = chemAtomSetDict[groupName]
    atomSetNames = []
    
    if len(atoms) == 1:
      atom = atoms[0]
      # not enough atoms for prochiral. Corrupt ChemComp
      makeAtomSet(atom.name, atoms, None, 'simple')
      continue
      
    for atom in atoms:
      name = chemAtomSetDict[atom].name
      makeAtomSet(name,(atom,),chemAtomSet,'stereo')
      atomSetNames.append(name)

    for n, atom in enumerate(atoms):
 
      #name = chemAtomSetDict[atom].name
      #name2 = makeNonStereoName(molType, name, n)
      # Shouldn't have to do this if non-equiv groups have paired names
      
      name2 = makeNonStereoName(molType, '%s%d' % (chemAtomSet.name[:-1], n), n)
        
      makeGuiMultiAtomSet(residue, name2, atomSetNames,
                          elementSymbol,'nonstereo',chemAtomSet)

    makeGuiMultiAtomSet(residue, groupName, atomSetNames,
                        elementSymbol,'ambiguous',chemAtomSet)

  for groupName in multiSet.keys():
    atomSetNames  = multiSet[groupName].keys()
    elementSymbol = elementSymbolDict[groupName]
    chemAtomSet   = chemAtomSetDict[groupName]
    if "|" in groupName:
      # we don't do these pseudoatoms in Analysis
      continue

    # e.g. for Val Hga*
    for n, atomSetName in enumerate(atomSetNames):
      name2 = makeNonStereoName(molType, atomSetName, n)
      makeGuiMultiAtomSet(residue, name2, atomSetNames,
                          elementSymbol,'nonstereo',chemAtomSet)
    
    makeGuiMultiAtomSet(residue, groupName, atomSetNames,
                        elementSymbol,'ambiguous',chemAtomSet)

def makeNonStereoName(molType, name, n=None):
  """Convert a sterospecific atom name into a non-stereospecific one for a GUI
  .. describe:: Input
  
  Word, Int (naming offset from start of alphabet)

  .. describe:: Output
  
  Word
  """

  match = re.match('(\w+)(\d|\'+)(\D*)', name)
  
  if not match:
    #print molType, name, n
    return 
  
  
  letters = match.group(1)
  number  = match.group(2)
  prime   = ''
  
  if number == '\'':
    number = 1
    prime = '\''
  elif number == '\'\'':
    number = 2
    prime = '\''
  
  if n is None:
    n = int(number) - 1

  name = letters + prime + chr(ord('a')+n)+ match.group(3)
    
  return name

def makeGuiMultiAtomSet(residue,multiGuiName,guiSetsNames,elementSymbol,mappingType,chemAtomSet):
  """Make atom set mappings for multiple atom set selections
  .. describe:: Input
  
  MolSystem.Residue, Word (Analysis.AtomSetMapping.name),
             List of Words (Analysis.AtomSetMapping.names),
             Word, Word, ChemComp.ChemAtomSet

  .. describe:: Output
  
  Analysis.AtomSetMapping
  """
  
  if "|" in multiGuiName:
    return
  
  residueMapping = getResidueMapping(residue)  
  molType = residue.molResidue.molType
  for guiName in guiSetsNames:
    atomSetMapping = residueMapping.findFirstAtomSetMapping(name=makeGuiName(guiName, elementSymbol, molType))
    if atomSetMapping is None:
      print "Non-existent group error in makeGuiMultiAtomSet for", residue.molResidue.ccpCode, residue.seqCode, guiName
      return
    #atomSet      = atomSetMapping.atomSets[0]
    chemAtomSet1 = atomSetMapping.chemAtomSet
    
    for guiName2 in guiSetsNames:
      atomSetMapping2 = residueMapping.findFirstAtomSetMapping(name=makeGuiName(guiName2, elementSymbol, molType))
      if atomSetMapping2 is None:
        print "Non-existent group error in makeGuiMultiAtomSet for", residue.molResidue.ccpCode
        return
      #atomSet      = atomSetMapping2.atomSets[0]
      chemAtomSet2 = atomSetMapping2.chemAtomSet
      if chemAtomSet2 and chemAtomSet1:
        if chemAtomSet1.isProchiral != chemAtomSet2.isProchiral:
          print "Prochiratity error in makeGuiMultiAtomSet for", residue.molResidue.ccpCode
          return
        if chemAtomSet1.isEquivalent != chemAtomSet2.isEquivalent:
          print "Equivalent error in makeGuiMultiAtomSet for ", residue.molResidue.ccpCode
          return

  atomSets = []
  for guiName in guiSetsNames:
    name0 = makeGuiName(guiName, elementSymbol, molType)
    atomSetSerials = residueMapping.findFirstAtomSetMapping(name=name0).atomSetSerials
    for atom in residue.atoms:
      atomSet = atom.atomSet
      if atomSet:
        if atomSet.serial in atomSetSerials and atomSet not in atomSets:
          atomSets.append(atomSet)
          break
    
  if not residueMapping.findFirstAtomSetMapping(name=multiGuiName):
    atomSetMapping = makeAtomSetMapping(residueMapping, multiGuiName, atomSets,
                                        chemAtomSet, mappingType)
  
  return atomSetMapping

def moveMolSystemChain(chain, molSystem):
  """Moves a chain from one molSystem to another
  .. describe:: Input
  
  MolSystem.Chain, MolSystem.MolSystem

  .. describe:: Output
  
  None
  """

  from MergeObjects import mergeObjects

  if chain.molSystem is molSystem:
    return

  molecule = chain.molecule
  project  = molSystem.root
  residues = chain.sortedResidues()
  seq      = []

  for residue in residues:
    seq.append( residue.ccpCode )

  newChain = makeChain(molSystem,molecule)
  
  for chainMapping in project.currentAnalysisProject.chainMappings:
    if chainMapping.chain is chain:
      chainMapping.delete()
  
  residues1 = residues
  
  for i in range(len(seq)):
    residue1 = residues1[i]
    residue2 = newChain.sortedResidues()[i]
    for atom in residue1.atoms:

      atom2 = residue2.findFirstAtom(name=atom.name)
      if atom2:
        mergeObjects(atom, atom2)
      else:
        print "missing %d %s %s" % (residue2.seqCode,residue2.ccpCode,atom.name) 
        
    mergeObjects(residue1, residue2)
    
  mergeObjects(chain, newChain)
      
def sequenceAlign(seq1,seq2,matrix,inspen=2,extpen=1):
  """Aligns two sequence strings (of one letter codes)
              - results are gapped with '-'; first tries an exact substring match,
             if that failts it then uses dynamicAlign
  .. describe:: Input
  
  String, String, Dict of Dicts (homology score matrix), Int, Int, Int, Int

  .. describe:: Output
  
  String, String (aligned & gapped sequence strings)
  """

  def substringAlign(s1, s2, n1, n2, n):
    
    if n1 < n2:
      score = -inspen - extpen*(n2-n1-1)
    else:
      score  = 0
    for x in s1:
      score += matrix[x][x]

    sA = n*'-' + s1 + (n2-n1-n)*'-'
    sB = s2

    return (sA, sB, score)

  nX = len(seq1)
  nY = len(seq2)

  if nX <= nY:
    n = seq2.find(seq1)
    if n >= 0:
      return substringAlign(seq1, seq2, nX, nY, n)
  else:
    n = seq1.find(seq2)
    if n >= 0:
      seqB, seqA, score = substringAlign(seq2, seq1, nY, nX, n)
      return seqA, seqB, score

  return dynamicAlign(seq1, seq2, matrix, inspen, extpen)

def dynamicAlign(seq1,seq2,matrix,inspen=2,extpen=1):
  """Aligns two sequence strings (of one letter codes) - results are gapped with '-'
  .. describe:: Input
  
  String, String, Dict of Dicts (homology score matrix), Int, Int, Int, Int

  .. describe:: Output
  
  String, String (aligned & gapped sequence strings)
  """

  seq1 = re.sub('-|\s','',seq1)
  seq2 = re.sub('-|\s','',seq2)

  seq1 = re.sub('\*','X',seq1)
  seq2 = re.sub('\*','X',seq2)

  maxScore = 0
          
  nX    = len(seq1)+1
  nY    = len(seq2)+1

  M     = [[0] * nY]
  R     = [[2] * nY]
  route = 0

  R[0][0] = 0
  for x in range(1,nX):
    M.append([0,])
    R.append([1,])
    for y in range(1,nY):
      p1 = inspen
      p2 = inspen
      if route == 1:
        p1 = extpen
      elif route == 2:
        p2 = extpen

      score = matrix[seq1[x-1]][seq2[y-1]]
      paths = [M[x-1][y-1]+score,
               M[x-1][y]-p1,
               M[x][y-1]-p2]

      best  = max(paths)
      route = paths.index(best)

      M[x].append( best )
      R[x].append( route )

      if best >= maxScore:
        maxScore = best

  best = M[-1][-1] - 1
  for x0 in range(1, nX):
    m = M[x0][-1]
    if m > best:
      best = m
      x = x0
      y = nY-1
  for y0 in range(1, nY):
    m = M[-1][y0]
    if m > best:
      best = m
      x = nX-1
      y = y0

  route = R[x][y]
  dx = nX - x - 1
  dy = nY - y - 1
  if dx > 0:
    seqA = seq1[-dx:]
    seqB = dx * '-'
  elif dy > 0:
    dy = nY - y - 1
    seqA = dy * '-'
    seqB = seq2[-dy:]
  else:
    seqA = ''
    seqB = ''

  while  x > 0 or y > 0 :
    if route == 0:
      seqA = seq1[x-1] + seqA
      seqB = seq2[y-1] + seqB
      x   -= 1
      y   -= 1

    elif route == 1:
      seqB = '-' + seqB
      seqA = seq1[x-1] + seqA
      x   -= 1

    elif route == 2:
      seqA = '-' + seqA
      seqB = seq2[y-1] + seqB
      y   -= 1

    route = R[x][y]

  return seqA, seqB, maxScore
  
def getRandomCoilShiftCorrectionsDict():
  """

  Citation

  Schwarzinger, S., Kroon, G. J. A., Foss, T. R., Chung, J., Wright, P. E., Dyson, H. J.
  "Sequence-Dependent Correlation of Random Coil NMR Chemical Shifts", 
  J. Am. Chem. Soc. 123, 2970-2978 (2001)

  Values obtained from a GGXGG sequence pentapeptide. 

  """

  data = """
  Ala    H     H   -0.01  -0.05   0.07  -0.10
  Ala    HA    H   -0.02  -0.03  -0.03   0.00
  Ala    C     C   -0.11  -0.77  -0.07  -0.02
  Ala    CA    C   -0.02  -0.17   0.06   0.01
  Ala    N     N   -0.12  -0.33  -0.57  -0.15
  Asn    H     H   -0.01  -0.03   0.13  -0.07
  Asn    HA    H   -0.01  -0.01  -0.02  -0.01
  Asn    C     C   -0.09  -0.66  -0.10  -0.03
  Asn    CA    C   -0.06  -0.03   0.23   0.01
  Asn    N     N   -0.18  -0.26   0.87  -0.17
  Asp    H     H   -0.02  -0.03   0.14  -0.11
  Asp    HA    H   -0.02  -0.01  -0.02  -0.01
  Asp    C     C   -0.08  -0.58  -0.13  -0.04
  Asp    CA    C   -0.03   0.00   0.25  -0.01
  Asp    N     N   -0.12  -0.20   0.86  -0.29
  Arg    H     H    0.00  -0.02   0.15  -0.06
  Arg    HA    H   -0.02  -0.02  -0.02   0.00
  Arg    C     C   -0.06  -0.49  -0.19  -0.03
  Arg    CA    C    0.00  -0.07  -0.01   0.02
  Arg    N     N   -0.06  -0.14   1.62  -0.06
  Cys    H     H    0.00  -0.02   0.20  -0.07
  Cys    HA    H   -0.01   0.02   0.00   0.00
  Cys    C     C   -0.08  -0.51  -0.28  -0.07
  Cys    CA    C   -0.03  -0.07   0.10  -0.01
  Cys    N     N   -0.06  -0.26   3.07   0.00
  Gln    H     H   -0.01  -0.02   0.15  -0.06
  Gln    HA    H   -0.01  -0.02  -0.01   0.00
  Gln    C     C   -0.05  -0.48  -0.18  -0.03
  Gln    CA    C   -0.02  -0.06   0.04   0.01
  Gln    N     N   -0.06  -0.14   1.62  -0.06
  Glu    H     H   -0.01  -0.03   0.15  -0.07
  Glu    HA    H   -0.02  -0.02  -0.02   0.00
  Glu    C     C   -0.09  -0.48  -0.20  -0.03
  Glu    CA    C   -0.01  -0.08   0.05   0.01
  Glu    N     N   -0.06  -0.20   1.51  -0.12
  Gly    H     H    0.00   0.00   0.00   0.00
  Gly    HA    H    0.00   0.00   0.00   0.00
  Gly    C     C    0.00   0.00   0.00   0.00
  Gly    CA    C    0.00   0.00   0.00   0.00
  Gly    N     N    0.00   0.00   0.00   0.00
  His    H     H   -0.01  -0.04   0.20   0.00
  His    HA    H   -0.03  -0.06   0.01   0.01
  His    C     C   -0.10  -0.65  -0.22  -0.07
  His    CA    C   -0.05  -0.09   0.02   0.01
  His    N     N   -0.12  -0.55   1.68   0.17
  Ile    H     H   -0.01  -0.06   0.17  -0.09
  Ile    HA    H   -0.03  -0.02  -0.02  -0.01
  Ile    C     C   -0.20  -0.58  -0.18  -0.02
  Ile    CA    C   -0.07  -0.20  -0.01   0.02
  Ile    N     N   -0.18  -0.14   4.87   0.00
  Leu    H     H    0.00  -0.03   0.14  -0.08
  Leu    HA    H   -0.04  -0.03  -0.05  -0.01
  Leu    C     C   -0.13  -0.50  -0.13  -0.01
  Leu    CA    C   -0.01  -0.10   0.03   0.02
  Leu    N     N   -0.06  -0.14   1.05  -0.06
  Lys    H     H    0.00  -0.03   0.14  -0.06
  Lys    HA    H   -0.02  -0.02  -0.01   0.00
  Lys    C     C   -0.08  -0.50  -0.18  -0.03
  Lys    CA    C   -0.01  -0.11  -0.02   0.02
  Lys    N     N   -0.06  -0.20   1.57  -0.06
  Met    H     H    0.00  -0.02   0.15  -0.06
  Met    HA    H   -0.02  -0.01  -0.01   0.00
  Met    C     C   -0.08  -0.41  -0.18  -0.02
  Met    CA    C    0.00   0.10  -0.06   0.01
  Met    N     N   -0.06  -0.20   1.57  -0.06
  Phe    H     H   -0.03  -0.12   0.10  -0.37
  Phe    HA    H   -0.06  -0.09  -0.08  -0.04
  Phe    C     C   -0.27  -0.83  -0.25  -0.10
  Phe    CA    C   -0.07  -0.23   0.06   0.01
  Phe    N     N   -0.18  -0.49   2.78  -0.46
  Pro    H     H   -0.04  -0.18   0.19  -0.12
  Pro    HA    H   -0.01   0.11  -0.03  -0.01
  Pro    C     C   -0.47  -2.84  -0.09  -0.02
  Pro    CA    C   -0.22  -2.00   0.02   0.04
  Pro    N     N   -0.18  -0.32   0.87  -0.17
  Ser    H     H    0.00  -0.03   0.16  -0.08
  Ser    HA    H   -0.01   0.02   0.00  -0.01
  Ser    C     C   -0.08  -0.40  -0.15  -0.06
  Ser    CA    C    0.00  -0.08   0.13   0.00
  Ser    N     N   -0.06  -0.03   2.55  -0.17
  Thr    H     H    0.01   0.00   0.14  -0.06
  Thr    HA    H   -0.01   0.05   0.00  -0.01
  Thr    C     C   -0.08  -0.19  -0.13  -0.05
  Thr    CA    C   -0.01  -0.04   0.12   0.00
  Thr    N     N   -0.06  -0.03   2.78  -0.12
  Trp    H     H   -0.08  -0.13   0.04  -0.62
  Trp    HA    H   -0.08  -0.10  -0.15  -0.16
  Trp    C     C   -0.26  -0.85  -0.30  -0.17
  Trp    CA    C   -0.02  -0.17   0.03  -0.08
  Trp    N     N    0.00  -0.26   3.19  -0.64
  Tyr    H     H   -0.04  -0.11   0.09  -0.42
  Tyr    HA    H   -0.05  -0.10  -0.08  -0.04
  Tyr    C     C   -0.28  -0.85  -0.24  -0.13
  Tyr    CA    C   -0.07  -0.22   0.06  -0.01
  Tyr    N     N   -0.24  -0.43   3.01  -0.52
  Val    H     H   -0.01  -0.05   0.17  -0.08
  Val    HA    H   -0.02  -0.01  -0.02  -0.01
  Val    C     C   -0.20  -0.57  -0.18  -0.03
  Val    CA    C   -0.07  -0.21  -0.02   0.01
  Val    N     N   -0.24  -0.14   4.34  -0.06
  """
  
  rcsDict = {}
  offsets = [-2,-1,1,2]
  for o in offsets:
    rcsDict[o] = {}
  
  lines = data.split('\n')
  for line in lines:
    array = line.split()
    if array:
      tlc    = array[0]
      atom   = array[1]
      values = [float(v) for v in array[3:]]
      
      for i in range(4):
        offset = offsets[i]
        if rcsDict[offset].get(tlc) is None:
          rcsDict[offset][tlc] = {}
  
        rcsDict[offset][tlc][atom] = values[i]

  return rcsDict
