/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the reusable ccl java library
 * (http://www.kclee.com/clemens/java/ccl/).
 *
 * The Initial Developer of the Original Code is
 * Chr. Clemens Lee.
 * Portions created by Chr. Clemens Lee are Copyright (C) 2002
 * Chr. Clemens Lee. All Rights Reserved.
 *
 * Contributor(s): Chr. Clemens Lee <clemens@kclee.com>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

package ccl.util.test;

import ccl.util.ClassPathUtil;
import ccl.util.Exitable;
import ccl.util.FileUtil;
import ccl.util.IniFile;
import ccl.util.Init;
import ccl.util.NullOutputStream;
import ccl.util.Test;
import ccl.util.Util;
import java.io.File;
import java.io.IOException;
import java.io.PrintStream;

/**
 * Test cases for ccl.util.Init class.
 * 
 * @version $Id: InitTest.java,v 1.24 2003/10/18 07:53:25 clemens Exp clemens $
 * @author <a href="http://www.kclee.com/clemens/">
 *         Chr. Clemens Lee</a>
 *         &lt;<a href="mailto:clemens@kclee.com">
 *         clemens@kclee.com
 *         </a>>
 */
public class InitTest extends Test implements Exitable 
{
    private Init _init = null;

    private void _testGetApplicationPath( String sRCSHeader_ ) 
        throws IOException
    {
        String sTempDir = FileUtil.createTempDir();
        String sAppDir = FileUtil.concatPath( sTempDir
                                              , "classes"
                                              + File.separator
                                              + "ccl9.26"      );
        String sTempJar = FileUtil.concatPath( sAppDir
                                               , "lib"  );
        FileUtil.md( sTempJar );

        // locate and copy ccl.jar
        Init initTemp = new Init( this, null, sRCSHeader_ );
        String sOrigJar = initTemp.getApplicationPath();
        sOrigJar = FileUtil.concatPath( sOrigJar
                                        , "lib"
                                        + File.separator
                                        + "ccl.jar"      );
        sTempJar = sTempJar + File.separator + "inittest.jar";
        FileUtil.copy( sOrigJar, sTempJar );

        // alter classpath temporarily
        String sOrigClasspath = System.getProperty( "java.class.path" );
        String sTempClasspath = sTempJar + File.pathSeparator + sOrigClasspath;
        System.setProperty( "java.class.path", sTempClasspath );

        // create new Init object
        Init init = new Init( this, null, sRCSHeader_ );
        Assert( init.getApplicationName().equals( "InitTest" )
                , "Application name should be 'Inittest' but is '" + init.getApplicationName() + "'!" );
        Assert( FileUtil.equalsPath( init.getApplicationPath()
                                     , sAppDir               )
                , "Application path should be '" + sAppDir + "' but is '" + init.getApplicationPath() + "'!" );

        System.setProperty( "java.class.path", sOrigClasspath );

        FileUtil.deleteRecursively( sTempDir );

        // create new Init object
        String sApplicationHome = ClassPathUtil.getApplicationHome( this );
        Util.debug( this, "_testGetApplicationPath(..).sApplicationHome: " + sApplicationHome );
        String sTestDir = FileUtil.concatPath( ClassPathUtil.getApplicationHome( this )
                                               , "test" );
        String sInittest2File = FileUtil.concatPath( sTestDir, "inittest2.ini" );
        String sInittest2Content = FileUtil.readFile( sInittest2File );
        init = new Init( this, null, sRCSHeader_, sInittest2Content );
        Assert( init.getKeyValue( "ConfigPath" ).equals( "/home/leecc/pp/config" )
                , "ConfigPath (should be '/home/leecc/pp/config'): '" + init.getKeyValue( "ConfigPath" ) + "'" );

        String[] asArg = new String[] { "-inifile", sInittest2File }; 
        init = new Init( this, asArg, sRCSHeader_ );
        Assert( init.getKeyValue( "ConfigPath" ).equals( "/home/leecc/pp/config" )
                , "ConfigPath (should be '/home/leecc/pp/config'): '" + init.getKeyValue( "ConfigPath" ) + "'" );
    }

    private void _testCommaInOptionDescription()
    {
        String sIniFileContent = "[Init]\n"
               + "Author=Chr. Clemens Lee\n"
               + "\n"
               + "[Help]\n"
               + "HelpUsage=\n"
               + "Options=localid,exchange\n"
               + "localid=s,m,The local id (e.g. wkn) of a stock.,,,\n"
               + "exchange=s,o,The stock exchange symbol (e.g. ETR for Xetra, NYS, etc.). By default ETR is used.\n"
               ;

        // create init object
        Init init = new Init( this
                              , new String[ 0 ]
                              , "$Header: /home/clemens/src/java/ccl/src/ccl/util/test/RCS/InitTest.java,v 1.24 2003/10/18 07:53:25 clemens Exp clemens $"
                              , sIniFileContent );
        String sHelpMessage = init.getHelpMessage();
        Util.debug( this, "_testCommaInOptionDescription().helpMessage: '" + sHelpMessage + "'" );
        bugIf( Util.isEmpty( sHelpMessage ) );
        bugIf( sHelpMessage.indexOf( "NYS" ) == -1 );
        bugIf( sHelpMessage.indexOf( ",,," ) == -1 );
    }

    private void _testHardCodedInitContent()
    {
        String sIniFileContent = "[Init]\n"
               + "Author=Chr. Clemens Lee\n"
               + "\n"
               + "[Help]\n"
               + "HelpUsage=some_parameters\n"
               + "Options=localid,exchange\n"
               + "localid=s,m,The local id (e.g. wkn) of a stock.,,,\n"
               + "exchange=s,o,The stock exchange symbol (e.g. ETR for Xetra, NYS, etc.). By default ETR is used.\n"
               ;

        // create init object
        Init init = new Init( this
                              , new String[ 0 ]
                              , "$Header: /home/clemens/src/java/ccl/src/ccl/util/test/RCS/InitTest.java,v 1.24 2003/10/18 07:53:25 clemens Exp clemens $"
                              , sIniFileContent );
        String sHelpMessage = init.getHelpMessage();
        Util.debug( this, "_testCommaInOptionDescription().helpMessage: '" + sHelpMessage + "'" );
        bugIf( sHelpMessage.indexOf( "some_parameters" ) == -1 );
    }

    public InitTest() 
    {
        super();
    }
    
    public InitTest(Test pTest_) 
    {
        super(pTest_);
    }
    
    /**
     * This is the main method where the action takes place.
     *
     * @exception   Exception   just to catch every possible bug.
     */
    protected void _doIt() 
        throws Exception
    {
        String sLocalDir = FileUtil.getClassPath(this);
        String sTestIniFile = FileUtil.concatPath(sLocalDir, "test.ini");
        
        // ist es auch wirklich weg?
        bugIf(FileUtil.delete(sTestIniFile),
              "InitTest: delete funktioniert auch nicht.");

        String sRCSHeader = "$Header: /home/clemens/src/java/ccl/src/ccl/util/test/RCS/InitTest.java,v 1.24 2003/10/18 07:53:25 clemens Exp clemens $";
        Init pInit = new Init(this, null, sRCSHeader );
        
        // Init auch erzeugt?
        bugIf(pInit == null, "InitTest");
        int version = pInit.getVersion();
        int release = pInit.getRelease();
        // versions- und releasnummer testen
        bugIf(version != 1, "version: " + version );
        bugIf(release < version, "VersionString, Version, Release: " +
              pInit.getVersionString() + ", " +
              version + ", " + release);
        pInit.setFileFullName(FileUtil.concatPath(sLocalDir, "test.proj"));
        
        FileUtil.delete( sTestIniFile );

        // test for non legal option that no exception gets thrown.
        String[] asArg = new String[ 3 ];
        asArg[ 0 ] = "-jedit";
        asArg[ 1 ] = "-inifile";
        asArg[ 2 ] = FileUtil.concatPath( sLocalDir, "test10.ini" );
        PrintStream out = System.out;
        try 
        {
            System.setOut( new PrintStream( new NullOutputStream() ) );
            pInit = new Init( this, asArg, sRCSHeader );
            System.setOut( out );
            bugIf( false );
        }
        catch( Exception pException ) 
        {
            System.setOut( out );
            Util.printlnErr( this, "_doIt().pException: " + pException );
            pException.printStackTrace();
            bugIf( true, "test10.ini failed!" );
        }

        // check ccl init object
        if ( _init != null ) 
        {
            // this can not work if started from jacob
            Assert( FileUtil.existsFile( FileUtil.concatPath( _init.getApplicationPath()
                                                              , "lib" + File.separator + "ccl.jar" ) )
                    ,   "Application path: "
                    + _init.getApplicationPath() );
            Assert( Util.endsWith( _init.getHelpBrokerURL()
                                   , File.separator
                                   + "doc"
                                   + File.separator
                                   + "ccl.hs" )
                    , "Help set: "
                    + _init.getHelpBrokerURL() );

            Assert( _init.getAuthor().equals( "Chr. Clemens Lee" ) );
        }

        // check that init object uses default content only
        // after nothing was found in ini file.
        {
            // create init object with clearcase.ini file
            // temporarily
            String sIniFileContent = FileUtil.getResourceAsString( this
                                                                   , "clearsight.ini" );
            Assert( sIniFileContent.indexOf( "\nAuthor=Chr. Clemens Lee\n" ) != -1 );

            String sIniFileName    = FileUtil.concatPath( FileUtil.getTempDir()
                                                          , "clearsight.ini" );

            FileUtil.delete   ( sIniFileName      );
            FileUtil.writeFile( sIniFileName
                                , "[Init]\n"
                                + "OldFiles=/home/clemens/bunin/clearsight/test.csml;\n"
                                + "File=/home/clemens/bunin/clearsight/pp.csml\n" );

            // create init object
            asArg = new String[] { "-inifile"
                                   , sIniFileName };
            Init init = new Init( this
                                  , asArg
                                  , sRCSHeader
                                  , null );
            // get old files should not be empty
            bugIf( Util.isEmpty( init.getKeyValue( "File" ) ) );


            asArg = new String[] { "-inifile"
                                   , sIniFileName };
            init = new Init( this
                             , asArg
                             , sRCSHeader
                             , sIniFileContent );
            // get old files should not be empty
            bugIf( Util.isEmpty( init.getKeyValue( "File" ) ) );

            // get Author should not be empty
            String sAuthor = IniFile.getKeyValueFromString( "Init"
                                                            , "Author"
                                                            , sIniFileContent );
            Assert( sAuthor.equals( "Chr. Clemens Lee" )
                    , "IniFile Author 0: " + sAuthor );

            sAuthor = IniFile.getKeyValue( null
                                           , "Init"
                                           , "Author"
                                           , sIniFileContent );
            Assert( sAuthor.equals( "Chr. Clemens Lee" )
                    , "IniFile Author 1: " + sAuthor );

            sAuthor = IniFile.getKeyValue( sIniFileName
                                           , "Init"
                                           , "Author"
                                           , sIniFileContent );
            Assert( sAuthor.equals( "Chr. Clemens Lee" )
                    , "IniFile Author 2: " + sAuthor );

            Assert( init.getAuthor().equals( "Chr. Clemens Lee" )
                    , "Author: " + init.getAuthor() );
        }

        _testGetApplicationPath( sRCSHeader );

        _testCommaInOptionDescription();

        _testHardCodedInitContent();
    }

    /**
     * The InitTest uses the ccl init object itself. The
     * init object is forwarded from outer test to the
     * next test.
     */
    public void setInit( Init init_ ) 
    {
        _init = init_;
    }

    /**
     * Start init test standalone.
     */
    public static void main(String[] args) 
    {
        Test pTest = (Test) (new InitTest());
        pTest.setVerbose( true );
        pTest.run();
        Test.printResult( pTest );

        System.exit( 0 );
    }

    /**
     * Dummy implementation for Exitable used by this test
     * itself.
     */
    public void setExit() 
    {
    }
}
