/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the reusable ccl java library
 * (http://www.kclee.com/clemens/java/ccl/).
 *
 * The Initial Developer of the Original Code is
 * Chr. Clemens Lee.
 * Portions created by Chr. Clemens Lee are Copyright (C) 2002
 * Chr. Clemens Lee. All Rights Reserved.
 *
 * Contributor(s): Chr. Clemens Lee <clemens@kclee.com>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

package ccl.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.io.IOException;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;
import java.util.zip.ZipEntry;
import java.util.zip.ZipException;
import java.util.zip.ZipFile;

/**
 * Used to load classes just to examine with the reflection api,
 * not to use them. Based on SimpleClassLoader from "Java Virtual
 * Machine" by Jon Meyer & Tom Downing.
 *
 * @version $Id: ReflectionClassLoader.java,v 1.10 2002/10/06 16:42:09 clemens Exp clemens $
 * @author <a href="http://www.kclee.com/clemens/">
 *         Chr. Clemens Lee</a>
 *         &lt;<a href="mailto:clemens@kclee.com">
 *         clemens@kclee.com
 *         </a>>
 */
public class ReflectionClassLoader extends ClassLoader 
{
    private String _sClasspath = System.getProperty("java.class.path");
    private Hashtable _htLoadedClasses = new Hashtable();
    private Hashtable _htResolvedClasses = new Hashtable();
    
    public String getAbsoluteClassFileName(String sFullClassName_) 
    {
        String sRelativeClassName = sFullClassName_.replace('.', File.separatorChar) + ".class";
        Util.debug("ReflectionClassLoader.getAbsoluteClassFileName(..).sRelativeClassName: " + sRelativeClassName);
        Util.debug("ReflectionClassLoader.getAbsoluteClassFileName(..)._sClasspath: " + _sClasspath);
        
        int pos = 0;
        int lastpos = 0;
        while(lastpos != -1) 
        {
            pos = _sClasspath.indexOf(File.pathSeparatorChar, lastpos);
            
            String sCurrentDir;
            if (pos == -1) 
            {
                sCurrentDir = _sClasspath.substring(lastpos);
                lastpos = -1;
            } 
            else 
            {
                sCurrentDir = _sClasspath.substring(lastpos, pos);
                lastpos = pos + 1;
                
                if        (lastpos >= _sClasspath.length()
                           ) 
                {
                    lastpos = -1;
                }
            }
            Util.debug("ReflectionClassLoader.getAbsoluteClassFileName(..).sCurrentDir: " + sCurrentDir);
            
            String sTestFile = FileUtil.concatPath(sCurrentDir, sRelativeClassName);
            Util.debug("ReflectionClassLoader.getAbsoluteClassFileName(..).sTestFile: " + sTestFile);
            if (FileUtil.existsFile(sTestFile)) 
            {
                return(sTestFile);
            }
        }
        
        return null;
    }
    
    public byte[] loadClassFile(String sAbsoluteClassFileName_)
        throws IOException
    {
        FileInputStream pFileInputStream = new FileInputStream
               (sAbsoluteClassFileName_);
        byte abtClass[] = new byte[pFileInputStream.available()];
        pFileInputStream.read(abtClass);
        
        return abtClass;
    }
    
    public byte[] loadClassFile(ZipFile zipOpen_, ZipEntry pZipEntry_)
        throws IOException, ZipException
    {
        Util.debug("ReflectionClassLoader.loadClassFile(..).zipOpen_: " + zipOpen_);
        Util.debug("ReflectionClassLoader.loadClassFile(..).pZipEntry_: " + pZipEntry_);
        Util.debug("ReflectionClassLoader.loadClassFile(..).pZipEntry_.getMethod(): " + pZipEntry_.getMethod());
        int compressedSize = (int) pZipEntry_.getCompressedSize();
        int uncompressedSize = (int) pZipEntry_.getSize();
        if (uncompressedSize == -1) 
        {
            throw new ZipException();
        }
        byte abtClass[] = new byte[uncompressedSize];
        Util.debug("ReflectionClassLoader.loadClassFile(..).abtClass.length: " + abtClass.length);
        InputStream pInputStream = zipOpen_.getInputStream(pZipEntry_);
        Util.debug("ReflectionClassLoader.loadClassFile(..).pInputStream: " + pInputStream);
        
        int bytesRead = 0;
        while(bytesRead < uncompressedSize) 
        {
            bytesRead += pInputStream.read(abtClass, bytesRead, uncompressedSize - bytesRead);
            Util.debug("ReflectionClassLoader.loadClassFile(..).bytesRead: " + bytesRead);
        }
        
        return abtClass;
    }
    
    public byte[] getClassFromJavaArchives(String sFullClassName_)
        throws IOException, ZipException
    {
        String sClassFileName = sFullClassName_.replace('.', '/') + ".class";
        
        // first get all archives from classpath
        Vector vClasspathElements = Util.stringToLines(_sClasspath,
                                                       File.pathSeparatorChar);
        for(Enumeration eElements = vClasspathElements.elements();
            eElements.hasMoreElements(); )
        {
            String sNextArchiv = (String) eElements.nextElement();
            Util.debug("ReflectionClassLoader.getClassFromJavaArchives(..).sNextArchiv: " + sNextArchiv);
            if ((Util.endsWith(sNextArchiv, ".zip") ||
                 Util.endsWith(sNextArchiv, ".jar")) &&
                FileUtil.existsFile(sNextArchiv))
            {
                try 
                {
                    ZipFile pZipFile = new ZipFile(sNextArchiv);
                    ZipEntry pZipEntry = pZipFile.getEntry(sClassFileName);
                    if (pZipEntry != null) 
                    {
                        byte abtClass[] = loadClassFile(pZipFile, pZipEntry);
                        pZipFile.close();
                        
                        return abtClass;
                    }
                    pZipFile.close();
                }
                catch(Exception pException) 
                {
                }
            }
        }
        
        return null;
    }
    
    public byte[] getClassFile(String sFullClassName_)
        throws IOException,
           ClassNotFoundException,
           ZipException
    {
        String sAbsoluteClassFileName = getAbsoluteClassFileName(sFullClassName_);
        Util.debug("ReflectionClassLoader.getClassFile(..).sAbsoluteClassFileName: " + sAbsoluteClassFileName);
        
        if (sAbsoluteClassFileName != null) 
        {
            return loadClassFile(sAbsoluteClassFileName);
        }
        
        byte abtClass[] = getClassFromJavaArchives(sFullClassName_);
        if (abtClass == null) 
        {
            throw new ClassNotFoundException();
        }
        
        return abtClass;
    }

    /**
     * Gets invoked from ClassLoader's loadClass(String) method.
     *
     * @throws   ClassNotFoundException   if the class was not found
     */
    protected Class loadClass(String sClassName_, boolean bResolve_)
        throws ClassNotFoundException
    {
        Util.debug("ReflectionClassLoader.loadClass(..).sClassName_: " + sClassName_);
        Util.debug("ReflectionClassLoader.loadClass(..).bResolve_: " + bResolve_);
        
        try 
        {
            if (!_htLoadedClasses.containsKey(sClassName_)) 
            {
                Class classGot = null;
                if ( sClassName_.startsWith( "java." )
                     || (sClassName_.startsWith( "javax." ) 
                         // jacob does not run with 1.0 anyway
                         && (!System.getProperty( "java.version" ).startsWith( "1.1" ))) )
                {
                    // special treatment for javax. in jdk 1.2 or higher?
                    classGot = findSystemClass(sClassName_);
                }
                else 
                {
                    byte abtClass[] = getClassFile(sClassName_);
                    classGot = defineClass(sClassName_, abtClass, 0, abtClass.length);
                }
                _htLoadedClasses.put(sClassName_, classGot);
            }
        }
        catch(ZipException pZipException) 
        {
            Util.debug("ReflectionClassLoader.loadClass(..).pZipException: " + pZipException);
            throw new ClassNotFoundException();
        }
        catch(IOException pIOException) 
        {
            Util.debug("ReflectionClassLoader.loadClass(..).pIOException: " + pIOException);
            throw new ClassNotFoundException();
        }
        catch(Error pError) 
        {
            Util.debug("ReflectionClassLoader.loadClass(..).pError: " + pError);
            throw new ClassNotFoundException();
        }
        
        /*if (bResolve_&& !_htResolvedClasses.containsKey(sClassName_)) {
          resolveClass((Class)_htLoadedClasses.get(sClassName_));
          _htResolvedClasses.put(sClassName_, "true");
          }*/
        
        return (Class) _htLoadedClasses.get(sClassName_);
    }
    
    public ReflectionClassLoader() 
    {
        super();
    }
    
    public ReflectionClassLoader(String sClasspath_) 
    {
        super();
        setClasspath(sClasspath_);
    }
    
    public void setClasspath(String sClasspath_) 
    {
        _sClasspath = new String(sClasspath_);
    }
}
