/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the reusable ccl java library
 * (http://www.kclee.com/clemens/java/ccl/).
 *
 * The Initial Developer of the Original Code is
 * Chr. Clemens Lee.
 * Portions created by Chr. Clemens Lee are Copyright (C) 2002
 * Chr. Clemens Lee. All Rights Reserved.
 *
 * Contributor(s): Chr. Clemens Lee <clemens@kclee.com>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

package ccl.swing;

import ccl.util.FileUtil;
import ccl.util.Testable;
import ccl.util.Test;
import ccl.util.Util;
import java.awt.Canvas;
import java.awt.Color;
import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Event;
import java.awt.EventQueue;
import java.awt.Font;
import java.awt.Frame;
import java.awt.Image;
import java.awt.MediaTracker;
import java.awt.Point;
import java.awt.Toolkit;
import java.awt.Window;
import java.awt.event.KeyEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.lang.reflect.InvocationTargetException;
import java.util.Date;
import java.util.Enumeration;
import java.util.MissingResourceException;
import java.util.Vector;
import javax.swing.DefaultListModel;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JFileChooser;
import javax.swing.JList;
import javax.swing.JPopupMenu;
import javax.swing.JRootPane;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.KeyStroke;
import javax.swing.MenuElement;
import javax.swing.SwingUtilities;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumnModel;
import javax.swing.text.Keymap;
import javax.swing.text.Segment;
import javax.swing.tree.DefaultMutableTreeNode;

/**
 * Utility class for gui/swing stuff. Instead of creating
 * and showing special dialogs yourself, this class provides
 * some shortcut methods for this task.
 *
 * @version  $Id: SwingUtil.java,v 1.21 2003/05/01 16:44:28 clemens Exp clemens $
 * @author <a href="http://www.kclee.com/clemens/">
 *         Chr. Clemens Lee</a>
 *         &lt;<a href="mailto:clemens@kclee.com">
 *         clemens@kclee.com
 *         </a>>
 */
public class SwingUtil 
{
    /** Constant for yes, no, and cancel. */
    public static final int YES = 0;
    /** Constant for yes, no, and cancel. */
    public static final int NO = 1;
    /** Constant for yes, no, and cancel. */
    public static final int CANCEL = 2;

    private static int _yesNoOrCancel( Frame frmParent_,
                                       YesNoCancelDialog
                                       pYesNoCancelDialog_ )
    {
        pYesNoCancelDialog_.dispose();
        frmParent_.requestFocus();
        if (pYesNoCancelDialog_.isCancel()) 
        {
            return CANCEL;
        }
        if (pYesNoCancelDialog_.isYes()) 
        {
            return YES;
        }
        
        return NO;
    }

    /**
     * @return   null if canceled.
     */
    public static String inputListCancel( Frame frmParent_,
                                          String sPrint_, 
                                          Vector vsItems_ )
    {
        if (Test.isTest()) 
        {
            return( (String) Test.getValue() );
        }

        ListCancelSelector dlgInput =
               new ListCancelSelector(frmParent_, sPrint_, vsItems_);
        dlgInput.show();
        String sRetVal = dlgInput.getValue();
        dlgInput.dispose();

        return sRetVal;
    }

    /**
     * @return   null for cancel, empty vector if nothing was
     *           selected, still the user pressed ok.
     */
    public static Vector inputDoubleListCancel(Frame frmParent_,
                                               String sPrint_,
                                               Vector vsFirst_,
                                               Vector vsSecond_,
                                               boolean bMultiSelect_)
    {
        Vector vRetVal = null;
        try 
        {
            DoubleListCancelSelector dlgInput =
                   new DoubleListCancelSelector
                   ( frmParent_, sPrint_,
                     vsFirst_, vsSecond_,
                     bMultiSelect_);
            dlgInput.show();
            if ( dlgInput.isOK() ) 
            {
                vRetVal = dlgInput.getValues();
            }
            dlgInput.dispose();
        }
        catch( Exception pException ) 
        {
            Util.println( "SwingUtil.inputDoubleListCancel(..).pException: " + 
                          pException );
            pException.printStackTrace();
        }
        Util.debug( "SwingUtil.inputDoubleListCancel(..).END" );
            
        return vRetVal;
    }

    public static Vector inputDoubleListCancel(Frame frmParent_,
                                               String sPrint_,
                                               Vector vsFirst_,
                                               Vector vsSecond_,
                                               boolean bMultiSelect_,
                                               Color clrListBackground_,
                                               Color clrListForeground_)
    {
        Vector vRetVal = null;
        try 
        {
            DoubleListCancelSelector dlgInput =
                   new DoubleListCancelSelector
                   ( frmParent_, sPrint_,
                     vsFirst_, vsSecond_,
                     bMultiSelect_);
            dlgInput.setListBackground( clrListBackground_ );
            dlgInput.setListForeground( clrListForeground_ );
            dlgInput.show();
            vRetVal = dlgInput.getValues();
            dlgInput.dispose();
        }
        catch( Exception pException ) 
        {
            Util.println( "SwingUtil.inputDoubleListCancel(..).pException: " + 
                          pException );
            pException.printStackTrace();
        }
        Util.debug( "SwingUtil.inputDoubleListCancel(..).END" );
            
        return vRetVal;
    }
    
    public static Image getResourcesImage( String sResourceImageName_ )
        throws MissingResourceException
    {
        ImageIcon pImageIcon = new ImageIcon
               ( SwingUtil.class.getResource
                 ( sResourceImageName_ ) );

        return pImageIcon.getImage();
    }

    /**
     * Requests focus for frmParent_ afterwards.
     */
    public static void showMessage(Frame frmParent_, String sMessages_) 
    {
        MessageBox dlgMessage = new MessageBox(frmParent_, sMessages_);
        _oTestValue = dlgMessage;
        dlgMessage.show();
        frmParent_.requestFocus();
    }

    /**
     * Sets the initial focus of a component when a window pops up.
     * component.requestFocus() won't work in a constructor, since
     * the window is not visible yet.
     */
    public static void setInitialFocus( Window pWindow_
                                        , final Component pComponent_ ) 
    {
        pWindow_.addWindowListener( new WindowAdapter() 
            {
                public void windowOpened( WindowEvent pWindowEvent_ ) 
                {
                    pComponent_.requestFocus();
                }
            } );
    }

    /**
     * This file choose selects peer class under windows,
     * swing file chooser on other systems.
     * Swing file chooser is pretty broken on windows,
     * while the native implementation is not bad.
     * On unix it's vice versa.
     *
     * @param   asFileSuffixes_   for example: { "jiml", "nr" }
     */
    public static String getFileName( Frame frmParent_, 
                                      String sTitle_,
                                      String[] asFileSuffixes_,
                                      String sDirectory_ )
    {
        JFileChooser pFileChooser = new JFileChooser
               ( sDirectory_ );
        pFileChooser.setDialogTitle( sTitle_ );

        Util.panicIf( asFileSuffixes_ == null ||
                      asFileSuffixes_.length < 1 );

        ExampleFileFilter pExampleFileFilter = new ExampleFileFilter
               ( asFileSuffixes_, "" );

        pFileChooser.addChoosableFileFilter( pExampleFileFilter );
        pFileChooser.setFileFilter( pExampleFileFilter );
        pFileChooser.setFileSelectionMode( JFileChooser.FILES_ONLY );

        int retVal = pFileChooser.showDialog( frmParent_, null );
        
        if ( retVal != JFileChooser.APPROVE_OPTION ) 
        {
            return null;
        }
        
        String sRetVal = pFileChooser.getSelectedFile().getAbsolutePath();
        
        return sRetVal;
    }

    /**
     * (This file choose selects peer class under windows,
     * swing file chooser on other systems. -- not true!)
     * Swing file chooser is pretty broken on windows,
     * while the native implementation is not bad.
     * On unix it's vice versa.
     *
     * @param   sFileName_   for example: *.proj
     */
    public static String getFileName( Frame frmParent_, 
                                      String sTitle_,
                                      String sFileName_,
                                      String sDirectory_ )
    {
        /*if ( Util.isOSWindows() ) {
            return AWTUtil.getFileName( frmParent_,
                                        sTitle_,
                                        sFileName_,
                                        sDirectory_ );
                                        }*/

        JFileChooser pFileChooser = new JFileChooser
               ( sDirectory_ );
        pFileChooser.setDialogTitle( sTitle_ );

        Util.panicIf( sFileName_.indexOf( '.' ) == -1 );

        String sFileExtension = sFileName_.substring
               ( sFileName_.lastIndexOf( '.' ) + 1 );
        ExampleFileFilter pExampleFileFilter = new ExampleFileFilter
               ( new String[] { sFileExtension }, "" );

        pFileChooser.addChoosableFileFilter( pExampleFileFilter );
        pFileChooser.setFileFilter( pExampleFileFilter );
        pFileChooser.setFileSelectionMode( JFileChooser.FILES_ONLY );

        int retVal = pFileChooser.showDialog( frmParent_, null );
        
        if ( retVal != JFileChooser.APPROVE_OPTION ) 
        {
            return null;
        }
        
        String sRetVal = pFileChooser.getSelectedFile().getAbsolutePath();
        
        return sRetVal;
    }

    /**
     * @param   sFileName_   for example: *.proj
     */
    public static String getFileName( Frame frmParent_, 
                                      String sTitle_,
                                      String sFileName_,
                                      String sDirectory_,
                                      String sOKButtonText_ )
    {
        /*if ( Util.isOSWindows() ) {
            return AWTUtil.getFileName( frmParent_,
                                        sTitle_,
                                        sFileName_,
                                        sDirectory_ );
                                        }*/

        JFileChooser pFileChooser = new JFileChooser
               ( sDirectory_ );
        pFileChooser.setDialogTitle( sTitle_ );
        pFileChooser.setApproveButtonText( sOKButtonText_ );

        Util.panicIf( sFileName_.indexOf( '.' ) == -1
                      , "sFileName_ was empty in method "
                        + "ccl.swing.Util.getFileName(..)." );

        String sFileExtension = sFileName_.substring
               ( sFileName_.lastIndexOf( '.' ) + 1 );
        ExampleFileFilter pExampleFileFilter = new ExampleFileFilter
               ( new String[] { sFileExtension }, "" );

        pFileChooser.addChoosableFileFilter( pExampleFileFilter );
        pFileChooser.setFileFilter( pExampleFileFilter );
        pFileChooser.setFileSelectionMode( JFileChooser.FILES_ONLY );

        int retVal = pFileChooser.showDialog( frmParent_, null );
        
        if ( retVal != JFileChooser.APPROVE_OPTION ) 
        {
            return null;
        }
        
        String sRetVal = pFileChooser.getSelectedFile().getAbsolutePath();
        
        return sRetVal;
    }

    /**
     * This file choose selects peer class under windows,
     * swing file chooser on other systems.
     * Swing file chooser is pretty broken on windows,
     * while the native implementation is not bad.
     * On unix it's vice versa.
     *
     * @return   null on cancel
     */
    public static String getDirName( Frame frmParent_, 
                                     String sTitle_,
                                     String sDirectory_ )
    {
        JFileChooser pFileChooser = new JFileChooser
               ( sDirectory_ );
        pFileChooser.setDialogTitle( sTitle_ );

        pFileChooser.setFileSelectionMode( JFileChooser.DIRECTORIES_ONLY );

        int retVal = pFileChooser.showDialog( frmParent_, null );
        
        if ( retVal != JFileChooser.APPROVE_OPTION ) 
        {
            return null;
        }
        
        String sRetVal = pFileChooser.getSelectedFile().getAbsolutePath();
        
        return sRetVal;
    }

    /**
     * @return   null on cancel
     */
    public static String getFileOrDirName( Frame frmParent_, 
                                           String sTitle_,
                                           String sDirectory_ )
    {
        JFileChooser pFileChooser = new JFileChooser
               ( sDirectory_ );
        pFileChooser.setDialogTitle( sTitle_ );

        pFileChooser.setFileSelectionMode( JFileChooser.FILES_AND_DIRECTORIES );

        int retVal = pFileChooser.showDialog( frmParent_, null );
        
        if ( retVal != JFileChooser.APPROVE_OPTION ) 
        {
            return null;
        }
        
        String sRetVal = pFileChooser.getSelectedFile().getAbsolutePath();
        
        return sRetVal;
    }

    /**
     * @return   null on cancel
     */
    public static String getFileName( Frame frmParent_, 
                                      String sTitle_,
                                      String sStartDirectory_ )
    {
        JFileChooser pFileChooser = new JFileChooser
               ( sStartDirectory_ );
        pFileChooser.setDialogTitle( sTitle_ );

        pFileChooser.setFileSelectionMode( JFileChooser.FILES_ONLY );

        int retVal = pFileChooser.showDialog( frmParent_, null );
        
        if ( retVal != JFileChooser.APPROVE_OPTION ) 
        {
            return null;
        }
        
        String sRetVal = pFileChooser.getSelectedFile().getAbsolutePath();
        
        return sRetVal;
    }

    /**
     * Opens a dialog and returns either true or false.
     */
    public static boolean isOKOrCancel( Frame frmParent_,
                                        String sMessage_ ) 
    {
        OKCancelDialog dlgOKCancel = new OKCancelDialog
               ( frmParent_, sMessage_ );
        dlgOKCancel.setVisible( true );
        boolean bRetVal = dlgOKCancel.isOK();
        dlgOKCancel.dispose();
        frmParent_.requestFocus();
        
        return bRetVal;
    }

    /** 
     * A swing workaround. Buttons react to enter. We need it for
     * default button.
     */
    public static void unregisterEnterAction( JButton pButton_ ) 
    {
        KeyStroke enterPressedKeyStroke = 
               KeyStroke.getKeyStroke( '\n', 0, false );
        KeyStroke enterReleasedKeyStroke = 
               KeyStroke.getKeyStroke( '\n', 0, true );
        pButton_.unregisterKeyboardAction( enterPressedKeyStroke );
        pButton_.unregisterKeyboardAction( enterReleasedKeyStroke );
    }

    /**
     * From sun's JTextField.java: [ccl 1998-1-8]
     * Need this for default button in dialogs.
     *
     * For compatibility with java.awt.TextField, the VK_ENTER
     * key fires the ActionEvent to the registered 
     * ActionListeners. However, awt didn't have default buttons
     * like swing does. If a text field has focus and the
     * VK_ENTER key is pressed, it will fire the fields
     * ActionEvent rather than activate the default button. To
     * disable the compatibility with awt for text fields, the
     * following code fragment will remove the binding of
     * VK_ENTER from the default keymap used by all JTextFields
     * if that is desired.
     */
    public static void removeEnterBinding( JTextField pTextField_ ) 
    {
        KeyStroke ksEnter = KeyStroke.getKeyStroke( KeyEvent.VK_ENTER, 0 );
        Keymap pKeymap = pTextField_.getKeymap();
        pKeymap.removeKeyStrokeBinding( ksEnter );
        KeyStroke ksAltC = KeyStroke.getKeyStroke( KeyEvent.VK_C, 
                                                   Event.META_MASK );
        pKeymap.removeKeyStrokeBinding( ksAltC );
    }

    private static CCLBorder       _pCCLBorder       = null;
    private static CCLButtonBorder _pCCLButtonBorder = null;

    /**
     * Instead of creating your own new CCLBorder object each
     * time you want to use a CCLBorder, you can simply reuse
     * the object returned by this method.
     */
    public static CCLBorder createCCLBorder() 
    {
        if ( _pCCLBorder == null) 
        {
            _pCCLBorder = new CCLBorder();
        }

        return _pCCLBorder;
    }

    /**
     * Instead of creating your own new CCLButtonBorder object each
     * time you want to use a CCLButtonBorder, you can simply reuse
     * the object returned by this method.
     */
    public static CCLButtonBorder createCCLButtonBorder() 
    {
        if ( _pCCLButtonBorder == null) 
        {
            _pCCLButtonBorder = new CCLButtonBorder();
        }

        return _pCCLButtonBorder;
    }

    /**
     * Uses ccl button border for the given table's headers.
     * 
     * @param table    the table to change the header borders on.
     *                 The table must have its columns already set.
     */
    static public void setCCLButtonBorderOnTableHeader( JTable table, Font font ) 
    {
        TableCellRenderer renderer = new CCLButtonBorderTableCellRenderer( font );
        TableColumnModel tableModel = table.getColumnModel();
        for( int index  = 0; index < tableModel.getColumnCount(); index ++ ) 
        {
            tableModel.getColumn( index ).setHeaderRenderer( renderer );
        }
    }

    /**
     * If not running in an AWT thread, then invoke later.
     * Otherwise just execute the code.
     */
    public static void invokeLaterIfNecessary( Runnable pRunnable_ ) 
    {
        Util.panicIf( pRunnable_ == null );

        if ( Thread.currentThread().getName().startsWith( "AWT-" ) ) 
        {
            pRunnable_.run();
        }
        else 
        {
            SwingUtilities.invokeLater( pRunnable_ );
        }
    }

    /**
     * If not running in an AWT thread, then invoke later.
     * Otherwise just execute the code.
     */
    public static void invokeAndWaitIfNecessary( Runnable pRunnable_ ) 
        throws InterruptedException
               , InvocationTargetException
    {
        Util.panicIf( pRunnable_ == null );

        if ( EventQueue.isDispatchThread() )
        {
            pRunnable_.run();
        }
        else 
        {
            SwingUtilities.invokeAndWait( pRunnable_ );
        }
    }

    /**
     * @return   null if canceled.
     */
    public static String inputCancel( Frame frmParent_, 
                                      String sPrint_, 
                                      String sInit_ ) 
    {
        if (Test.isTest()) 
        {
            return((String) Test.getValue());
        }

        String sRetVal = null;
        InputCancelDialog dlgInput = new InputCancelDialog( frmParent_
                                                            , sPrint_
                                                            , sInit_ );
        
        _oTestValue = dlgInput;
        dlgInput.show();
        if (dlgInput.isOK())
        {
            sRetVal = dlgInput.getValue();
        }
        dlgInput.dispose();
        
        return sRetVal;
    }

    /**
     * @return   null if canceled.
     */
    public static String inputCancel( Frame frmParent_, 
                                      String sPrint_ )
    {
        return inputCancel( frmParent_, sPrint_, "" );
    }

    /**
     * @param sPath_        String representation of the new child node.
     *                      For example "Notes/Ideas/Crazy". This will
     *                      Search for node Notes->Ideas and inserts the
     *                      new child assuming its string representation
     *                      is "Crazy".
     * @param cDelimiter_   '/' for example.
     */
    public static void insertNodeToTree( DefaultMutableTreeNode tnRoot_,
                                  DefaultMutableTreeNode tnChild_,
                                  String sPath_, 
                                  char cDelimiter_ )
    {
        Util.panicIf( Util.isEmpty( sPath_ ) );

        Vector vChild = Util.stringToLines( sPath_, cDelimiter_ );
        Util.panicIf( vChild == null || 
                      vChild.size() < 2 );

        String sChild = (String) vChild.elementAt( 1 );
        int childrenCount = tnRoot_.getChildCount();
        for( int child = 0; child < childrenCount; child++ ) 
        {
            DefaultMutableTreeNode tnNextChild = 
                   (DefaultMutableTreeNode) tnRoot_.getChildAt( child );
            if ( sChild.equals( tnNextChild.toString() ) ) 
            {
                insertNodeToTree( tnNextChild
                                  , tnChild_
                                  , sPath_.substring
                                        ( sPath_.indexOf( cDelimiter_ )
                                          + 1 )
                                  , cDelimiter_ );
                
                return;
            }
            else if ( sChild.compareTo( tnNextChild.
                                        toString() ) < 0 ) 
            {
                tnRoot_.insert( tnChild_, child );
                
                return;
            }
        }
        tnRoot_.insert( tnChild_, childrenCount );
    }

    public static Vector toVector( DefaultListModel pDefaultListModel_ ) 
    {
        Vector pVector = new Vector();
        Enumeration pEnumeration = pDefaultListModel_.elements();
        while( pEnumeration.hasMoreElements() ) 
        {
            pVector.addElement( pEnumeration.nextElement() );
        }

        return pVector;
    }

    /**
     * For example: getFirstComponent( pFrame, "name_text" );
     * or:          getFirstComponent( pFrame, "javax.swing.JTextField" );
     */
    public static Component getFirstComponent( Container pContainer_
                                               , String sName_ ) 
    {
        return getComponent( pContainer_, sName_, 0 );
    }

    /**
     * Simliar to a stack trace a string containing the tree of awt components
     * in the given container.
     */
    public static String getContainerDump( Container container )
    {
        return getContainerDump( container, 0 );
    }


    /**
     * Simliar to a stack trace a string containing the tree of awt components
     * in the given container.
     */
    private static String getContainerDump( Container container, int indent )
    {
        String sRetVal = Util.getSpaces( indent ) + container.toString();
        Component[] acmpChildren = container.getComponents();
        for( int child = 0; child < acmpChildren.length; child++ )
        {
            if ( acmpChildren[ child ] instanceof Container )
            {
                sRetVal += "\n" + getContainerDump( (Container)acmpChildren[ child ]
                                                    , indent + 2 );
            }
            else
            {
                sRetVal += "\n" + Util.getSpaces( indent + 2 ) + acmpChildren[ child ].toString();
            }
        }

        return sRetVal;
    }

    /**
     * For example: getComponent( pFrame, "name_text", 0 );
     * or:          getComponent( pFrame, "javax.swing.JTextField", 2 );
     *
     * @param n_   >= 0; the n-1st component of this type will be returned.
     */
    public static Component getComponent( Container pContainer_
                                          , String sName_
                                          , int n_ ) 
    {
        Util.panicIf ( n_ < 0 );

        int found = 0;

        Component[] acmpChildren = pContainer_.getComponents();
        if ( acmpChildren.length == 0 ) 
        {
            return null;
        }
        for( int component = 0; component < acmpChildren.length; component++ ) 
        {
            // has component requested name
            String sNextName = acmpChildren[ component ].getName();
            Util.debug( "SwingUtil.getFirstComponent(..).sNextName: " +
                        sNextName );
            if ( sNextName != null && sNextName.equals( sName_ ) ) 
            {
                if ( found == n_ ) 
                {
                    return acmpChildren[ component ];
                }
                found++;
            } 
            else 
            {
                // has component requested class name?
                sNextName = acmpChildren[ component ].getClass().getName();
                Util.debug( "SwingUtil.getFirstComponent(..).sNextName: " +
                            sNextName );
                if ( sNextName != null && sNextName.equals( sName_ ) ) 
                {
                    if ( found == n_ ) 
                    {
                        return acmpChildren[ component ];
                    }
                    found++;
                } 
                else 
                {
                    // has component requested 'toString()' result?
                    sNextName = acmpChildren[ component ].toString();
                    Util.debug( "SwingUtil.getFirstComponent(..).sNextName: " +
                                sNextName );
                    if ( sNextName != null && sNextName.startsWith( sName_ ) ) 
                    {
                        if ( found == n_ ) 
                        {
                            return acmpChildren[ component ];
                        }
                        found++;
                    }
                }
            }
            if ( acmpChildren[ component ] instanceof Container ) 
            {
                Component cmpRetVal = getComponent
                       ( (Container) acmpChildren[ component ], sName_,
                         n_ - found );
                if ( cmpRetVal != null ) 
                {
                    return cmpRetVal;
                }
            }
        }
        
        return null;
    }

    /**
     * @param n_   >= 0; the n-1st component of this type will be returned.
     */
    public static Component getComponent( Container pContainer_, 
                                          Testable pTestable_, 
                                          int n_ ) 
    {
        Util.panicIf ( pTestable_ == null );
        Util.panicIf ( n_ < 0 );

        int found = 0;

        Component[] acmpChildren = pContainer_.getComponents();
        if ( acmpChildren.length == 0 ) 
        {
            return null;
        }
        for( int component = 0; component < acmpChildren.length; component++ ) 
        {
            if ( pTestable_.test( acmpChildren[ component ] ) ) 
            {
                if ( found == n_ ) 
                {
                    return acmpChildren[ component ];
                }
                found++;
            }
            if ( acmpChildren[ component ] instanceof Container ) 
            {
                Component cmpRetVal = getComponent
                       ( (Container) acmpChildren[ component ], 
                         pTestable_,
                         n_ - found );
                if ( cmpRetVal != null ) 
                {
                    return cmpRetVal;
                }
            }
        }

        return null;
    }

    /**
     * Might return earlier than the action has finished.
     */
    public static void pressOKButton( Window pWindow_ ) 
    {
        Testable pTestable = new Testable() 
            {
                public boolean test( Object oComponent_ ) 
                {
                    if ( oComponent_ instanceof JButton &&
                         ((JButton) oComponent_).getText().equals( "OK" ) )
                    {
                        return true;
                    }
                    
                    return false;
                }
            };
        final JButton btnOK = (JButton) SwingUtil.getComponent
               ( pWindow_,
                 pTestable,
                 0 );
        if ( btnOK != null ) 
        {
            invokeLaterIfNecessary( new Runnable() 
                {
                    public void run() 
                    {
                        btnOK.doClick();
                    }
                } );
        }
    }

    public static MenuElement getMenuElement( MenuElement pMenuElement_,
                                              String[] asMenuName_ ) 
    {
        Util.debug( "SwingUtil.getMenuElement(..).pMenuElement_: " +
                    pMenuElement_ );
        Util.panicIf( pMenuElement_ == null );
        Util.panicIf( asMenuName_ == null );
        Util.panicIf( asMenuName_.length < 1 );

        MenuElement[] apMenuElement = pMenuElement_.getSubElements();
        if ( apMenuElement.length == 1 &&
             apMenuElement[ 0 ] instanceof JPopupMenu )
        {
            apMenuElement = apMenuElement[ 0 ].getSubElements();
        }

        for( int menu = 0; menu < apMenuElement.length; menu++ ) 
        {
            MenuElement meChild = apMenuElement[ menu ];
            Util.debug( "SwingUtil.getMenuElement(..).meChild: " + meChild );
            String sName = ((JComponent) meChild).getName();
            Util.debug( "SwingUtil.getMenuElement(..).name: " + sName );
            if ( sName != null && sName.equals( asMenuName_[ 0 ] ) )
            {
                if ( asMenuName_.length == 1 ) 
                {
                    return apMenuElement[ menu ];
                }
                else 
                {
                    String[] asNewMenuName = new String[ asMenuName_.length 
                                                         - 1 ];
                    for( int name = 1; name < asMenuName_.length; name++ ) 
                    {
                        asNewMenuName[ name - 1 ] = asMenuName_[ name ];
                    }
                    
                    return getMenuElement( apMenuElement[ menu ],
                                           asNewMenuName );
                }
            }            
        }
        
        return null;
    }

    /**
     * Send characters interactively to components, 
     * espacially text components.
     */
    public static void dispatchKeyEvent( final Component pComponent_, 
                                         final char c_ ) 
    {
        invokeLaterIfNecessary( new Runnable() 
            {
                public void run() 
                {
                    KeyEvent pKeyEvent = new KeyEvent( pComponent_,
                                                       KeyEvent.KEY_TYPED,
                                                       new Date().getTime(),
                                                       0,
                                                       KeyEvent.VK_UNDEFINED,
                                                       c_ );
                    pComponent_.dispatchEvent( pKeyEvent );
                }
            } );
    }

    /**
     * Send pKeyEvent_ to component, 
     * espacially text component.
     */
    public static void dispatchKeyEvent( final Component pComponent_,
                                         final int virtualKey_ )
    {
        invokeLaterIfNecessary( new Runnable() 
            {
                public void run() 
                {
                    KeyEvent pKeyEvent = new KeyEvent( pComponent_,
                                                       KeyEvent.KEY_PRESSED,
                                                       new Date().getTime(),
                                                       0,
                                                       virtualKey_ );
                    pComponent_.dispatchEvent( pKeyEvent );
                    pKeyEvent = new KeyEvent( pComponent_,
                                              KeyEvent.KEY_RELEASED,
                                              new Date().getTime(),
                                              0,
                                              virtualKey_);
                    pComponent_.dispatchEvent( pKeyEvent );
                }
            } );
    }

    /**
     * Send enter key event to component, 
     * espacially text component.
     */
    public static void dispatchEnterKeyEvent( final Component pComponent_ ) 
    {
        dispatchKeyEvent( pComponent_, KeyEvent.VK_ENTER );
    }

    /**
     * Using this method you don't need to create an inner class
     * yourself to invoke it later in awt thread.
     */
    public static void doClick( final JButton btnClick_ ) 
    {
        invokeLaterIfNecessary( new Runnable() {
            public void run() 
            {
                btnClick_.doClick();
            }
        } );
    }

    /**
     * Like String.lastIndexOf() but working on Segment instead
     * of String.
     */
    public static int lastIndexOf( Segment pSegment_, int c_ ) 
    {
        return lastIndexOf( pSegment_, c_, pSegment_.count - 1 );
    }

    /**
     * Like String.lastIndexOf() but working on Segment instead
     * of String.
     */
    public static int lastIndexOf( Segment pSegment_,
                                   int c_, int fromIndex_ )
    {
        int min = pSegment_.offset;
        
        for( int i = pSegment_.offset 
                     + ((fromIndex_ >= pSegment_.count) 
                        ? pSegment_.count - 1 
                        : fromIndex_)
             ; i >= min 
             ; i-- )
        {
            if ( pSegment_.array[i] == c_ ) 
            {
                return i - pSegment_.offset;
            }
        }

        return -1;
    }

    public static void showAboutDialog( MainJFrame pMainFrame_ ) 
    {
        AboutDialog dlgAbout = new AboutDialog
               ( pMainFrame_ );
        pMainFrame_.requestFocus();
        dlgAbout.dispose();
    }

    /**
     * @param pList_   expects list with DefaultListModel.
     */
    public static void setList( JList pList_, Vector vValues_ ) 
    {
        DefaultListModel dlmList = (DefaultListModel) pList_.getModel();
        pList_.clearSelection();
        dlmList.removeAllElements();
        Enumeration eValues = vValues_.elements();
        while( eValues.hasMoreElements() ) 
        {
            dlmList.addElement( eValues.nextElement() );
        }
    }

    /**
     * @return AWTUtil.YES, AWTUtil.NO or AWTUtil.CANCEL
     *
     * @deprecated   Use ccl.swing.SwingUtil instead.
     * @see YesNoCancelDialog
     */
    public static int wannaSave( Frame frmParent_ ) 
    {
        YesNoCancelDialog pYesNoCancelDialog =
               new YesNoCancelDialog( frmParent_ );
        
        return _yesNoOrCancel( frmParent_, pYesNoCancelDialog );
    }

    /* -------------------------------------------------------
     * --- former awt util routines -----------------------------
     * -----------------------------------------------------*/

    public static Dimension getScreenSize() 
    {
        Toolkit pToolkit = Toolkit.getDefaultToolkit();
        Dimension pDimension = pToolkit.getScreenSize();
        
        return( pDimension );
    }

    public static void maximizeWindow( Window pWindow_ ) 
    {
        pWindow_.setLocation( 0, 0 );
        pWindow_.setSize( getScreenSize() );
    }

    public static void maximizeWindow( Window window, float percentage ) 
    {
        Dimension dimScreen = getScreenSize();
        Dimension dimWindow = new Dimension
               ( (int) ((float) dimScreen.width * percentage)
                 , (int) ((float) dimScreen.height * percentage) );
        Point ptLocation = new Point
               ( (dimScreen.width - dimWindow.width) / 2
                 , (dimScreen.height - dimWindow.height) / 2 );

        window.setLocation( ptLocation.x, ptLocation.y );
        window.setSize( dimWindow );
    }

    public static void centerComponent( Component cmpObject_
                                        , Component cmpParent_
                                        , int heightFraction_ )
    {
        Util.panicIf(cmpObject_ == null ||
                     cmpParent_ == null ||
                     heightFraction_ == 0);
        
        if ( !cmpParent_.isVisible() )
        {
            centerComponent(cmpObject_);
            return;
        }
        
        Dimension dimObject = cmpObject_.getSize();
        Dimension dimParent = cmpParent_.getSize();
        Point ptLocation = cmpParent_.getLocation();

        int posX = (dimParent.width - dimObject.width)   / 2;
        int posY = (dimParent.height - dimObject.height) / heightFraction_;
        
        if (posX < 0 || posY < 0) 
        {
            centerComponent(cmpObject_);
            return;
        }
        
        cmpObject_.setLocation(ptLocation.x + posX,
                               ptLocation.y + posY);
    }

    public static void centerComponent( Component cmpObject ) 
    {
        Dimension dimObject = cmpObject.getSize();
        Dimension dimScreen = getScreenSize();
        Util.debug( "ccl.swing.SwingUtil"
                    + ".centerComponent(..).dimScreen: " 
                    + dimScreen );
        int posX;
        int posY;
        
        posX = (dimScreen.width - dimObject.width) / 2;
        if (posX < 0) 
        {
            posX = 0;
        }
        posY = (dimScreen.height - dimObject.height) / 2;
        if (posY < 0) 
        {
            posY = 0;
        }
        Util.debug( "ccl.swing.SwingUtil.centerComponent(..).posX: " + posX );
        Util.debug( "ccl.swing.SwingUtil.centerComponent(..).posY: " + posY );
        
        cmpObject.setLocation(posX, posY);
    }

    public static Image loadImage( String sImageFileName_ ) 
    {
        Util.panicIf( Util.isEmpty( sImageFileName_ ) );

        if ( !FileUtil.existsFile( sImageFileName_ ) ) 
        {
            return null;
        }

        Component cmpDummy = new Canvas();
        MediaTracker pMediaTracker = new MediaTracker( cmpDummy );
        Toolkit pToolkit = cmpDummy.getToolkit();
        Image imgRetVal = null;
        try 
        {
            imgRetVal = pToolkit.getImage( sImageFileName_ );

            pMediaTracker.addImage( imgRetVal, 0 );
            pMediaTracker.waitForAll();
        }
        catch( Exception pException ) 
        {
            return null;
        }

        if ( pMediaTracker.isErrorAny() ) 
        {
            imgRetVal = null;
        }

        return imgRetVal;
    }

    static private Frame _frmMain = null;

    /**
     * @deprecated   Testing should not rely on this hack.
     */
    public static void setMainFrame(Frame pFrame_) 
    {
        _frmMain = pFrame_;
    }

    static private Object _oTestValue = null;
    
    /**
     * @deprecated   Test hack.
     */
    public static void getValue(Test pTest_) 
    {
        pTest_.setValue(_oTestValue);
    }

    public static Dimension max( Dimension dimA_, Dimension dimB_ ) 
    {
        if ( dimA_ == null && dimB_ == null ) 
        {
            return new Dimension( 0, 0 );
        }

        if ( dimA_ == null ) 
        {
            return new Dimension( dimB_ );
        }
        if ( dimB_ == null ) 
        {
            return new Dimension( dimA_ );
        }

        Dimension dimRetVal = new Dimension( dimA_ );
        dimRetVal.width = Math.max( dimRetVal.width, dimB_.width );
        dimB_.height = Math.max( dimRetVal.height, dimB_.height );

        return dimRetVal;
    }

    public static void setDefaultButton( Component pComponent_,
                                         JButton pButton_ )
    {
        JRootPane pRootPane = SwingUtilities.getRootPane( pComponent_ );
        pRootPane.setDefaultButton( pButton_ );
    }

    public static boolean isFileSavable( Frame frmParent_,
                                         String sSaveFile_ )
    {
        boolean bSavable = false;

        if ( Util.isEmpty( sSaveFile_ ) ) 
        {
            return bSavable;
        }
        
        if ( FileUtil.existsDir( sSaveFile_ ) ) 
        {
            SwingUtil.showMessage( frmParent_,
                                   "File\n\n\"" +
                                   sSaveFile_ +
                                   "\"\n\nexists already as a directory!\n" +
                                   "File saving aborted!" );
            
            return bSavable;
        }

        
        bSavable  = !FileUtil.existsFile( sSaveFile_ );
        if ( !bSavable ) 
        {
            bSavable = SwingUtil.isOKOrCancel
                   ( frmParent_,
                     "File\n\n\"" +
                     sSaveFile_ +
                     "\"\n\nexists already.\n" +
                     "Is it OK to overwrite it?" );
        }

        return bSavable;
    }
}
