mod edge;
mod gcs;
mod server_config;
mod socket;
mod tls;
mod utils;

pub use server_config::Protocol;
pub use server_config::ServerConfig;

use crate::build;
use crate::settings::global_user::GlobalUser;
use crate::settings::toml::{DeployConfig, Target};
use crate::terminal::{message, styles};

/// `wrangler dev` starts a server on a dev machine that routes incoming HTTP requests
/// to a Cloudflare Workers runtime and returns HTTP responses
pub fn dev(
    target: Target,
    deploy_config: DeployConfig,
    user: Option<GlobalUser>,
    server_config: ServerConfig,
    local_protocol: Protocol,
    upstream_protocol: Protocol,
    verbose: bool,
) -> Result<(), failure::Error> {
    // before serving requests we must first build the Worker
    build(&target)?;

    let host_str = styles::highlight("--host");
    let local_str = styles::highlight("--local-protocol");
    let upstream_str = styles::highlight("--upstream-protocol");

    if server_config.host.is_https() != upstream_protocol.is_https() {
        failure::bail!(format!(
            "Protocol mismatch: protocol in {} and protocol in {} must match",
            host_str, upstream_str
        ))
    } else if local_protocol.is_https() && upstream_protocol.is_http() {
        failure::bail!("{} cannot be https if {} is http", local_str, upstream_str)
    }

    if let Some(user) = user {
        if server_config.host.is_default() {
            // Authenticated and no host provided, run on edge with user's zone
            return edge::dev(
                target,
                user,
                server_config,
                deploy_config,
                local_protocol,
                upstream_protocol,
                verbose,
            );
        }

        // If user is authenticated but host is provided, use gcs with given host
        message::warn(
            format!(
                "{} provided, will run unauthenticated and upstream to provided host",
                host_str
            )
            .as_str(),
        );
    }

    gcs::dev(target, server_config, local_protocol, verbose)
}
