use std::{collections::HashMap, default, fmt, ops::Deref, path::PathBuf, thread::current};

use chrono::Duration;
use color_eyre::eyre::Result;
use config::{ConfigError, Value};
use crossterm::event::{KeyCode, KeyEvent, KeyModifiers};
use derive_deref::{Deref, DerefMut};
use ratatui::style::{Color, Modifier, Style};
use serde::{
    de::{self, Deserializer, MapAccess, Visitor},
    Deserialize, Serialize,
};
use serde_json::Value as JsonValue;

use crate::{
    action::Action,
    mode::{self, Mode},
    old_config::{General as OldGeneral, OldConfig},
};

#[derive(Clone, Debug)]
pub struct RuntimeConfig {
    pub interval: Duration,
    pub command: Vec<String>,
}

const CONFIG: &str = include_str!("../.config/config.json5");

#[derive(Clone, Debug, Deserialize, Default)]
pub struct AppConfig {
    #[serde(default)]
    pub _data_dir: PathBuf,
    #[serde(default)]
    pub _config_dir: PathBuf,
}

#[derive(Clone, Debug, Default, Deserialize)]
pub struct General {
    #[serde(default)]
    pub no_shell: Option<bool>,
    #[serde(default)]
    pub shell: Option<String>,
    #[serde(default)]
    pub shell_options: Option<String>,
    #[serde(default)]
    pub skip_empty_diffs: Option<bool>,
}

impl From<OldGeneral> for General {
    fn from(value: OldGeneral) -> Self {
        Self {
            no_shell: value.no_shell,
            shell: value.shell,
            shell_options: value.shell_options,
            skip_empty_diffs: value.skip_empty_diffs,
        }
    }
}

#[derive(Clone, Debug, Default, Deserialize)]
pub struct Config {
    #[serde(default, flatten)]
    pub config: AppConfig,
    #[serde(default)]
    pub keybindings: KeyBindings,
    #[serde(default)]
    pub styles: Styles,
    #[serde(default)]
    pub general: General,
}

impl Config {
    pub fn new() -> Result<Self, config::ConfigError> {
        let data_dir = crate::utils::get_data_dir();
        let config_dir = crate::utils::get_config_dir();
        let mut builder = config::Config::builder()
            .set_default("_data_dir", data_dir.to_str().unwrap())?
            .set_default("_config_dir", config_dir.to_str().unwrap())?;

        let config_files = [
            ("config.json5", config::FileFormat::Json5),
            ("config.json", config::FileFormat::Json),
            ("config.yaml", config::FileFormat::Yaml),
            ("config.toml", config::FileFormat::Toml),
            ("config.ini", config::FileFormat::Ini),
        ];
        let mut found_config = false;
        for (file, format) in &config_files {
            builder = builder.add_source(
                config::File::from(config_dir.join(file))
                    .format(*format)
                    .required(false),
            );
            if config_dir.join(file).exists() {
                found_config = true
            }
        }
        if !found_config {
            log::error!("No configuration file found. Application may not behave as expected");
        }

        let mut cfg: Self = builder.build()?.try_deserialize()?;

        cfg.defaulting();

        Ok(cfg)
    }

    pub fn defaulting(&mut self) {
        let default_config: Config = json5::from_str(CONFIG).unwrap();

        let current_binding_actions = self
            .keybindings
            .iter()
            .flat_map(|(mode, bindings)| {
                let actions = bindings
                    .iter()
                    .map(|(keys, action)| (keys.clone(), action.clone()))
                    .collect::<HashMap<_, _>>();
                actions
                    .into_iter()
                    .map(|(keys, action)| (*mode, action))
                    .collect::<Vec<_>>()
            })
            .collect::<Vec<_>>();

        for (mode, default_bindings) in default_config.keybindings.iter() {
            let user_bindings = self.keybindings.entry(*mode).or_default();
            for (key, cmd) in default_bindings.iter() {
                if current_binding_actions
                    .iter()
                    .any(|(m, a)| m == mode && a == cmd)
                {
                    continue;
                }

                user_bindings
                    .entry(key.clone())
                    .or_insert_with(|| cmd.clone());
            }
        }

        for (mode, default_styles) in default_config.styles.iter() {
            let user_styles = self.styles.entry(*mode).or_default();
            for (style_key, style) in default_styles.iter() {
                user_styles
                    .entry(style_key.clone())
                    .or_insert_with(|| *style);
            }
        }

        if self.general.no_shell.is_none() {
            self.general.no_shell = default_config.general.no_shell;
        }
        if self.general.shell.is_none() {
            self.general.shell = default_config.general.shell;
        }
        if self.general.shell_options.is_none() {
            self.general.shell_options = default_config.general.shell_options;
        }
        if self.general.skip_empty_diffs.is_none() {
            self.general.skip_empty_diffs = default_config.general.skip_empty_diffs;
        }
    }

    pub fn get_style(&self, style: &str) -> Style {
        *self
            .styles
            .get(&Mode::All)
            .and_then(|styles| styles.get(style))
            .unwrap_or(&Style::default())
    }
}

impl From<OldConfig> for Config {
    fn from(old_config: OldConfig) -> Self {
        let mut keybindings = HashMap::new();
        let mut styles = HashMap::new();

        let general = old_config.general.unwrap_or_default();
        let keymap = old_config.keymap.unwrap_or_default();
        let color = old_config.color.unwrap_or_default();

        let wrap = |s: &str| {
            s.trim()
                .split(" ")
                .map(|s| format!("<{}>", s))
                .collect::<Vec<_>>()
                .join("")
        };
        let mut all_keybindings = HashMap::new();
        let mut insert_keybinding = |key: Option<String>, action: Action| {
            if let Some(s) = key {
                all_keybindings.insert(parse_key_sequence(&wrap(&s)).unwrap(), action);
            }
        };

        insert_keybinding(keymap.timemachine_go_to_future, Action::GoToFuture);
        insert_keybinding(keymap.timemachine_go_to_more_future, Action::GoToMoreFuture);
        insert_keybinding(keymap.timemachine_go_to_past, Action::GoToPast);
        insert_keybinding(keymap.timemachine_go_to_more_past, Action::GoToMorePast);
        insert_keybinding(keymap.timemachine_go_to_now, Action::GoToCurrent);
        insert_keybinding(keymap.timemachine_go_to_oldest, Action::GoToOldest);
        insert_keybinding(keymap.toggle_timemachine, Action::SwitchTimemachineMode);
        insert_keybinding(keymap.scroll_left, Action::ScrollLeft);
        insert_keybinding(keymap.scroll_right, Action::ScrollRight);
        insert_keybinding(keymap.scroll_up, Action::ResultScrollUp);
        insert_keybinding(keymap.scroll_down, Action::ResultScrollDown);
        insert_keybinding(keymap.scroll_half_page_up, Action::ResultHalfPageUp);
        insert_keybinding(keymap.scroll_half_page_down, Action::ResultHalfPageDown);
        insert_keybinding(keymap.scroll_page_up, Action::ResultPageUp);
        insert_keybinding(keymap.scroll_page_down, Action::ResultPageDown);
        insert_keybinding(keymap.scroll_top_of_page, Action::TopOfPage);
        insert_keybinding(keymap.scroll_bottom_of_page, Action::BottomOfPage);

        keybindings.insert(Mode::All, all_keybindings);

        let mut all_styles = HashMap::new();
        if let Some(bg) = color.background {
            if !bg.is_empty() {
                all_styles.insert("background".to_string(), parse_style(&format!("on {}", bg)));
            }
        }
        styles.insert(Mode::All, all_styles);

        Config {
            keybindings: KeyBindings(keybindings),
            styles: Styles(styles),
            general: General::from(general),
            ..Default::default()
        }
    }
}

#[derive(Clone, Debug, Default, Deref, DerefMut)]
pub struct KeyBindings(pub HashMap<Mode, HashMap<Vec<KeyEvent>, Action>>);

impl<'de> Deserialize<'de> for KeyBindings {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error>
    where
        D: Deserializer<'de>,
    {
        let parsed_map = HashMap::<Mode, HashMap<String, Action>>::deserialize(deserializer)?;

        let keybindings = parsed_map
            .into_iter()
            .map(|(mode, inner_map)| {
                let converted_inner_map = inner_map
                    .into_iter()
                    .map(|(key_str, cmd)| (parse_key_sequence(&key_str).unwrap(), cmd))
                    .collect();
                (mode, converted_inner_map)
            })
            .collect();

        Ok(KeyBindings(keybindings))
    }
}

fn parse_key_event(raw: &str) -> Result<KeyEvent, String> {
    let raw_lower = raw.to_ascii_lowercase();
    let (remaining, modifiers) = extract_modifiers(&raw_lower);
    parse_key_code_with_modifiers(remaining, modifiers)
}

fn extract_modifiers(raw: &str) -> (&str, KeyModifiers) {
    let mut modifiers = KeyModifiers::empty();
    let mut current = raw;

    loop {
        match current {
            rest if rest.starts_with("ctrl-") => {
                modifiers.insert(KeyModifiers::CONTROL);
                current = &rest[5..];
            }
            rest if rest.starts_with("alt-") => {
                modifiers.insert(KeyModifiers::ALT);
                current = &rest[4..];
            }
            rest if rest.starts_with("shift-") => {
                modifiers.insert(KeyModifiers::SHIFT);
                current = &rest[6..];
            }
            _ => break, // break out of the loop if no known prefix is detected
        };
    }

    (current, modifiers)
}

fn parse_key_code_with_modifiers(
    raw: &str,
    mut modifiers: KeyModifiers,
) -> Result<KeyEvent, String> {
    let c = match raw {
        "esc" => KeyCode::Esc,
        "enter" => KeyCode::Enter,
        "left" => KeyCode::Left,
        "right" => KeyCode::Right,
        "up" => KeyCode::Up,
        "down" => KeyCode::Down,
        "home" => KeyCode::Home,
        "end" => KeyCode::End,
        "pageup" => KeyCode::PageUp,
        "pagedown" => KeyCode::PageDown,
        "backtab" => {
            modifiers.insert(KeyModifiers::SHIFT);
            KeyCode::BackTab
        }
        "backspace" => KeyCode::Backspace,
        "delete" => KeyCode::Delete,
        "insert" => KeyCode::Insert,
        "f1" => KeyCode::F(1),
        "f2" => KeyCode::F(2),
        "f3" => KeyCode::F(3),
        "f4" => KeyCode::F(4),
        "f5" => KeyCode::F(5),
        "f6" => KeyCode::F(6),
        "f7" => KeyCode::F(7),
        "f8" => KeyCode::F(8),
        "f9" => KeyCode::F(9),
        "f10" => KeyCode::F(10),
        "f11" => KeyCode::F(11),
        "f12" => KeyCode::F(12),
        "space" => KeyCode::Char(' '),
        "hyphen" => KeyCode::Char('-'),
        "minus" => KeyCode::Char('-'),
        "tab" => KeyCode::Tab,
        c if c.len() == 1 => {
            let mut c = c.chars().next().unwrap();
            if modifiers.contains(KeyModifiers::SHIFT) {
                c = c.to_ascii_uppercase();
            }
            KeyCode::Char(c)
        }
        _ => return Err(format!("Unable to parse {raw}")),
    };
    Ok(KeyEvent::new(c, modifiers))
}

pub fn key_event_to_string(key_event: &KeyEvent) -> String {
    let char;
    let key_code = match key_event.code {
        KeyCode::Backspace => "backspace",
        KeyCode::Enter => "enter",
        KeyCode::Left => "left",
        KeyCode::Right => "right",
        KeyCode::Up => "up",
        KeyCode::Down => "down",
        KeyCode::Home => "home",
        KeyCode::End => "end",
        KeyCode::PageUp => "pageup",
        KeyCode::PageDown => "pagedown",
        KeyCode::Tab => "tab",
        KeyCode::BackTab => "backtab",
        KeyCode::Delete => "delete",
        KeyCode::Insert => "insert",
        KeyCode::F(c) => {
            char = format!("f({c})");
            &char
        }
        KeyCode::Char(' ') => "space",
        KeyCode::Char(c) => {
            char = c.to_string();
            &char
        }
        KeyCode::Esc => "esc",
        KeyCode::Null => "",
        KeyCode::CapsLock => "",
        KeyCode::Menu => "",
        KeyCode::ScrollLock => "",
        KeyCode::Media(_) => "",
        KeyCode::NumLock => "",
        KeyCode::PrintScreen => "",
        KeyCode::Pause => "",
        KeyCode::KeypadBegin => "",
        KeyCode::Modifier(_) => "",
    };

    let mut modifiers = Vec::with_capacity(3);

    if key_event.modifiers.intersects(KeyModifiers::CONTROL) {
        modifiers.push("ctrl");
    }

    if key_event.modifiers.intersects(KeyModifiers::SHIFT) {
        modifiers.push("shift");
    }

    if key_event.modifiers.intersects(KeyModifiers::ALT) {
        modifiers.push("alt");
    }

    let mut key = modifiers.join("-");

    if !key.is_empty() {
        key.push('-');
    }
    key.push_str(key_code);

    key
}

pub fn parse_key_sequence(raw: &str) -> Result<Vec<KeyEvent>, String> {
    if raw.chars().filter(|c| *c == '>').count() != raw.chars().filter(|c| *c == '<').count() {
        return Err(format!("Unable to parse `{}`", raw));
    }
    let raw = if !raw.contains("><") {
        let raw = raw.strip_prefix('<').unwrap_or(raw);
        let raw = raw.strip_prefix('>').unwrap_or(raw);
        raw
    } else {
        raw
    };
    let sequences = raw
        .split("><")
        .map(|seq| {
            if let Some(s) = seq.strip_prefix('<') {
                s
            } else if let Some(s) = seq.strip_suffix('>') {
                s
            } else {
                seq
            }
        })
        .collect::<Vec<_>>();

    sequences.into_iter().map(parse_key_event).collect()
}

#[derive(Clone, Debug, Default, Deref, DerefMut)]
pub struct Styles(pub HashMap<Mode, HashMap<String, Style>>);

impl<'de> Deserialize<'de> for Styles {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error>
    where
        D: Deserializer<'de>,
    {
        let parsed_map = HashMap::<Mode, HashMap<String, String>>::deserialize(deserializer)?;

        let styles = parsed_map
            .into_iter()
            .map(|(mode, inner_map)| {
                let converted_inner_map = inner_map
                    .into_iter()
                    .map(|(str, style)| (str, parse_style(&style)))
                    .collect();
                (mode, converted_inner_map)
            })
            .collect();

        Ok(Styles(styles))
    }
}

pub fn parse_style(line: &str) -> Style {
    let (foreground, background) =
        line.split_at(line.to_lowercase().find("on ").unwrap_or(line.len()));
    let foreground = process_color_string(foreground);
    let background = process_color_string(&background.replace("on ", ""));

    let mut style = Style::default();
    if let Some(fg) = parse_color(&foreground.0) {
        style = style.fg(fg);
    }
    if let Some(bg) = parse_color(&background.0) {
        style = style.bg(bg);
    }
    style = style.add_modifier(foreground.1 | background.1);
    style
}

fn process_color_string(color_str: &str) -> (String, Modifier) {
    let color = color_str
        .replace("grey", "gray")
        .replace("bright ", "")
        .replace("bold ", "")
        .replace("underline ", "")
        .replace("inverse ", "");

    let mut modifiers = Modifier::empty();
    if color_str.contains("underline") {
        modifiers |= Modifier::UNDERLINED;
    }
    if color_str.contains("bold") {
        modifiers |= Modifier::BOLD;
    }
    if color_str.contains("inverse") {
        modifiers |= Modifier::REVERSED;
    }

    (color, modifiers)
}

fn parse_color(s: &str) -> Option<Color> {
    let s = s.trim_start();
    let s = s.trim_end();
    if s.contains("bright color") {
        let s = s.trim_start_matches("bright ");
        let c = s
            .trim_start_matches("color")
            .parse::<u8>()
            .unwrap_or_default();
        Some(Color::Indexed(c.wrapping_shl(8)))
    } else if s.contains("color") {
        let c = s
            .trim_start_matches("color")
            .parse::<u8>()
            .unwrap_or_default();
        Some(Color::Indexed(c))
    } else if s.contains("gray") {
        let c = 232
            + s.trim_start_matches("gray")
                .parse::<u8>()
                .unwrap_or_default();
        Some(Color::Indexed(c))
    } else if s.contains("rgb") {
        let red = (s.as_bytes()[3] as char).to_digit(10).unwrap_or_default() as u8;
        let green = (s.as_bytes()[4] as char).to_digit(10).unwrap_or_default() as u8;
        let blue = (s.as_bytes()[5] as char).to_digit(10).unwrap_or_default() as u8;
        let c = 16 + red * 36 + green * 6 + blue;
        Some(Color::Indexed(c))
    } else if s == "bold black" {
        Some(Color::Indexed(8))
    } else if s == "bold red" {
        Some(Color::Indexed(9))
    } else if s == "bold green" {
        Some(Color::Indexed(10))
    } else if s == "bold yellow" {
        Some(Color::Indexed(11))
    } else if s == "bold blue" {
        Some(Color::Indexed(12))
    } else if s == "bold magenta" {
        Some(Color::Indexed(13))
    } else if s == "bold cyan" {
        Some(Color::Indexed(14))
    } else if s == "bold white" {
        Some(Color::Indexed(15))
    } else if s == "black" {
        Some(Color::Indexed(0))
    } else if s == "red" {
        Some(Color::Indexed(1))
    } else if s == "green" {
        Some(Color::Indexed(2))
    } else if s == "yellow" {
        Some(Color::Indexed(3))
    } else if s == "blue" {
        Some(Color::Indexed(4))
    } else if s == "magenta" {
        Some(Color::Indexed(5))
    } else if s == "cyan" {
        Some(Color::Indexed(6))
    } else if s == "white" {
        Some(Color::Indexed(7))
    } else {
        None
    }
}

#[cfg(test)]
mod tests {
    use pretty_assertions::assert_eq;

    use super::*;

    #[test]
    fn test_parse_style_default() {
        let style = parse_style("");
        assert_eq!(style, Style::default());
    }

    #[test]
    fn test_parse_style_foreground() {
        let style = parse_style("red");
        assert_eq!(style.fg, Some(Color::Indexed(1)));
    }

    #[test]
    fn test_parse_style_background() {
        let style = parse_style("on blue");
        assert_eq!(style.bg, Some(Color::Indexed(4)));
    }

    #[test]
    fn test_parse_style_modifiers() {
        let style = parse_style("underline red on blue");
        assert_eq!(style.fg, Some(Color::Indexed(1)));
        assert_eq!(style.bg, Some(Color::Indexed(4)));
    }

    #[test]
    fn test_process_color_string() {
        let (color, modifiers) = process_color_string("underline bold inverse gray");
        assert_eq!(color, "gray");
        assert!(modifiers.contains(Modifier::UNDERLINED));
        assert!(modifiers.contains(Modifier::BOLD));
        assert!(modifiers.contains(Modifier::REVERSED));
    }

    #[test]
    fn test_parse_color_rgb() {
        let color = parse_color("rgb123");
        let expected = 16 + 36 + 2 * 6 + 3;
        assert_eq!(color, Some(Color::Indexed(expected)));
    }

    #[test]
    fn test_parse_color_unknown() {
        let color = parse_color("unknown");
        assert_eq!(color, None);
    }

    #[test]
    fn test_config() -> Result<()> {
        let c = Config::new()?;
        assert_eq!(
            c.keybindings
                .get(&Mode::All)
                .unwrap()
                .get(&parse_key_sequence("<q>").unwrap_or_default())
                .unwrap(),
            &Action::Quit
        );
        Ok(())
    }

    #[test]
    fn test_simple_keys() {
        assert_eq!(
            parse_key_event("a").unwrap(),
            KeyEvent::new(KeyCode::Char('a'), KeyModifiers::empty())
        );

        assert_eq!(
            parse_key_event("enter").unwrap(),
            KeyEvent::new(KeyCode::Enter, KeyModifiers::empty())
        );

        assert_eq!(
            parse_key_event("esc").unwrap(),
            KeyEvent::new(KeyCode::Esc, KeyModifiers::empty())
        );
    }

    #[test]
    fn test_with_modifiers() {
        assert_eq!(
            parse_key_event("ctrl-a").unwrap(),
            KeyEvent::new(KeyCode::Char('a'), KeyModifiers::CONTROL)
        );

        assert_eq!(
            parse_key_event("alt-enter").unwrap(),
            KeyEvent::new(KeyCode::Enter, KeyModifiers::ALT)
        );

        assert_eq!(
            parse_key_event("shift-esc").unwrap(),
            KeyEvent::new(KeyCode::Esc, KeyModifiers::SHIFT)
        );
    }

    #[test]
    fn test_multiple_modifiers() {
        assert_eq!(
            parse_key_event("ctrl-alt-a").unwrap(),
            KeyEvent::new(
                KeyCode::Char('a'),
                KeyModifiers::CONTROL | KeyModifiers::ALT
            )
        );

        assert_eq!(
            parse_key_event("ctrl-shift-enter").unwrap(),
            KeyEvent::new(KeyCode::Enter, KeyModifiers::CONTROL | KeyModifiers::SHIFT)
        );
    }

    #[test]
    fn test_reverse_multiple_modifiers() {
        assert_eq!(
            key_event_to_string(&KeyEvent::new(
                KeyCode::Char('a'),
                KeyModifiers::CONTROL | KeyModifiers::ALT
            )),
            "ctrl-alt-a".to_string()
        );
    }

    #[test]
    fn test_invalid_keys() {
        assert!(parse_key_event("invalid-key").is_err());
        assert!(parse_key_event("ctrl-invalid-key").is_err());
    }

    #[test]
    fn test_case_insensitivity() {
        assert_eq!(
            parse_key_event("CTRL-a").unwrap(),
            KeyEvent::new(KeyCode::Char('a'), KeyModifiers::CONTROL)
        );

        assert_eq!(
            parse_key_event("AlT-eNtEr").unwrap(),
            KeyEvent::new(KeyCode::Enter, KeyModifiers::ALT)
        );
    }

    #[test]
    fn test_old_config_defaulting() {
        let config_str = r#"
    [general]
    skip_empty_diffs = true
    "#;

        let config = OldConfig::new_from_str(config_str).unwrap();
        let mut config = Config::from(config);
        config.defaulting();

        assert_eq!(config.general.skip_empty_diffs, Some(true));
    }

    #[test]
    fn test_old_config_keymap() {
        let config_str = r#"
[keymap]
timemachine_go_to_past = "Down"
timemachine_go_to_more_past = "Shift-Down"
timemachine_go_to_future = "Up"
timemachine_go_to_more_future = "Shift-Up"
timemachine_go_to_now = "Ctrl-Shift-Up"
timemachine_go_to_oldest = "Ctrl-Shift-Down"
scroll_left = "h"
scroll_right = "l"
scroll_up = "k"
scroll_down = "j"
scroll_half_page_up = "Ctrl-u"
scroll_half_page_down = "Ctrl-d"
scroll_page_up = "Ctrl-b"
scroll_page_down = "Ctrl-f"
scroll_bottom_of_page = "Shift-g"
scroll_top_of_page = "g g"
"#;

        let config = OldConfig::new_from_str(config_str).unwrap();
        let mut config = Config::from(config);

        config.defaulting();

        let assert_action = |config: &Config, key: &str, action: Action| {
            assert_eq!(
                config
                    .keybindings
                    .get(&Mode::All)
                    .unwrap()
                    .get(&parse_key_sequence(key).unwrap_or_default())
                    .unwrap(),
                &action
            );
        };

        assert_action(&config, "<Down>", Action::GoToPast);
        assert_action(&config, "<Shift-Down>", Action::GoToMorePast);
        assert_action(&config, "<Up>", Action::GoToFuture);
        assert_action(&config, "<Shift-Up>", Action::GoToMoreFuture);
        assert_action(&config, "<Ctrl-Shift-Up>", Action::GoToCurrent);
        assert_action(&config, "<Ctrl-Shift-Down>", Action::GoToOldest);
        assert_action(&config, "<h>", Action::ScrollLeft);
        assert_action(&config, "<l>", Action::ScrollRight);
        assert_action(&config, "<k>", Action::ResultScrollUp);
        assert_action(&config, "<j>", Action::ResultScrollDown);
        assert_action(&config, "<Ctrl-u>", Action::ResultHalfPageUp);
        assert_action(&config, "<Ctrl-d>", Action::ResultHalfPageDown);
        assert_action(&config, "<Ctrl-b>", Action::ResultPageUp);
        assert_action(&config, "<Ctrl-f>", Action::ResultPageDown);
        assert_action(&config, "<Shift-g>", Action::BottomOfPage);
        assert_action(&config, "<g><g>", Action::TopOfPage);
    }
}
