use insta::{assert_debug_snapshot, assert_snapshot};

use super::*;

/// Check platform tag ordering.
/// The list is displayed in decreasing priority.
///
/// A reference list can be generated with:
/// ```text
/// $ python -c "from packaging import tags; [print(tag) for tag in tags.platform_tags()]"`
/// ````
#[test]
fn test_platform_tags_manylinux() {
    let tags = compatible_tags(&Platform::new(
        Os::Manylinux {
            major: 2,
            minor: 20,
        },
        Arch::X86_64,
    ))
    .unwrap();
    assert_debug_snapshot!(
        tags,
        @r###"
    [
        "manylinux_2_20_x86_64",
        "manylinux_2_19_x86_64",
        "manylinux_2_18_x86_64",
        "manylinux_2_17_x86_64",
        "manylinux2014_x86_64",
        "manylinux_2_16_x86_64",
        "manylinux_2_15_x86_64",
        "manylinux_2_14_x86_64",
        "manylinux_2_13_x86_64",
        "manylinux_2_12_x86_64",
        "manylinux2010_x86_64",
        "manylinux_2_11_x86_64",
        "manylinux_2_10_x86_64",
        "manylinux_2_9_x86_64",
        "manylinux_2_8_x86_64",
        "manylinux_2_7_x86_64",
        "manylinux_2_6_x86_64",
        "manylinux_2_5_x86_64",
        "manylinux1_x86_64",
        "linux_x86_64",
    ]
    "###
    );
}

#[test]
fn test_platform_tags_macos() {
    let tags = compatible_tags(&Platform::new(
        Os::Macos {
            major: 21,
            minor: 6,
        },
        Arch::X86_64,
    ))
    .unwrap();
    assert_debug_snapshot!(
        tags,
        @r###"
    [
        "macosx_21_0_x86_64",
        "macosx_21_0_intel",
        "macosx_21_0_fat64",
        "macosx_21_0_fat32",
        "macosx_21_0_universal2",
        "macosx_21_0_universal",
        "macosx_20_0_x86_64",
        "macosx_20_0_intel",
        "macosx_20_0_fat64",
        "macosx_20_0_fat32",
        "macosx_20_0_universal2",
        "macosx_20_0_universal",
        "macosx_19_0_x86_64",
        "macosx_19_0_intel",
        "macosx_19_0_fat64",
        "macosx_19_0_fat32",
        "macosx_19_0_universal2",
        "macosx_19_0_universal",
        "macosx_18_0_x86_64",
        "macosx_18_0_intel",
        "macosx_18_0_fat64",
        "macosx_18_0_fat32",
        "macosx_18_0_universal2",
        "macosx_18_0_universal",
        "macosx_17_0_x86_64",
        "macosx_17_0_intel",
        "macosx_17_0_fat64",
        "macosx_17_0_fat32",
        "macosx_17_0_universal2",
        "macosx_17_0_universal",
        "macosx_16_0_x86_64",
        "macosx_16_0_intel",
        "macosx_16_0_fat64",
        "macosx_16_0_fat32",
        "macosx_16_0_universal2",
        "macosx_16_0_universal",
        "macosx_15_0_x86_64",
        "macosx_15_0_intel",
        "macosx_15_0_fat64",
        "macosx_15_0_fat32",
        "macosx_15_0_universal2",
        "macosx_15_0_universal",
        "macosx_14_0_x86_64",
        "macosx_14_0_intel",
        "macosx_14_0_fat64",
        "macosx_14_0_fat32",
        "macosx_14_0_universal2",
        "macosx_14_0_universal",
        "macosx_13_0_x86_64",
        "macosx_13_0_intel",
        "macosx_13_0_fat64",
        "macosx_13_0_fat32",
        "macosx_13_0_universal2",
        "macosx_13_0_universal",
        "macosx_12_0_x86_64",
        "macosx_12_0_intel",
        "macosx_12_0_fat64",
        "macosx_12_0_fat32",
        "macosx_12_0_universal2",
        "macosx_12_0_universal",
        "macosx_11_0_x86_64",
        "macosx_11_0_intel",
        "macosx_11_0_fat64",
        "macosx_11_0_fat32",
        "macosx_11_0_universal2",
        "macosx_11_0_universal",
        "macosx_10_16_x86_64",
        "macosx_10_16_intel",
        "macosx_10_16_fat64",
        "macosx_10_16_fat32",
        "macosx_10_16_universal2",
        "macosx_10_16_universal",
        "macosx_10_15_x86_64",
        "macosx_10_15_intel",
        "macosx_10_15_fat64",
        "macosx_10_15_fat32",
        "macosx_10_15_universal2",
        "macosx_10_15_universal",
        "macosx_10_14_x86_64",
        "macosx_10_14_intel",
        "macosx_10_14_fat64",
        "macosx_10_14_fat32",
        "macosx_10_14_universal2",
        "macosx_10_14_universal",
        "macosx_10_13_x86_64",
        "macosx_10_13_intel",
        "macosx_10_13_fat64",
        "macosx_10_13_fat32",
        "macosx_10_13_universal2",
        "macosx_10_13_universal",
        "macosx_10_12_x86_64",
        "macosx_10_12_intel",
        "macosx_10_12_fat64",
        "macosx_10_12_fat32",
        "macosx_10_12_universal2",
        "macosx_10_12_universal",
        "macosx_10_11_x86_64",
        "macosx_10_11_intel",
        "macosx_10_11_fat64",
        "macosx_10_11_fat32",
        "macosx_10_11_universal2",
        "macosx_10_11_universal",
        "macosx_10_10_x86_64",
        "macosx_10_10_intel",
        "macosx_10_10_fat64",
        "macosx_10_10_fat32",
        "macosx_10_10_universal2",
        "macosx_10_10_universal",
        "macosx_10_9_x86_64",
        "macosx_10_9_intel",
        "macosx_10_9_fat64",
        "macosx_10_9_fat32",
        "macosx_10_9_universal2",
        "macosx_10_9_universal",
        "macosx_10_8_x86_64",
        "macosx_10_8_intel",
        "macosx_10_8_fat64",
        "macosx_10_8_fat32",
        "macosx_10_8_universal2",
        "macosx_10_8_universal",
        "macosx_10_7_x86_64",
        "macosx_10_7_intel",
        "macosx_10_7_fat64",
        "macosx_10_7_fat32",
        "macosx_10_7_universal2",
        "macosx_10_7_universal",
        "macosx_10_6_x86_64",
        "macosx_10_6_intel",
        "macosx_10_6_fat64",
        "macosx_10_6_fat32",
        "macosx_10_6_universal2",
        "macosx_10_6_universal",
        "macosx_10_5_x86_64",
        "macosx_10_5_intel",
        "macosx_10_5_fat64",
        "macosx_10_5_fat32",
        "macosx_10_5_universal2",
        "macosx_10_5_universal",
        "macosx_10_4_x86_64",
        "macosx_10_4_intel",
        "macosx_10_4_fat64",
        "macosx_10_4_fat32",
        "macosx_10_4_universal2",
        "macosx_10_4_universal",
    ]
    "###
    );

    let tags = compatible_tags(&Platform::new(
        Os::Macos {
            major: 14,
            minor: 0,
        },
        Arch::X86_64,
    ))
    .unwrap();
    assert_debug_snapshot!(
        tags,
        @r###"
    [
        "macosx_14_0_x86_64",
        "macosx_14_0_intel",
        "macosx_14_0_fat64",
        "macosx_14_0_fat32",
        "macosx_14_0_universal2",
        "macosx_14_0_universal",
        "macosx_13_0_x86_64",
        "macosx_13_0_intel",
        "macosx_13_0_fat64",
        "macosx_13_0_fat32",
        "macosx_13_0_universal2",
        "macosx_13_0_universal",
        "macosx_12_0_x86_64",
        "macosx_12_0_intel",
        "macosx_12_0_fat64",
        "macosx_12_0_fat32",
        "macosx_12_0_universal2",
        "macosx_12_0_universal",
        "macosx_11_0_x86_64",
        "macosx_11_0_intel",
        "macosx_11_0_fat64",
        "macosx_11_0_fat32",
        "macosx_11_0_universal2",
        "macosx_11_0_universal",
        "macosx_10_16_x86_64",
        "macosx_10_16_intel",
        "macosx_10_16_fat64",
        "macosx_10_16_fat32",
        "macosx_10_16_universal2",
        "macosx_10_16_universal",
        "macosx_10_15_x86_64",
        "macosx_10_15_intel",
        "macosx_10_15_fat64",
        "macosx_10_15_fat32",
        "macosx_10_15_universal2",
        "macosx_10_15_universal",
        "macosx_10_14_x86_64",
        "macosx_10_14_intel",
        "macosx_10_14_fat64",
        "macosx_10_14_fat32",
        "macosx_10_14_universal2",
        "macosx_10_14_universal",
        "macosx_10_13_x86_64",
        "macosx_10_13_intel",
        "macosx_10_13_fat64",
        "macosx_10_13_fat32",
        "macosx_10_13_universal2",
        "macosx_10_13_universal",
        "macosx_10_12_x86_64",
        "macosx_10_12_intel",
        "macosx_10_12_fat64",
        "macosx_10_12_fat32",
        "macosx_10_12_universal2",
        "macosx_10_12_universal",
        "macosx_10_11_x86_64",
        "macosx_10_11_intel",
        "macosx_10_11_fat64",
        "macosx_10_11_fat32",
        "macosx_10_11_universal2",
        "macosx_10_11_universal",
        "macosx_10_10_x86_64",
        "macosx_10_10_intel",
        "macosx_10_10_fat64",
        "macosx_10_10_fat32",
        "macosx_10_10_universal2",
        "macosx_10_10_universal",
        "macosx_10_9_x86_64",
        "macosx_10_9_intel",
        "macosx_10_9_fat64",
        "macosx_10_9_fat32",
        "macosx_10_9_universal2",
        "macosx_10_9_universal",
        "macosx_10_8_x86_64",
        "macosx_10_8_intel",
        "macosx_10_8_fat64",
        "macosx_10_8_fat32",
        "macosx_10_8_universal2",
        "macosx_10_8_universal",
        "macosx_10_7_x86_64",
        "macosx_10_7_intel",
        "macosx_10_7_fat64",
        "macosx_10_7_fat32",
        "macosx_10_7_universal2",
        "macosx_10_7_universal",
        "macosx_10_6_x86_64",
        "macosx_10_6_intel",
        "macosx_10_6_fat64",
        "macosx_10_6_fat32",
        "macosx_10_6_universal2",
        "macosx_10_6_universal",
        "macosx_10_5_x86_64",
        "macosx_10_5_intel",
        "macosx_10_5_fat64",
        "macosx_10_5_fat32",
        "macosx_10_5_universal2",
        "macosx_10_5_universal",
        "macosx_10_4_x86_64",
        "macosx_10_4_intel",
        "macosx_10_4_fat64",
        "macosx_10_4_fat32",
        "macosx_10_4_universal2",
        "macosx_10_4_universal",
    ]
    "###
    );

    let tags = compatible_tags(&Platform::new(
        Os::Macos {
            major: 10,
            minor: 6,
        },
        Arch::X86_64,
    ))
    .unwrap();
    assert_debug_snapshot!(
        tags,
        @r###"
    [
        "macosx_10_6_x86_64",
        "macosx_10_6_intel",
        "macosx_10_6_fat64",
        "macosx_10_6_fat32",
        "macosx_10_6_universal2",
        "macosx_10_6_universal",
        "macosx_10_5_x86_64",
        "macosx_10_5_intel",
        "macosx_10_5_fat64",
        "macosx_10_5_fat32",
        "macosx_10_5_universal2",
        "macosx_10_5_universal",
        "macosx_10_4_x86_64",
        "macosx_10_4_intel",
        "macosx_10_4_fat64",
        "macosx_10_4_fat32",
        "macosx_10_4_universal2",
        "macosx_10_4_universal",
    ]
    "###
    );
}

/// Ensure the tags returned do not include the `manylinux` tags
/// when `manylinux_incompatible` is set to `false`.
#[test]
fn test_manylinux_incompatible() {
    let tags = Tags::from_env(
        &Platform::new(
            Os::Manylinux {
                major: 2,
                minor: 28,
            },
            Arch::X86_64,
        ),
        (3, 9),
        "cpython",
        (3, 9),
        false,
        false,
    )
    .unwrap();
    assert_snapshot!(
        tags,
        @r###"
    cp39-cp39-linux_x86_64
    cp39-abi3-linux_x86_64
    cp39-none-linux_x86_64
    cp38-abi3-linux_x86_64
    cp37-abi3-linux_x86_64
    cp36-abi3-linux_x86_64
    cp35-abi3-linux_x86_64
    cp34-abi3-linux_x86_64
    cp33-abi3-linux_x86_64
    cp32-abi3-linux_x86_64
    py39-none-linux_x86_64
    py3-none-linux_x86_64
    py38-none-linux_x86_64
    py37-none-linux_x86_64
    py36-none-linux_x86_64
    py35-none-linux_x86_64
    py34-none-linux_x86_64
    py33-none-linux_x86_64
    py32-none-linux_x86_64
    py31-none-linux_x86_64
    py30-none-linux_x86_64
    cp39-none-any
    py39-none-any
    py3-none-any
    py38-none-any
    py37-none-any
    py36-none-any
    py35-none-any
    py34-none-any
    py33-none-any
    py32-none-any
    py31-none-any
    py30-none-any
    "###);
}

/// Check full tag ordering.
/// The list is displayed in decreasing priority.
///
/// A reference list can be generated with:
/// ```text
/// $ python -c "from packaging import tags; [print(tag) for tag in tags.sys_tags()]"`
/// ```
#[test]
fn test_system_tags_manylinux() {
    let tags = Tags::from_env(
        &Platform::new(
            Os::Manylinux {
                major: 2,
                minor: 28,
            },
            Arch::X86_64,
        ),
        (3, 9),
        "cpython",
        (3, 9),
        true,
        false,
    )
    .unwrap();
    assert_snapshot!(
        tags,
        @r###"
    cp39-cp39-manylinux_2_28_x86_64
    cp39-cp39-manylinux_2_27_x86_64
    cp39-cp39-manylinux_2_26_x86_64
    cp39-cp39-manylinux_2_25_x86_64
    cp39-cp39-manylinux_2_24_x86_64
    cp39-cp39-manylinux_2_23_x86_64
    cp39-cp39-manylinux_2_22_x86_64
    cp39-cp39-manylinux_2_21_x86_64
    cp39-cp39-manylinux_2_20_x86_64
    cp39-cp39-manylinux_2_19_x86_64
    cp39-cp39-manylinux_2_18_x86_64
    cp39-cp39-manylinux_2_17_x86_64
    cp39-cp39-manylinux2014_x86_64
    cp39-cp39-manylinux_2_16_x86_64
    cp39-cp39-manylinux_2_15_x86_64
    cp39-cp39-manylinux_2_14_x86_64
    cp39-cp39-manylinux_2_13_x86_64
    cp39-cp39-manylinux_2_12_x86_64
    cp39-cp39-manylinux2010_x86_64
    cp39-cp39-manylinux_2_11_x86_64
    cp39-cp39-manylinux_2_10_x86_64
    cp39-cp39-manylinux_2_9_x86_64
    cp39-cp39-manylinux_2_8_x86_64
    cp39-cp39-manylinux_2_7_x86_64
    cp39-cp39-manylinux_2_6_x86_64
    cp39-cp39-manylinux_2_5_x86_64
    cp39-cp39-manylinux1_x86_64
    cp39-cp39-linux_x86_64
    cp39-abi3-manylinux_2_28_x86_64
    cp39-abi3-manylinux_2_27_x86_64
    cp39-abi3-manylinux_2_26_x86_64
    cp39-abi3-manylinux_2_25_x86_64
    cp39-abi3-manylinux_2_24_x86_64
    cp39-abi3-manylinux_2_23_x86_64
    cp39-abi3-manylinux_2_22_x86_64
    cp39-abi3-manylinux_2_21_x86_64
    cp39-abi3-manylinux_2_20_x86_64
    cp39-abi3-manylinux_2_19_x86_64
    cp39-abi3-manylinux_2_18_x86_64
    cp39-abi3-manylinux_2_17_x86_64
    cp39-abi3-manylinux2014_x86_64
    cp39-abi3-manylinux_2_16_x86_64
    cp39-abi3-manylinux_2_15_x86_64
    cp39-abi3-manylinux_2_14_x86_64
    cp39-abi3-manylinux_2_13_x86_64
    cp39-abi3-manylinux_2_12_x86_64
    cp39-abi3-manylinux2010_x86_64
    cp39-abi3-manylinux_2_11_x86_64
    cp39-abi3-manylinux_2_10_x86_64
    cp39-abi3-manylinux_2_9_x86_64
    cp39-abi3-manylinux_2_8_x86_64
    cp39-abi3-manylinux_2_7_x86_64
    cp39-abi3-manylinux_2_6_x86_64
    cp39-abi3-manylinux_2_5_x86_64
    cp39-abi3-manylinux1_x86_64
    cp39-abi3-linux_x86_64
    cp39-none-manylinux_2_28_x86_64
    cp39-none-manylinux_2_27_x86_64
    cp39-none-manylinux_2_26_x86_64
    cp39-none-manylinux_2_25_x86_64
    cp39-none-manylinux_2_24_x86_64
    cp39-none-manylinux_2_23_x86_64
    cp39-none-manylinux_2_22_x86_64
    cp39-none-manylinux_2_21_x86_64
    cp39-none-manylinux_2_20_x86_64
    cp39-none-manylinux_2_19_x86_64
    cp39-none-manylinux_2_18_x86_64
    cp39-none-manylinux_2_17_x86_64
    cp39-none-manylinux2014_x86_64
    cp39-none-manylinux_2_16_x86_64
    cp39-none-manylinux_2_15_x86_64
    cp39-none-manylinux_2_14_x86_64
    cp39-none-manylinux_2_13_x86_64
    cp39-none-manylinux_2_12_x86_64
    cp39-none-manylinux2010_x86_64
    cp39-none-manylinux_2_11_x86_64
    cp39-none-manylinux_2_10_x86_64
    cp39-none-manylinux_2_9_x86_64
    cp39-none-manylinux_2_8_x86_64
    cp39-none-manylinux_2_7_x86_64
    cp39-none-manylinux_2_6_x86_64
    cp39-none-manylinux_2_5_x86_64
    cp39-none-manylinux1_x86_64
    cp39-none-linux_x86_64
    cp38-abi3-manylinux_2_28_x86_64
    cp38-abi3-manylinux_2_27_x86_64
    cp38-abi3-manylinux_2_26_x86_64
    cp38-abi3-manylinux_2_25_x86_64
    cp38-abi3-manylinux_2_24_x86_64
    cp38-abi3-manylinux_2_23_x86_64
    cp38-abi3-manylinux_2_22_x86_64
    cp38-abi3-manylinux_2_21_x86_64
    cp38-abi3-manylinux_2_20_x86_64
    cp38-abi3-manylinux_2_19_x86_64
    cp38-abi3-manylinux_2_18_x86_64
    cp38-abi3-manylinux_2_17_x86_64
    cp38-abi3-manylinux2014_x86_64
    cp38-abi3-manylinux_2_16_x86_64
    cp38-abi3-manylinux_2_15_x86_64
    cp38-abi3-manylinux_2_14_x86_64
    cp38-abi3-manylinux_2_13_x86_64
    cp38-abi3-manylinux_2_12_x86_64
    cp38-abi3-manylinux2010_x86_64
    cp38-abi3-manylinux_2_11_x86_64
    cp38-abi3-manylinux_2_10_x86_64
    cp38-abi3-manylinux_2_9_x86_64
    cp38-abi3-manylinux_2_8_x86_64
    cp38-abi3-manylinux_2_7_x86_64
    cp38-abi3-manylinux_2_6_x86_64
    cp38-abi3-manylinux_2_5_x86_64
    cp38-abi3-manylinux1_x86_64
    cp38-abi3-linux_x86_64
    cp37-abi3-manylinux_2_28_x86_64
    cp37-abi3-manylinux_2_27_x86_64
    cp37-abi3-manylinux_2_26_x86_64
    cp37-abi3-manylinux_2_25_x86_64
    cp37-abi3-manylinux_2_24_x86_64
    cp37-abi3-manylinux_2_23_x86_64
    cp37-abi3-manylinux_2_22_x86_64
    cp37-abi3-manylinux_2_21_x86_64
    cp37-abi3-manylinux_2_20_x86_64
    cp37-abi3-manylinux_2_19_x86_64
    cp37-abi3-manylinux_2_18_x86_64
    cp37-abi3-manylinux_2_17_x86_64
    cp37-abi3-manylinux2014_x86_64
    cp37-abi3-manylinux_2_16_x86_64
    cp37-abi3-manylinux_2_15_x86_64
    cp37-abi3-manylinux_2_14_x86_64
    cp37-abi3-manylinux_2_13_x86_64
    cp37-abi3-manylinux_2_12_x86_64
    cp37-abi3-manylinux2010_x86_64
    cp37-abi3-manylinux_2_11_x86_64
    cp37-abi3-manylinux_2_10_x86_64
    cp37-abi3-manylinux_2_9_x86_64
    cp37-abi3-manylinux_2_8_x86_64
    cp37-abi3-manylinux_2_7_x86_64
    cp37-abi3-manylinux_2_6_x86_64
    cp37-abi3-manylinux_2_5_x86_64
    cp37-abi3-manylinux1_x86_64
    cp37-abi3-linux_x86_64
    cp36-abi3-manylinux_2_28_x86_64
    cp36-abi3-manylinux_2_27_x86_64
    cp36-abi3-manylinux_2_26_x86_64
    cp36-abi3-manylinux_2_25_x86_64
    cp36-abi3-manylinux_2_24_x86_64
    cp36-abi3-manylinux_2_23_x86_64
    cp36-abi3-manylinux_2_22_x86_64
    cp36-abi3-manylinux_2_21_x86_64
    cp36-abi3-manylinux_2_20_x86_64
    cp36-abi3-manylinux_2_19_x86_64
    cp36-abi3-manylinux_2_18_x86_64
    cp36-abi3-manylinux_2_17_x86_64
    cp36-abi3-manylinux2014_x86_64
    cp36-abi3-manylinux_2_16_x86_64
    cp36-abi3-manylinux_2_15_x86_64
    cp36-abi3-manylinux_2_14_x86_64
    cp36-abi3-manylinux_2_13_x86_64
    cp36-abi3-manylinux_2_12_x86_64
    cp36-abi3-manylinux2010_x86_64
    cp36-abi3-manylinux_2_11_x86_64
    cp36-abi3-manylinux_2_10_x86_64
    cp36-abi3-manylinux_2_9_x86_64
    cp36-abi3-manylinux_2_8_x86_64
    cp36-abi3-manylinux_2_7_x86_64
    cp36-abi3-manylinux_2_6_x86_64
    cp36-abi3-manylinux_2_5_x86_64
    cp36-abi3-manylinux1_x86_64
    cp36-abi3-linux_x86_64
    cp35-abi3-manylinux_2_28_x86_64
    cp35-abi3-manylinux_2_27_x86_64
    cp35-abi3-manylinux_2_26_x86_64
    cp35-abi3-manylinux_2_25_x86_64
    cp35-abi3-manylinux_2_24_x86_64
    cp35-abi3-manylinux_2_23_x86_64
    cp35-abi3-manylinux_2_22_x86_64
    cp35-abi3-manylinux_2_21_x86_64
    cp35-abi3-manylinux_2_20_x86_64
    cp35-abi3-manylinux_2_19_x86_64
    cp35-abi3-manylinux_2_18_x86_64
    cp35-abi3-manylinux_2_17_x86_64
    cp35-abi3-manylinux2014_x86_64
    cp35-abi3-manylinux_2_16_x86_64
    cp35-abi3-manylinux_2_15_x86_64
    cp35-abi3-manylinux_2_14_x86_64
    cp35-abi3-manylinux_2_13_x86_64
    cp35-abi3-manylinux_2_12_x86_64
    cp35-abi3-manylinux2010_x86_64
    cp35-abi3-manylinux_2_11_x86_64
    cp35-abi3-manylinux_2_10_x86_64
    cp35-abi3-manylinux_2_9_x86_64
    cp35-abi3-manylinux_2_8_x86_64
    cp35-abi3-manylinux_2_7_x86_64
    cp35-abi3-manylinux_2_6_x86_64
    cp35-abi3-manylinux_2_5_x86_64
    cp35-abi3-manylinux1_x86_64
    cp35-abi3-linux_x86_64
    cp34-abi3-manylinux_2_28_x86_64
    cp34-abi3-manylinux_2_27_x86_64
    cp34-abi3-manylinux_2_26_x86_64
    cp34-abi3-manylinux_2_25_x86_64
    cp34-abi3-manylinux_2_24_x86_64
    cp34-abi3-manylinux_2_23_x86_64
    cp34-abi3-manylinux_2_22_x86_64
    cp34-abi3-manylinux_2_21_x86_64
    cp34-abi3-manylinux_2_20_x86_64
    cp34-abi3-manylinux_2_19_x86_64
    cp34-abi3-manylinux_2_18_x86_64
    cp34-abi3-manylinux_2_17_x86_64
    cp34-abi3-manylinux2014_x86_64
    cp34-abi3-manylinux_2_16_x86_64
    cp34-abi3-manylinux_2_15_x86_64
    cp34-abi3-manylinux_2_14_x86_64
    cp34-abi3-manylinux_2_13_x86_64
    cp34-abi3-manylinux_2_12_x86_64
    cp34-abi3-manylinux2010_x86_64
    cp34-abi3-manylinux_2_11_x86_64
    cp34-abi3-manylinux_2_10_x86_64
    cp34-abi3-manylinux_2_9_x86_64
    cp34-abi3-manylinux_2_8_x86_64
    cp34-abi3-manylinux_2_7_x86_64
    cp34-abi3-manylinux_2_6_x86_64
    cp34-abi3-manylinux_2_5_x86_64
    cp34-abi3-manylinux1_x86_64
    cp34-abi3-linux_x86_64
    cp33-abi3-manylinux_2_28_x86_64
    cp33-abi3-manylinux_2_27_x86_64
    cp33-abi3-manylinux_2_26_x86_64
    cp33-abi3-manylinux_2_25_x86_64
    cp33-abi3-manylinux_2_24_x86_64
    cp33-abi3-manylinux_2_23_x86_64
    cp33-abi3-manylinux_2_22_x86_64
    cp33-abi3-manylinux_2_21_x86_64
    cp33-abi3-manylinux_2_20_x86_64
    cp33-abi3-manylinux_2_19_x86_64
    cp33-abi3-manylinux_2_18_x86_64
    cp33-abi3-manylinux_2_17_x86_64
    cp33-abi3-manylinux2014_x86_64
    cp33-abi3-manylinux_2_16_x86_64
    cp33-abi3-manylinux_2_15_x86_64
    cp33-abi3-manylinux_2_14_x86_64
    cp33-abi3-manylinux_2_13_x86_64
    cp33-abi3-manylinux_2_12_x86_64
    cp33-abi3-manylinux2010_x86_64
    cp33-abi3-manylinux_2_11_x86_64
    cp33-abi3-manylinux_2_10_x86_64
    cp33-abi3-manylinux_2_9_x86_64
    cp33-abi3-manylinux_2_8_x86_64
    cp33-abi3-manylinux_2_7_x86_64
    cp33-abi3-manylinux_2_6_x86_64
    cp33-abi3-manylinux_2_5_x86_64
    cp33-abi3-manylinux1_x86_64
    cp33-abi3-linux_x86_64
    cp32-abi3-manylinux_2_28_x86_64
    cp32-abi3-manylinux_2_27_x86_64
    cp32-abi3-manylinux_2_26_x86_64
    cp32-abi3-manylinux_2_25_x86_64
    cp32-abi3-manylinux_2_24_x86_64
    cp32-abi3-manylinux_2_23_x86_64
    cp32-abi3-manylinux_2_22_x86_64
    cp32-abi3-manylinux_2_21_x86_64
    cp32-abi3-manylinux_2_20_x86_64
    cp32-abi3-manylinux_2_19_x86_64
    cp32-abi3-manylinux_2_18_x86_64
    cp32-abi3-manylinux_2_17_x86_64
    cp32-abi3-manylinux2014_x86_64
    cp32-abi3-manylinux_2_16_x86_64
    cp32-abi3-manylinux_2_15_x86_64
    cp32-abi3-manylinux_2_14_x86_64
    cp32-abi3-manylinux_2_13_x86_64
    cp32-abi3-manylinux_2_12_x86_64
    cp32-abi3-manylinux2010_x86_64
    cp32-abi3-manylinux_2_11_x86_64
    cp32-abi3-manylinux_2_10_x86_64
    cp32-abi3-manylinux_2_9_x86_64
    cp32-abi3-manylinux_2_8_x86_64
    cp32-abi3-manylinux_2_7_x86_64
    cp32-abi3-manylinux_2_6_x86_64
    cp32-abi3-manylinux_2_5_x86_64
    cp32-abi3-manylinux1_x86_64
    cp32-abi3-linux_x86_64
    py39-none-manylinux_2_28_x86_64
    py39-none-manylinux_2_27_x86_64
    py39-none-manylinux_2_26_x86_64
    py39-none-manylinux_2_25_x86_64
    py39-none-manylinux_2_24_x86_64
    py39-none-manylinux_2_23_x86_64
    py39-none-manylinux_2_22_x86_64
    py39-none-manylinux_2_21_x86_64
    py39-none-manylinux_2_20_x86_64
    py39-none-manylinux_2_19_x86_64
    py39-none-manylinux_2_18_x86_64
    py39-none-manylinux_2_17_x86_64
    py39-none-manylinux2014_x86_64
    py39-none-manylinux_2_16_x86_64
    py39-none-manylinux_2_15_x86_64
    py39-none-manylinux_2_14_x86_64
    py39-none-manylinux_2_13_x86_64
    py39-none-manylinux_2_12_x86_64
    py39-none-manylinux2010_x86_64
    py39-none-manylinux_2_11_x86_64
    py39-none-manylinux_2_10_x86_64
    py39-none-manylinux_2_9_x86_64
    py39-none-manylinux_2_8_x86_64
    py39-none-manylinux_2_7_x86_64
    py39-none-manylinux_2_6_x86_64
    py39-none-manylinux_2_5_x86_64
    py39-none-manylinux1_x86_64
    py39-none-linux_x86_64
    py3-none-manylinux_2_28_x86_64
    py3-none-manylinux_2_27_x86_64
    py3-none-manylinux_2_26_x86_64
    py3-none-manylinux_2_25_x86_64
    py3-none-manylinux_2_24_x86_64
    py3-none-manylinux_2_23_x86_64
    py3-none-manylinux_2_22_x86_64
    py3-none-manylinux_2_21_x86_64
    py3-none-manylinux_2_20_x86_64
    py3-none-manylinux_2_19_x86_64
    py3-none-manylinux_2_18_x86_64
    py3-none-manylinux_2_17_x86_64
    py3-none-manylinux2014_x86_64
    py3-none-manylinux_2_16_x86_64
    py3-none-manylinux_2_15_x86_64
    py3-none-manylinux_2_14_x86_64
    py3-none-manylinux_2_13_x86_64
    py3-none-manylinux_2_12_x86_64
    py3-none-manylinux2010_x86_64
    py3-none-manylinux_2_11_x86_64
    py3-none-manylinux_2_10_x86_64
    py3-none-manylinux_2_9_x86_64
    py3-none-manylinux_2_8_x86_64
    py3-none-manylinux_2_7_x86_64
    py3-none-manylinux_2_6_x86_64
    py3-none-manylinux_2_5_x86_64
    py3-none-manylinux1_x86_64
    py3-none-linux_x86_64
    py38-none-manylinux_2_28_x86_64
    py38-none-manylinux_2_27_x86_64
    py38-none-manylinux_2_26_x86_64
    py38-none-manylinux_2_25_x86_64
    py38-none-manylinux_2_24_x86_64
    py38-none-manylinux_2_23_x86_64
    py38-none-manylinux_2_22_x86_64
    py38-none-manylinux_2_21_x86_64
    py38-none-manylinux_2_20_x86_64
    py38-none-manylinux_2_19_x86_64
    py38-none-manylinux_2_18_x86_64
    py38-none-manylinux_2_17_x86_64
    py38-none-manylinux2014_x86_64
    py38-none-manylinux_2_16_x86_64
    py38-none-manylinux_2_15_x86_64
    py38-none-manylinux_2_14_x86_64
    py38-none-manylinux_2_13_x86_64
    py38-none-manylinux_2_12_x86_64
    py38-none-manylinux2010_x86_64
    py38-none-manylinux_2_11_x86_64
    py38-none-manylinux_2_10_x86_64
    py38-none-manylinux_2_9_x86_64
    py38-none-manylinux_2_8_x86_64
    py38-none-manylinux_2_7_x86_64
    py38-none-manylinux_2_6_x86_64
    py38-none-manylinux_2_5_x86_64
    py38-none-manylinux1_x86_64
    py38-none-linux_x86_64
    py37-none-manylinux_2_28_x86_64
    py37-none-manylinux_2_27_x86_64
    py37-none-manylinux_2_26_x86_64
    py37-none-manylinux_2_25_x86_64
    py37-none-manylinux_2_24_x86_64
    py37-none-manylinux_2_23_x86_64
    py37-none-manylinux_2_22_x86_64
    py37-none-manylinux_2_21_x86_64
    py37-none-manylinux_2_20_x86_64
    py37-none-manylinux_2_19_x86_64
    py37-none-manylinux_2_18_x86_64
    py37-none-manylinux_2_17_x86_64
    py37-none-manylinux2014_x86_64
    py37-none-manylinux_2_16_x86_64
    py37-none-manylinux_2_15_x86_64
    py37-none-manylinux_2_14_x86_64
    py37-none-manylinux_2_13_x86_64
    py37-none-manylinux_2_12_x86_64
    py37-none-manylinux2010_x86_64
    py37-none-manylinux_2_11_x86_64
    py37-none-manylinux_2_10_x86_64
    py37-none-manylinux_2_9_x86_64
    py37-none-manylinux_2_8_x86_64
    py37-none-manylinux_2_7_x86_64
    py37-none-manylinux_2_6_x86_64
    py37-none-manylinux_2_5_x86_64
    py37-none-manylinux1_x86_64
    py37-none-linux_x86_64
    py36-none-manylinux_2_28_x86_64
    py36-none-manylinux_2_27_x86_64
    py36-none-manylinux_2_26_x86_64
    py36-none-manylinux_2_25_x86_64
    py36-none-manylinux_2_24_x86_64
    py36-none-manylinux_2_23_x86_64
    py36-none-manylinux_2_22_x86_64
    py36-none-manylinux_2_21_x86_64
    py36-none-manylinux_2_20_x86_64
    py36-none-manylinux_2_19_x86_64
    py36-none-manylinux_2_18_x86_64
    py36-none-manylinux_2_17_x86_64
    py36-none-manylinux2014_x86_64
    py36-none-manylinux_2_16_x86_64
    py36-none-manylinux_2_15_x86_64
    py36-none-manylinux_2_14_x86_64
    py36-none-manylinux_2_13_x86_64
    py36-none-manylinux_2_12_x86_64
    py36-none-manylinux2010_x86_64
    py36-none-manylinux_2_11_x86_64
    py36-none-manylinux_2_10_x86_64
    py36-none-manylinux_2_9_x86_64
    py36-none-manylinux_2_8_x86_64
    py36-none-manylinux_2_7_x86_64
    py36-none-manylinux_2_6_x86_64
    py36-none-manylinux_2_5_x86_64
    py36-none-manylinux1_x86_64
    py36-none-linux_x86_64
    py35-none-manylinux_2_28_x86_64
    py35-none-manylinux_2_27_x86_64
    py35-none-manylinux_2_26_x86_64
    py35-none-manylinux_2_25_x86_64
    py35-none-manylinux_2_24_x86_64
    py35-none-manylinux_2_23_x86_64
    py35-none-manylinux_2_22_x86_64
    py35-none-manylinux_2_21_x86_64
    py35-none-manylinux_2_20_x86_64
    py35-none-manylinux_2_19_x86_64
    py35-none-manylinux_2_18_x86_64
    py35-none-manylinux_2_17_x86_64
    py35-none-manylinux2014_x86_64
    py35-none-manylinux_2_16_x86_64
    py35-none-manylinux_2_15_x86_64
    py35-none-manylinux_2_14_x86_64
    py35-none-manylinux_2_13_x86_64
    py35-none-manylinux_2_12_x86_64
    py35-none-manylinux2010_x86_64
    py35-none-manylinux_2_11_x86_64
    py35-none-manylinux_2_10_x86_64
    py35-none-manylinux_2_9_x86_64
    py35-none-manylinux_2_8_x86_64
    py35-none-manylinux_2_7_x86_64
    py35-none-manylinux_2_6_x86_64
    py35-none-manylinux_2_5_x86_64
    py35-none-manylinux1_x86_64
    py35-none-linux_x86_64
    py34-none-manylinux_2_28_x86_64
    py34-none-manylinux_2_27_x86_64
    py34-none-manylinux_2_26_x86_64
    py34-none-manylinux_2_25_x86_64
    py34-none-manylinux_2_24_x86_64
    py34-none-manylinux_2_23_x86_64
    py34-none-manylinux_2_22_x86_64
    py34-none-manylinux_2_21_x86_64
    py34-none-manylinux_2_20_x86_64
    py34-none-manylinux_2_19_x86_64
    py34-none-manylinux_2_18_x86_64
    py34-none-manylinux_2_17_x86_64
    py34-none-manylinux2014_x86_64
    py34-none-manylinux_2_16_x86_64
    py34-none-manylinux_2_15_x86_64
    py34-none-manylinux_2_14_x86_64
    py34-none-manylinux_2_13_x86_64
    py34-none-manylinux_2_12_x86_64
    py34-none-manylinux2010_x86_64
    py34-none-manylinux_2_11_x86_64
    py34-none-manylinux_2_10_x86_64
    py34-none-manylinux_2_9_x86_64
    py34-none-manylinux_2_8_x86_64
    py34-none-manylinux_2_7_x86_64
    py34-none-manylinux_2_6_x86_64
    py34-none-manylinux_2_5_x86_64
    py34-none-manylinux1_x86_64
    py34-none-linux_x86_64
    py33-none-manylinux_2_28_x86_64
    py33-none-manylinux_2_27_x86_64
    py33-none-manylinux_2_26_x86_64
    py33-none-manylinux_2_25_x86_64
    py33-none-manylinux_2_24_x86_64
    py33-none-manylinux_2_23_x86_64
    py33-none-manylinux_2_22_x86_64
    py33-none-manylinux_2_21_x86_64
    py33-none-manylinux_2_20_x86_64
    py33-none-manylinux_2_19_x86_64
    py33-none-manylinux_2_18_x86_64
    py33-none-manylinux_2_17_x86_64
    py33-none-manylinux2014_x86_64
    py33-none-manylinux_2_16_x86_64
    py33-none-manylinux_2_15_x86_64
    py33-none-manylinux_2_14_x86_64
    py33-none-manylinux_2_13_x86_64
    py33-none-manylinux_2_12_x86_64
    py33-none-manylinux2010_x86_64
    py33-none-manylinux_2_11_x86_64
    py33-none-manylinux_2_10_x86_64
    py33-none-manylinux_2_9_x86_64
    py33-none-manylinux_2_8_x86_64
    py33-none-manylinux_2_7_x86_64
    py33-none-manylinux_2_6_x86_64
    py33-none-manylinux_2_5_x86_64
    py33-none-manylinux1_x86_64
    py33-none-linux_x86_64
    py32-none-manylinux_2_28_x86_64
    py32-none-manylinux_2_27_x86_64
    py32-none-manylinux_2_26_x86_64
    py32-none-manylinux_2_25_x86_64
    py32-none-manylinux_2_24_x86_64
    py32-none-manylinux_2_23_x86_64
    py32-none-manylinux_2_22_x86_64
    py32-none-manylinux_2_21_x86_64
    py32-none-manylinux_2_20_x86_64
    py32-none-manylinux_2_19_x86_64
    py32-none-manylinux_2_18_x86_64
    py32-none-manylinux_2_17_x86_64
    py32-none-manylinux2014_x86_64
    py32-none-manylinux_2_16_x86_64
    py32-none-manylinux_2_15_x86_64
    py32-none-manylinux_2_14_x86_64
    py32-none-manylinux_2_13_x86_64
    py32-none-manylinux_2_12_x86_64
    py32-none-manylinux2010_x86_64
    py32-none-manylinux_2_11_x86_64
    py32-none-manylinux_2_10_x86_64
    py32-none-manylinux_2_9_x86_64
    py32-none-manylinux_2_8_x86_64
    py32-none-manylinux_2_7_x86_64
    py32-none-manylinux_2_6_x86_64
    py32-none-manylinux_2_5_x86_64
    py32-none-manylinux1_x86_64
    py32-none-linux_x86_64
    py31-none-manylinux_2_28_x86_64
    py31-none-manylinux_2_27_x86_64
    py31-none-manylinux_2_26_x86_64
    py31-none-manylinux_2_25_x86_64
    py31-none-manylinux_2_24_x86_64
    py31-none-manylinux_2_23_x86_64
    py31-none-manylinux_2_22_x86_64
    py31-none-manylinux_2_21_x86_64
    py31-none-manylinux_2_20_x86_64
    py31-none-manylinux_2_19_x86_64
    py31-none-manylinux_2_18_x86_64
    py31-none-manylinux_2_17_x86_64
    py31-none-manylinux2014_x86_64
    py31-none-manylinux_2_16_x86_64
    py31-none-manylinux_2_15_x86_64
    py31-none-manylinux_2_14_x86_64
    py31-none-manylinux_2_13_x86_64
    py31-none-manylinux_2_12_x86_64
    py31-none-manylinux2010_x86_64
    py31-none-manylinux_2_11_x86_64
    py31-none-manylinux_2_10_x86_64
    py31-none-manylinux_2_9_x86_64
    py31-none-manylinux_2_8_x86_64
    py31-none-manylinux_2_7_x86_64
    py31-none-manylinux_2_6_x86_64
    py31-none-manylinux_2_5_x86_64
    py31-none-manylinux1_x86_64
    py31-none-linux_x86_64
    py30-none-manylinux_2_28_x86_64
    py30-none-manylinux_2_27_x86_64
    py30-none-manylinux_2_26_x86_64
    py30-none-manylinux_2_25_x86_64
    py30-none-manylinux_2_24_x86_64
    py30-none-manylinux_2_23_x86_64
    py30-none-manylinux_2_22_x86_64
    py30-none-manylinux_2_21_x86_64
    py30-none-manylinux_2_20_x86_64
    py30-none-manylinux_2_19_x86_64
    py30-none-manylinux_2_18_x86_64
    py30-none-manylinux_2_17_x86_64
    py30-none-manylinux2014_x86_64
    py30-none-manylinux_2_16_x86_64
    py30-none-manylinux_2_15_x86_64
    py30-none-manylinux_2_14_x86_64
    py30-none-manylinux_2_13_x86_64
    py30-none-manylinux_2_12_x86_64
    py30-none-manylinux2010_x86_64
    py30-none-manylinux_2_11_x86_64
    py30-none-manylinux_2_10_x86_64
    py30-none-manylinux_2_9_x86_64
    py30-none-manylinux_2_8_x86_64
    py30-none-manylinux_2_7_x86_64
    py30-none-manylinux_2_6_x86_64
    py30-none-manylinux_2_5_x86_64
    py30-none-manylinux1_x86_64
    py30-none-linux_x86_64
    cp39-none-any
    py39-none-any
    py3-none-any
    py38-none-any
    py37-none-any
    py36-none-any
    py35-none-any
    py34-none-any
    py33-none-any
    py32-none-any
    py31-none-any
    py30-none-any
    "###
    );
}

#[test]
fn test_system_tags_macos() {
    let tags = Tags::from_env(
        &Platform::new(
            Os::Macos {
                major: 14,
                minor: 0,
            },
            Arch::Aarch64,
        ),
        (3, 9),
        "cpython",
        (3, 9),
        false,
        false,
    )
    .unwrap();
    assert_snapshot!(
        tags,
        @r###"
    cp39-cp39-macosx_14_0_arm64
    cp39-cp39-macosx_14_0_universal2
    cp39-cp39-macosx_13_0_arm64
    cp39-cp39-macosx_13_0_universal2
    cp39-cp39-macosx_12_0_arm64
    cp39-cp39-macosx_12_0_universal2
    cp39-cp39-macosx_11_0_arm64
    cp39-cp39-macosx_11_0_universal2
    cp39-cp39-macosx_10_16_universal2
    cp39-cp39-macosx_10_15_universal2
    cp39-cp39-macosx_10_14_universal2
    cp39-cp39-macosx_10_13_universal2
    cp39-cp39-macosx_10_12_universal2
    cp39-cp39-macosx_10_11_universal2
    cp39-cp39-macosx_10_10_universal2
    cp39-cp39-macosx_10_9_universal2
    cp39-cp39-macosx_10_8_universal2
    cp39-cp39-macosx_10_7_universal2
    cp39-cp39-macosx_10_6_universal2
    cp39-cp39-macosx_10_5_universal2
    cp39-cp39-macosx_10_4_universal2
    cp39-abi3-macosx_14_0_arm64
    cp39-abi3-macosx_14_0_universal2
    cp39-abi3-macosx_13_0_arm64
    cp39-abi3-macosx_13_0_universal2
    cp39-abi3-macosx_12_0_arm64
    cp39-abi3-macosx_12_0_universal2
    cp39-abi3-macosx_11_0_arm64
    cp39-abi3-macosx_11_0_universal2
    cp39-abi3-macosx_10_16_universal2
    cp39-abi3-macosx_10_15_universal2
    cp39-abi3-macosx_10_14_universal2
    cp39-abi3-macosx_10_13_universal2
    cp39-abi3-macosx_10_12_universal2
    cp39-abi3-macosx_10_11_universal2
    cp39-abi3-macosx_10_10_universal2
    cp39-abi3-macosx_10_9_universal2
    cp39-abi3-macosx_10_8_universal2
    cp39-abi3-macosx_10_7_universal2
    cp39-abi3-macosx_10_6_universal2
    cp39-abi3-macosx_10_5_universal2
    cp39-abi3-macosx_10_4_universal2
    cp39-none-macosx_14_0_arm64
    cp39-none-macosx_14_0_universal2
    cp39-none-macosx_13_0_arm64
    cp39-none-macosx_13_0_universal2
    cp39-none-macosx_12_0_arm64
    cp39-none-macosx_12_0_universal2
    cp39-none-macosx_11_0_arm64
    cp39-none-macosx_11_0_universal2
    cp39-none-macosx_10_16_universal2
    cp39-none-macosx_10_15_universal2
    cp39-none-macosx_10_14_universal2
    cp39-none-macosx_10_13_universal2
    cp39-none-macosx_10_12_universal2
    cp39-none-macosx_10_11_universal2
    cp39-none-macosx_10_10_universal2
    cp39-none-macosx_10_9_universal2
    cp39-none-macosx_10_8_universal2
    cp39-none-macosx_10_7_universal2
    cp39-none-macosx_10_6_universal2
    cp39-none-macosx_10_5_universal2
    cp39-none-macosx_10_4_universal2
    cp38-abi3-macosx_14_0_arm64
    cp38-abi3-macosx_14_0_universal2
    cp38-abi3-macosx_13_0_arm64
    cp38-abi3-macosx_13_0_universal2
    cp38-abi3-macosx_12_0_arm64
    cp38-abi3-macosx_12_0_universal2
    cp38-abi3-macosx_11_0_arm64
    cp38-abi3-macosx_11_0_universal2
    cp38-abi3-macosx_10_16_universal2
    cp38-abi3-macosx_10_15_universal2
    cp38-abi3-macosx_10_14_universal2
    cp38-abi3-macosx_10_13_universal2
    cp38-abi3-macosx_10_12_universal2
    cp38-abi3-macosx_10_11_universal2
    cp38-abi3-macosx_10_10_universal2
    cp38-abi3-macosx_10_9_universal2
    cp38-abi3-macosx_10_8_universal2
    cp38-abi3-macosx_10_7_universal2
    cp38-abi3-macosx_10_6_universal2
    cp38-abi3-macosx_10_5_universal2
    cp38-abi3-macosx_10_4_universal2
    cp37-abi3-macosx_14_0_arm64
    cp37-abi3-macosx_14_0_universal2
    cp37-abi3-macosx_13_0_arm64
    cp37-abi3-macosx_13_0_universal2
    cp37-abi3-macosx_12_0_arm64
    cp37-abi3-macosx_12_0_universal2
    cp37-abi3-macosx_11_0_arm64
    cp37-abi3-macosx_11_0_universal2
    cp37-abi3-macosx_10_16_universal2
    cp37-abi3-macosx_10_15_universal2
    cp37-abi3-macosx_10_14_universal2
    cp37-abi3-macosx_10_13_universal2
    cp37-abi3-macosx_10_12_universal2
    cp37-abi3-macosx_10_11_universal2
    cp37-abi3-macosx_10_10_universal2
    cp37-abi3-macosx_10_9_universal2
    cp37-abi3-macosx_10_8_universal2
    cp37-abi3-macosx_10_7_universal2
    cp37-abi3-macosx_10_6_universal2
    cp37-abi3-macosx_10_5_universal2
    cp37-abi3-macosx_10_4_universal2
    cp36-abi3-macosx_14_0_arm64
    cp36-abi3-macosx_14_0_universal2
    cp36-abi3-macosx_13_0_arm64
    cp36-abi3-macosx_13_0_universal2
    cp36-abi3-macosx_12_0_arm64
    cp36-abi3-macosx_12_0_universal2
    cp36-abi3-macosx_11_0_arm64
    cp36-abi3-macosx_11_0_universal2
    cp36-abi3-macosx_10_16_universal2
    cp36-abi3-macosx_10_15_universal2
    cp36-abi3-macosx_10_14_universal2
    cp36-abi3-macosx_10_13_universal2
    cp36-abi3-macosx_10_12_universal2
    cp36-abi3-macosx_10_11_universal2
    cp36-abi3-macosx_10_10_universal2
    cp36-abi3-macosx_10_9_universal2
    cp36-abi3-macosx_10_8_universal2
    cp36-abi3-macosx_10_7_universal2
    cp36-abi3-macosx_10_6_universal2
    cp36-abi3-macosx_10_5_universal2
    cp36-abi3-macosx_10_4_universal2
    cp35-abi3-macosx_14_0_arm64
    cp35-abi3-macosx_14_0_universal2
    cp35-abi3-macosx_13_0_arm64
    cp35-abi3-macosx_13_0_universal2
    cp35-abi3-macosx_12_0_arm64
    cp35-abi3-macosx_12_0_universal2
    cp35-abi3-macosx_11_0_arm64
    cp35-abi3-macosx_11_0_universal2
    cp35-abi3-macosx_10_16_universal2
    cp35-abi3-macosx_10_15_universal2
    cp35-abi3-macosx_10_14_universal2
    cp35-abi3-macosx_10_13_universal2
    cp35-abi3-macosx_10_12_universal2
    cp35-abi3-macosx_10_11_universal2
    cp35-abi3-macosx_10_10_universal2
    cp35-abi3-macosx_10_9_universal2
    cp35-abi3-macosx_10_8_universal2
    cp35-abi3-macosx_10_7_universal2
    cp35-abi3-macosx_10_6_universal2
    cp35-abi3-macosx_10_5_universal2
    cp35-abi3-macosx_10_4_universal2
    cp34-abi3-macosx_14_0_arm64
    cp34-abi3-macosx_14_0_universal2
    cp34-abi3-macosx_13_0_arm64
    cp34-abi3-macosx_13_0_universal2
    cp34-abi3-macosx_12_0_arm64
    cp34-abi3-macosx_12_0_universal2
    cp34-abi3-macosx_11_0_arm64
    cp34-abi3-macosx_11_0_universal2
    cp34-abi3-macosx_10_16_universal2
    cp34-abi3-macosx_10_15_universal2
    cp34-abi3-macosx_10_14_universal2
    cp34-abi3-macosx_10_13_universal2
    cp34-abi3-macosx_10_12_universal2
    cp34-abi3-macosx_10_11_universal2
    cp34-abi3-macosx_10_10_universal2
    cp34-abi3-macosx_10_9_universal2
    cp34-abi3-macosx_10_8_universal2
    cp34-abi3-macosx_10_7_universal2
    cp34-abi3-macosx_10_6_universal2
    cp34-abi3-macosx_10_5_universal2
    cp34-abi3-macosx_10_4_universal2
    cp33-abi3-macosx_14_0_arm64
    cp33-abi3-macosx_14_0_universal2
    cp33-abi3-macosx_13_0_arm64
    cp33-abi3-macosx_13_0_universal2
    cp33-abi3-macosx_12_0_arm64
    cp33-abi3-macosx_12_0_universal2
    cp33-abi3-macosx_11_0_arm64
    cp33-abi3-macosx_11_0_universal2
    cp33-abi3-macosx_10_16_universal2
    cp33-abi3-macosx_10_15_universal2
    cp33-abi3-macosx_10_14_universal2
    cp33-abi3-macosx_10_13_universal2
    cp33-abi3-macosx_10_12_universal2
    cp33-abi3-macosx_10_11_universal2
    cp33-abi3-macosx_10_10_universal2
    cp33-abi3-macosx_10_9_universal2
    cp33-abi3-macosx_10_8_universal2
    cp33-abi3-macosx_10_7_universal2
    cp33-abi3-macosx_10_6_universal2
    cp33-abi3-macosx_10_5_universal2
    cp33-abi3-macosx_10_4_universal2
    cp32-abi3-macosx_14_0_arm64
    cp32-abi3-macosx_14_0_universal2
    cp32-abi3-macosx_13_0_arm64
    cp32-abi3-macosx_13_0_universal2
    cp32-abi3-macosx_12_0_arm64
    cp32-abi3-macosx_12_0_universal2
    cp32-abi3-macosx_11_0_arm64
    cp32-abi3-macosx_11_0_universal2
    cp32-abi3-macosx_10_16_universal2
    cp32-abi3-macosx_10_15_universal2
    cp32-abi3-macosx_10_14_universal2
    cp32-abi3-macosx_10_13_universal2
    cp32-abi3-macosx_10_12_universal2
    cp32-abi3-macosx_10_11_universal2
    cp32-abi3-macosx_10_10_universal2
    cp32-abi3-macosx_10_9_universal2
    cp32-abi3-macosx_10_8_universal2
    cp32-abi3-macosx_10_7_universal2
    cp32-abi3-macosx_10_6_universal2
    cp32-abi3-macosx_10_5_universal2
    cp32-abi3-macosx_10_4_universal2
    py39-none-macosx_14_0_arm64
    py39-none-macosx_14_0_universal2
    py39-none-macosx_13_0_arm64
    py39-none-macosx_13_0_universal2
    py39-none-macosx_12_0_arm64
    py39-none-macosx_12_0_universal2
    py39-none-macosx_11_0_arm64
    py39-none-macosx_11_0_universal2
    py39-none-macosx_10_16_universal2
    py39-none-macosx_10_15_universal2
    py39-none-macosx_10_14_universal2
    py39-none-macosx_10_13_universal2
    py39-none-macosx_10_12_universal2
    py39-none-macosx_10_11_universal2
    py39-none-macosx_10_10_universal2
    py39-none-macosx_10_9_universal2
    py39-none-macosx_10_8_universal2
    py39-none-macosx_10_7_universal2
    py39-none-macosx_10_6_universal2
    py39-none-macosx_10_5_universal2
    py39-none-macosx_10_4_universal2
    py3-none-macosx_14_0_arm64
    py3-none-macosx_14_0_universal2
    py3-none-macosx_13_0_arm64
    py3-none-macosx_13_0_universal2
    py3-none-macosx_12_0_arm64
    py3-none-macosx_12_0_universal2
    py3-none-macosx_11_0_arm64
    py3-none-macosx_11_0_universal2
    py3-none-macosx_10_16_universal2
    py3-none-macosx_10_15_universal2
    py3-none-macosx_10_14_universal2
    py3-none-macosx_10_13_universal2
    py3-none-macosx_10_12_universal2
    py3-none-macosx_10_11_universal2
    py3-none-macosx_10_10_universal2
    py3-none-macosx_10_9_universal2
    py3-none-macosx_10_8_universal2
    py3-none-macosx_10_7_universal2
    py3-none-macosx_10_6_universal2
    py3-none-macosx_10_5_universal2
    py3-none-macosx_10_4_universal2
    py38-none-macosx_14_0_arm64
    py38-none-macosx_14_0_universal2
    py38-none-macosx_13_0_arm64
    py38-none-macosx_13_0_universal2
    py38-none-macosx_12_0_arm64
    py38-none-macosx_12_0_universal2
    py38-none-macosx_11_0_arm64
    py38-none-macosx_11_0_universal2
    py38-none-macosx_10_16_universal2
    py38-none-macosx_10_15_universal2
    py38-none-macosx_10_14_universal2
    py38-none-macosx_10_13_universal2
    py38-none-macosx_10_12_universal2
    py38-none-macosx_10_11_universal2
    py38-none-macosx_10_10_universal2
    py38-none-macosx_10_9_universal2
    py38-none-macosx_10_8_universal2
    py38-none-macosx_10_7_universal2
    py38-none-macosx_10_6_universal2
    py38-none-macosx_10_5_universal2
    py38-none-macosx_10_4_universal2
    py37-none-macosx_14_0_arm64
    py37-none-macosx_14_0_universal2
    py37-none-macosx_13_0_arm64
    py37-none-macosx_13_0_universal2
    py37-none-macosx_12_0_arm64
    py37-none-macosx_12_0_universal2
    py37-none-macosx_11_0_arm64
    py37-none-macosx_11_0_universal2
    py37-none-macosx_10_16_universal2
    py37-none-macosx_10_15_universal2
    py37-none-macosx_10_14_universal2
    py37-none-macosx_10_13_universal2
    py37-none-macosx_10_12_universal2
    py37-none-macosx_10_11_universal2
    py37-none-macosx_10_10_universal2
    py37-none-macosx_10_9_universal2
    py37-none-macosx_10_8_universal2
    py37-none-macosx_10_7_universal2
    py37-none-macosx_10_6_universal2
    py37-none-macosx_10_5_universal2
    py37-none-macosx_10_4_universal2
    py36-none-macosx_14_0_arm64
    py36-none-macosx_14_0_universal2
    py36-none-macosx_13_0_arm64
    py36-none-macosx_13_0_universal2
    py36-none-macosx_12_0_arm64
    py36-none-macosx_12_0_universal2
    py36-none-macosx_11_0_arm64
    py36-none-macosx_11_0_universal2
    py36-none-macosx_10_16_universal2
    py36-none-macosx_10_15_universal2
    py36-none-macosx_10_14_universal2
    py36-none-macosx_10_13_universal2
    py36-none-macosx_10_12_universal2
    py36-none-macosx_10_11_universal2
    py36-none-macosx_10_10_universal2
    py36-none-macosx_10_9_universal2
    py36-none-macosx_10_8_universal2
    py36-none-macosx_10_7_universal2
    py36-none-macosx_10_6_universal2
    py36-none-macosx_10_5_universal2
    py36-none-macosx_10_4_universal2
    py35-none-macosx_14_0_arm64
    py35-none-macosx_14_0_universal2
    py35-none-macosx_13_0_arm64
    py35-none-macosx_13_0_universal2
    py35-none-macosx_12_0_arm64
    py35-none-macosx_12_0_universal2
    py35-none-macosx_11_0_arm64
    py35-none-macosx_11_0_universal2
    py35-none-macosx_10_16_universal2
    py35-none-macosx_10_15_universal2
    py35-none-macosx_10_14_universal2
    py35-none-macosx_10_13_universal2
    py35-none-macosx_10_12_universal2
    py35-none-macosx_10_11_universal2
    py35-none-macosx_10_10_universal2
    py35-none-macosx_10_9_universal2
    py35-none-macosx_10_8_universal2
    py35-none-macosx_10_7_universal2
    py35-none-macosx_10_6_universal2
    py35-none-macosx_10_5_universal2
    py35-none-macosx_10_4_universal2
    py34-none-macosx_14_0_arm64
    py34-none-macosx_14_0_universal2
    py34-none-macosx_13_0_arm64
    py34-none-macosx_13_0_universal2
    py34-none-macosx_12_0_arm64
    py34-none-macosx_12_0_universal2
    py34-none-macosx_11_0_arm64
    py34-none-macosx_11_0_universal2
    py34-none-macosx_10_16_universal2
    py34-none-macosx_10_15_universal2
    py34-none-macosx_10_14_universal2
    py34-none-macosx_10_13_universal2
    py34-none-macosx_10_12_universal2
    py34-none-macosx_10_11_universal2
    py34-none-macosx_10_10_universal2
    py34-none-macosx_10_9_universal2
    py34-none-macosx_10_8_universal2
    py34-none-macosx_10_7_universal2
    py34-none-macosx_10_6_universal2
    py34-none-macosx_10_5_universal2
    py34-none-macosx_10_4_universal2
    py33-none-macosx_14_0_arm64
    py33-none-macosx_14_0_universal2
    py33-none-macosx_13_0_arm64
    py33-none-macosx_13_0_universal2
    py33-none-macosx_12_0_arm64
    py33-none-macosx_12_0_universal2
    py33-none-macosx_11_0_arm64
    py33-none-macosx_11_0_universal2
    py33-none-macosx_10_16_universal2
    py33-none-macosx_10_15_universal2
    py33-none-macosx_10_14_universal2
    py33-none-macosx_10_13_universal2
    py33-none-macosx_10_12_universal2
    py33-none-macosx_10_11_universal2
    py33-none-macosx_10_10_universal2
    py33-none-macosx_10_9_universal2
    py33-none-macosx_10_8_universal2
    py33-none-macosx_10_7_universal2
    py33-none-macosx_10_6_universal2
    py33-none-macosx_10_5_universal2
    py33-none-macosx_10_4_universal2
    py32-none-macosx_14_0_arm64
    py32-none-macosx_14_0_universal2
    py32-none-macosx_13_0_arm64
    py32-none-macosx_13_0_universal2
    py32-none-macosx_12_0_arm64
    py32-none-macosx_12_0_universal2
    py32-none-macosx_11_0_arm64
    py32-none-macosx_11_0_universal2
    py32-none-macosx_10_16_universal2
    py32-none-macosx_10_15_universal2
    py32-none-macosx_10_14_universal2
    py32-none-macosx_10_13_universal2
    py32-none-macosx_10_12_universal2
    py32-none-macosx_10_11_universal2
    py32-none-macosx_10_10_universal2
    py32-none-macosx_10_9_universal2
    py32-none-macosx_10_8_universal2
    py32-none-macosx_10_7_universal2
    py32-none-macosx_10_6_universal2
    py32-none-macosx_10_5_universal2
    py32-none-macosx_10_4_universal2
    py31-none-macosx_14_0_arm64
    py31-none-macosx_14_0_universal2
    py31-none-macosx_13_0_arm64
    py31-none-macosx_13_0_universal2
    py31-none-macosx_12_0_arm64
    py31-none-macosx_12_0_universal2
    py31-none-macosx_11_0_arm64
    py31-none-macosx_11_0_universal2
    py31-none-macosx_10_16_universal2
    py31-none-macosx_10_15_universal2
    py31-none-macosx_10_14_universal2
    py31-none-macosx_10_13_universal2
    py31-none-macosx_10_12_universal2
    py31-none-macosx_10_11_universal2
    py31-none-macosx_10_10_universal2
    py31-none-macosx_10_9_universal2
    py31-none-macosx_10_8_universal2
    py31-none-macosx_10_7_universal2
    py31-none-macosx_10_6_universal2
    py31-none-macosx_10_5_universal2
    py31-none-macosx_10_4_universal2
    py30-none-macosx_14_0_arm64
    py30-none-macosx_14_0_universal2
    py30-none-macosx_13_0_arm64
    py30-none-macosx_13_0_universal2
    py30-none-macosx_12_0_arm64
    py30-none-macosx_12_0_universal2
    py30-none-macosx_11_0_arm64
    py30-none-macosx_11_0_universal2
    py30-none-macosx_10_16_universal2
    py30-none-macosx_10_15_universal2
    py30-none-macosx_10_14_universal2
    py30-none-macosx_10_13_universal2
    py30-none-macosx_10_12_universal2
    py30-none-macosx_10_11_universal2
    py30-none-macosx_10_10_universal2
    py30-none-macosx_10_9_universal2
    py30-none-macosx_10_8_universal2
    py30-none-macosx_10_7_universal2
    py30-none-macosx_10_6_universal2
    py30-none-macosx_10_5_universal2
    py30-none-macosx_10_4_universal2
    cp39-none-any
    py39-none-any
    py3-none-any
    py38-none-any
    py37-none-any
    py36-none-any
    py35-none-any
    py34-none-any
    py33-none-any
    py32-none-any
    py31-none-any
    py30-none-any
    "###
    );
}
