//
// Copyright (c) Pirmin Kalberer. All rights reserved.
// Licensed under the MIT License. See LICENSE file in the project root for full license information.
//

use crate::core::feature::FeatureAttrValType;
use crate::core::feature::{Feature, FeatureAttr};
use crate::core::geom;
use crate::core::geom::{GeometryType, Point};
use crate::core::layer::Layer;
use crate::core::screen;
use crate::mvt::geom_encoder::EncodableGeom;
use crate::mvt::tile::{ScreenGeom, Tile};
use crate::mvt::vector_tile;
use std::fs::File;
use tile_grid::Extent;

#[test]
fn test_point_to_screen_coords() {
    let zh_mercator = geom::Point::new(960000.0, 6002729.0, Some(3857));
    //let zh_wgs84 = postgis::Point::new(47.3703149, 8.5285874, Some(4326));
    let extent = Extent {
        minx: 958826.08,
        miny: 5987771.04,
        maxx: 978393.96,
        maxy: 6007338.92,
    };
    let mut tile = Tile::new(&extent, true);
    let mut layer = Layer::new("points");
    layer.tile_size = 256;
    let _ = tile.new_layer(&layer);

    let screen_pt = tile.point(&zh_mercator);
    assert_eq!(screen_pt, screen::Point { x: 15, y: 61 });
    assert_eq!(screen_pt.encode().vec(), &[9, 30, 122]);
    assert_eq!(
        tile.point(&geom::Point::new(extent.minx, extent.maxy, Some(3857))),
        screen::Point { x: 0, y: 0 }
    );
    assert_eq!(
        tile.point(&geom::Point::new(extent.maxx, extent.miny, Some(3857))),
        screen::Point { x: 256, y: 256 }
    );
    assert_eq!(
        tile.point(&geom::Point::new(
            extent.minx - 100.,
            extent.miny - 100.,
            Some(3857)
        )),
        screen::Point { x: -1, y: 257 }
    );
    assert_eq!(
        tile.point(&geom::Point::new(
            extent.maxx + 100.,
            extent.maxy + 100.,
            Some(3857)
        )),
        screen::Point { x: 257, y: -1 }
    );
    assert_eq!(
        tile.point(&geom::Point::new(f64::MAX, f64::MIN, Some(3857))),
        screen::Point {
            x: i32::MAX,
            y: i32::MAX
        }
    );
    assert_eq!(
        tile.point(&geom::Point::new(f64::MIN, f64::MAX, Some(3857))),
        screen::Point {
            x: i32::MIN,
            y: i32::MIN + 257
        }
    );
}

#[test]
fn test_clipped_polygon() {
    use postgis::ewkb::{self, EwkbRead};
    use tile_grid::Grid;

    fn hex_to_vec(hexstr: &str) -> Vec<u8> {
        hexstr
            .as_bytes()
            .chunks(2)
            .map(|chars| {
                let hb = if chars[0] <= 57 {
                    chars[0] - 48
                } else {
                    chars[0] - 55
                };
                let lb = if chars[1] <= 57 {
                    chars[1] - 48
                } else {
                    chars[1] - 55
                };
                hb * 16 + lb
            })
            .collect::<Vec<_>>()
    }

    fn poly_to_wkt(rings: &Vec<screen::LineString>) -> String {
        format!(
            "POLYGON(({}))",
            rings[0]
                .points
                .iter()
                .map(|p| format!("{} {}", p.x, p.y))
                .collect::<Vec<_>>()
                .join(",")
        )
    }

    let tile_extent = Grid::web_mercator().tile_extent_xyz(68, 40, 7);
    let mut tile = Tile::new(&tile_extent, false);
    let mut layer = Layer::new("");
    layer.tile_size = 512;
    let _ = tile.new_layer(&layer);

    // v0.14
    // WITH t AS (
    //   SELECT ST_GeomFromEWKT('SRID=3857;POLYGON((1376278.48 7231699.26,1375968.72 7231083.12,1375725.77 7230672.73,1375248.86 7229798.640000001,1374856.24 7229145.29,1374449.04 7228521.47,1374046.56 7227878.7,1373424.99 7226879.94,1373104.05 7226485.36,1372554.57 7225902.890000001,1371800.6500000001 7225205.61,1371399.86 7224893.53,1370174.55 7223920.13,1369766.53 7223656.16,1369197.29 7223296.8,1367858.6500000001 7222560.62,1367623.7 7222453.33,1367300.59 7222303.65,1366718.26 7222030.11,1366292.19 7221788.73,1365218.5 7221039.01,1364301.42 7220401.890000001,1362927.33 7219571.890000001,1362731.97 7219461.65,1361693.27 7218875.38,1360735.8900000001 7218391.3100000005,1360091.6500000001 7218004.23,1358722.91 7217074.41,1358288.85 7216761.08,1357698 7216309.350000001,1357412.51 7216069.5600000005,1356819.82 7215458.73,1355795.3800000001 7214248.8,1355367.8900000001 7213687.24,1354775.25 7212688.2,1354383.61 7211970.2700000005,1354060.24 7211326.07,1353751.22 7210710.66,1353546.51 7210332.29,1352930.1 7209241.75,1352348.74 7208338.5600000005,1351811.61 7207501.09,1351279.33 7206800.3,1350894.3 7206354.29,1349890.1 7205429.91,1349364.77 7205031.45,1348717.43 7204586.11,1348021.05 7204155.49,1347722.96 7204086.86,1347262.32 7204146.48,1347034.1400000001 7204235.18,1346752.09 7204434.390000001,1346552.6300000001 7204590.8,1346392.87 7204770.26,1346151.49 7204697.67,1346126.92 7204690.600000001,1346102.35 7204683.54,1345905.35 7204613.4,1345658.01 7204526.5,1345477.9000000001 7205186.51,1344732.72 7204700.25,1344271.47 7204414.59,1343865.32 7204233.67,1342492.02 7203938.38,1341069.97 7203751.53,1339556.8 7203687.63,1338968.07 7203735.8,1338440.91 7203794.600000001,1337756.99 7203938.61,1337378.3900000001 7204011.350000001,1337093.9000000001 7204018.93,1336758.03 7203931.88,1335684.8900000001 7203461.87,1335135.22 7203251.01,1332953.31 7202294.18,1331657.92 7201765.01,1331101.61 7201637.75,1330363.23 7201522.59,1330008.84 7201437.36,1329254.33 7201192.63,1328305.52 7200854.29,1327552.26 7200580.37,1326562.09 7200210.21,1324656.3900000001 7199348.66,1323806.78 7199009.69,1322709.54 7198632.640000001,1321195.57 7198188.140000001,1320132.42 7197866.83,1319587.96 7197736.57,1319186.62 7197675.69,1318792.56 7197640.26,1318139.96 7197623.2,1317575.26 7197662.5200000005,1316204.52 7197771.0200000005,1315437.1400000001 7197849.69,1314689.82 7197926.24,1312806.92 7198260.100000001,1312098.36 7198453.5200000005,1311597.36 7198634.72,1311191.46 7198800.57,1310789.59 7199003.9,1310343.61 7199218.22,1309870.35 7199433,1309441.91 7199580.390000001,1309171.6400000001 7199614.07,1308675.59 7199593.55,1307326.1300000001 7199372.390000001,1306429.3900000001 7199327.79,1305440.2 7199356.350000001,1303849.48 7199370.850000001,1302694.36 7199461.57,1301941.66 7199473.88,1300855.53 7199407.850000001,1300454.21 7199344.33,1300164.1 7199203.88,1299822.6400000001 7198941.32,1299079.36 7198149.15,1298399.72 7197388.41,1297338.93 7196201.890000001,1296571.06 7195329.94,1295328.83 7193934.74,1294755.56 7193293.57,1294235.8800000001 7192590.18,1293777.74 7192010.4,1293349.74 7191506.93,1292795.4000000001 7190971.2700000005,1292143.3900000001 7190385.93,1291496.1300000001 7189884.0600000005,1290914.07 7189527.850000001,1290451.1500000001 7189287.5,1290024.84 7189104.57,1289386.6500000001 7188972.48,1288280.42 7188941,1287902 7188888.55,1287584.44 7188809.62,1287146.31 7188626.72,1286849.3800000001 7188443.46,1286324.76 7188038.42,1286007.36 7187729.28,1285851.61 7187542.28,1285244.26 7186693.48,1285115.1300000001 7186461.79,1284977.36 7186219.99,1284793.69 7185978.86,1283962.73 7185053.2,1282973.37 7183848.600000001,1282927.19 7183706.11,1282934.17 7183596.95,1282941.04 7183479.28,1283048.56 7183387.47,1282802.5 7183174.51,1282658.79 7183231.04,1282551.21 7183196.75,1282451.27 7183099.32,1281504.86 7181028.83,1281428.11 7180760.75,1281398.1500000001 7180435.82,1281368.81 7179039.61,1281494.03 7177641.5,1281536.46 7177514.91,1281635.43 7177423.3,1281742.29 7177411.61,1281868.61 7177455.83,1282318.24 7176187.98,1282217.75 7176044.67,1282143.95 7175866.94,1282039.12 7175408.76,1282026.33 7175219.98,1282078.67 7175074.5600000005,1282202.81 7174963.9,1282400.86 7174916.890000001,1283217.56 7174905.49,1283782.58 7174870.32,1283917.53 7174805.44,1283997.43 7174690.2700000005,1284005.1300000001 7174513.140000001,1283972.26 7174227.65,1283896.3 7173896.83,1283812.9000000001 7173761.84,1283650.12 7173656.91,1283180.87 7173491.600000001,1282413.71 7173269.15,1282133.36 7173131.79,1281883.12 7172961.67,1281401.8900000001 7172536.07,1281225.76 7172320.68,1281106.46 7172162.37,1281022.75 7172000.17,1280921.29 7171780.350000001,1280834.01 7171356.17,1280788.42 7171125.41,1280702.67 7170936.0200000005,1280247.1400000001 7170265.13,1280161.44 7170079.17,1279911.43 7169414.03,1279822.71 7169253.640000001,1279691.03 7169178.88,1279393.22 7169118.28,1279264.57 7169016.26,1278939.6400000001 7169968.140000001,1279077.96 7170030.92,1279177.96 7170146.95,1279270.69 7170353.25,1279675.01 7172391.22,1279782.83 7173082.37,1279787.11 7173398.850000001,1279704.79 7173966.72,1279635.92 7174148.07,1279502.03 7174292.850000001,1279325.11 7174395.67,1278868.3800000001 7174527.79,1278177.45 7174671.54,1277210.09 7174846.140000001,1276850.49 7174878.140000001,1276500.49 7174865.73,1275833.6 7174774.07,1275321.24 7174682.04,1245040.3900000001 7172903.5200000005,1240772.3900000001 7187705.07,1240703.6400000001 7187943.23,1240634.82 7188181.68,1240566.1500000001 7188419.57,1240497.4000000001 7188657.75,1240428.6400000001 7188895.94,1239895.73 7190739.29,1234976.42 7207758.55,1235010.18 7207927.46,1235011.3900000001 7208077.84,1234922.31 7208449.41,1234753.74 7209327.55,1234701.62 7210695.45,1234578.24 7211900.0200000005,1234438.28 7212744.09,1234375.85 7213058.44,1234070.85 7214593.95,1233926.47 7215323.72,1233780.74 7216315.3,1233726.74 7216970.38,1233698.19 7217316.75,1233686.11 7218379.36,1233954.1500000001 7222685.16,1233969.01 7223051.41,1233953.09 7223412.79,1233893.44 7223861.86,1233796.6300000001 7224364.32,1233168.3 7227025.55,1233074.98 7227548.73,1232904.59 7228709.84,1232788.46 7232232.19,1232774.37 7232619.67,1232487.68 7234903.61,1232543.52 7235226.12,1232615.91 7235587.08,1232689.98 7235799.91,1232764.59 7235891.91,1232890.56 7235987.62,1232940.2 7236036.61,1232965.1500000001 7236137.21,1233185.98 7236206.86,1233270.76 7236458.79,1233468.75 7236911.93,1233640.96 7237289.51,1233971.74 7237669.890000001,1234122.07 7237825.09,1234494.99 7238207.7,1234967.75 7238645.3,1235167.1400000001 7238780.76,1235929.11 7239144.63,1236114.74 7239266.47,1236266.41 7239416.01,1236460.18 7239695.42,1236625.8800000001 7239885.97,1237149.3 7240327.16,1237472.61 7240550.66,1237637.31 7240617.82,1237877.25 7240684.350000001,1237980.24 7240731.46,1238069.72 7240799.25,1238421.54 7241152.74,1238532.1300000001 7241288.91,1238678.79 7241644.15,1238770.92 7242020.44,1238892.53 7242643.34,1238941.76 7242800.62,1239060.46 7243080.74,1239109.99 7243272.3100000005,1239146.42 7243532.57,1239150.41 7243882.58,1243547.6400000001 7246279.47,1243972.8800000001 7246512.51,1244229.32 7246642.350000001,1244463.91 7246817.03,1244609.09 7246956.48,1245285.33 7247827.640000001,1245325.3800000001 7247895.95,1246345.9000000001 7247829.67,1246472.54 7247814.73,1246616.91 7247861.44,1246860.28 7248049.73,1247011.91 7248139.3,1248004.9000000001 7248414.8100000005,1248684.33 7248507.8,1249600.1 7248586.390000001,1251231.78 7248630.2,1251951.12 7248597.15,1253376.76 7248266.5200000005,1254284.6 7247910.18,1254727.24 7247639.37,1255032.37 7247491.91,1257154.83 7246988.44,1259102.59 7246666.66,1259282.85 7246700.72,1259349.99 7246730.890000001,1282171.35 7249567.38,1284434.18 7247587.94,1290031.33 7241108.84,1296246.95 7233043.98,1298736.96 7233079.43,1308609.5 7233213.850000001,1309064.09 7233219.75,1309518.69 7233225.63,1309973.29 7233231.48,1310427.8800000001 7233237.3100000005,1330447.6500000001 7233468.78,1336280.54 7236051.140000001,1337635.67 7236650.37,1345865.5 7240283.82,1351003.31 7242547.12,1352804.74 7243339.76,1354227.19 7243339.79,1364617.08 7255935.07,1366011.75 7259425.41,1389790.43 7262043.53,1393281.54 7262422.2,1393015.71 7262307.32,1392726.67 7262096.71,1392431 7261831.22,1392054.72 7261378.65,1391713.74 7260833.94,1391257.6400000001 7259857.49,1390835.44 7258953.49,1390099.67 7257253.54,1389188.41 7254757.41,1388914.54 7253907.8,1388335.4000000001 7252110.86,1388096.77 7251266.58,1387851.9000000001 7250572.04,1387497.1500000001 7249741.32,1386741.06 7248253.12,1386231.58 7247395.79,1385798.23 7246821.61,1385459.96 7246406.37,1385264.86 7246155.65,1383830.3 7244311.48,1383716.42 7244214.890000001,1383379.02 7243955.92,1382897.96 7243573.65,1381359.04 7242290.26,1381107.94 7242051.33,1380874.67 7241772.45,1380714.6 7241476.23,1380431.7 7240865.68,1379873.7 7239610.17,1379308.86 7238348.13,1379021.87 7237770.48,1378710.42 7237164.99,1378370.32 7236454.29,1377313.34 7233922.43,1377024.6500000001 7233266.19,1376817.24 7232777.66,1376485.54 7232108.88,1376278.48 7231699.26),(1278685.48 7174945.58,1278786.95 7174913.57,1279196.79 7174955.71,1279469.86 7174929.88,1279565.08 7174938.8100000005,1279628.6 7174989.01,1279654.96 7175055.04,1279737.8800000001 7175535.66,1279788.16 7175611.58,1279851.61 7175656.68,1280048.23 7175752.74,1280105.09 7175813.25,1280116.11 7175874.390000001,1280076.4000000001 7175951.54,1279767.05 7176185.57,1279675.48 7176197.03,1279588.46 7176167.57,1279521.4000000001 7176107.19,1278684.91 7175156.66,1278648.21 7175078.850000001,1278645.44 7174998.890000001,1278685.48 7174945.58))') AS wkb_geometry
    // )
    // SELECT ST_CollectionExtract(ST_MakeValid(ST_Multi(ST_Buffer(ST_SnapToGrid(ST_Multi(ST_Buffer(ST_Intersection(wkb_geometry,ST_MakeEnvelope(1252344.271424327-5*1222.9924525628203, 7200979.560689885-5*1222.9924525628203, 1565430.3392804116+5*1222.9924525628203, 7514065.62854597+5*1222.9924525628203,3857)), 0.0)), 1222.9924525628203/2), 0.0))),3)::geometry(MULTIPOLYGON,3857)::bytea
    let ewkb = hex_to_vec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
    let geom = ewkb::MultiPolygonT::<ewkb::Point>::read_ewkb(&mut ewkb.as_slice()).unwrap();
    assert_eq!(geom.polygons[0].rings[0].points.len(), 171);
    assert_eq!(
        geom.polygons[0].rings[0].points[0],
        ewkb::Point {
            x: 1376478.0053594543,
            y: 7231554.372003957,
            srid: None
        }
    );
    let screen_geom = screen::MultiPolygon::from_geom(&tile, &geom);
    assert_eq!(screen_geom.polygons[0].rings.len(), 1);
    assert_eq!(poly_to_wkt(&screen_geom.polygons[0].rings), "POLYGON((203 50,202 49,201 47,199 46,199 45,198 44,197 42,196 42,196 41,194 40,194 39,192 38,192 37,191 36,189 35,188 35,187 34,185 34,184 33,182 32,181 30,179 29,177 28,176 28,174 26,173 26,171 25,171 24,169 22,168 21,168 19,167 18,166 17,166 16,165 15,164 14,164 12,163 11,162 10,161 9,160 7,159 7,158 6,156 5,155 5,154 6,153 6,152 7,151 6,150 6,150 5,147 5,145 5,143 4,142 5,141 5,140 5,139 5,138 5,136 4,135 4,132 2,130 1,129 1,128 1,127 1,126 0,124 0,123 0,121 0,118 -2,117 -2,115 -3,113 -4,111 -4,110 -4,109 -4,108 -4,107 -4,104 -4,103 -4,102 -4,99 -3,98 -3,97 -3,96 -3,96 -2,95 -2,94 -2,93 -1,92 -1,90 -2,88 -2,87 -2,84 -2,82 -1,81 -1,79 -2,78 -2,76 -4,75 -5,74 -7,72 -8,72 -9,-9 -9,-9 77,-8 77,-6 78,-5 78,-3 78,-1 78,0 78,2 77,3 77,4 76,8 75,11 75,49 79,52 76,62 66,72 52,76 52,92 53,93 53,94 53,95 53,128 53,137 57,139 58,153 64,161 68,164 68,167 68,183 89,185 95,224 100,228 100,227 99,227 98,226 95,225 95,224 91,223 87,222 87,221 84,221 82,221 81,220 80,219 77,218 76,217 75,217 74,216 74,214 71,213 70,212 70,210 68,210 66,209 66,208 64,208 63,207 61,206 60,206 59,205 58,203 54,203 53,203 52,203 51,203 50))");
    // SELECT ST_GeomFromText('POLYGON((203 50,202 49,201 47,199 46,199 45,198 44,197 42,196 42,196 41,194 40,194 39,192 38,192 37,191 36,189 35,188 35,187 34,185 34,184 33,182 32,181 30,179 29,177 28,176 28,174 26,173 26,171 25,171 24,169 22,168 21,168 19,167 18,166 17,166 16,165 15,164 14,164 12,163 11,162 10,161 9,160 7,159 7,158 6,156 5,155 5,154 6,153 6,152 7,151 6,150 6,150 5,147 5,145 5,143 4,142 5,141 5,140 5,139 5,138 5,136 4,135 4,132 2,130 1,129 1,128 1,127 1,126 0,124 0,123 0,121 0,118 -2,117 -2,115 -3,113 -4,111 -4,110 -4,109 -4,108 -4,107 -4,104 -4,103 -4,102 -4,99 -3,98 -3,97 -3,96 -3,96 -2,95 -2,94 -2,93 -1,92 -1,90 -2,88 -2,87 -2,84 -2,82 -1,81 -1,79 -2,78 -2,76 -4,75 -5,74 -7,72 -8,72 -9,-9 -9,-9 77,-8 77,-6 78,-5 78,-3 78,-1 78,0 78,2 77,3 77,4 76,8 75,11 75,49 79,52 76,62 66,72 52,76 52,92 53,93 53,94 53,95 53,128 53,137 57,139 58,153 64,161 68,164 68,167 68,183 89,185 95,224 100,228 100,227 99,227 98,226 95,225 95,224 91,223 87,222 87,221 84,221 82,221 81,220 80,219 77,218 76,217 75,217 74,216 74,214 71,213 70,212 70,210 68,210 66,209 66,208 64,208 63,207 61,206 60,206 59,205 58,203 54,203 53,203 52,203 51,203 50))')
    assert_eq!(screen_geom.polygons[0].rings[0].points.len(), 170);

    // v0.13
    // WITH t AS (
    //   SELECT ST_GeomFromEWKT('SRID=3857;POLYGON((1376278.48 7231699.26,1375968.72 7231083.12,1375725.77 7230672.73,1375248.86 7229798.640000001,1374856.24 7229145.29,1374449.04 7228521.47,1374046.56 7227878.7,1373424.99 7226879.94,1373104.05 7226485.36,1372554.57 7225902.890000001,1371800.6500000001 7225205.61,1371399.86 7224893.53,1370174.55 7223920.13,1369766.53 7223656.16,1369197.29 7223296.8,1367858.6500000001 7222560.62,1367623.7 7222453.33,1367300.59 7222303.65,1366718.26 7222030.11,1366292.19 7221788.73,1365218.5 7221039.01,1364301.42 7220401.890000001,1362927.33 7219571.890000001,1362731.97 7219461.65,1361693.27 7218875.38,1360735.8900000001 7218391.3100000005,1360091.6500000001 7218004.23,1358722.91 7217074.41,1358288.85 7216761.08,1357698 7216309.350000001,1357412.51 7216069.5600000005,1356819.82 7215458.73,1355795.3800000001 7214248.8,1355367.8900000001 7213687.24,1354775.25 7212688.2,1354383.61 7211970.2700000005,1354060.24 7211326.07,1353751.22 7210710.66,1353546.51 7210332.29,1352930.1 7209241.75,1352348.74 7208338.5600000005,1351811.61 7207501.09,1351279.33 7206800.3,1350894.3 7206354.29,1349890.1 7205429.91,1349364.77 7205031.45,1348717.43 7204586.11,1348021.05 7204155.49,1347722.96 7204086.86,1347262.32 7204146.48,1347034.1400000001 7204235.18,1346752.09 7204434.390000001,1346552.6300000001 7204590.8,1346392.87 7204770.26,1346151.49 7204697.67,1346126.92 7204690.600000001,1346102.35 7204683.54,1345905.35 7204613.4,1345658.01 7204526.5,1345477.9000000001 7205186.51,1344732.72 7204700.25,1344271.47 7204414.59,1343865.32 7204233.67,1342492.02 7203938.38,1341069.97 7203751.53,1339556.8 7203687.63,1338968.07 7203735.8,1338440.91 7203794.600000001,1337756.99 7203938.61,1337378.3900000001 7204011.350000001,1337093.9000000001 7204018.93,1336758.03 7203931.88,1335684.8900000001 7203461.87,1335135.22 7203251.01,1332953.31 7202294.18,1331657.92 7201765.01,1331101.61 7201637.75,1330363.23 7201522.59,1330008.84 7201437.36,1329254.33 7201192.63,1328305.52 7200854.29,1327552.26 7200580.37,1326562.09 7200210.21,1324656.3900000001 7199348.66,1323806.78 7199009.69,1322709.54 7198632.640000001,1321195.57 7198188.140000001,1320132.42 7197866.83,1319587.96 7197736.57,1319186.62 7197675.69,1318792.56 7197640.26,1318139.96 7197623.2,1317575.26 7197662.5200000005,1316204.52 7197771.0200000005,1315437.1400000001 7197849.69,1314689.82 7197926.24,1312806.92 7198260.100000001,1312098.36 7198453.5200000005,1311597.36 7198634.72,1311191.46 7198800.57,1310789.59 7199003.9,1310343.61 7199218.22,1309870.35 7199433,1309441.91 7199580.390000001,1309171.6400000001 7199614.07,1308675.59 7199593.55,1307326.1300000001 7199372.390000001,1306429.3900000001 7199327.79,1305440.2 7199356.350000001,1303849.48 7199370.850000001,1302694.36 7199461.57,1301941.66 7199473.88,1300855.53 7199407.850000001,1300454.21 7199344.33,1300164.1 7199203.88,1299822.6400000001 7198941.32,1299079.36 7198149.15,1298399.72 7197388.41,1297338.93 7196201.890000001,1296571.06 7195329.94,1295328.83 7193934.74,1294755.56 7193293.57,1294235.8800000001 7192590.18,1293777.74 7192010.4,1293349.74 7191506.93,1292795.4000000001 7190971.2700000005,1292143.3900000001 7190385.93,1291496.1300000001 7189884.0600000005,1290914.07 7189527.850000001,1290451.1500000001 7189287.5,1290024.84 7189104.57,1289386.6500000001 7188972.48,1288280.42 7188941,1287902 7188888.55,1287584.44 7188809.62,1287146.31 7188626.72,1286849.3800000001 7188443.46,1286324.76 7188038.42,1286007.36 7187729.28,1285851.61 7187542.28,1285244.26 7186693.48,1285115.1300000001 7186461.79,1284977.36 7186219.99,1284793.69 7185978.86,1283962.73 7185053.2,1282973.37 7183848.600000001,1282927.19 7183706.11,1282934.17 7183596.95,1282941.04 7183479.28,1283048.56 7183387.47,1282802.5 7183174.51,1282658.79 7183231.04,1282551.21 7183196.75,1282451.27 7183099.32,1281504.86 7181028.83,1281428.11 7180760.75,1281398.1500000001 7180435.82,1281368.81 7179039.61,1281494.03 7177641.5,1281536.46 7177514.91,1281635.43 7177423.3,1281742.29 7177411.61,1281868.61 7177455.83,1282318.24 7176187.98,1282217.75 7176044.67,1282143.95 7175866.94,1282039.12 7175408.76,1282026.33 7175219.98,1282078.67 7175074.5600000005,1282202.81 7174963.9,1282400.86 7174916.890000001,1283217.56 7174905.49,1283782.58 7174870.32,1283917.53 7174805.44,1283997.43 7174690.2700000005,1284005.1300000001 7174513.140000001,1283972.26 7174227.65,1283896.3 7173896.83,1283812.9000000001 7173761.84,1283650.12 7173656.91,1283180.87 7173491.600000001,1282413.71 7173269.15,1282133.36 7173131.79,1281883.12 7172961.67,1281401.8900000001 7172536.07,1281225.76 7172320.68,1281106.46 7172162.37,1281022.75 7172000.17,1280921.29 7171780.350000001,1280834.01 7171356.17,1280788.42 7171125.41,1280702.67 7170936.0200000005,1280247.1400000001 7170265.13,1280161.44 7170079.17,1279911.43 7169414.03,1279822.71 7169253.640000001,1279691.03 7169178.88,1279393.22 7169118.28,1279264.57 7169016.26,1278939.6400000001 7169968.140000001,1279077.96 7170030.92,1279177.96 7170146.95,1279270.69 7170353.25,1279675.01 7172391.22,1279782.83 7173082.37,1279787.11 7173398.850000001,1279704.79 7173966.72,1279635.92 7174148.07,1279502.03 7174292.850000001,1279325.11 7174395.67,1278868.3800000001 7174527.79,1278177.45 7174671.54,1277210.09 7174846.140000001,1276850.49 7174878.140000001,1276500.49 7174865.73,1275833.6 7174774.07,1275321.24 7174682.04,1245040.3900000001 7172903.5200000005,1240772.3900000001 7187705.07,1240703.6400000001 7187943.23,1240634.82 7188181.68,1240566.1500000001 7188419.57,1240497.4000000001 7188657.75,1240428.6400000001 7188895.94,1239895.73 7190739.29,1234976.42 7207758.55,1235010.18 7207927.46,1235011.3900000001 7208077.84,1234922.31 7208449.41,1234753.74 7209327.55,1234701.62 7210695.45,1234578.24 7211900.0200000005,1234438.28 7212744.09,1234375.85 7213058.44,1234070.85 7214593.95,1233926.47 7215323.72,1233780.74 7216315.3,1233726.74 7216970.38,1233698.19 7217316.75,1233686.11 7218379.36,1233954.1500000001 7222685.16,1233969.01 7223051.41,1233953.09 7223412.79,1233893.44 7223861.86,1233796.6300000001 7224364.32,1233168.3 7227025.55,1233074.98 7227548.73,1232904.59 7228709.84,1232788.46 7232232.19,1232774.37 7232619.67,1232487.68 7234903.61,1232543.52 7235226.12,1232615.91 7235587.08,1232689.98 7235799.91,1232764.59 7235891.91,1232890.56 7235987.62,1232940.2 7236036.61,1232965.1500000001 7236137.21,1233185.98 7236206.86,1233270.76 7236458.79,1233468.75 7236911.93,1233640.96 7237289.51,1233971.74 7237669.890000001,1234122.07 7237825.09,1234494.99 7238207.7,1234967.75 7238645.3,1235167.1400000001 7238780.76,1235929.11 7239144.63,1236114.74 7239266.47,1236266.41 7239416.01,1236460.18 7239695.42,1236625.8800000001 7239885.97,1237149.3 7240327.16,1237472.61 7240550.66,1237637.31 7240617.82,1237877.25 7240684.350000001,1237980.24 7240731.46,1238069.72 7240799.25,1238421.54 7241152.74,1238532.1300000001 7241288.91,1238678.79 7241644.15,1238770.92 7242020.44,1238892.53 7242643.34,1238941.76 7242800.62,1239060.46 7243080.74,1239109.99 7243272.3100000005,1239146.42 7243532.57,1239150.41 7243882.58,1243547.6400000001 7246279.47,1243972.8800000001 7246512.51,1244229.32 7246642.350000001,1244463.91 7246817.03,1244609.09 7246956.48,1245285.33 7247827.640000001,1245325.3800000001 7247895.95,1246345.9000000001 7247829.67,1246472.54 7247814.73,1246616.91 7247861.44,1246860.28 7248049.73,1247011.91 7248139.3,1248004.9000000001 7248414.8100000005,1248684.33 7248507.8,1249600.1 7248586.390000001,1251231.78 7248630.2,1251951.12 7248597.15,1253376.76 7248266.5200000005,1254284.6 7247910.18,1254727.24 7247639.37,1255032.37 7247491.91,1257154.83 7246988.44,1259102.59 7246666.66,1259282.85 7246700.72,1259349.99 7246730.890000001,1282171.35 7249567.38,1284434.18 7247587.94,1290031.33 7241108.84,1296246.95 7233043.98,1298736.96 7233079.43,1308609.5 7233213.850000001,1309064.09 7233219.75,1309518.69 7233225.63,1309973.29 7233231.48,1310427.8800000001 7233237.3100000005,1330447.6500000001 7233468.78,1336280.54 7236051.140000001,1337635.67 7236650.37,1345865.5 7240283.82,1351003.31 7242547.12,1352804.74 7243339.76,1354227.19 7243339.79,1364617.08 7255935.07,1366011.75 7259425.41,1389790.43 7262043.53,1393281.54 7262422.2,1393015.71 7262307.32,1392726.67 7262096.71,1392431 7261831.22,1392054.72 7261378.65,1391713.74 7260833.94,1391257.6400000001 7259857.49,1390835.44 7258953.49,1390099.67 7257253.54,1389188.41 7254757.41,1388914.54 7253907.8,1388335.4000000001 7252110.86,1388096.77 7251266.58,1387851.9000000001 7250572.04,1387497.1500000001 7249741.32,1386741.06 7248253.12,1386231.58 7247395.79,1385798.23 7246821.61,1385459.96 7246406.37,1385264.86 7246155.65,1383830.3 7244311.48,1383716.42 7244214.890000001,1383379.02 7243955.92,1382897.96 7243573.65,1381359.04 7242290.26,1381107.94 7242051.33,1380874.67 7241772.45,1380714.6 7241476.23,1380431.7 7240865.68,1379873.7 7239610.17,1379308.86 7238348.13,1379021.87 7237770.48,1378710.42 7237164.99,1378370.32 7236454.29,1377313.34 7233922.43,1377024.6500000001 7233266.19,1376817.24 7232777.66,1376485.54 7232108.88,1376278.48 7231699.26),(1278685.48 7174945.58,1278786.95 7174913.57,1279196.79 7174955.71,1279469.86 7174929.88,1279565.08 7174938.8100000005,1279628.6 7174989.01,1279654.96 7175055.04,1279737.8800000001 7175535.66,1279788.16 7175611.58,1279851.61 7175656.68,1280048.23 7175752.74,1280105.09 7175813.25,1280116.11 7175874.390000001,1280076.4000000001 7175951.54,1279767.05 7176185.57,1279675.48 7176197.03,1279588.46 7176167.57,1279521.4000000001 7176107.19,1278684.91 7175156.66,1278648.21 7175078.850000001,1278645.44 7174998.890000001,1278685.48 7174945.58))') AS wkb_geometry
    // )
    // SELECT ST_CollectionExtract(ST_MakeValid(ST_Multi(ST_Buffer(ST_SnapToGrid(ST_Multi(ST_Buffer(ST_Intersection(wkb_geometry,ST_Buffer(ST_MakeEnvelope(1252344.271424327, 7200979.560689885, 1565430.3392804116, 7514065.62854597,3857),10*1222.9924525628203)), 0.0)), 1222.9924525628203/2), 0.0))),3)::geometry(MULTIPOLYGON,3857)::bytea AS wkb_geometry
    // FROM t
    // WHERE wkb_geometry && ST_Buffer(ST_MakeEnvelope(1252344.271424327, 7200979.560689885, 1565430.3392804116, 7514065.62854597,3857),10*1222.9924525628203)
    let ewkb = hex_to_vec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
    let geom = ewkb::MultiPolygonT::<ewkb::Point>::read_ewkb(&mut ewkb.as_slice()).unwrap();
    assert_eq!(geom.polygons[0].rings[0].points.len(), 197);
    let screen_geom = screen::MultiPolygon::from_geom(&tile, &geom);
    assert_eq!(screen_geom.polygons[0].rings.len(), 1);
    assert_eq!(poly_to_wkt(&screen_geom.polygons[0].rings), "POLYGON((203 50,202 49,201 47,199 46,199 45,198 44,197 42,196 42,196 41,194 40,194 39,192 38,192 37,191 36,189 35,188 35,187 34,185 34,184 33,182 32,181 30,179 29,177 28,176 28,174 26,173 26,171 25,171 24,169 22,168 21,168 19,167 18,166 17,166 16,165 15,164 14,164 12,163 11,162 10,161 9,160 7,159 7,158 6,156 5,155 5,154 6,153 6,152 7,151 6,150 6,150 5,147 5,145 5,143 4,142 5,141 5,140 5,139 5,138 5,136 4,135 4,132 2,130 1,129 1,128 1,127 1,126 0,124 0,123 0,121 0,118 -2,117 -2,115 -3,113 -4,111 -4,110 -4,109 -4,108 -4,107 -4,104 -4,103 -4,102 -4,99 -3,98 -3,97 -3,96 -3,96 -2,95 -2,94 -2,93 -1,92 -1,90 -2,88 -2,87 -2,84 -2,82 -1,81 -1,79 -2,78 -2,76 -4,75 -5,74 -7,72 -8,70 -11,69 -12,69 -13,68 -14,67 -14,66 -15,65 -16,64 -17,63 -18,62 -18,61 -19,59 -19,58 -19,57 -19,0 -19,-3 -19,-7 -17,-10 -16,-13 -13,-16 -10,-17 -7,-19 -3,-19 0,-19 71,-13 74,-12 75,-11 77,-10 77,-9 77,-8 77,-6 78,-5 78,-3 78,-1 78,0 78,2 77,3 77,4 76,8 75,11 75,49 79,52 76,62 66,72 52,76 52,92 53,93 53,94 53,95 53,128 53,137 57,139 58,153 64,161 68,164 68,167 68,183 89,185 95,224 100,228 100,227 99,227 98,226 95,225 95,224 91,223 87,222 87,221 84,221 82,221 81,220 80,219 77,218 76,217 75,217 74,216 74,214 71,213 70,212 70,210 68,210 66,209 66,208 64,208 63,207 61,206 60,206 59,205 58,203 54,203 53,203 52,203 51,203 50))");
    // SELECT ST_GeomFromText('POLYGON((203 50,202 49,201 47,199 46,199 45,198 44,197 42,196 42,196 41,194 40,194 39,192 38,192 37,191 36,189 35,188 35,187 34,185 34,184 33,182 32,181 30,179 29,177 28,176 28,174 26,173 26,171 25,171 24,169 22,168 21,168 19,167 18,166 17,166 16,165 15,164 14,164 12,163 11,162 10,161 9,160 7,159 7,158 6,156 5,155 5,154 6,153 6,152 7,151 6,150 6,150 5,147 5,145 5,143 4,142 5,141 5,140 5,139 5,138 5,136 4,135 4,132 2,130 1,129 1,128 1,127 1,126 0,124 0,123 0,121 0,118 -2,117 -2,115 -3,113 -4,111 -4,110 -4,109 -4,108 -4,107 -4,104 -4,103 -4,102 -4,99 -3,98 -3,97 -3,96 -3,96 -2,95 -2,94 -2,93 -1,92 -1,90 -2,88 -2,87 -2,84 -2,82 -1,81 -1,79 -2,78 -2,76 -4,75 -5,74 -7,72 -8,70 -11,69 -12,69 -13,68 -14,67 -14,66 -15,65 -16,64 -17,63 -18,62 -18,61 -19,59 -19,58 -19,57 -19,0 -19,-3 -19,-7 -17,-10 -16,-13 -13,-16 -10,-17 -7,-19 -3,-19 0,-19 71,-13 74,-12 75,-11 77,-10 77,-9 77,-8 77,-6 78,-5 78,-3 78,-1 78,0 78,2 77,3 77,4 76,8 75,11 75,49 79,52 76,62 66,72 52,76 52,92 53,93 53,94 53,95 53,128 53,137 57,139 58,153 64,161 68,164 68,167 68,183 89,185 95,224 100,228 100,227 99,227 98,226 95,225 95,224 91,223 87,222 87,221 84,221 82,221 81,220 80,219 77,218 76,217 75,217 74,216 74,214 71,213 70,212 70,210 68,210 66,209 66,208 64,208 63,207 61,206 60,206 59,205 58,203 54,203 53,203 52,203 51,203 50))')
    assert_eq!(screen_geom.polygons[0].rings[0].points.len(), 196);

    // v0.11
    // WITH t AS (
    //   SELECT ST_GeomFromEWKT('SRID=3857;POLYGON((1376278.48 7231699.26,1375968.72 7231083.12,1375725.77 7230672.73,1375248.86 7229798.640000001,1374856.24 7229145.29,1374449.04 7228521.47,1374046.56 7227878.7,1373424.99 7226879.94,1373104.05 7226485.36,1372554.57 7225902.890000001,1371800.6500000001 7225205.61,1371399.86 7224893.53,1370174.55 7223920.13,1369766.53 7223656.16,1369197.29 7223296.8,1367858.6500000001 7222560.62,1367623.7 7222453.33,1367300.59 7222303.65,1366718.26 7222030.11,1366292.19 7221788.73,1365218.5 7221039.01,1364301.42 7220401.890000001,1362927.33 7219571.890000001,1362731.97 7219461.65,1361693.27 7218875.38,1360735.8900000001 7218391.3100000005,1360091.6500000001 7218004.23,1358722.91 7217074.41,1358288.85 7216761.08,1357698 7216309.350000001,1357412.51 7216069.5600000005,1356819.82 7215458.73,1355795.3800000001 7214248.8,1355367.8900000001 7213687.24,1354775.25 7212688.2,1354383.61 7211970.2700000005,1354060.24 7211326.07,1353751.22 7210710.66,1353546.51 7210332.29,1352930.1 7209241.75,1352348.74 7208338.5600000005,1351811.61 7207501.09,1351279.33 7206800.3,1350894.3 7206354.29,1349890.1 7205429.91,1349364.77 7205031.45,1348717.43 7204586.11,1348021.05 7204155.49,1347722.96 7204086.86,1347262.32 7204146.48,1347034.1400000001 7204235.18,1346752.09 7204434.390000001,1346552.6300000001 7204590.8,1346392.87 7204770.26,1346151.49 7204697.67,1346126.92 7204690.600000001,1346102.35 7204683.54,1345905.35 7204613.4,1345658.01 7204526.5,1345477.9000000001 7205186.51,1344732.72 7204700.25,1344271.47 7204414.59,1343865.32 7204233.67,1342492.02 7203938.38,1341069.97 7203751.53,1339556.8 7203687.63,1338968.07 7203735.8,1338440.91 7203794.600000001,1337756.99 7203938.61,1337378.3900000001 7204011.350000001,1337093.9000000001 7204018.93,1336758.03 7203931.88,1335684.8900000001 7203461.87,1335135.22 7203251.01,1332953.31 7202294.18,1331657.92 7201765.01,1331101.61 7201637.75,1330363.23 7201522.59,1330008.84 7201437.36,1329254.33 7201192.63,1328305.52 7200854.29,1327552.26 7200580.37,1326562.09 7200210.21,1324656.3900000001 7199348.66,1323806.78 7199009.69,1322709.54 7198632.640000001,1321195.57 7198188.140000001,1320132.42 7197866.83,1319587.96 7197736.57,1319186.62 7197675.69,1318792.56 7197640.26,1318139.96 7197623.2,1317575.26 7197662.5200000005,1316204.52 7197771.0200000005,1315437.1400000001 7197849.69,1314689.82 7197926.24,1312806.92 7198260.100000001,1312098.36 7198453.5200000005,1311597.36 7198634.72,1311191.46 7198800.57,1310789.59 7199003.9,1310343.61 7199218.22,1309870.35 7199433,1309441.91 7199580.390000001,1309171.6400000001 7199614.07,1308675.59 7199593.55,1307326.1300000001 7199372.390000001,1306429.3900000001 7199327.79,1305440.2 7199356.350000001,1303849.48 7199370.850000001,1302694.36 7199461.57,1301941.66 7199473.88,1300855.53 7199407.850000001,1300454.21 7199344.33,1300164.1 7199203.88,1299822.6400000001 7198941.32,1299079.36 7198149.15,1298399.72 7197388.41,1297338.93 7196201.890000001,1296571.06 7195329.94,1295328.83 7193934.74,1294755.56 7193293.57,1294235.8800000001 7192590.18,1293777.74 7192010.4,1293349.74 7191506.93,1292795.4000000001 7190971.2700000005,1292143.3900000001 7190385.93,1291496.1300000001 7189884.0600000005,1290914.07 7189527.850000001,1290451.1500000001 7189287.5,1290024.84 7189104.57,1289386.6500000001 7188972.48,1288280.42 7188941,1287902 7188888.55,1287584.44 7188809.62,1287146.31 7188626.72,1286849.3800000001 7188443.46,1286324.76 7188038.42,1286007.36 7187729.28,1285851.61 7187542.28,1285244.26 7186693.48,1285115.1300000001 7186461.79,1284977.36 7186219.99,1284793.69 7185978.86,1283962.73 7185053.2,1282973.37 7183848.600000001,1282927.19 7183706.11,1282934.17 7183596.95,1282941.04 7183479.28,1283048.56 7183387.47,1282802.5 7183174.51,1282658.79 7183231.04,1282551.21 7183196.75,1282451.27 7183099.32,1281504.86 7181028.83,1281428.11 7180760.75,1281398.1500000001 7180435.82,1281368.81 7179039.61,1281494.03 7177641.5,1281536.46 7177514.91,1281635.43 7177423.3,1281742.29 7177411.61,1281868.61 7177455.83,1282318.24 7176187.98,1282217.75 7176044.67,1282143.95 7175866.94,1282039.12 7175408.76,1282026.33 7175219.98,1282078.67 7175074.5600000005,1282202.81 7174963.9,1282400.86 7174916.890000001,1283217.56 7174905.49,1283782.58 7174870.32,1283917.53 7174805.44,1283997.43 7174690.2700000005,1284005.1300000001 7174513.140000001,1283972.26 7174227.65,1283896.3 7173896.83,1283812.9000000001 7173761.84,1283650.12 7173656.91,1283180.87 7173491.600000001,1282413.71 7173269.15,1282133.36 7173131.79,1281883.12 7172961.67,1281401.8900000001 7172536.07,1281225.76 7172320.68,1281106.46 7172162.37,1281022.75 7172000.17,1280921.29 7171780.350000001,1280834.01 7171356.17,1280788.42 7171125.41,1280702.67 7170936.0200000005,1280247.1400000001 7170265.13,1280161.44 7170079.17,1279911.43 7169414.03,1279822.71 7169253.640000001,1279691.03 7169178.88,1279393.22 7169118.28,1279264.57 7169016.26,1278939.6400000001 7169968.140000001,1279077.96 7170030.92,1279177.96 7170146.95,1279270.69 7170353.25,1279675.01 7172391.22,1279782.83 7173082.37,1279787.11 7173398.850000001,1279704.79 7173966.72,1279635.92 7174148.07,1279502.03 7174292.850000001,1279325.11 7174395.67,1278868.3800000001 7174527.79,1278177.45 7174671.54,1277210.09 7174846.140000001,1276850.49 7174878.140000001,1276500.49 7174865.73,1275833.6 7174774.07,1275321.24 7174682.04,1245040.3900000001 7172903.5200000005,1240772.3900000001 7187705.07,1240703.6400000001 7187943.23,1240634.82 7188181.68,1240566.1500000001 7188419.57,1240497.4000000001 7188657.75,1240428.6400000001 7188895.94,1239895.73 7190739.29,1234976.42 7207758.55,1235010.18 7207927.46,1235011.3900000001 7208077.84,1234922.31 7208449.41,1234753.74 7209327.55,1234701.62 7210695.45,1234578.24 7211900.0200000005,1234438.28 7212744.09,1234375.85 7213058.44,1234070.85 7214593.95,1233926.47 7215323.72,1233780.74 7216315.3,1233726.74 7216970.38,1233698.19 7217316.75,1233686.11 7218379.36,1233954.1500000001 7222685.16,1233969.01 7223051.41,1233953.09 7223412.79,1233893.44 7223861.86,1233796.6300000001 7224364.32,1233168.3 7227025.55,1233074.98 7227548.73,1232904.59 7228709.84,1232788.46 7232232.19,1232774.37 7232619.67,1232487.68 7234903.61,1232543.52 7235226.12,1232615.91 7235587.08,1232689.98 7235799.91,1232764.59 7235891.91,1232890.56 7235987.62,1232940.2 7236036.61,1232965.1500000001 7236137.21,1233185.98 7236206.86,1233270.76 7236458.79,1233468.75 7236911.93,1233640.96 7237289.51,1233971.74 7237669.890000001,1234122.07 7237825.09,1234494.99 7238207.7,1234967.75 7238645.3,1235167.1400000001 7238780.76,1235929.11 7239144.63,1236114.74 7239266.47,1236266.41 7239416.01,1236460.18 7239695.42,1236625.8800000001 7239885.97,1237149.3 7240327.16,1237472.61 7240550.66,1237637.31 7240617.82,1237877.25 7240684.350000001,1237980.24 7240731.46,1238069.72 7240799.25,1238421.54 7241152.74,1238532.1300000001 7241288.91,1238678.79 7241644.15,1238770.92 7242020.44,1238892.53 7242643.34,1238941.76 7242800.62,1239060.46 7243080.74,1239109.99 7243272.3100000005,1239146.42 7243532.57,1239150.41 7243882.58,1243547.6400000001 7246279.47,1243972.8800000001 7246512.51,1244229.32 7246642.350000001,1244463.91 7246817.03,1244609.09 7246956.48,1245285.33 7247827.640000001,1245325.3800000001 7247895.95,1246345.9000000001 7247829.67,1246472.54 7247814.73,1246616.91 7247861.44,1246860.28 7248049.73,1247011.91 7248139.3,1248004.9000000001 7248414.8100000005,1248684.33 7248507.8,1249600.1 7248586.390000001,1251231.78 7248630.2,1251951.12 7248597.15,1253376.76 7248266.5200000005,1254284.6 7247910.18,1254727.24 7247639.37,1255032.37 7247491.91,1257154.83 7246988.44,1259102.59 7246666.66,1259282.85 7246700.72,1259349.99 7246730.890000001,1282171.35 7249567.38,1284434.18 7247587.94,1290031.33 7241108.84,1296246.95 7233043.98,1298736.96 7233079.43,1308609.5 7233213.850000001,1309064.09 7233219.75,1309518.69 7233225.63,1309973.29 7233231.48,1310427.8800000001 7233237.3100000005,1330447.6500000001 7233468.78,1336280.54 7236051.140000001,1337635.67 7236650.37,1345865.5 7240283.82,1351003.31 7242547.12,1352804.74 7243339.76,1354227.19 7243339.79,1364617.08 7255935.07,1366011.75 7259425.41,1389790.43 7262043.53,1393281.54 7262422.2,1393015.71 7262307.32,1392726.67 7262096.71,1392431 7261831.22,1392054.72 7261378.65,1391713.74 7260833.94,1391257.6400000001 7259857.49,1390835.44 7258953.49,1390099.67 7257253.54,1389188.41 7254757.41,1388914.54 7253907.8,1388335.4000000001 7252110.86,1388096.77 7251266.58,1387851.9000000001 7250572.04,1387497.1500000001 7249741.32,1386741.06 7248253.12,1386231.58 7247395.79,1385798.23 7246821.61,1385459.96 7246406.37,1385264.86 7246155.65,1383830.3 7244311.48,1383716.42 7244214.890000001,1383379.02 7243955.92,1382897.96 7243573.65,1381359.04 7242290.26,1381107.94 7242051.33,1380874.67 7241772.45,1380714.6 7241476.23,1380431.7 7240865.68,1379873.7 7239610.17,1379308.86 7238348.13,1379021.87 7237770.48,1378710.42 7237164.99,1378370.32 7236454.29,1377313.34 7233922.43,1377024.6500000001 7233266.19,1376817.24 7232777.66,1376485.54 7232108.88,1376278.48 7231699.26),(1278685.48 7174945.58,1278786.95 7174913.57,1279196.79 7174955.71,1279469.86 7174929.88,1279565.08 7174938.8100000005,1279628.6 7174989.01,1279654.96 7175055.04,1279737.8800000001 7175535.66,1279788.16 7175611.58,1279851.61 7175656.68,1280048.23 7175752.74,1280105.09 7175813.25,1280116.11 7175874.390000001,1280076.4000000001 7175951.54,1279767.05 7176185.57,1279675.48 7176197.03,1279588.46 7176167.57,1279521.4000000001 7176107.19,1278684.91 7175156.66,1278648.21 7175078.850000001,1278645.44 7174998.890000001,1278685.48 7174945.58))') AS wkb_geometry
    // )
    // SELECT COALESCE(ST_SnapToGrid(ST_Multi(ST_Buffer(ST_Intersection(ST_MakeValid(wkb_geometry),ST_Buffer(ST_MakeEnvelope(1252344.271424327, 7200979.560689885, 1565430.3392804116, 7514065.62854597,3857),10*1222.9924525628203)), 0.0)), 1222.9924525628203/2),ST_GeomFromText('MULTIPOLYGON EMPTY',3857))::geometry(MULTIPOLYGON,3857)::bytea AS geometry
    // FROM t
    // WHERE wkb_geometry && ST_Buffer(ST_MakeEnvelope(1252344.271424327, 7200979.560689885, 1565430.3392804116, 7514065.62854597,3857),10*1222.9924525628203)
    let ewkb = hex_to_vec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
    let geom = ewkb::MultiPolygonT::<ewkb::Point>::read_ewkb(&mut ewkb.as_slice()).unwrap();
    assert_eq!(geom.polygons[0].rings[0].points.len(), 198);
    let screen_geom = screen::MultiPolygon::from_geom(&tile, &geom);
    assert_eq!(screen_geom.polygons[0].rings.len(), 1);
    assert_eq!(poly_to_wkt(&screen_geom.polygons[0].rings), "POLYGON((203 50,202 49,201 47,199 46,199 45,198 44,197 42,196 42,196 41,194 40,194 39,192 38,192 37,191 36,189 35,188 35,187 34,185 34,184 33,182 32,181 30,179 29,177 28,176 28,174 26,173 26,171 25,171 24,169 22,168 21,168 19,167 18,166 17,166 16,165 15,164 14,164 12,163 11,162 10,161 9,160 7,159 7,158 6,156 5,155 5,154 6,153 6,152 7,151 6,150 6,150 5,147 5,145 5,143 4,142 5,141 5,140 5,139 5,138 5,136 4,135 4,132 2,130 1,129 1,128 1,127 1,126 0,124 0,123 0,121 0,118 -2,117 -2,115 -3,113 -4,111 -4,110 -4,109 -4,108 -4,107 -4,104 -4,103 -4,102 -4,99 -3,98 -3,97 -3,96 -3,96 -2,95 -2,94 -2,93 -1,92 -1,90 -2,88 -2,87 -2,84 -2,82 -1,81 -1,79 -2,78 -2,76 -4,75 -5,74 -7,72 -8,70 -11,69 -12,69 -13,68 -14,67 -14,66 -15,65 -16,64 -17,63 -18,62 -18,61 -19,59 -19,58 -19,57 -19,0 -19,-3 -19,-7 -17,-10 -16,-13 -13,-16 -10,-17 -7,-19 -3,-19 0,-19 71,-13 74,-12 75,-11 77,-10 77,-9 77,-8 77,-6 78,-5 78,-3 78,-1 78,0 78,2 77,3 77,4 76,8 75,11 75,49 79,52 76,62 66,72 52,76 52,92 53,93 53,94 53,95 53,128 53,137 57,139 58,153 64,161 68,164 68,167 68,183 89,185 95,224 100,229 100,228 100,227 99,227 98,226 95,225 95,224 91,223 87,222 87,221 84,221 82,221 81,220 80,219 77,218 76,217 75,217 74,216 74,214 71,213 70,212 70,210 68,210 66,209 66,208 64,208 63,207 61,206 60,206 59,205 58,203 54,203 53,203 52,203 51,203 50))");
    // SELECT ST_GeomFromText('POLYGON((203 50,202 49,201 47,199 46,199 45,198 44,197 42,196 42,196 41,194 40,194 39,192 38,192 37,191 36,189 35,188 35,187 34,185 34,184 33,182 32,181 30,179 29,177 28,176 28,174 26,173 26,171 25,171 24,169 22,168 21,168 19,167 18,166 17,166 16,165 15,164 14,164 12,163 11,162 10,161 9,160 7,159 7,158 6,156 5,155 5,154 6,153 6,152 7,151 6,150 6,150 5,147 5,145 5,143 4,142 5,141 5,140 5,139 5,138 5,136 4,135 4,132 2,130 1,129 1,128 1,127 1,126 0,124 0,123 0,121 0,118 -2,117 -2,115 -3,113 -4,111 -4,110 -4,109 -4,108 -4,107 -4,104 -4,103 -4,102 -4,99 -3,98 -3,97 -3,96 -3,96 -2,95 -2,94 -2,93 -1,92 -1,90 -2,88 -2,87 -2,84 -2,82 -1,81 -1,79 -2,78 -2,76 -4,75 -5,74 -7,72 -8,70 -11,69 -12,69 -13,68 -14,67 -14,66 -15,65 -16,64 -17,63 -18,62 -18,61 -19,59 -19,58 -19,57 -19,0 -19,-3 -19,-7 -17,-10 -16,-13 -13,-16 -10,-17 -7,-19 -3,-19 0,-19 71,-13 74,-12 75,-11 77,-10 77,-9 77,-8 77,-6 78,-5 78,-3 78,-1 78,0 78,2 77,3 77,4 76,8 75,11 75,49 79,52 76,62 66,72 52,76 52,92 53,93 53,94 53,95 53,128 53,137 57,139 58,153 64,161 68,164 68,167 68,183 89,185 95,224 100,229 100,228 100,227 99,227 98,226 95,225 95,224 91,223 87,222 87,221 84,221 82,221 81,220 80,219 77,218 76,217 75,217 74,216 74,214 71,213 70,212 70,210 68,210 66,209 66,208 64,208 63,207 61,206 60,206 59,205 58,203 54,203 53,203 52,203 51,203 50))')
    assert_eq!(screen_geom.polygons[0].rings[0].points.len(), 197);
}

#[test]
fn test_tile_values() {
    let mut value = vector_tile::Tile_Value::new();
    assert_eq!(value, Default::default());
    assert!(!value.has_string_value());
    value.set_string_value(String::from("Hello, world!"));
    println!("{:?}", value);
    assert!(value.has_string_value());
    assert_eq!(value.get_string_value(), String::from("Hello, world!"));
}

#[test]
fn test_read_from_file() {
    // Command line decoding:
    // protoc --decode=vector_tile.Tile src/mvt/vector_tile.proto <src/test/tile.pbf
    let mut f = File::open("../t-rex-service/src/test/tile.pbf").unwrap();
    let tile = Tile::read_from(&mut f).unwrap();
    println!("{:#?}", tile);
    let ref layer = tile.get_layers()[0];
    assert_eq!(layer.get_name(), "roads");
    let ref feature = layer.get_features()[1];
    assert_eq!(
        feature.get_field_type(),
        vector_tile::Tile_GeomType::POLYGON
    );
    let ref geometry = feature.get_geometry();
    assert_eq!(
        geometry,
        &[9, 8236, 4926, 34, 9, 24, 37, 21, 10, 7, 4, 19, 15]
    );
}

// https://github.com/mapbox/vector-tile-spec/tree/master/2.1#45-example
#[cfg(test)]
const TILE_EXAMPLE: &'static str = r#"Tile {
    layers: [
        Tile_Layer {
            version: Some(
                2,
            ),
            name: Some("points"),
            features: [
                Tile_Feature {
                    id: Some(
                        1,
                    ),
                    tags: [
                        0,
                        0,
                        1,
                        0,
                        2,
                        1,
                    ],
                    field_type: Some(
                        POINT,
                    ),
                    geometry: [
                        9,
                        490,
                        6262,
                    ],
                    unknown_fields: UnknownFields {
                        fields: None,
                    },
                    cached_size: CachedSize {
                        size: 0,
                    },
                },
                Tile_Feature {
                    id: Some(
                        2,
                    ),
                    tags: [
                        0,
                        2,
                        2,
                        3,
                    ],
                    field_type: Some(
                        POINT,
                    ),
                    geometry: [
                        9,
                        490,
                        6262,
                    ],
                    unknown_fields: UnknownFields {
                        fields: None,
                    },
                    cached_size: CachedSize {
                        size: 0,
                    },
                },
            ],
            keys: [
                "hello",
                "h",
                "count",
            ],
            values: [
                Tile_Value {
                    string_value: Some("world"),
                    float_value: None,
                    double_value: None,
                    int_value: None,
                    uint_value: None,
                    sint_value: None,
                    bool_value: None,
                    unknown_fields: UnknownFields {
                        fields: None,
                    },
                    cached_size: CachedSize {
                        size: 0,
                    },
                },
                Tile_Value {
                    string_value: None,
                    float_value: None,
                    double_value: Some(
                        1.23,
                    ),
                    int_value: None,
                    uint_value: None,
                    sint_value: None,
                    bool_value: None,
                    unknown_fields: UnknownFields {
                        fields: None,
                    },
                    cached_size: CachedSize {
                        size: 0,
                    },
                },
                Tile_Value {
                    string_value: Some("again"),
                    float_value: None,
                    double_value: None,
                    int_value: None,
                    uint_value: None,
                    sint_value: None,
                    bool_value: None,
                    unknown_fields: UnknownFields {
                        fields: None,
                    },
                    cached_size: CachedSize {
                        size: 0,
                    },
                },
                Tile_Value {
                    string_value: None,
                    float_value: None,
                    double_value: None,
                    int_value: Some(
                        2,
                    ),
                    uint_value: None,
                    sint_value: None,
                    bool_value: None,
                    unknown_fields: UnknownFields {
                        fields: None,
                    },
                    cached_size: CachedSize {
                        size: 0,
                    },
                },
            ],
            extent: Some(
                4096,
            ),
            unknown_fields: UnknownFields {
                fields: None,
            },
            cached_size: CachedSize {
                size: 0,
            },
        },
    ],
    unknown_fields: UnknownFields {
        fields: None,
    },
    cached_size: CachedSize {
        size: 0,
    },
}"#;

#[test]
fn test_build_mvt() {
    // https://github.com/mapbox/vector-tile-spec/tree/master/2.1#45-example
    let mut mvt_tile = vector_tile::Tile::new();

    let mut mvt_layer = vector_tile::Tile_Layer::new();
    mvt_layer.set_version(2);
    mvt_layer.set_name(String::from("points"));
    mvt_layer.set_extent(4096);

    let mut mvt_feature = vector_tile::Tile_Feature::new();
    mvt_feature.set_id(1);
    mvt_feature.set_field_type(vector_tile::Tile_GeomType::POINT);
    mvt_feature.set_geometry([9, 490, 6262].to_vec());

    let mut mvt_value = vector_tile::Tile_Value::new();
    mvt_value.set_string_value(String::from("world"));
    Tile::add_feature_attribute(
        &mut mvt_layer,
        &mut mvt_feature,
        String::from("hello"),
        mvt_value,
    );
    let mut mvt_value = vector_tile::Tile_Value::new();
    mvt_value.set_string_value(String::from("world"));
    Tile::add_feature_attribute(
        &mut mvt_layer,
        &mut mvt_feature,
        String::from("h"),
        mvt_value,
    );
    let mut mvt_value = vector_tile::Tile_Value::new();
    mvt_value.set_double_value(1.23);
    Tile::add_feature_attribute(
        &mut mvt_layer,
        &mut mvt_feature,
        String::from("count"),
        mvt_value,
    );

    mvt_layer.mut_features().push(mvt_feature);

    mvt_feature = vector_tile::Tile_Feature::new();
    mvt_feature.set_id(2);
    mvt_feature.set_field_type(vector_tile::Tile_GeomType::POINT);
    mvt_feature.set_geometry([9, 490, 6262].to_vec());

    let mut mvt_value = vector_tile::Tile_Value::new();
    mvt_value.set_string_value(String::from("again"));
    Tile::add_feature_attribute(
        &mut mvt_layer,
        &mut mvt_feature,
        String::from("hello"),
        mvt_value,
    );
    let mut mvt_value = vector_tile::Tile_Value::new();
    mvt_value.set_int_value(2);
    Tile::add_feature_attribute(
        &mut mvt_layer,
        &mut mvt_feature,
        String::from("count"),
        mvt_value,
    );

    mvt_layer.mut_features().push(mvt_feature);

    mvt_tile.mut_layers().push(mvt_layer);
    println!("{:#?}", mvt_tile);
    // Ignore trailing commas because of https://github.com/rust-lang/rust/pull/59076/
    assert_eq!(
        TILE_EXAMPLE.replace(",\n", "\n"),
        &*format!("{:#?}", mvt_tile).replace(",\n", "\n")
    );
}

/// Basic Feature implementation
struct FeatureStruct {
    fid: Option<u64>,
    attributes: Vec<FeatureAttr>,
    geometry: GeometryType,
}

impl Feature for FeatureStruct {
    fn fid(&self) -> Option<u64> {
        self.fid
    }
    fn attributes(&self) -> Vec<FeatureAttr> {
        self.attributes.clone()
    }
    fn geometry(&self) -> Result<GeometryType, String> {
        match self.geometry {
            GeometryType::Point(ref g) => Ok(GeometryType::Point(g.clone())),
            // Return dummy geometry because of missing Clone impl
            _ => Ok(GeometryType::Point(Point::new(0.0, 0.0, None))),
        }
    }
}

#[test]
fn test_build_mvt_with_helpers() {
    use std::env;

    let extent = Extent {
        minx: 958826.08,
        miny: 5987771.04,
        maxx: 978393.96,
        maxy: 6007338.92,
    };
    let mut tile = Tile::new(&extent, false);
    let layer = Layer::new("points");
    let mut mvt_layer = tile.new_layer(&layer);

    let geom = GeometryType::Point(geom::Point::new(960000.0, 6002729.0, Some(3857)));
    let feature = FeatureStruct {
        fid: Some(1),
        attributes: vec![
            FeatureAttr {
                key: String::from("hello"),
                value: FeatureAttrValType::String(String::from("world")),
            },
            FeatureAttr {
                key: String::from("h"),
                value: FeatureAttrValType::String(String::from("world")),
            },
            FeatureAttr {
                key: String::from("count"),
                value: FeatureAttrValType::Double(1.23),
            },
        ],
        geometry: geom,
    };
    tile.add_feature(&mut mvt_layer, &feature);

    let geom = GeometryType::Point(geom::Point::new(960000.0, 6002729.0, Some(3857)));
    let feature = FeatureStruct {
        fid: Some(2),
        attributes: vec![
            FeatureAttr {
                key: String::from("hello"),
                value: FeatureAttrValType::String(String::from("again")),
            },
            FeatureAttr {
                key: String::from("count"),
                value: FeatureAttrValType::Int(2),
            },
        ],
        geometry: geom,
    };
    tile.add_feature(&mut mvt_layer, &feature);

    tile.add_layer(mvt_layer);
    println!("{:#?}", tile.mvt_tile);
    assert_eq!(
        TILE_EXAMPLE.replace(",\n", "\n"),
        &*format!("{:#?}", tile.mvt_tile).replace(",\n", "\n")
    );

    let mut path = env::temp_dir();
    path.push("out.pbf");
    tile.to_file(&format!("{}", &path.display()));
}

#[test]
fn clip_points() {
    let extent = Extent {
        minx: 958826.08,
        miny: 5987771.04,
        maxx: 978393.96,
        maxy: 6007338.92,
    };
    let mut tile = Tile::new(&extent, true);
    let mut layer = Layer::new("points");
    layer.tile_size = 256;
    let mut mvt_layer = tile.new_layer(&layer);

    let point = geom::Point::new(960000.0, 6002729.0, Some(3857));
    assert_eq!(tile.point(&point), screen::Point { x: 15, y: 61 });
    let feature = FeatureStruct {
        fid: Some(1),
        attributes: Vec::new(),
        geometry: GeometryType::Point(point),
    };
    tile.add_feature(&mut mvt_layer, &feature);

    assert_eq!(mvt_layer.get_features().len(), 1);

    let point = geom::Point::new(990000.0, 6002729.0, Some(3857));
    assert_eq!(tile.point(&point), screen::Point { x: 407, y: 61 });
    let feature = FeatureStruct {
        fid: Some(1),
        attributes: Vec::new(),
        geometry: GeometryType::Point(point),
    };
    tile.add_feature(&mut mvt_layer, &feature);

    assert_eq!(mvt_layer.get_features().len(), 1);
}
