use std::fs::File;
use std::io::BufReader;
use std::path::PathBuf;
use std::process::Command;

use anyhow::{bail, Result};
use clap::Parser;
use log::*;
use merge::Merge;
use rpassword::{prompt_password, read_password_from_bufread};
use serde::Deserialize;

use crate::backend::{
    Cache, CachedBackend, ChooseBackend, DecryptBackend, DecryptReadBackend, DecryptWriteBackend,
    FileType, HotColdBackend, ReadBackend,
};
use crate::crypto::Key;
use crate::repofile::{find_key_in_backend, ConfigFile};

#[derive(Default, Parser, Deserialize, Merge)]
#[serde(default, rename_all = "kebab-case")]
pub struct RepositoryOptions {
    /// Repository to use
    #[clap(short, long, global = true, env = "RUSTIC_REPOSITORY")]
    repository: Option<String>,

    /// Repository to use as hot storage
    #[clap(long, global = true, env = "RUSTIC_REPO_HOT")]
    repo_hot: Option<String>,

    /// Password of the repository - WARNING: Using --password can reveal the password in the process list!
    #[clap(long, global = true, env = "RUSTIC_PASSWORD")]
    password: Option<String>,

    /// File to read the password from
    #[clap(
        short,
        long,
        global = true,
        parse(from_os_str),
        env = "RUSTIC_PASSWORD_FILE",
        conflicts_with = "password"
    )]
    password_file: Option<PathBuf>,

    /// Command to read the password from
    #[clap(
        long,
        global = true,
        env = "RUSTIC_PASSWORD_COMMAND",
        conflicts_with_all = &["password", "password-file"],
    )]
    password_command: Option<String>,

    /// Don't use a cache.
    #[clap(long, global = true, env = "RUSTIC_NO_CACHE")]
    #[merge(strategy = merge::bool::overwrite_false)]
    no_cache: bool,

    /// Use this dir as cache dir instead of the standard cache dir
    #[clap(
        long,
        global = true,
        parse(from_os_str),
        conflicts_with = "no-cache",
        env = "RUSTIC_CACHE_DIR"
    )]
    cache_dir: Option<PathBuf>,
}

pub struct Repository {
    pub(crate) name: String,
    pub(crate) be: HotColdBackend<ChooseBackend>,
    pub(crate) be_hot: Option<ChooseBackend>,
    pub(crate) opts: RepositoryOptions,
}

impl Repository {
    pub fn new(opts: RepositoryOptions) -> Result<Self> {
        let be = match &opts.repository {
            Some(repo) => ChooseBackend::from_url(repo)?,
            None => bail!("No repository given. Please use the --repository option."),
        };

        let be_hot = opts
            .repo_hot
            .as_ref()
            .map(|repo| ChooseBackend::from_url(repo))
            .transpose()?;

        let be = HotColdBackend::new(be, be_hot.clone());
        let mut name = opts.repository.as_ref().unwrap().clone();
        if let Some(repo_hot) = &opts.repo_hot {
            name.push('#');
            name.push_str(repo_hot);
        }

        Ok(Self {
            name,
            be,
            be_hot,
            opts,
        })
    }

    pub fn password(&self) -> Result<Option<String>> {
        match (
            &self.opts.password,
            &self.opts.password_file,
            &self.opts.password_command,
        ) {
            (Some(pwd), _, _) => Ok(Some(pwd.clone())),
            (_, Some(file), _) => {
                let mut file = BufReader::new(File::open(file)?);
                Ok(Some(read_password_from_bufread(&mut file)?))
            }
            (_, _, Some(command)) => {
                let mut commands: Vec<_> = command.split(' ').collect();
                let output = Command::new(commands[0])
                    .args(&mut commands[1..])
                    .output()?;

                let mut pwd = BufReader::new(&*output.stdout);
                Ok(Some(read_password_from_bufread(&mut pwd)?))
            }
            (None, None, None) => Ok(None),
        }
    }

    pub fn open(self) -> Result<OpenRepository> {
        let config_ids = self.be.list(FileType::Config)?;

        match config_ids.len() {
            1 => {} // ok, continue
            0 => bail!("No config file found. Is there a repo at {}?", self.name),
            _ => bail!("More than one config file at {}. Aborting.", self.name),
        }

        if let Some(be_hot) = &self.be_hot {
            let mut keys = self.be.list_with_size(FileType::Key)?;
            keys.sort_unstable_by_key(|key| key.0);
            let mut hot_keys = be_hot.list_with_size(FileType::Key)?;
            hot_keys.sort_unstable_by_key(|key| key.0);
            if keys != hot_keys {
                bail!(
                    "keys from repo and repo-hot do not match for {}. Aborting.",
                    self.name
                );
            }
        }

        let key = get_key(&self.be, self.password()?)?;
        info!("repository {}: password is correct.", self.name);

        let dbe = DecryptBackend::new(&self.be, key.clone());
        let config: ConfigFile = dbe.get_file(&config_ids[0])?;
        match (config.is_hot == Some(true), self.be_hot.is_some()) {
                (true, false) => bail!("repository is a hot repository!\nPlease use as --repo-hot in combination with the normal repo. Aborting."),
                (false, true) => bail!("repo-hot is not a hot repository! Aborting."),
                _ => {}
            }
        let cache = (!self.opts.no_cache)
            .then(|| Cache::new(config.id, self.opts.cache_dir).ok())
            .flatten();
        match &cache {
            None => info!("using no cache"),
            Some(cache) => info!("using cache at {}", cache.location()),
        }
        let be_cached = CachedBackend::new(self.be.clone(), cache.clone());
        let mut dbe = DecryptBackend::new(&be_cached, key.clone());
        let zstd = config.zstd()?;
        dbe.set_zstd(zstd);

        Ok(OpenRepository {
            name: self.name,
            key,
            dbe,
            cache,
            be: self.be,
            be_hot: self.be_hot,
            config,
        })
    }
}

pub struct OpenRepository {
    pub(crate) name: String,
    pub(crate) be: HotColdBackend<ChooseBackend>,
    pub(crate) be_hot: Option<ChooseBackend>,
    pub(crate) key: Key,
    pub(crate) cache: Option<Cache>,
    pub(crate) dbe: DecryptBackend<CachedBackend<HotColdBackend<ChooseBackend>>, Key>,
    pub(crate) config: ConfigFile,
}

const MAX_PASSWORD_RETRIES: usize = 5;
pub fn get_key(be: &impl ReadBackend, password: Option<String>) -> Result<Key> {
    for _ in 0..MAX_PASSWORD_RETRIES {
        match password {
            // if password is given, directly return the result of find_key_in_backend and don't retry
            Some(pass) => return find_key_in_backend(be, &pass, None),
            None => {
                // TODO: Differentiate between wrong password and other error!
                if let Ok(key) =
                    find_key_in_backend(be, &prompt_password("enter repository password: ")?, None)
                {
                    return Ok(key);
                }
            }
        }
    }
    bail!("incorrect password!");
}
