//! Various batch processing tasks, intended primarily for debugging.

pub mod flags;
pub mod load_cargo;
mod parse;
mod symbols;
mod highlight;
mod analysis_stats;
mod diagnostics;
mod ssr;
mod lsif;

mod progress_report;

use std::io::Read;

use anyhow::Result;
use ide::AnalysisHost;
use vfs::Vfs;

#[derive(Clone, Copy)]
pub enum Verbosity {
    Spammy,
    Verbose,
    Normal,
    Quiet,
}

impl Verbosity {
    pub fn is_verbose(self) -> bool {
        matches!(self, Verbosity::Verbose | Verbosity::Spammy)
    }
    pub fn is_spammy(self) -> bool {
        matches!(self, Verbosity::Spammy)
    }
}

fn read_stdin() -> Result<String> {
    let mut buff = String::new();
    std::io::stdin().read_to_string(&mut buff)?;
    Ok(buff)
}

fn report_metric(metric: &str, value: u64, unit: &str) {
    if std::env::var("RA_METRICS").is_err() {
        return;
    }
    println!("METRIC:{}:{}:{}", metric, value, unit)
}

fn print_memory_usage(mut host: AnalysisHost, vfs: Vfs) {
    let mut mem = host.per_query_memory_usage();

    let before = profile::memory_usage();
    drop(vfs);
    let vfs = before.allocated - profile::memory_usage().allocated;
    mem.push(("VFS".into(), vfs));

    let before = profile::memory_usage();
    drop(host);
    mem.push(("Unaccounted".into(), before.allocated - profile::memory_usage().allocated));

    mem.push(("Remaining".into(), profile::memory_usage().allocated));

    for (name, bytes) in mem {
        // NOTE: Not a debug print, so avoid going through the `eprintln` defined above.
        eprintln!("{:>8} {}", bytes, name);
    }
}
