//! Grammar for the command-line arguments.
#![allow(unreachable_pub)]
use std::{env, path::PathBuf};

use ide_ssr::{SsrPattern, SsrRule};
use rust_analyzer::cli::{BenchWhat, Position, Verbosity};
use vfs::AbsPathBuf;

xflags::xflags! {
    src "./src/bin/flags.rs"

    /// LSP server for the Rust programming language.
    cmd rust-analyzer {
        /// Verbosity level, can be repeated multiple times.
        repeated -v, --verbose
        /// Verbosity level.
        optional -q, --quiet

        /// Log to the specified file instead of stderr.
        optional --log-file path: PathBuf
        /// Flush log records to the file immediately.
        optional --no-log-buffering

        /// Wait until a debugger is attached to (requires debug build).
        optional --wait-dbg

        default cmd lsp-server {
            /// Print version.
            optional --version
            /// Print help.
            optional -h, --help

            /// Dump a LSP config JSON schema.
            optional --print-config-schema
        }

        /// Parse stdin.
        cmd parse {
            /// Suppress printing.
            optional --no-dump
        }

        /// Parse stdin and print the list of symbols.
        cmd symbols {}

        /// Highlight stdin as html.
        cmd highlight {
            /// Enable rainbow highlighting of identifiers.
            optional --rainbow
        }

        /// Batch typecheck project and print summary statistics
        cmd analysis-stats
            /// Directory with Cargo.toml.
            required path: PathBuf
        {
            /// Randomize order in which crates, modules, and items are processed.
            optional --randomize
            /// Run type inference in parallel.
            optional --parallel
            /// Collect memory usage statistics.
            optional --memory-usage

            /// Only analyze items matching this path.
            optional -o, --only path: String
            /// Also analyze all dependencies.
            optional --with-deps

            /// Load OUT_DIR values by running `cargo check` before analysis.
            optional --load-output-dirs
            /// Use proc-macro-srv for proc-macro expanding.
            optional --with-proc-macro
        }

        /// Benchmark specific analysis operation
        cmd analysis-bench
            /// Directory with Cargo.toml.
            required path: PathBuf
        {
            /// Collect memory usage statistics.
            optional --memory-usage

            /// Compute syntax highlighting for this file
            optional --highlight path: PathBuf
            /// Compute completions at file:line:column location.
            optional --complete location: Position
            /// Compute goto definition at file:line:column location.
            optional --goto-def location: Position

            /// Load OUT_DIR values by running `cargo check` before analysis.
            optional --load-output-dirs
            /// Use proc-macro-srv for proc-macro expanding.
            optional --with-proc-macro
        }

        cmd diagnostics
            /// Directory with Cargo.toml.
            required path: PathBuf
        {
            /// Load OUT_DIR values by running `cargo check` before analysis.
            optional --load-output-dirs
            /// Use proc-macro-srv for proc-macro expanding.
            optional --with-proc-macro
        }

        cmd ssr
            /// A structured search replace rule (`$a.foo($b) ==> bar($a, $b)`)
            repeated rule: SsrRule
        {}

        cmd search
            /// A structured search replace pattern (`$a.foo($b)`)
            repeated pattern: SsrPattern
        {
            /// Prints debug information for any nodes with source exactly equal to snippet.
            optional --debug snippet: String
        }

        cmd proc-macro {}
    }
}

// generated start
// The following code is generated by `xflags` macro.
// Run `env UPDATE_XFLAGS=1 cargo build` to regenerate.
#[derive(Debug)]
pub struct RustAnalyzer {
    pub verbose: u32,
    pub quiet: bool,
    pub log_file: Option<PathBuf>,
    pub no_log_buffering: bool,
    pub wait_dbg: bool,
    pub subcommand: RustAnalyzerCmd,
}

#[derive(Debug)]
pub enum RustAnalyzerCmd {
    LspServer(LspServer),
    Parse(Parse),
    Symbols(Symbols),
    Highlight(Highlight),
    AnalysisStats(AnalysisStats),
    AnalysisBench(AnalysisBench),
    Diagnostics(Diagnostics),
    Ssr(Ssr),
    Search(Search),
    ProcMacro(ProcMacro),
}

#[derive(Debug)]
pub struct LspServer {
    pub version: bool,
    pub help: bool,
    pub print_config_schema: bool,
}

#[derive(Debug)]
pub struct Parse {
    pub no_dump: bool,
}

#[derive(Debug)]
pub struct Symbols;

#[derive(Debug)]
pub struct Highlight {
    pub rainbow: bool,
}

#[derive(Debug)]
pub struct AnalysisStats {
    pub path: PathBuf,

    pub randomize: bool,
    pub parallel: bool,
    pub memory_usage: bool,
    pub only: Option<String>,
    pub with_deps: bool,
    pub load_output_dirs: bool,
    pub with_proc_macro: bool,
}

#[derive(Debug)]
pub struct AnalysisBench {
    pub path: PathBuf,

    pub memory_usage: bool,
    pub highlight: Option<PathBuf>,
    pub complete: Option<Position>,
    pub goto_def: Option<Position>,
    pub load_output_dirs: bool,
    pub with_proc_macro: bool,
}

#[derive(Debug)]
pub struct Diagnostics {
    pub path: PathBuf,

    pub load_output_dirs: bool,
    pub with_proc_macro: bool,
}

#[derive(Debug)]
pub struct Ssr {
    pub rule: Vec<SsrRule>,
}

#[derive(Debug)]
pub struct Search {
    pub pattern: Vec<SsrPattern>,

    pub debug: Option<String>,
}

#[derive(Debug)]
pub struct ProcMacro;

impl RustAnalyzer {
    pub const HELP: &'static str = Self::HELP_;

    pub fn from_env() -> xflags::Result<Self> {
        Self::from_env_()
    }
}
// generated end

impl RustAnalyzer {
    pub(crate) fn verbosity(&self) -> Verbosity {
        if self.quiet {
            return Verbosity::Quiet;
        }
        match self.verbose {
            0 => Verbosity::Normal,
            1 => Verbosity::Verbose,
            _ => Verbosity::Spammy,
        }
    }
}

impl AnalysisBench {
    pub(crate) fn what(&self) -> BenchWhat {
        match (&self.highlight, &self.complete, &self.goto_def) {
            (Some(path), None, None) => {
                let path = env::current_dir().unwrap().join(path);
                BenchWhat::Highlight { path: AbsPathBuf::assert(path) }
            }
            (None, Some(position), None) => BenchWhat::Complete(position.clone()),
            (None, None, Some(position)) => BenchWhat::GotoDef(position.clone()),
            _ => panic!("exactly one of  `--highlight`, `--complete` or `--goto-def` must be set"),
        }
    }
}
