use clap::{SubCommand, ArgMatches, Arg};

use super::{BasicOptions, StaticSubcommand, set_current_branch, get_current_branch, default_explain};
use error::{ErrorKind, Result};
use libpijul::patch::UnsignedPatch;
use libpijul::{RecordState, FileStatus};
use std::collections::HashSet;
use rand;

pub fn invocation() -> StaticSubcommand {
    return SubCommand::with_name("checkout")
        .about("Change the current branch")
        .arg(Arg::with_name("repository")
             .long("repository")
             .help("Local repository.")
             .takes_value(true)
        )
        .arg(Arg::with_name("branch")
             .help("Branch to switch to.")
             .takes_value(true)
        )
        .arg(Arg::with_name("force")
             .short("f")
             .long("force")
             .takes_value(false)
             .help("Only check files moves, deletions and additions (much faster).")
        )
}

pub fn run(args: &ArgMatches) -> Result<()> {
    let opts = BasicOptions::from_args(args)?;
    if let Some(branch) = args.value_of("branch") {
        let repo = opts.open_and_grow_repo(409600)?;
        let mut txn = repo.mut_txn_begin(rand::thread_rng())?;
        let current_branch = get_current_branch(&opts.repo_root)?;
        // We need to check at least that there are no file
        // moves/additions/deletions, because these would be
        // overwritten by the checkout, sometimes causing Pijul to
        // panic.
        if args.is_present("force") {
            // Check whether there are file moves.
            if txn.iter_inodes(None).any(|(_, ch)| ch.status != FileStatus::Ok) {
                return Err(ErrorKind::PendingChanges.into());
            }
        } else {
            // Check whether there are more general changes.
            let mut record = RecordState::new();
            txn.record(&mut record, &current_branch, &opts.repo_root, None)?;
            let (changes, _) = record.finish();

            if !changes.is_empty() {
                return Err(ErrorKind::PendingChanges.into());
            }
        }

        if !txn.get_branch(branch).is_some() {
            return Err(ErrorKind::NoSuchBranch.into());
        }

        txn.output_repository(
            &branch,
            &opts.repo_root,
            None,
            &UnsignedPatch::empty().leave_unsigned(),
            &HashSet::new()
        )?;
        txn.commit()?;

        set_current_branch(&opts.repo_root, branch)?;

        println!("Current branch: {:?}", get_current_branch(&opts.repo_root)?);
        Ok(())

    } else {
        Err(ErrorKind::NoSuchBranch.into())
    }
}

pub fn explain(res: Result<()>) {
    default_explain(res)
}
