use nu_engine::eval_expression;
use nu_protocol::{
    ast::Call,
    engine::{EngineState, Stack},
    FromValue, ShellError, Span, Spanned, Value,
};
use serde::{Deserialize, Serialize};

/// A representation of the plugin's invocation command including command line args
///
/// The `EvaluatedCall` contains information about the way a [Plugin](crate::Plugin) was invoked
/// representing the [`Span`] corresponding to the invocation as well as the arguments
/// it was invoked with. It is one of three items passed to [`run`](crate::Plugin::run()) along with
/// `name` which command that was invoked and a [`Value`] that represents the input.
///
/// The evaluated call is used with the Plugins because the plugin doesn't have
/// access to the Stack and the EngineState the way a built in command might. For that
/// reason, before encoding the message to the plugin all the arguments to the original
/// call (which are expressions) are evaluated and passed to Values
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct EvaluatedCall {
    /// Span of the command invocation
    pub head: Span,
    /// Values of positional arguments
    pub positional: Vec<Value>,
    /// Names and values of named arguments
    pub named: Vec<(Spanned<String>, Option<Value>)>,
}

impl EvaluatedCall {
    pub(crate) fn try_from_call(
        call: &Call,
        engine_state: &EngineState,
        stack: &mut Stack,
    ) -> Result<Self, ShellError> {
        let positional = call
            .positional_iter()
            .map(|expr| eval_expression(engine_state, stack, expr))
            .collect::<Result<Vec<Value>, ShellError>>()?;

        let mut named = Vec::with_capacity(call.named_len());
        for (string, _, expr) in call.named_iter() {
            let value = match expr {
                None => None,
                Some(expr) => Some(eval_expression(engine_state, stack, expr)?),
            };

            named.push((string.clone(), value))
        }

        Ok(Self {
            head: call.head,
            positional,
            named,
        })
    }

    /// Indicates whether named parameter is present in the arguments
    ///
    /// Typically this method would be used on a flag parameter, a named parameter
    /// that does not take a value.
    ///
    /// # Examples
    /// Invoked as `my_command --foo`:
    /// ```
    /// # use nu_protocol::{Spanned, Span, Value};
    /// # use nu_plugin::EvaluatedCall;
    /// # let null_span = Span::new(0, 0);
    /// # let call = EvaluatedCall {
    /// #     head: null_span,
    /// #     positional: Vec::new(),
    /// #     named: vec![(
    /// #         Spanned { item: "foo".to_owned(), span: null_span},
    /// #         None
    /// #     )],
    /// # };
    /// assert!(call.has_flag("foo"));
    /// ```
    ///
    /// Invoked as `my_command --bar`:
    /// ```
    /// # use nu_protocol::{Spanned, Span, Value};
    /// # use nu_plugin::EvaluatedCall;
    /// # let null_span = Span::new(0, 0);
    /// # let call = EvaluatedCall {
    /// #     head: null_span,
    /// #     positional: Vec::new(),
    /// #     named: vec![(
    /// #         Spanned { item: "bar".to_owned(), span: null_span},
    /// #         None
    /// #     )],
    /// # };
    /// assert!(!call.has_flag("foo"));
    /// ```
    pub fn has_flag(&self, flag_name: &str) -> bool {
        for name in &self.named {
            if flag_name == name.0.item {
                return true;
            }
        }

        false
    }

    /// Returns the [`Value`] of an optional named argument
    ///
    /// # Examples
    /// Invoked as `my_command --foo 123`:
    /// ```
    /// # use nu_protocol::{Spanned, Span, Value};
    /// # use nu_plugin::EvaluatedCall;
    /// # let null_span = Span::new(0, 0);
    /// # let call = EvaluatedCall {
    /// #     head: null_span,
    /// #     positional: Vec::new(),
    /// #     named: vec![(
    /// #         Spanned { item: "foo".to_owned(), span: null_span},
    /// #         Some(Value::int(123, null_span))
    /// #     )],
    /// # };
    /// let opt_foo = match call.get_flag_value("foo") {
    ///     Some(Value::Int { val, .. }) => Some(val),
    ///     None => None,
    ///     _ => panic!(),
    /// };
    /// assert_eq!(opt_foo, Some(123));
    /// ```
    ///
    /// Invoked as `my_command`:
    /// ```
    /// # use nu_protocol::{Spanned, Span, Value};
    /// # use nu_plugin::EvaluatedCall;
    /// # let null_span = Span::new(0, 0);
    /// # let call = EvaluatedCall {
    /// #     head: null_span,
    /// #     positional: Vec::new(),
    /// #     named: vec![],
    /// # };
    /// let opt_foo = match call.get_flag_value("foo") {
    ///     Some(Value::Int { val, .. }) => Some(val),
    ///     None => None,
    ///     _ => panic!(),
    /// };
    /// assert_eq!(opt_foo, None);
    /// ```
    pub fn get_flag_value(&self, flag_name: &str) -> Option<Value> {
        for name in &self.named {
            if flag_name == name.0.item {
                return name.1.clone();
            }
        }

        None
    }

    /// Returns the [`Value`] of a given (zero indexed) positional argument, if present
    ///
    /// Examples:
    /// Invoked as `my_command a b c`:
    /// ```
    /// # use nu_protocol::{Spanned, Span, Value};
    /// # use nu_plugin::EvaluatedCall;
    /// # let null_span = Span::new(0, 0);
    /// # let call = EvaluatedCall {
    /// #     head: null_span,
    /// #     positional: vec![
    /// #         Value::string("a".to_owned(), null_span),
    /// #         Value::string("b".to_owned(), null_span),
    /// #         Value::string("c".to_owned(), null_span),
    /// #     ],
    /// #     named: vec![],
    /// # };
    /// let arg = match call.nth(1) {
    ///     Some(Value::String { val, .. }) => val,
    ///     _ => panic!(),
    /// };
    /// assert_eq!(arg, "b".to_owned());
    ///
    /// let arg = call.nth(7);
    /// assert!(arg.is_none());
    /// ```
    pub fn nth(&self, pos: usize) -> Option<Value> {
        self.positional.get(pos).cloned()
    }

    /// Returns the value of a named argument interpreted as type `T`
    ///
    /// # Examples
    /// Invoked as `my_command --foo 123`:
    /// ```
    /// # use nu_protocol::{Spanned, Span, Value};
    /// # use nu_plugin::EvaluatedCall;
    /// # let null_span = Span::new(0, 0);
    /// # let call = EvaluatedCall {
    /// #     head: null_span,
    /// #     positional: Vec::new(),
    /// #     named: vec![(
    /// #         Spanned { item: "foo".to_owned(), span: null_span},
    /// #         Some(Value::int(123, null_span))
    /// #     )],
    /// # };
    /// let foo = call.get_flag::<i64>("foo");
    /// assert_eq!(foo.unwrap(), Some(123));
    /// ```
    ///
    /// Invoked as `my_command --bar 123`:
    /// ```
    /// # use nu_protocol::{Spanned, Span, Value};
    /// # use nu_plugin::EvaluatedCall;
    /// # let null_span = Span::new(0, 0);
    /// # let call = EvaluatedCall {
    /// #     head: null_span,
    /// #     positional: Vec::new(),
    /// #     named: vec![(
    /// #         Spanned { item: "bar".to_owned(), span: null_span},
    /// #         Some(Value::int(123, null_span))
    /// #     )],
    /// # };
    /// let foo = call.get_flag::<i64>("foo");
    /// assert_eq!(foo.unwrap(), None);
    /// ```
    ///
    /// Invoked as `my_command --foo abc`:
    /// ```
    /// # use nu_protocol::{Spanned, Span, Value};
    /// # use nu_plugin::EvaluatedCall;
    /// # let null_span = Span::new(0, 0);
    /// # let call = EvaluatedCall {
    /// #     head: null_span,
    /// #     positional: Vec::new(),
    /// #     named: vec![(
    /// #         Spanned { item: "foo".to_owned(), span: null_span},
    /// #         Some(Value::string("abc".to_owned(), null_span))
    /// #     )],
    /// # };
    /// let foo = call.get_flag::<i64>("foo");
    /// assert!(foo.is_err());
    /// ```
    pub fn get_flag<T: FromValue>(&self, name: &str) -> Result<Option<T>, ShellError> {
        if let Some(value) = self.get_flag_value(name) {
            FromValue::from_value(value).map(Some)
        } else {
            Ok(None)
        }
    }

    /// Retrieve the Nth and all following positional arguments as type `T`
    ///
    /// # Example
    /// Invoked as `my_command zero one two three`:
    /// ```
    /// # use nu_protocol::{Spanned, Span, Value};
    /// # use nu_plugin::EvaluatedCall;
    /// # let null_span = Span::new(0, 0);
    /// # let call = EvaluatedCall {
    /// #     head: null_span,
    /// #     positional: vec![
    /// #         Value::string("zero".to_owned(), null_span),
    /// #         Value::string("one".to_owned(), null_span),
    /// #         Value::string("two".to_owned(), null_span),
    /// #         Value::string("three".to_owned(), null_span),
    /// #     ],
    /// #     named: Vec::new(),
    /// # };
    /// let args = call.rest::<String>(0);
    /// assert_eq!(args.unwrap(), vec!["zero", "one", "two", "three"]);
    ///
    /// let args = call.rest::<String>(2);
    /// assert_eq!(args.unwrap(), vec!["two", "three"]);
    /// ```
    pub fn rest<T: FromValue>(&self, starting_pos: usize) -> Result<Vec<T>, ShellError> {
        self.positional
            .iter()
            .skip(starting_pos)
            .map(|value| FromValue::from_value(value.clone()))
            .collect()
    }

    /// Retrieve the value of an optional positional argument interpreted as type `T`
    ///
    /// Returns the value of a (zero indexed) positional argument of type `T`.
    /// Alternatively returns [`None`] if the positional argument does not exist
    /// or an error that can be passed back to the shell on error.
    pub fn opt<T: FromValue>(&self, pos: usize) -> Result<Option<T>, ShellError> {
        if let Some(value) = self.nth(pos) {
            FromValue::from_value(value).map(Some)
        } else {
            Ok(None)
        }
    }

    /// Retrieve the value of a mandatory positional argument as type `T`
    ///
    /// Expect a positional argument of type `T` and return its value or, if the
    /// argument does not exist or is of the wrong type, return an error that can
    /// be passed back to the shell.
    pub fn req<T: FromValue>(&self, pos: usize) -> Result<T, ShellError> {
        if let Some(value) = self.nth(pos) {
            FromValue::from_value(value)
        } else if self.positional.is_empty() {
            Err(ShellError::AccessEmptyContent { span: self.head })
        } else {
            Err(ShellError::AccessBeyondEnd {
                max_idx: self.positional.len() - 1,
                span: self.head,
            })
        }
    }
}

#[cfg(test)]
mod test {
    use super::*;
    use nu_protocol::{Span, Spanned, Value};

    #[test]
    fn call_to_value() {
        let call = EvaluatedCall {
            head: Span::new(0, 10),
            positional: vec![
                Value::float(1.0, Span::new(0, 10)),
                Value::string("something", Span::new(0, 10)),
            ],
            named: vec![
                (
                    Spanned {
                        item: "name".to_string(),
                        span: Span::new(0, 10),
                    },
                    Some(Value::float(1.0, Span::new(0, 10))),
                ),
                (
                    Spanned {
                        item: "flag".to_string(),
                        span: Span::new(0, 10),
                    },
                    None,
                ),
            ],
        };

        let name: Option<f64> = call.get_flag("name").unwrap();
        assert_eq!(name, Some(1.0));

        assert!(call.has_flag("flag"));

        let required: f64 = call.req(0).unwrap();
        assert!((required - 1.0).abs() < f64::EPSILON);

        let optional: Option<String> = call.opt(1).unwrap();
        assert_eq!(optional, Some("something".to_string()));

        let rest: Vec<String> = call.rest(1).unwrap();
        assert_eq!(rest, vec!["something".to_string()]);
    }
}
