//! Various minor utilities.

use cfg_if::cfg_if;
use once_cell::sync::Lazy;

use crate::String;

/// The current version of fif, as defined in Cargo.toml.
pub const VERSION: Option<&'static str> = option_env!("CARGO_PKG_VERSION");

cfg_if! {
	if #[cfg(any(all(unix, feature = "infer-backend"), all(not(unix), not(feature = "xdg-mime-backend"))))] {
		/// The backend being used; either "Infer" or "XDG-Mime".
		pub const BACKEND: &str = "Infer";
	} else {
		/// The backend being used; either "Infer" or "XDG-Mime".
		pub const BACKEND: &str = "XDG-Mime";
	}
}

/// The version defined in Cargo.toml, prefixed with a v (e.g. "v0.3.1")
pub(crate) static CLAP_VERSION: Lazy<String> = Lazy::new(|| String::from("v") + VERSION.unwrap_or("???"));

/// Similar to [`CLAP_VERSION`], followed by the chosen backend and abbreviated git commit hash in parentheses - For
/// example, "v0.3.6 (XDG-Mime backend, commit #043e097)"
pub static CLAP_LONG_VERSION: Lazy<String> = Lazy::new(|| {
	format!(
		"v{} ({} backend, commit #{})",
		VERSION.unwrap_or("???"),
		BACKEND,
		option_env!("GIT_SHA").unwrap_or("???")
	)
	.into()
});

/// Returns the name of the target operating system with proper casing, like "Windows" or "macOS".
#[allow(clippy::option_map_unit_fn)]
pub fn os_name() -> String {
	match std::env::consts::OS {
		// special cases: "ios" should not be capitalised into "Ios", for example
		"ios" => "iOS".into(),
		"macos" => "macOS".into(),
		"freebsd" => "FreeBSD".into(),
		"openbsd" => "OpenBSD".into(),
		"netbsd" => "NetBSD".into(),
		"vxworks" => "VxWorks".into(),
		os => {
			// generic case: return consts::OS with the first letter in uppercase ("linux" -> "Linux")
			let mut os_upper = String::from(os);
			os_upper.get_mut(0..1).map(|first| first.make_ascii_uppercase());
			os_upper
		}
	}
}
