(* 
  HereDoc
  Alain Frisch  <Alain.Frisch@ens.fr>
  modified by remi vanicat <vanicat@labri.u-bordeaux.fr>
*)


open Pcaml

(*** Processing variables ***)

(* var_handler is the name of the function to be called to resolve
   "variables" in here documents *)
let var_handler = ref "var"

(* template_file is the default filename for template locator *)
let template_file = ref "templates.tpl"

let unit loc = let e = <:expr< () >> in <:str_item< $exp:e$ >>

EXTEND
 str_item: AFTER "top"
 [[ "VAR_HANDLER"; s = STRING ->
	 var_handler := s;
         unit loc
  | "TEMPLATE_FILE"; s = STRING ->
         template_file := s;
	 unit loc
  ]];
END

(*** Stack of parsed files ***)

let tpl_files = ref ([] : string list)

let enter_file s =  tpl_files := s :: !tpl_files 
let leave_file () =  tpl_files := List.tl !tpl_files
let cur_file () =  match !tpl_files with [] -> !input_file | t::_ -> t
let cur_tpl_file () = match !tpl_files with [] -> !template_file | t::_ -> t


(*** Parsing an Caml expression (with current grammar); ***)

let caml_parse entry s =
  let pars = Grammar.Entry.parse entry in
  let strea= Stream.of_string s in
  pars strea


let caml_parse_expr = caml_parse Pcaml.expr

let caml_parse_implem s =
  List.map fst (fst (caml_parse Pcaml.implem s))

(*
let caml_parse_implem s =
  List.map fst (caml_parse Pcaml.implem s)
*)

let here_abstr = Grammar.Entry.create gram "here_abstr"

EXTEND
  here_abstr:  [[ e = expr; pl =  LIST0 [ "->" ; p = patt -> p]  -> (e,pl)  ]];
END

let caml_parse_here_abstr =  caml_parse here_abstr

let build_fun loc args e =
  List.fold_right (fun arg e -> <:expr< fun [~ $arg$ -> $anti:e$] >>) args e



(*** The parsing function for here document ***)

let loc = (Lexing.dummy_pos,Lexing.dummy_pos)

let here_expr args s =
  let buf = Lexing.from_string s in

  let text_repr e = <:expr< $uid:"Text"$ . $lid: "repr"$ $e$ >> in

  let text_of_list = function
    | [] ->    text_repr <:expr< $str:""$ >>
    | [e] ->   text_repr e
    | toks ->  text_repr <:expr< [| $list:toks$ |]>>
  in
  
  let empty = text_of_list [] in

  let condition (cond,loc) e =
    if cond="" then e
    else 
      let cond = caml_parse_expr cond in
      <:expr< if $cond$ then $e$ else $empty$ >> 
  in
  
  let pos i = { Lexing.pos_fname = ""; Lexing.pos_lnum = 0;
	       Lexing.pos_bol = 0; Lexing.pos_cnum = i } in
  let loc = (pos 0, pos (String.length s)) in

  let transl_loc (i,j) = (pos i, pos j) in

  let rec parse toplevel = 
    text_of_list (parse_list toplevel)

  and parse_list toplevel =
  
    let cur_cond = ref (Some ("",loc)) in
    
    let rec tok () =  match HereDoc_lexer.token buf with
      | HereDoc_lexer.Literal s ->
	  if s = "" then tok ()
	  else 
	    let e = text_repr <:expr< $str: s$>> in 
	    e :: (tok ())
      | HereDoc_lexer.Expr (s,loc) ->
	  let e = text_repr (caml_parse_expr s) in 
	  e :: (tok ())
      | HereDoc_lexer.ExtVar (s,loc) ->
	  let loc = transl_loc loc in
	  let e = text_repr <:expr< $lid: !var_handler$ $str:s$ >> in
	  e :: (tok ())
      | HereDoc_lexer.Special (s,loc) ->
	  let loc = transl_loc loc in
	  let (cmd, arg) =
	    try
	      let i = String.index s ':' in
	      (String.sub s 0 i, String.sub s (i+1) (String.length s - i - 1))
	    with Not_found -> (s,"") in
	  begin match cmd with
	    | "filename" -> 
		let e = text_repr <:expr< $str:cur_file ()$ >> in
		e :: tok ()
	    | "lastmod" ->
		let st = Unix.stat (cur_file ()) in
		let mtime = st.Unix.st_mtime in
		let tm = Unix.localtime mtime in
		let y = tm.Unix.tm_year + 1900
		and m = tm.Unix.tm_mon + 1
		and d = tm.Unix.tm_mday in
		let e =
		  if arg = "" then
		    let s = Printf.sprintf "%i-%02i-%02i" y m d in
		    text_repr <:expr< $str:s$ >>
		  else
		    let f = caml_parse_expr arg in
		    text_repr <:expr< $f$ 
		      $int:string_of_int y$ $int:string_of_int m$ 
		      $int:string_of_int d$>> in
		e :: tok ()
	    | _ -> failwith ("Unknown Special "^s^" in quotation")
	  end
      | HereDoc_lexer.Conditional (s,loc) ->
	  let loc = transl_loc loc in
	  cur_cond := Some (s,loc);
	  []
      | HereDoc_lexer.Abstraction (s,loc) ->
	  let loc = transl_loc loc in
	  if s = "" then 
	    if toplevel then 
	      failwith "Unexpected end of abstraction in quotation" 
	    else
	      (cur_cond := None; [])
	  else
	    let (abs_e,abs_pl) = caml_parse_here_abstr s in
	    let e = parse false in
	    let e = List.fold_right 
		      (fun p e -> <:expr< fun $p$ -> $e$ >>) abs_pl e in
	    let e = text_repr <:expr< $abs_e$ $e$ >> in
	    
	    e :: (tok ())
      | HereDoc_lexer.Textend -> 
	  cur_cond := None;
	  []
    in

    let rec aux () = match !cur_cond with
      | Some (cond,loc) -> 
	  let e = text_of_list (tok ()) in
	  let e = condition (cond,loc) e in
	  e :: (aux())
      | None -> 
	  []
    in
    aux ()

  in
  let e = parse true in
  build_fun loc args e


let here_pat x = failwith "HereDoc quotation can't match anything"

let _ =
  Quotation.add "here" (Quotation.ExAst (here_expr [], here_pat));
  Quotation.default := "here"


(*** External files and templates ***)

let open_file filename =
  if Filename.is_implicit filename then
    open_in (Filename.concat (Filename.dirname !input_file) filename)
  else
    open_in filename

let file filename =
  let f = open_file filename in
  let n = in_channel_length f in
  let s = String.create n in
  really_input f s 0 n;
  close_in f;
  s

let split_args_re = Str.regexp "[ \t]+"

let file_part filename part =
  let f = open_file filename in

  let start_re = Str.regexp ("^=="^part^"\\( .*\\)?==$") in

  let rec find_start () =
    let s = input_line f in
    if (Str.string_match start_re s 0) then
      try Str.split split_args_re (Str.matched_group 1 s)
      with Not_found -> []
    else
      find_start ()
  in	

  let rec find s = 
    let p = pos_in f in
    if (input_line f = s) then p else find s in
  let (args,pos,n) = 
    try
      let args = find_start () in
      let pos = pos_in f  in 
      let n = find "====" - pos - 1 in
      (args,pos,n)
    with End_of_file -> failwith 
	(Printf.sprintf "Template \"%s\".\"%s\" not found" filename part)
  in
  seek_in f pos;
  let s = String.create n in
  really_input f s 0 n;
  close_in f;
  (args,s)




let tpl_loc = Grammar.Entry.create gram "template locator"
let str_items_semi = Grammar.Entry.create gram "str_items_semi"
let str_item_semi = Grammar.Entry.create gram "str_item_semi"

DELETE_RULE
  implem: str_item_semi;  SELF
END
DELETE_RULE
  implem: "#"; LIDENT; OPT expr; ";;"
END
DELETE_RULE
  implem: EOI
END
(*
LIST0 [ str_item; OPT ";;" ]; EOI *)
DELETE_RULE
  module_expr: "struct"; LIST0 [ str_item; OPT ";;" ]; "end"
END

EXTEND
expr: BEFORE "simple"
 [[ "TPL"; tpl = tpl_loc ->
      let (args,tpl) = tpl in
      let e = here_expr args tpl in
      leave_file ();
      e
  | "VERBATIM"; tpl = tpl_loc ->
      let (args,tpl) = tpl in
      let ast = <:expr< $str:tpl$ >> in
      leave_file ();
      ast
  | "EXPR"; tpl = tpl_loc ->
      try
      	let (args,tpl) = tpl in
      	let ast = caml_parse_expr tpl in
	leave_file ();
      	build_fun loc args ast
      with
	| Stdpp.Exc_located ((a,b), Stream.Error msg) ->
	    Printf.eprintf "File %s, loc(%i,%i) :%s" (cur_file ()) a.Lexing.pos_cnum b.Lexing.pos_cnum msg;
	    raise (Stdpp.Exc_located (loc, Stream.Error msg))

 ]]; 

  implem:
    [ [ si = str_items_semi; (sil, stopped) = SELF -> ((si @ sil), stopped)
      | "#"; n = LIDENT; dp = OPT expr; ";;" ->
          ([(<:str_item< # $n$ $opt:dp$ >>, loc)], true)
      | EOI -> ([], false) 
      ] ]
  ;


 module_expr:
   [[ "struct"; stl = LIST0 [ s = str_items_semi -> s ]; "end" ->
	let st = List.map fst (List.concat stl) in
        <:module_expr< struct $list:st$ end >> 
   ]];


 str_item_semi:
    [ [ si = str_item; OPT ";;" -> (si, loc) ] ]
  ;

 str_items_semi:
   [[ "INCLUDE"; tpl = tpl_loc; OPT ";;"  ->
      	let (args,tpl) = tpl in
      	let ast = caml_parse_implem tpl in
      	leave_file ();
      	List.map (fun x -> (x, (Lexing.dummy_pos,Lexing.dummy_pos))) ast
    | s = str_item_semi -> [s]
    ]];
 

 tpl_loc:
 [[ "."; p = STRING -> 
      let s = cur_tpl_file () in
      enter_file s;
      file_part s p
  | s = STRING ; "."; p = STRING ->
      enter_file s;
      file_part s p
  | s = STRING -> 
      enter_file s;
      [], (file s)
 ]];
END

