Template file for doc.ml (html documentation for HereDoc)
The sections

==document_init==
let title = <<HereDoc : "Here Document" facility for OCaml>>
and subtitle = <<release $$VERBATIM "RELEASE"$$>>

let footer = TPL."footer"

let sections = EXPR."sections"
====

==sections==

[
    "Introduction", TPL."intro";
    "Installation", TPL."install";
    "The module Text: Convenient text manipulation", TPL."text";
    "Quotation", TPL."quot"; 
    "Verbatim includes and templates", TPL."tpl"; 
]

====

==intro==
This file <tt>${filename}</tt>, last modified on ${lastmod}.

<p>Without any doubt, $ocaml is a great langage for Web publishing.
I used it for a long time to produce my static web pages and more
recently to write some CGI scripts.

<p>HereDoc is an attempt to provide syntactic sugar for such applications.

<p>In $ocaml, strings constants can span multiple lines. Of course, this
is most useful for the applications we have in mind. But OCaml
lacks some imporant features.

<ul>
 <li>you have to escape every " (the string delimiter in OCaml) in a string;
     for instance, the produce the string 
                 &lt;a href="..">
     you have to write 
                 &lt;a href=\"...\">
     and, believe me, it becomes quickly annoying.
 </li>

 <li>if you want to insert the content of an identifier or the result of
     a Caml expression (computed value), you have to do things like :
       <ul>
        <li> <pre>print_string ("The value of x is " ^ x ^ ".");</pre> </li>
        <li> <pre>Printf.printf "The value of x is %s." x;</pre> </li>
       </ul>
     In Perl and similar languages, you just write :
         <ul><li><pre>print "The value of x is \$x.";</pre></li></ul>
     which looks nicer.
 </li>

 <li>there is no easy way to include file, at compile time.
     We would like to include files verbatim or use them as templates
     (text with gaps to be filled).
  </li>
</ul>

HereDoc relies on $pfour, the powerful preprocessor for OCaml, to
extend the concrete syntax of the language and provide convenient
notations for all these operations. Most of the work is done
at compile time, so the code is fast.
====

==install==
HereDoc has been developped on a Linux box, with:
<ul>
 <li> $ocaml 3.00 </li>
 <li> $pfour 3.00 </li>
</ul>
It has been adapted to work with recent version of OCaml (3.09 currently).

HereDoc supports a findlib-based installation. Just do:
<pre>make all && make install</pre>

To compile a file with the HereDoc syntax:
<pre>ocamlfind ocamlc -package HereDoc -syntax camlp4o ...</pre>

<p>
To gain a better understanding of HereDoc, you may want to
see the plain OCaml code it produces:
<pre>
camlp4o `ocamlfind query HereDoc`/pa_HereDoc.cma pr_o.cmo my_file.ml
</pre>

<p>
Have a look at the files doc.ml, doc.layout.tpl,
doc.sections.tpl (the source code for this page). 
They illustrate the use of HereDoc.
====

==text==
<p>Before we describe the new syntax, we have to introduce a tiny
module : Text (what an original name). With this module, every
Caml value "is" a string. To understand what it means, let's define
the "string content" S(x) of a value x. It is a string, defined by:
<ul>
 <li>if x is a string, S(x)=x</li>
 <li>if x is an int, a char, or a constant constructor of a concrete type, S(x)=""</li>
 <li>if x is a block {x1,...,xn}, S(x) is the concatenation S(x1)^...^S(xn)</li>
</ul>

Similarly, the "string components" SC(x) is defined by:
<ul>
 <li>if x is a string, SC(x)=[x]</li>
 <li>if x is an int, a char, or a constant constructor of a concrete type, SC(x)=[]</li>
 <li>if x is a block {x1,...,xn}, SC(x) is SC(x1)@...@SC(xn)</li>
</ul>

The module defines an opaque type <tt>Text.t</tt> with
an "universal constructor" <tt>Text.repr: 'a -> t</tt>.
The function <tt>Text.iter:(string -> unit) -> t -> unit</tt>  
applies a given function to the string components of a <tt>Text.t</tt> value.
There are also two functions to compute S(x) and its length.

<p>
So, for instance, we have (oh yes, this is not well-typed; you have
to add <tt>Text.repr</tt> everywhere):
<ul>
  <li><pre>S("Hello world !") = "Hello world !"</pre></li>
  <li><pre>S(["Hello";(" ","World");Some " !"]) = "Hello world !"</pre></li>
  <li><pre>S([|"Hello"; 5; Some (5," "); "World !"|]) = "Hello world !"</pre></li>
</ul>

<p>
Typically, an application builds its output by concatenating strings.
This involves many copy operations so it may be quite slow.
With the module Text, just put your substrings in a list (or better,
an array: only one big block), and see it as a string with
<tt>Text.repr</tt>. When you want to finally output the "string", use
<tt>Text.iter</tt> or <tt>Text.to_string</tt> on the result.

<p>
Usually, a closure block is simply ignored:
if you don't give a function enough arguments, you get an empty
string component. You can tell <tt>Text</tt> to raise
the exception <tt>Text.Closure</tt> whenever it reach a closure
by using <tt>Text.set_fail_on_closur true</tt>.

<p>
Another feature of the module Text is the manipulation of "postponed
texts". Suppose you want to create a text with a hole to be filled
later, when an extra piece of information is available. A simple way
to use a reference:

<pre>
   let post = ref Text.empty in
   repr [Text.repr "Total size :"; Text.repr post];
   (* ... *)
   post := Text.repr (string_of_int (total_size ()));
</pre>

Sometimes it is better to put the expression to be computed later
at the position of the postponed text, then to activate it later;
also, having a global reference is not easy to manage.
You can do that:

<pre>
   repr [Text.repr "Total size :"; Text.postponed "totalsize" (fun ()
-> Text.repr (string_of_int (total_size ())))];
   (* ... *)
   Text.activate "totalsize"
</pre>

Here, "totalsize" identifies the delayed computation and Text.activate
triggers the evaluation.
====

==quot==
<p>
HereDoc defines the quotation &lt;:here&lt;...>>. It returns
a value of type <tt>Text.t</tt> (see <a href="#3">the Text
module</a>). HereDoc declare this quotation as the 
default quotation, so you can simply write &lt;&lt;...>>.

<p>
The quotation content is interpreted with the following rules:
<ul>
 <li> <u>Anti-quoted expression:</u><p>
The sequence \$$ marks the beginning and the end of an expression
to be evaluated inside the quotation. For instance:
  <pre> &lt;&lt;The winner is \$$[|"John";"Bob"|].(win)\$$. >> </pre>
You can use any identifier in the scope at the quotation position.
The expression can return a value of any type, and it will
be interpreted with the module Text (with <tt>Text.repr</tt>).
For instance, it can return a list
of strings, and it will be interpreted as you would expect:
  <pre> &lt;&lt;The winners are : \$$List.map String.uppercase \
["John";"Bob"]\$$>> </pre>
Note that inside the expression, you can use another quotation
(yet another stupid example):
 <pre> &lt;&lt; Tada \$$ (&lt;&lt;Hello>>, " World") \$$ >> </pre>
 </li>

 <li> <u>Anti-quoted identifier</u>:<p>
There is a shorter form when the expression is only a
identifier, possibly qualified with a module path and/or
a record path:
  <pre> &lt;&lt;And the winner is ... \$winner !>> </pre>
  <pre> &lt;&lt;And the winner is ... \$Game.winner !>> </pre>
 </li>

 <li> <u>Runtime variable</u>:<p>
The \$$ form is expanded at compile time, so the name of
identifiers must be known to the compiler. You can't
access directly "run-time" variables, like CGI arguments.
Suppose you have a function 
<pre>val var : string -> 'a = &lt;fun></pre>
which can expand these run-time variables (for instance, it can
lookup in the table of CGI arguments) where, as usual, the result
is to be interpreted through the module Text. You can write:
 <pre> &lt;&lt; \$$var "street"\$$>> </pre>
You have a short form for this:
 <pre> &lt;&lt; \$(street)>> </pre>
The name of the expanding function is "var" by default; but you
can change this with the toplevel directive:
 <pre>VAR_HANDLER "myfun"</pre>
It is important to understand that the name of the expanding function
is static, but the real function depends on the scope of the
quotation. Here's the traditionnaly stupid example:

 <pre>
    let var = function
       | "birthday" -> "May 13th"
       | "firstname -> "Alain"
    in	
    &lt;&lt; This year, \$(firstname)'s birthday is on \$(birthday). >>
 </pre>

 </li>

 <li> <u>Conditional sections</u>:
    <p>
    you can define conditional sections with the notation:
    <pre> \$?expr: </pre>
    where <tt>expr</tt> is a boolean expression. If the expression
    evaluates to true (during runtime), the text between this
    conditional and the next one (or the end of the quotation) is
    included. Otherwise, it is discarded.
    <p>
    The empty conditional <tt>\$?:</tt> is equivalent to
<tt>\$?true:</tt>.
    <p>Here is an example:
     <pre>
       let (fr,en) = (true,false) in
       let name = "John" in
       &lt;&lt; \$?fr:Bonjour\$?en:Hello\$?: \$name >>
     </pre>
     You can use more complex expressions:
     <pre>
       << \$?Random.int 10 = 0 :lucky man\$?: >>
     </pre>
 </li>

 <li> <u>Abstraction</u>:
   <p>
   Consider the following example:
   <pre>
	&lt;ul>
	\$$List.map (fun x -> &lt;&lt; &lt;li>\$x&lt;/li> 
	>>) l\$$
	&lt;/ul>
   </pre>

   HereDoc provides a nicer notation for this kind of iteration:
   <pre>
       \$[e -> p1 -> p2 ... -> pn]\$
       ....
       \$[]\$
   </pre>
   is equivalent to:
   <pre>
      \$$e (fun p1 ... pn -> &lt;&lt;....>>)\$$
   </pre>

   <tt>e</tt> is an expression which evaluates to a function with
   n curryfied arguments. <tt>p1, ..., pn</tt> are patterns.
   The last <tt>....</tt> is called the abstracted text.

   For instance, suppose you have defined:
   <pre>
        let map l f= List.map f l
        let rec mapi n l f = 
          match l with
            | [] -> []
            | t::q -> (f n t)::(mapi (n+1) q f)
   </pre>
   You can write this html table renderer:
   <pre>
       &lt;table>
       \$[mapi 1 t -> i -> ligne]\$
       &lt;tr>
       &lt;td>&lt;b>\$$string_of_int i\$$:&lt;/b>&lt;/td>
       \$[map ligne -> case]\$
       &lt;td>\$case&lt;/td>
       \$[]\$
       &lt;/tr>
       \$[]\$
       &lt;/table>
   </pre>

   The case <tt>n=0</tt> is useful to filter the text through a
   function. For instance, if <tt>box: Text.t -> Text.t</tt> is
   a function which put a frame around its argument, you can write:
   <pre>
       \$[box]\$ Long long text ...  \$[]\$
   </pre>

   <strong>Note:</strong>
   Conditional sections are local to the abstracted text.

   
 </li>

 <li> <u>Specials</u>:
      <p>
      You have a few metavariables accessible with the notation
      <tt>\${meta}</tt> or <tt>\${meta:arg}</tt>.

      Here are the possible values for meta:
      <table>
      <tr>
        <td valign=top> filename </td>
        <td> Returns the current file being processed (the ml source
file or template file, see later);  arg is ignored. </td>
      </tr>
        <tr>
        <td valign=top> lastmod </td>
        <td> Computes the last modification date for the current file;
if arg is empty or absent, the date is formatted as "yyyy-mm-dd";
otherwise, arg is parsed as a Caml expression yielding a formatting
function <tt>int -> int -> int -> 'a</tt> (whose arguments are in the order
year, month, day).  </td>
     </tr>
      </table>

 </li>

 <li> You can of course escape (protect) \$$ and \$ by a backslash:
\\\$$ \\\$. The backslah itself can be escaped: \\\\.

 <li> A backslash at the end of the line hide the following \\n
character. All other backslash sequence, such as \\n, \\t, 
etc... are disabled. You don't need to escape the quotation mark.</li>

</ul>
====

==tpl==
It is sometimes convenient to put some text in external file. HereDoc
extends the syntax to provides four kind of includes:
verbatim, templates, declaration includes and expression includes.
<ul>
 <li>
   <tt>VERBATIM "file.txt"</tt>
  returns a string with the content of the given file.
 </li>

 <li>
   <tt>TPL "file.txt"</tt>
   returns the content of the file, interpreted as a quotation (of
type <tt>Text.t</tt>).
 </li>

 <li>
   <tt>INCLUDE "file.txt"</tt> (at the position of a toplevel declaration or
   of a structure implementation item) includes the content of the
   file.
 </li>

 <li>
   <tt>EXPR "file.txt"</tt> same as <tt>INCLUDE</tt>, but for an expression.
 </li>
</ul>

To print a template, use:
<pre>Text.iter print_string TPL "file.txt"</pre>

You can put several chunks of text in a file. Use the notation:
<pre>"file.txt"."chunk"</pre>
to designate the section "chunk" in the file "file.txt", that is all the
lines between the line "==chunk==" (two signs equal, the name of the
chunk, two signs equal, alone on their line), and the line "===="
(four signs equal).

Templates and expression includes may take arguments: if the opening
ligne is ==chunk arg1 arg2 .. argn== instead of ==chunk==, the
chunk returns a function with labeled arguments arg1,...,argn.
For instance, if the template file is:
<pre>
 ==add x y==
 x + y
 ====

 ==link dest txt==
 &lt;a href="\$dest">\$txt&lt;/a>
 ====

</pre>
you can declare:
<pre>
let add = EXPR "file.txt"."add"
let link = TPL "file.txt"."link"
</pre>

and get:
<pre>
val add : x:int -> y:int -> int
val link : dest:'a -> txt:'b -> Text.t
</pre>

<p>
If many of the chunks you need are in the same file, you can use
the toplevel phrase:
<pre>TEMPLATE_FILE "file.txt"</pre>
and then omit the file name, for instance:
<pre>TPL ."chunck"</pre>
When parsing an external template, the default file is the current file.
By default, the template filename is "templates.tpl".

<p>
The commands "VERBATIM", "EXPR" and "TPL" are expressions; 
you can use them inside "expression in quotation". So you can for instance
include a template from a template :
<pre>
 ==chunk==
 Want to see another template ?
 \$$TPL ."another"\$$
 ====
</pre>

====


==footer==
<hr>
Author's mail : $$mailto Author.mail$$<br>
$$link "Author homepage" Author.homepage$$<br>
The program took $time seconds to generate this page.
====

