/**
 * @file bob/core/array_random.h
 * @date Tue Apr 2 13:00:00 2013 +0200
 * @author Laurent El Shafey <Laurent.El-Shafey@idiap.ch>
 *
 * Copyright (C) 2011-2013 Idiap Research Institute, Martigny, Switzerland
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 3 of the License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef BOB_CORE_ARRAY_RANDOM_H 
#define BOB_CORE_ARRAY_RANDOM_H

#include <blitz/array.h>
#include <boost/random.hpp>

namespace bob { namespace core { namespace array {
/**
 * @ingroup CORE_ARRAY
 * @{
 */

/**
 * @brief Fills in a blitz++ array with random numbers generated by a normal
 * distribution. The array should be of type float or double.
 * @warning Because of the use of the blitz iterator, the array is filled in
 * according to the memory order.
 * @warning Please note that we rely on boost rather than blitz++ ranlib
 * because of design considerations.
 */
template <typename T, int N>
void randn(boost::mt19937& rng, blitz::Array<T,N>& dest, 
  const T mean=0., const T var=1.)
{
  boost::normal_distribution<T> distrib(mean, var);
  boost::variate_generator<boost::mt19937, boost::normal_distribution<T> > 
    vg(rng, distrib);
  typedef typename blitz::Array<T,N>::iterator BA_iterator;
  for (BA_iterator it=dest.begin(); it !=dest.end(); ++it)
    *it = vg();
}

/**
 * @}
 */
}}}

#endif /* BOB_CORE_ARRAY_RANDOM_H */
