/*-*- mode:c;indent-tabs-mode:nil;c-basic-offset:2;tab-width:8;coding:utf-8 -*-│
│vi: set net ft=c ts=2 sts=2 sw=2 fenc=utf-8                                :vi│
╞══════════════════════════════════════════════════════════════════════════════╡
│ Copyright 2022 Justine Alexandra Roberts Tunney                              │
│                                                                              │
│ Permission to use, copy, modify, and/or distribute this software for         │
│ any purpose with or without fee is hereby granted, provided that the         │
│ above copyright notice and this permission notice appear in all copies.      │
│                                                                              │
│ THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL                │
│ WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED                │
│ WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE             │
│ AUTHOR BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL         │
│ DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR        │
│ PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER               │
│ TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR             │
│ PERFORMANCE OF THIS SOFTWARE.                                                │
╚─────────────────────────────────────────────────────────────────────────────*/
#include <inttypes.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "blink/assert.h"
#include "blink/bus.h"
#include "blink/endian.h"
#include "blink/flag.h"
#include "blink/high.h"
#include "blink/machine.h"
#include "blink/macros.h"
#include "blink/pml4t.h"
#include "blink/rde.h"
#include "blink/util.h"
#include "blink/x86.h"

#define INTERESTING_FLAGS (PAGE_U | PAGE_RW | PAGE_XD)

#define BYTES       16384
#define APPEND(...) u->o += snprintf(u->b + u->o, BYTES - u->o, __VA_ARGS__)

struct MapMaker {
  bool t;
  int o;
  int count;
  int committed;
  long lines;
  char *b;
  i64 start;
  u64 flags;
};

static i64 MakeAddress(u16 a[4]) {
  u64 x;
  x = 0;
  x |= a[0];
  x <<= 9;
  x |= a[1];
  x <<= 9;
  x |= a[2];
  x <<= 9;
  x |= a[3];
  x <<= 12;
  return x;
}

static void FormatStartPage(struct MapMaker *u, i64 start) {
  u->t = true;
  u->start = start;
  u->count = 0;
  u->committed = 0;
}

static void FormatEndPage(struct Machine *m, struct MapMaker *u, i64 end) {
  int i;
  char size[16];
  struct FileMap *fm;
  bool isreading, iswriting, isexecuting;
  u->t = false;
  if (u->lines++) APPEND("\n");
  isexecuting = m->xedd && MAX(u->start, m->ip) <
                               MIN(m->ip + Oplength(m->xedd->op.rde), end);
  isreading = MAX(u->start, m->readaddr) < MIN(m->readaddr + m->readsize, end);
  iswriting =
      MAX(u->start, m->writeaddr) < MIN(m->writeaddr + m->writesize, end);
  if (g_high.enabled) {
    if (isexecuting) APPEND("\033[7m");
    if (isreading || iswriting) APPEND("\033[1m");
  }
  APPEND("%012" PRIx64 "-%012" PRIx64, u->start, end - 1);
  if (g_high.enabled && (isreading || iswriting || isexecuting)) {
    APPEND("\033[0m");
  }
  FormatSize(size, end - u->start, 1024);
  APPEND(" %5s ", size);
  if (FLAG_nolinear) {
    APPEND("%3d%% ", (int)ceil((double)u->committed / u->count * 100));
  }
  i = 0;
  if (u->flags & PAGE_U) {
    APPEND("r");
    ++i;
  }
  if (u->flags & PAGE_RW) {
    APPEND("w");
    ++i;
  }
  if (~u->flags & PAGE_XD) {
    APPEND("x");
    ++i;
  }
  while (i++ < 4) {
    APPEND(" ");
  }
  if ((fm = GetFileMap(m->system, u->start))) {
    APPEND("%s", fm->path);
  }
}

static void FormatPdeOrPte(struct Machine *m, struct MapMaker *u, u64 entry,
                           u16 a[4], int n) {
  if (u->t && (u->flags != (entry & INTERESTING_FLAGS))) {
    FormatEndPage(m, u, MakeAddress(a));
  }
  if (!u->t) {
    FormatStartPage(u, MakeAddress(a));
    u->flags = entry & INTERESTING_FLAGS;
  }
  u->count += n;
  if (~entry & PAGE_RSRV) {
    u->committed += n;
  }
}

static u8 *GetPt(struct Machine *m, u64 entry, bool is_cr3) {
  return GetPageAddress(m->system, entry, is_cr3);
}

char *FormatPml4t(struct Machine *m) {
  _Thread_local static char b[BYTES];
  u8 *pd[4];
  u64 entry;
  u16 i, a[4];
  struct MapMaker u = {.b = b};
  u16 range[][2] = {{256, 512}, {0, 256}};
  b[0] = 0;
  if (m->mode != XED_MODE_LONG) return b;
  unassert(m->system->cr3);
  pd[0] = GetPt(m, m->system->cr3, true);
  for (i = 0; i < ARRAYLEN(range); ++i) {
    a[0] = range[i][0];
    do {
      a[1] = a[2] = a[3] = 0;
      entry = ReadPte(pd[0] + a[0] * 8);
      if (!(entry & PAGE_V)) {
        if (u.t) FormatEndPage(m, &u, MakeAddress(a));
      } else if (entry & PAGE_PS) {
        FormatPdeOrPte(m, &u, entry, a, 1 << 27);
      } else {
        pd[1] = GetPt(m, entry, false);
        do {
          a[2] = a[3] = 0;
          entry = ReadPte(pd[1] + a[1] * 8);
          if (!(entry & PAGE_V)) {
            if (u.t) FormatEndPage(m, &u, MakeAddress(a));
          } else if (entry & PAGE_PS) {
            FormatPdeOrPte(m, &u, entry, a, 1 << 18);
          } else {
            pd[2] = GetPt(m, entry, false);
            do {
              a[3] = 0;
              entry = ReadPte(pd[2] + a[2] * 8);
              if (!(entry & PAGE_V)) {
                if (u.t) FormatEndPage(m, &u, MakeAddress(a));
              } else if (entry & PAGE_PS) {
                FormatPdeOrPte(m, &u, entry, a, 1 << 9);
              } else {
                pd[3] = GetPt(m, entry, false);
                do {
                  entry = ReadPte(pd[3] + a[3] * 8);
                  if (~entry & PAGE_V) {
                    if (u.t) FormatEndPage(m, &u, MakeAddress(a));
                  } else {
                    FormatPdeOrPte(m, &u, entry, a, 1);
                  }
                } while (++a[3] != 512);
              }
            } while (++a[2] != 512);
          }
        } while (++a[1] != 512);
      }
    } while (++a[0] != range[i][1]);
  }
  if (u.t) {
    FormatEndPage(m, &u, 0x800000000000);
  }
  return b;
}
