%{
/*
 * Copyright (c) 2006-2008 BitMover, Inc.
 */
#include <stdio.h>
#include "Lcompile.h"

/* L_lex is generated by flex. */
extern int	L_lex (void);

#define YYERROR_VERBOSE
#define L_error L_synerr
%}

%expect 0

%union {
	long	i;
	char	*s;
	Tcl_Obj	*obj;
	Type	*Type;
	Expr	*Expr;
	Block	*Block;
	ForEach	*ForEach;
	Switch	*Switch;
	Case	*Case;
	FnDecl	*FnDecl;
	Cond	*Cond;
	Loop	*Loop;
	Stmt	*Stmt;
	TopLev	*TopLev;
	VarDecl	*VarDecl;
	ClsDecl	*ClsDecl;
	struct {
		Type	*t;
		char	*s;
	} Typename;
}

%token T_ANDAND "&&"
%token T_ARROW "=>"
%token T_ATTRIBUTE "_attribute"
%token T_BANG "!"
%token T_BANGTWID "!~"
%token T_BITAND "&"
%token T_BITOR "|"
%token T_BITNOT "~"
%token T_BITXOR "^"
%token T_BREAK "break"
%token T_CLASS "class"
%token T_COLON ":"
%token T_COMMA ","
%token T_CONSTRUCTOR "constructor"
%token T_CONTINUE "continue"
%token T_DEFINED "defined"
%token T_DESTRUCTOR "destructor"
%token T_DO "do"
%token T_DOT "."
%token T_DOTDOT ".."
%token T_ELLIPSIS "..."
%token T_ELSE "else"
%token T_EQ "eq"
%token T_EQBITAND "&="
%token T_EQBITOR "|="
%token T_EQBITXOR "^="
%token T_EQDOT ".="
%token T_EQLSHIFT "<<="
%token T_EQMINUS "-="
%token T_EQPERC "%="
%token T_EQPLUS "+="
%token T_EQRSHIFT ">>="
%token T_EQSTAR "*="
%token T_EQSLASH "/="
%token T_EQTWID "=~"
%token T_EQUALS "="
%token T_EQUALEQUAL "=="
%token T_EXPAND "(expand)"
%token T_EXTERN "extern"
%token T_FLOAT "float"
%token <s> T_FLOAT_LITERAL "float constant"
%token T_FOR "for"
%token T_FOREACH "foreach"
%token T_GOTO "goto"
%token T_GE "ge"
%token T_GREATER ">"
%token T_GREATEREQ ">="
%token T_GT "gt"
%token <s> T_HTML
%token <s> T_ID "id"
%token T_IF "if"
%token T_INSTANCE "instance"
%token T_INT "int"
%token <s> T_INT_LITERAL "integer constant"
%token <s> T_LHTML_EXPR_START "<?="
%token <s> T_LHTML_EXPR_END "?>"
%token T_LBRACE "{"
%token T_LBRACKET "["
%token T_LE "le"
%token <s> T_LEFT_INTERPOL "${"
%token <s> T_LEFT_INTERPOL_RE "${ (in re)"
%token T_LESSTHAN "<"
%token T_LESSTHANEQ "<="
%token T_LPAREN "("
%token T_LSHIFT "<<"
%token T_LT "lt"
%token T_MINUS "-"
%token T_MINUSMINUS "--"
%token T_NE "ne"
%token T_NOTEQUAL "!="
%token T_OROR "||"
%token <s> T_PATTERN "pattern function"
%token T_PERC "%"
%token T_PLUS "+"
%token T_PLUSPLUS "++"
%token T_POINTS "->"
%token T_POLY "poly"
%token T_PRIVATE "private"
%token T_PUBLIC "public"
%token T_QUESTION "?"
%token T_RBRACE "}"
%token T_RBRACKET "]"
%token <s> T_RE "regular expression"
%token <s> T_RE_MODIFIER "regexp modifier"
%token T_RETURN "return"
%token T_RIGHT_INTERPOL "} (end of interpolation)"
%token T_RIGHT_INTERPOL_RE "} (end of interpolation in re)"
%token T_RPAREN ")"
%token T_RSHIFT ">>"
%token T_TRY "try"
%token T_SEMI ";"
%token T_SLASH "/"
%token T_SPLIT "split"
%token T_STAR "*"
%token <s> T_START_BACKTICK "backtick"
%token <s> T_STR_BACKTICK "`"
%token <s> T_STR_LITERAL "string constant"
%token T_STRCAT " . "
%token T_STRING "string"
%token T_STRUCT "struct"
%token <s> T_SUBST "=~ s/a/b/"
%token <Typename> T_TYPE "type name"
%token T_TYPEDEF "typedef"
%token T_UNLESS "unless"
%token T_ARGUSED "_argused"
%token T_OPTIONAL "_optional"
%token T_MUSTBETYPE "_mustbetype"
%token T_VOID "void"
%token T_WIDGET "widget"
%token T_WHILE "while"
%token T_PRAGMA "#pragma"
%token T_SWITCH "switch"
%token T_CASE "case"
%token T_DEFAULT "default"
%token END 0 "end of file"

/*
 * This follows the C operator-precedence rules, from lowest to
 * highest precedence.
 */
%left LOWEST
// The following %nonassoc lines are defined to resolve a conflict with
// labeled statements (see the stmt nonterm).
%nonassoc T_IF T_UNLESS T_RETURN T_ID T_STR_LITERAL T_LEFT_INTERPOL
%nonassoc T_STR_BACKTICK T_INT_LITERAL T_FLOAT_LITERAL T_TYPE T_WHILE
%nonassoc T_FOR T_DO T_DEFINED T_STRING T_FOREACH T_BREAK T_CONTINUE
%nonassoc T_SPLIT T_GOTO T_WIDGET T_PRAGMA T_SWITCH T_START_BACKTICK T_TRY
%nonassoc T_HTML T_LHTML_EXPR_START T_LPAREN
%left T_COMMA
%nonassoc T_ELSE T_SEMI
%right T_EQUALS T_EQPLUS T_EQMINUS T_EQSTAR T_EQSLASH T_EQPERC
       T_EQBITAND T_EQBITOR T_EQBITXOR T_EQLSHIFT T_EQRSHIFT T_EQDOT
%right T_QUESTION
%left T_OROR
%left T_ANDAND
%left T_BITOR
%left T_BITXOR
%left T_BITAND
%left T_EQ T_NE T_EQUALEQUAL T_NOTEQUAL T_EQTWID T_BANGTWID
%left T_GT T_GE T_LT T_LE T_GREATER T_GREATEREQ T_LESSTHAN T_LESSTHANEQ
%left T_LSHIFT T_RSHIFT
%left T_PLUS T_MINUS T_STRCAT
%left T_STAR T_SLASH T_PERC
%right PREFIX_INCDEC UPLUS UMINUS T_BANG T_BITNOT ADDRESS
%left T_LBRACKET T_LBRACE T_RBRACE T_DOT T_POINTS T_PLUSPLUS T_MINUSMINUS
%left HIGHEST

%type <TopLev> toplevel_code
%type <ClsDecl> class_decl class_decl_tail
%type <FnDecl> function_decl fundecl_tail fundecl_tail1
%type <Stmt> stmt single_stmt compound_stmt stmt_list opt_stmt_list
%type <Stmt> unlabeled_stmt optional_else
%type <Cond> selection_stmt
%type <Loop> iteration_stmt
%type <ForEach> foreach_stmt
%type <Switch> switch_stmt
%type <Case> switch_cases switch_case
%type <Expr> expr expression_stmt argument_expr_list pragma_expr_list
%type <Expr> id id_list string_literal cmdsubst_literal dotted_id
%type <Expr> regexp_literal regexp_literal_mod subst_literal interpolated_expr
%type <Expr> interpolated_expr_re list list_element case_expr option_arg
%type <Expr> here_doc_backtick opt_attribute pragma
%type <VarDecl> parameter_list parameter_decl_list parameter_decl
%type <VarDecl> declaration_list declaration declaration2
%type <VarDecl> init_declarator_list declarator_list init_declarator
%type <VarDecl> declarator opt_declarator struct_decl_list struct_decl
%type <VarDecl> struct_declarator_list
%type <Type> array_or_hash_type type_specifier scalar_type_specifier
%type <Type> struct_specifier
%type <obj> dotted_id_1
%type <i> decl_qualifier parameter_attrs

%%

start:	  toplevel_code
	{
		REVERSE(TopLev, next, $1);
		L->ast = $1;
	}
	;

toplevel_code:
	  toplevel_code class_decl
	{
		if ($2) {
			$$ = ast_mkTopLevel(L_TOPLEVEL_CLASS, $1, @2, @2);
			$$->u.class = $2;
		} else {
			// Don't create a node for a forward class declaration.
			$$ = $1;
		}
	}
	| toplevel_code function_decl
	{
		$$ = ast_mkTopLevel(L_TOPLEVEL_FUN, $1, @2, @2);
		$2->decl->flags |= DECL_FN;
		if ($2->decl->flags & DECL_PRIVATE) {
			$2->decl->flags |= SCOPE_SCRIPT;
		} else {
			$2->decl->flags |= SCOPE_GLOBAL;
		}
		$$->u.fun = $2;
	}
	| toplevel_code T_TYPEDEF type_specifier declarator ";"
	{
		L_set_declBaseType($4, $3);
		L_typedef_store($4);
		$$ = $1;  // nothing more to do
	}
	| toplevel_code declaration
	{
		// Global variable declaration.
		VarDecl *v;
		$$ = ast_mkTopLevel(L_TOPLEVEL_GLOBAL, $1, @2, @2);
		for (v = $2; v; v = v->next) {
			v->flags |= DECL_GLOBAL_VAR;
			if ($2->flags & DECL_PRIVATE) {
				v->flags |= SCOPE_SCRIPT;
			} else {
				v->flags |= SCOPE_GLOBAL;
			}
		}
		$$->u.global = $2;
	}
	| toplevel_code stmt
	{
		// Top-level statement.
		$$ = ast_mkTopLevel(L_TOPLEVEL_STMT, $1, @2, @2);
		$$->u.stmt = $2;
	}
	| /* epsilon */		{ $$ = NULL; }
	;

class_decl:
	  T_CLASS id "{"
	{
		/*
		 * This is a new class declaration.
		 * Alloc the VarDecl now and associate it with
		 * the class name so that it is available while
		 * parsing the class body.
		 */
		Type	*t = type_mkClass();
		VarDecl	*d = ast_mkVarDecl(t, $2, @1, @1);
		ClsDecl	*c = ast_mkClsDecl(d, @1, @1);
		t->u.class.clsdecl = c;
		ASSERT(!L_typedef_lookup($2->str));
		L_typedef_store(d);
		$<ClsDecl>$ = c;
	} class_decl_tail
	{
		$$ = $5;
		/* silence unused warning */
		(void)$<ClsDecl>4;
	}
	| T_CLASS T_TYPE "{"
	{
		/*
		 * This is a class declaration where the type name was
		 * previously declared.  Use the ClsDecl from the
		 * prior decl.
		 */
		ClsDecl	*c = $2.t->u.class.clsdecl;
		unless (c->decl->flags & DECL_FORWARD) {
			L_err("redeclaration of %s", $2.s);
		}
		ASSERT(isclasstype(c->decl->type));
		c->decl->flags &= ~DECL_FORWARD;
		$<ClsDecl>$ = c;
	} class_decl_tail
	{
		$$ = $5;
		/* silence unused warning */
		(void)$<ClsDecl>4;
	}
	| T_CLASS id ";"
	{
		/* This is a forward class declaration. */
		Type	*t = type_mkClass();
		VarDecl	*d = ast_mkVarDecl(t, $2, @1, @3);
		ClsDecl	*c = ast_mkClsDecl(d, @1, @3);
		ASSERT(!L_typedef_lookup($2->str));
		t->u.class.clsdecl = c;
		d->flags |= DECL_FORWARD;
		L_typedef_store(d);
		$<ClsDecl>$ = NULL;
	}
	| T_CLASS T_TYPE ";"
	{
		/* Empty declaration of an already declared type. */
		unless (isclasstype($2.t)) {
			L_err("%s not a class type", $2.s);
		}
		$<ClsDecl>$ = NULL;
	}
	;

class_decl_tail:
	class_code "}"
	{
		$$ = $<ClsDecl>0;
		$$->node.loc.end       = @2.end;
		$$->decl->node.loc.end = @2.end;
		/* If constructor or destructor were omitted, make defaults. */
		unless ($$->constructors) {
			$$->constructors = ast_mkConstructor($$);
		}
		unless ($$->destructors) {
			$$->destructors = ast_mkDestructor($$);
		}
	}
	;

class_code:
	  class_code T_INSTANCE "{" declaration_list "}" opt_semi
	{
		VarDecl	*v;
		ClsDecl	*clsdecl = $<ClsDecl>0;
		REVERSE(VarDecl, next, $4);
		for (v = $4; v; v = v->next) {
			v->clsdecl = clsdecl;
			v->flags  |= SCOPE_CLASS | DECL_CLASS_INST_VAR;
			unless (v->flags & (DECL_PUBLIC | DECL_PRIVATE)) {
				L_errf(v, "class instance variable %s not "
				       "declared public or private",
				       v->id->str);
				v->flags |= DECL_PUBLIC;
			}
		}
		APPEND_OR_SET(VarDecl, next, clsdecl->instvars, $4);
	}
	| class_code T_INSTANCE "{" "}" opt_semi
	| class_code declaration
	{
		VarDecl	*v;
		ClsDecl	*clsdecl = $<ClsDecl>0;
		REVERSE(VarDecl, next, $2);
		for (v = $2; v; v = v->next) {
			v->clsdecl = clsdecl;
			v->flags  |= SCOPE_CLASS | DECL_CLASS_VAR;
			unless (v->flags & (DECL_PUBLIC | DECL_PRIVATE)) {
				L_errf(v, "class variable %s not "
				       "declared public or private",
				       v->id->str);
				v->flags |= DECL_PUBLIC;
			}
		}
		APPEND_OR_SET(VarDecl, next, clsdecl->clsvars, $2);
	}
	| class_code T_TYPEDEF type_specifier declarator ";"
	{
		L_set_declBaseType($4, $3);
		L_typedef_store($4);
	}
	| class_code function_decl
	{
		ClsDecl	*clsdecl = $<ClsDecl>0;
		$2->decl->clsdecl = clsdecl;
		$2->decl->flags  |= DECL_CLASS_FN;
		unless ($2->decl->flags & DECL_PRIVATE) {
			$2->decl->flags |= SCOPE_GLOBAL | DECL_PUBLIC;
		} else {
			$2->decl->flags |= SCOPE_CLASS;
			$2->decl->tclprefix = cksprintf("_L_class_%s_",
						clsdecl->decl->id->str);
		}
		APPEND_OR_SET(FnDecl, next, clsdecl->fns, $2);
	}
	| class_code T_CONSTRUCTOR fundecl_tail
	{
		ClsDecl	*clsdecl = $<ClsDecl>0;
		$3->decl->type->base_type = clsdecl->decl->type;
		$3->decl->clsdecl = clsdecl;
		$3->decl->flags  |= SCOPE_GLOBAL | DECL_CLASS_FN | DECL_PUBLIC |
			DECL_CLASS_CONST;
		APPEND_OR_SET(FnDecl, next, clsdecl->constructors, $3);
	}
	| class_code T_DESTRUCTOR fundecl_tail
	{
		ClsDecl	*clsdecl = $<ClsDecl>0;
		$3->decl->type->base_type = L_void;
		$3->decl->clsdecl = clsdecl;
		$3->decl->flags  |= SCOPE_GLOBAL | DECL_CLASS_FN | DECL_PUBLIC |
			DECL_CLASS_DESTR;
		APPEND_OR_SET(FnDecl, next, clsdecl->destructors, $3);
	}
	| class_code pragma
	{
		/*
		 * We don't store the things that make up class_code
		 * in order, so there's no place in which to
		 * interleave #pragmas.  So don't create an AST node,
		 * just update L->options now; it gets used when other
		 * AST nodes are created.
		 */
		L_compile_attributes(L->options, $2, L_attrs_pragma);
	}
	| /* epsilon */
	;

opt_semi:
	  ";"
	| /* epsilon */
	;

function_decl:
	  type_specifier fundecl_tail
	{
		$2->decl->type->base_type = $1;
		$$ = $2;
		$$->node.loc = @1;
	}
	| decl_qualifier type_specifier fundecl_tail
	{
		$3->decl->type->base_type = $2;
		$3->decl->flags |= $1;
		$$ = $3;
		$$->node.loc = @1;
	}
	;

fundecl_tail:
	  id fundecl_tail1
	{
		$$ = $2;
		$$->decl->id = $1;
		$$->node.loc = @1;
	}
	| T_PATTERN fundecl_tail1
	{
		VarDecl	*new_param;
		Expr	*dollar1 = ast_mkId("$cmd", @2, @2);

		$$ = $2;
		$$->decl->id = ast_mkId($1, @1, @1);
		ckfree($1);
		$$->node.loc = @1;
		/* Prepend a new arg "$1" as the first formal. */
		new_param = ast_mkVarDecl(L_string, dollar1, @1, @2);
		new_param->flags = SCOPE_LOCAL | DECL_LOCAL_VAR;
		new_param->next = $2->decl->type->u.func.formals;
		$$->decl->type->u.func.formals = new_param;
	}
	;

fundecl_tail1:
	  "(" parameter_list ")" opt_attribute compound_stmt
	{
		Type	*type = type_mkFunc(NULL, $2);
		VarDecl	*decl = ast_mkVarDecl(type, NULL, @1, @3);
		decl->attrs = $4;
		$$ = ast_mkFnDecl(decl, $5->u.block, @1, @5);
	}
	| "(" parameter_list ")" opt_attribute ";"
	{
		Type	*type = type_mkFunc(NULL, $2);
		VarDecl	*decl = ast_mkVarDecl(type, NULL, @1, @3);
		decl->attrs = $4;
		$$ = ast_mkFnDecl(decl, NULL, @1, @5);
	}
	;

stmt:
	  T_ID ":" stmt
	{
		$$ = ast_mkStmt(L_STMT_LABEL, NULL, @1, @3);
		$$->u.label = $1;
		$$->next = $3;
	}
	| T_ID ":" %prec LOWEST
	{
		$$ = ast_mkStmt(L_STMT_LABEL, NULL, @1, @2);
		$$->u.label = $1;
	}
	| unlabeled_stmt
	| pragma
	{
		L_compile_attributes(L->options, $1, L_attrs_pragma);
		$$ = NULL;
	}
	| T_HTML
	{
		// Wrap the html in a puts(-nonewline) call.
		Expr	*fn = ast_mkId("puts", @1, @1);
		Expr	*arg = ast_mkConst(L_string, "-nonewline", @1, @1);
		arg->next = ast_mkConst(L_string, $1, @1, @1);
		$$ = ast_mkStmt(L_STMT_EXPR, NULL, @1, @1);
		$$->u.expr = ast_mkFnCall(fn, arg, @1, @1);
	}
	| T_LHTML_EXPR_START expr T_LHTML_EXPR_END
	{
		// Wrap expr in a puts(-nonewline) call.
		Expr	*fn = ast_mkId("puts", @2, @2);
		Expr	*arg = ast_mkConst(L_string, "-nonewline", @2, @2);
		arg->next = $2;
		$$ = ast_mkStmt(L_STMT_EXPR, NULL, @1, @3);
		$$->u.expr = ast_mkFnCall(fn, arg, @1, @3);
	}
	;

pragma_expr_list:
	  id
	| id "=" id
	{
		$$ = ast_mkBinOp(L_OP_EQUALS, $1, $3, @1, @3);
	}
	| id "=" T_INT_LITERAL
	{
		Expr	*lit = ast_mkConst(L_int, $3, @3, @3);
		$$ = ast_mkBinOp(L_OP_EQUALS, $1, lit, @1, @3);
	}
	| pragma_expr_list "," id
	{
		$3->next = $1;
		$$ = $3;
		$$->node.loc.beg = @1.beg;
	}
	| pragma_expr_list "," id "=" id
	{
		$$ = ast_mkBinOp(L_OP_EQUALS, $3, $5, @3, @5);
		$$->next = $1;
		$$->node.loc.beg = @1.beg;
	}
	| pragma_expr_list "," id "=" T_INT_LITERAL
	{
		Expr	*lit = ast_mkConst(L_int, $5, @5, @5);
		$$ = ast_mkBinOp(L_OP_EQUALS, $3, lit, @3, @5);
		$$->next = $1;
		$$->node.loc.beg = @1.beg;
	}
	;

pragma:
	  T_PRAGMA pragma_expr_list
	{
		REVERSE(Expr, next, $2);
		$$ = $2;
		$$->node.loc.beg = @1.beg;
	}
	;

opt_attribute:
	  T_ATTRIBUTE "(" argument_expr_list ")"
	{
		REVERSE(Expr, next, $3);
		$$ = $3;
		$$->node.loc.beg = @1.beg;
		$$->node.loc.end = @4.end;
	}
	|	{ $$ = NULL; }
	;

unlabeled_stmt:
	  single_stmt
	| compound_stmt
	;

single_stmt:
	  selection_stmt
	{
		$$ = ast_mkStmt(L_STMT_COND, NULL, @1, @1);
		$$->u.cond = $1;
	}
	| iteration_stmt
	{
		$$ = ast_mkStmt(L_STMT_LOOP, NULL, @1, @1);
		$$->u.loop = $1;
	}
	| switch_stmt
	{
		$$ = ast_mkStmt(L_STMT_SWITCH, NULL, @1, @1);
		$$->u.swich = $1;
	}
	| foreach_stmt
	{
		$$ = ast_mkStmt(L_STMT_FOREACH, NULL, @1, @1);
		$$->u.foreach = $1;
	}
	| expr ";"
	{
		$$ = ast_mkStmt(L_STMT_EXPR, NULL, @1, @1);
		$$->u.expr = $1;
	}
	| T_BREAK ";"
	{
		$$ = ast_mkStmt(L_STMT_BREAK, NULL, @1, @1);
	}
	| T_CONTINUE ";"
	{
		$$ = ast_mkStmt(L_STMT_CONTINUE, NULL, @1, @1);
	}
	| T_RETURN ";"
	{
		$$ = ast_mkStmt(L_STMT_RETURN, NULL, @1, @1);
	}
	| T_RETURN expr ";"
	{
		$$ = ast_mkStmt(L_STMT_RETURN, NULL, @1, @2);
		$$->u.expr = $2;
	}
	| T_GOTO T_ID ";"
	{
		$$ = ast_mkStmt(L_STMT_GOTO, NULL, @1, @3);
		$$->u.label = $2;
	}
	| "try" compound_stmt T_ID "(" expr ")" compound_stmt
	{
		/*
		 * We don't want to make "catch" a keyword since it's a Tcl
		 * function name, so allow any ID here but check it.
		 */
		unless (!strcmp($3, "catch")) {
			L_synerr2("syntax error -- expected 'catch'", @3.beg);
		}
		$$ = ast_mkStmt(L_STMT_TRY, NULL, @1, @7);
		$$->u.try = ast_mkTry($2, $5, $7);
	}
	| "try" compound_stmt T_ID compound_stmt
	{
		$$ = ast_mkStmt(L_STMT_TRY, NULL, @1, @4);
		$$->u.try = ast_mkTry($2, NULL, $4);
	}
	| ";"	{ $$ = NULL; }
	;

selection_stmt:
	  T_IF "(" expr ")" compound_stmt optional_else
	{
		$$ = ast_mkIfUnless($3, $5, $6, @1, @6);
	}
	/* If you have no curly braces, you get no else. */
	| T_IF "(" expr ")" single_stmt
	{
		$$ = ast_mkIfUnless($3, $5, NULL, @1, @5);
	}
	| T_UNLESS "(" expr ")" compound_stmt optional_else
	{
		$$ = ast_mkIfUnless($3, $6, $5, @1, @6);
	}
	| T_UNLESS "(" expr ")" single_stmt
	{
		$$ = ast_mkIfUnless($3, NULL, $5, @1, @5);
	}
	;

switch_stmt:
	  T_SWITCH "(" expr ")" "{" switch_cases "}"
	{
		Case	*c, *def;

		for (c = $6, def = NULL; c; c = c->next) {
			if (c->expr) continue;
			if (def) {
				L_errf(c,
				"multiple default cases in switch statement");
			}
			def = c;
		}
		$$ = ast_mkSwitch($3, $6, @1, @7);
	}
	;

switch_cases:
	  switch_cases switch_case
	{
		if ($1) {
			APPEND(Case, next, $1, $2);
			$$ = $1;
		} else {
			$$ = $2;
		}
	}
	| /* epsilon */		{ $$ = NULL; }
	;

switch_case:
	  "case" re_start_case case_expr ":" opt_stmt_list
	{
		REVERSE(Stmt, next, $5);
		$$ = ast_mkCase($3, $5, @1, @5);
	}
	| "default" ":" opt_stmt_list
	{
		/* The default case is distinguished by a NULL expr. */
		REVERSE(Stmt, next, $3);
		$$ = ast_mkCase(NULL, $3, @1, @2);
	}
	;

case_expr:
	  regexp_literal_mod
	{
		if ($1->flags & L_EXPR_RE_G) {
			L_errf($1, "illegal regular expression modifier");
		}
	}
	| expr
	;

optional_else:
	/* Else clause must either have curly braces or be another if/unless. */
	  T_ELSE compound_stmt
	{
		$$ = $2;
		$$->node.loc = @1;
	}
	| T_ELSE selection_stmt
	{
		$$ = ast_mkStmt(L_STMT_COND, NULL, @1, @2);
		$$->u.cond = $2;
	}
	| /* epsilon */		{ $$ = NULL; }
	;

iteration_stmt:
	  T_WHILE "(" expr ")" stmt
	{
		$$ = ast_mkLoop(L_LOOP_WHILE, NULL, $3, NULL, $5, @1, @5);
	}
	| T_DO stmt T_WHILE "(" expr ")" ";"
	{
		$$ = ast_mkLoop(L_LOOP_DO, NULL, $5, NULL, $2, @1, @6);
	}
	| T_FOR "(" expression_stmt expression_stmt ")" stmt
	{
		$$ = ast_mkLoop(L_LOOP_FOR, $3, $4, NULL, $6, @1, @6);
	}
	| T_FOR "(" expression_stmt expression_stmt expr ")" stmt
	{
		$$ = ast_mkLoop(L_LOOP_FOR, $3, $4, $5, $7, @1, @7);
	}
	;

foreach_stmt:
	  T_FOREACH "(" id "=>" id id expr ")" stmt
	{
		$$ = ast_mkForeach($7, $3, $5, $9, @1, @9);
		unless (isid($6, "in")) {
			L_synerr2("syntax error -- expected 'in' in foreach",
				  @6.beg);
		}
	}
	| T_FOREACH "(" id_list id expr ")" stmt
	{
		$$ = ast_mkForeach($5, $3, NULL, $7, @1, @7);
		unless (isid($4, "in")) {
			L_synerr2("syntax error -- expected 'in' in foreach",
				  @4.beg);
		}
	}
	;

expression_stmt:
	  ";"		{ $$ = NULL; }
	| expr ";"
	;

opt_stmt_list:
	  stmt_list
	| { $$ = NULL; }
	;

stmt_list:
	  stmt
	{
		REVERSE(Stmt, next, $1);
		$$ = $1;
	}
	| stmt_list stmt
	{
		if ($2) {
			REVERSE(Stmt, next, $2);
			APPEND(Stmt, next, $2, $1);
			$$ = $2;
		} else {
			// Empty stmt.
			$$ = $1;
		}
	}
	;

parameter_list:
	  parameter_decl_list
	{
		VarDecl *v;
		REVERSE(VarDecl, next, $1);
		for (v = $1; v; v = v->next) {
			v->flags |= SCOPE_LOCAL | DECL_LOCAL_VAR;
		}
		$$ = $1;
		/*
		 * Special case a parameter list of "void" -- a single
		 * formal of type void with no arg name.  This really
		 * means there are no args.
		 */
		if ($1 && !$1->next && !$1->id && ($1->type == L_void)) {
			$$ = NULL;
		}
	}
	| /* epsilon */	{ $$ = NULL; }
	;

parameter_decl_list:
	  parameter_decl
	| parameter_decl_list "," parameter_decl
	{
		$3->next = $1;
		$$ = $3;
		$$->node.loc = @1;
	}
	;

parameter_decl:
	  parameter_attrs type_specifier opt_declarator
	{
		if ($3) {
			L_set_declBaseType($3, $2);
			$$ = $3;
		} else {
			$$ = ast_mkVarDecl($2, NULL, @2, @2);
			if (isnameoftype($2)) $$->flags |= DECL_REF;
		}
		$$->flags |= $1;
		$$->node.loc = @1;
	}
	| parameter_attrs T_ELLIPSIS id
	{
		Type *t = type_mkArray(NULL, L_poly);
		$$ = ast_mkVarDecl(t, $3, @1, @3);
		$$->flags |= $1 | DECL_REST_ARG;
	}
	;

parameter_attrs:
	  parameter_attrs T_ARGUSED	{ $$ = $1 | DECL_ARGUSED; }
	| parameter_attrs T_OPTIONAL	{ $$ = $1 | DECL_OPTIONAL; }
	| parameter_attrs T_MUSTBETYPE	{ $$ = $1 | DECL_NAME_EQUIV; }
	| /* epsilon */	{ $$ = 0; }
	;

argument_expr_list:
	  expr %prec T_COMMA
	| option_arg
	| option_arg expr %prec T_COMMA
	{
		$2->next = $1;
		$$ = $2;
		$$->node.loc = @1;
	}
	| argument_expr_list "," expr
	{
		$3->next = $1;
		$$ = $3;
		$$->node.loc.end = @3.end;
	}
	| argument_expr_list "," option_arg
	{
		$3->next = $1;
		$$ = $3;
		$$->node.loc.end = @3.end;
	}
	| argument_expr_list "," option_arg expr %prec T_COMMA
	{
		$4->next = $3;
		$3->next = $1;
		$$ = $4;
		$$->node.loc.end = @4.end;
	}
	;

/*
 * option_arg is an actual parameter "arg:" that becomes "-arg".
 * Allow both "T_ID:" and "default:" to overcome a nasty grammar
 * conflict with statement labels that otherwise results.  The scanner
 * returns T_DEFAULT T_COLON when it sees "default:" but returns T_ID
 * T_COLON for the other cases even if they include a reserved word.
 */
option_arg:
	  T_ID ":"
	{
		char	*s = cksprintf("-%s", $1);
		$$ = ast_mkConst(L_string, s, @1, @2);
		ckfree($1);
	}
	| "default" ":"
	{
		char	*s = cksprintf("-default");
		$$ = ast_mkConst(L_string, s, @1, @2);
	}
	;

expr:
	  "(" expr ")"
	{
		$$ = $2;
		$$->node.loc = @1;
		$$->node.loc.end = @3.end;
	}
	| "(" type_specifier ")" expr %prec PREFIX_INCDEC
	{
		// This is a binop where an arg is a Type*.
		$$ = ast_mkBinOp(L_OP_CAST, (Expr *)$2, $4, @1, @4);
	}
	| "(" T_EXPAND ")" expr %prec PREFIX_INCDEC
	{
		$$ = ast_mkUnOp(L_OP_EXPAND, $4, @1, @4);
	}
	| T_BANG expr
	{
		$$ = ast_mkUnOp(L_OP_BANG, $2, @1, @2);
	}
	| T_BITNOT expr
	{
		$$ = ast_mkUnOp(L_OP_BITNOT, $2, @1, @2);
	}
	| T_BITAND expr %prec ADDRESS
	{
		$$ = ast_mkUnOp(L_OP_ADDROF, $2, @1, @2);
	}
	| T_MINUS expr %prec UMINUS
	{
		$$ = ast_mkUnOp(L_OP_UMINUS, $2, @1, @2);
	}
	| T_PLUS expr %prec UPLUS
	{
		$$ = ast_mkUnOp(L_OP_UPLUS, $2, @1, @2);
	}
	| T_PLUSPLUS expr %prec PREFIX_INCDEC
	{
		$$ = ast_mkUnOp(L_OP_PLUSPLUS_PRE, $2, @1, @2);
	}
	| T_MINUSMINUS expr %prec PREFIX_INCDEC
	{
		$$ = ast_mkUnOp(L_OP_MINUSMINUS_PRE, $2, @1, @2);
	}
	| expr T_PLUSPLUS
	{
		$$ = ast_mkUnOp(L_OP_PLUSPLUS_POST, $1, @1, @2);
	}
	| expr T_MINUSMINUS
	{
		$$ = ast_mkUnOp(L_OP_MINUSMINUS_POST, $1, @1, @2);
	}
	| expr T_EQTWID regexp_literal_mod
	{
		$$ = ast_mkBinOp(L_OP_EQTWID, $1, $3, @1, @3);
	}
	| expr T_BANGTWID regexp_literal_mod
	{
		$$ = ast_mkBinOp(L_OP_BANGTWID, $1, $3, @1, @3);
	}
	| expr T_EQTWID regexp_literal subst_literal T_RE_MODIFIER
	{
		if (strchr($5, 'i')) $3->flags |= L_EXPR_RE_I;
		if (strchr($5, 'g')) $3->flags |= L_EXPR_RE_G;
		$$ = ast_mkTrinOp(L_OP_EQTWID, $1, $3, $4, @1, @5);
		ckfree($5);
	}
	| expr T_STAR expr
	{
		$$ = ast_mkBinOp(L_OP_STAR, $1, $3, @1, @3);
	}
	| expr T_SLASH expr
	{
		$$ = ast_mkBinOp(L_OP_SLASH, $1, $3, @1, @3);
	}
	| expr T_PERC expr
	{
		$$ = ast_mkBinOp(L_OP_PERC, $1, $3, @1, @3);
	}
	| expr T_PLUS expr
	{
		$$ = ast_mkBinOp(L_OP_PLUS, $1, $3, @1, @3);
	}
	| expr T_MINUS expr
	{
		$$ = ast_mkBinOp(L_OP_MINUS, $1, $3, @1, @3);
	}
	| expr T_EQ expr
	{
		$$ = ast_mkBinOp(L_OP_STR_EQ, $1, $3, @1, @3);
	}
	| expr T_NE expr
	{
		$$ = ast_mkBinOp(L_OP_STR_NE, $1, $3, @1, @3);
	}
	| expr T_LT expr
	{
		$$ = ast_mkBinOp(L_OP_STR_LT, $1, $3, @1, @3);
	}
	| expr T_LE expr
	{
		$$ = ast_mkBinOp(L_OP_STR_LE, $1, $3, @1, @3);
	}
	| expr T_GT expr
	{
		$$ = ast_mkBinOp(L_OP_STR_GT, $1, $3, @1, @3);
	}
	| expr T_GE expr
	{
		$$ = ast_mkBinOp(L_OP_STR_GE, $1, $3, @1, @3);
	}
	| expr T_EQUALEQUAL expr
	{
		$$ = ast_mkBinOp(L_OP_EQUALEQUAL, $1, $3, @1, @3);
	}
	| T_EQ "(" expr "," expr ")"
	{
		$$ = ast_mkBinOp(L_OP_EQUALEQUAL, $3, $5, @1, @6);
	}
	| expr T_NOTEQUAL expr
	{
		$$ = ast_mkBinOp(L_OP_NOTEQUAL, $1, $3, @1, @3);
	}
	| expr T_GREATER expr
	{
		$$ = ast_mkBinOp(L_OP_GREATER, $1, $3, @1, @3);
	}
	| expr T_GREATEREQ expr
	{
		$$ = ast_mkBinOp(L_OP_GREATEREQ, $1, $3, @1, @3);
	}
	| expr T_LESSTHAN expr
	{
		$$ = ast_mkBinOp(L_OP_LESSTHAN, $1, $3, @1, @3);
	}
	| expr T_LESSTHANEQ expr
	{
		$$ = ast_mkBinOp(L_OP_LESSTHANEQ, $1, $3, @1, @3);
	}
	| expr T_ANDAND expr
	{
		$$ = ast_mkBinOp(L_OP_ANDAND, $1, $3, @1, @3);
	}
	| expr T_OROR expr
	{
		$$ = ast_mkBinOp(L_OP_OROR, $1, $3, @1, @3);
	}
	| expr T_LSHIFT expr
	{
		$$ = ast_mkBinOp(L_OP_LSHIFT, $1, $3, @1, @3);
	}
	| expr T_RSHIFT expr
	{
		$$ = ast_mkBinOp(L_OP_RSHIFT, $1, $3, @1, @3);
	}
	| expr T_BITOR expr
	{
		$$ = ast_mkBinOp(L_OP_BITOR, $1, $3, @1, @3);
	}
	| expr T_BITAND expr
	{
		$$ = ast_mkBinOp(L_OP_BITAND, $1, $3, @1, @3);
	}
	| expr T_BITXOR expr
	{
		$$ = ast_mkBinOp(L_OP_BITXOR, $1, $3, @1, @3);
	}
	| id
	| string_literal
	| cmdsubst_literal
	| T_INT_LITERAL
	{
		$$ = ast_mkConst(L_int, $1, @1, @1);
	}
	| T_FLOAT_LITERAL
	{
		$$ = ast_mkConst(L_float, $1, @1, @1);
	}
	| id "(" argument_expr_list ")"
	{
		REVERSE(Expr, next, $3);
		$$ = ast_mkFnCall($1, $3, @1, @4);
	}
	| id "(" ")"
	{
		$$ = ast_mkFnCall($1, NULL, @1, @3);
	}
	| T_STRING "(" argument_expr_list ")"
	{
		Expr *id = ast_mkId("string", @1, @1);
		REVERSE(Expr, next, $3);
		$$ = ast_mkFnCall(id, $3, @1, @4);
	}
	| T_SPLIT "(" re_start_split regexp_literal_mod "," argument_expr_list ")"
	{
		Expr *id = ast_mkId("split", @1, @1);
		REVERSE(Expr, next, $6);
		$4->next = $6;
		$$ = ast_mkFnCall(id, $4, @1, @7);
	}
	/*
	 * Even though there is no regexp arg in this form, the
	 * re_start_split is necessary to be able to scan either a
	 * regexp or a non-regexp first argument.  The scanner makes
	 * that decision based on the first one or two characters and
	 * then returns appropriate tokens.
	 */
	| T_SPLIT "(" re_start_split argument_expr_list ")"
	{
		Expr *id = ast_mkId("split", @1, @1);
		REVERSE(Expr, next, $4);
		$$ = ast_mkFnCall(id, $4, @1, @5);
	}
	/* this is to allow calling Tk widget functions */
	| dotted_id "(" argument_expr_list ")"
	{
		REVERSE(Expr, next, $3);
		$$ = ast_mkFnCall($1, $3, @1, @4);
	}
	| dotted_id "(" ")"
	{
		$$ = ast_mkFnCall($1, NULL, @1, @3);
	}
	| expr T_EQUALS expr
	{
		$$ = ast_mkBinOp(L_OP_EQUALS, $1, $3, @1, @3);
	}
	| expr T_EQPLUS expr
	{
		$$ = ast_mkBinOp(L_OP_EQPLUS, $1, $3, @1, @3);
	}
	| expr T_EQMINUS expr
	{
		$$ = ast_mkBinOp(L_OP_EQMINUS, $1, $3, @1, @3);
	}
	| expr T_EQSTAR expr
	{
		$$ = ast_mkBinOp(L_OP_EQSTAR, $1, $3, @1, @3);
	}
	| expr T_EQSLASH expr
	{
		$$ = ast_mkBinOp(L_OP_EQSLASH, $1, $3, @1, @3);
	}
	| expr T_EQPERC expr
	{
		$$ = ast_mkBinOp(L_OP_EQPERC, $1, $3, @1, @3);
	}
	| expr T_EQBITAND expr
	{
		$$ = ast_mkBinOp(L_OP_EQBITAND, $1, $3, @1, @3);
	}
	| expr T_EQBITOR expr
	{
		$$ = ast_mkBinOp(L_OP_EQBITOR, $1, $3, @1, @3);
	}
	| expr T_EQBITXOR expr
	{
		$$ = ast_mkBinOp(L_OP_EQBITXOR, $1, $3, @1, @3);
	}
	| expr T_EQLSHIFT expr
	{
		$$ = ast_mkBinOp(L_OP_EQLSHIFT, $1, $3, @1, @3);
	}
	| expr T_EQRSHIFT expr
	{
		$$ = ast_mkBinOp(L_OP_EQRSHIFT, $1, $3, @1, @3);
	}
	| expr T_EQDOT expr
	{
		$$ = ast_mkBinOp(L_OP_EQDOT, $1, $3, @1, @3);
	}
	| T_DEFINED "(" expr ")"
	{
		$$ = ast_mkUnOp(L_OP_DEFINED, $3, @1, @4);
	}
	| expr "[" expr "]"
	{
		$$ = ast_mkBinOp(L_OP_ARRAY_INDEX, $1, $3, @1, @4);
	}
	| expr "{" expr "}"
	{
		$$ = ast_mkBinOp(L_OP_HASH_INDEX, $1, $3, @1, @4);
	}
	| expr T_STRCAT expr
	{
		$$ = ast_mkBinOp(L_OP_CONCAT, $1, $3, @1, @3);
	}
	| expr "." T_ID
	{
		$$ = ast_mkBinOp(L_OP_DOT, $1, NULL, @1, @3);
		$$->str = $3;
	}
	| expr "->" T_ID
	{
		$$ = ast_mkBinOp(L_OP_POINTS, $1, NULL, @1, @3);
		$$->str = $3;
	}
	| T_TYPE "." T_ID
	{
		// This is a binop where an arg is a Type*.
		$$ = ast_mkBinOp(L_OP_CLASS_INDEX, (Expr *)$1.t, NULL, @1, @3);
		$$->str = $3;
	}
	| T_TYPE "->" T_ID
	{
		// This is a binop where an arg is a Type*.
		$$ = ast_mkBinOp(L_OP_CLASS_INDEX, (Expr *)$1.t, NULL, @1, @3);
		$$->str = $3;
	}
	| expr "," expr
	{
		$$ = ast_mkBinOp(L_OP_COMMA, $1, $3, @1, @3);
	}
	| expr "[" expr T_DOTDOT expr "]"
	{
		$$ = ast_mkTrinOp(L_OP_ARRAY_SLICE, $1, $3, $5, @1, @3);
	}
	/*
	 * We don't really need to open a scope here, but it doesn't hurt, and
	 * it avoids a shift/reduce conflict with a compound_stmt production.
	 */
	| "{" enter_scope list "}"
	{
		$$ = $3;
		$$->node.loc = @1;
		$$->node.loc.end = @4.end;
		L_scope_leave();
	}
	| "{" "}"
	{
		$$ = ast_mkBinOp(L_OP_LIST, NULL, NULL, @1, @2);
	}
	| expr "?" expr ":" expr %prec T_QUESTION
	{
		$$ = ast_mkTrinOp(L_OP_TERNARY_COND, $1, $3, $5, @1, @5);
	}
	| "<" expr ">"
	{
		$$ = ast_mkUnOp(L_OP_FILE, $2, @1, @3);
	}
	| "<" ">"
	{
		$$ = ast_mkUnOp(L_OP_FILE, NULL, @1, @2);
	}
	;

re_start_split:
		{ L_lex_begReArg(0); }
	;

re_start_case:
		{ L_lex_begReArg(1); }
	;

id:
	  T_ID
	{
		$$ = ast_mkId($1, @1, @1);
		ckfree($1);
	}
	;

id_list:
	id
	| id "," id_list
	{
		$$ = $1;
		$$->next = $3;
		$$->node.loc.end = @3.end;
	}
	;

compound_stmt:
	  "{" enter_scope "}"
	{
		$$ = ast_mkStmt(L_STMT_BLOCK, NULL, @1, @3);
		$$->u.block = ast_mkBlock(NULL, NULL, @1, @3);
		L_scope_leave();
	}
	| "{" enter_scope stmt_list "}"
	{
		REVERSE(Stmt, next, $3);
		$$ = ast_mkStmt(L_STMT_BLOCK, NULL, @1, @4);
		$$->u.block = ast_mkBlock(NULL, $3, @1, @4);
		L_scope_leave();
	}
	| "{" enter_scope declaration_list "}"
	{
		VarDecl	*v;
		REVERSE(VarDecl, next, $3);
		for (v = $3; v; v = v->next) {
			v->flags |= SCOPE_LOCAL | DECL_LOCAL_VAR;
		}
		$$ = ast_mkStmt(L_STMT_BLOCK, NULL, @1, @4);
		$$->u.block = ast_mkBlock($3, NULL, @1, @4);
		L_scope_leave();
	}
	| "{" enter_scope declaration_list stmt_list "}"
	{
		VarDecl	*v;
		REVERSE(VarDecl, next, $3);
		for (v = $3; v; v = v->next) {
			v->flags |= SCOPE_LOCAL | DECL_LOCAL_VAR;
		}
		REVERSE(Stmt, next, $4);
		$$ = ast_mkStmt(L_STMT_BLOCK, NULL, @1, @5);
		$$->u.block = ast_mkBlock($3, $4, @1, @5);
		L_scope_leave();
	}
	;

enter_scope:
	   /* epsilon */ %prec HIGHEST { L_scope_enter(); }
	;

declaration_list:
	  declaration
	| declaration_list declaration
	{
		/*
		 * Each declaration is a list of declarators.  Here we
		 * append the lists.
		 */
		APPEND(VarDecl, next, $2, $1);
		$$ = $2;
	}
	;

declaration:
	  declaration2
	| decl_qualifier declaration2
	{
		VarDecl *v;
		for (v = $2; v; v = v->next) {
			v->flags |= $1;
		}
		$$ = $2;
		$$->node.loc = @1;
	}
	;

decl_qualifier:
	  T_PRIVATE	{ $$ = DECL_PRIVATE; }
	| T_PUBLIC	{ $$ = DECL_PUBLIC; }
	| T_EXTERN	{ $$ = DECL_EXTERN; }
	;

declaration2:
	  type_specifier init_declarator_list ";"
	{
		/* Don't REVERSE $2; it's done as part of declaration_list. */
		VarDecl *v;
		for (v = $2; v; v = v->next) {
			L_set_declBaseType(v, $1);
		}
		$$ = $2;
	}
	| type_specifier ";" { $$ = NULL; }
	;

init_declarator_list:
	  init_declarator
	| init_declarator_list "," init_declarator
	{
		$3->next = $1;
		$$ = $3;
	}
	;

declarator_list:
	  declarator
	| declarator_list "," declarator
	{
		$3->next = $1;
		$$ = $3;
	}
	;

init_declarator:
	  declarator
	| declarator T_EQUALS expr
	{
		$1->initializer = ast_mkBinOp(L_OP_EQUALS, $1->id, $3, @3, @3);
		$$ = $1;
		$$->node.loc.end = @3.end;
	}
	;

opt_declarator:
	  declarator
	| { $$ = NULL; }
	;

declarator:
	  id array_or_hash_type
	{
		$$ = ast_mkVarDecl($2, $1, @1, @2);
	}
	| T_TYPE array_or_hash_type
	{
		Expr *id = ast_mkId($1.s, @1, @1);
		$$ = ast_mkVarDecl($2, id, @1, @2);
		if (isnameoftype($1.t)) $$->flags |= DECL_REF;
		ckfree($1.s);
	}
	| T_BITAND id array_or_hash_type
	{
		Type *t = type_mkNameOf($3);
		$$ = ast_mkVarDecl(t, $2, @1, @3);
		$$->flags |= DECL_REF;
	}
	| T_BITAND id "(" parameter_list ")"
	{
		Type *tf = type_mkFunc(NULL, $4);
		Type *tn = type_mkNameOf(tf);
		$$ = ast_mkVarDecl(tn, $2, @1, @5);
		$$->flags |= DECL_REF;
	}
	;

/* Right recursion OK here since depth is typically low. */
array_or_hash_type:
	  /* epsilon */
	{
		$$ = NULL;
	}
	| "[" expr "]" array_or_hash_type
	{
		$$ = type_mkArray($2, $4);
	}
	| "[" "]" array_or_hash_type
	{
		$$ = type_mkArray(NULL, $3);
	}
	| "{" scalar_type_specifier "}" array_or_hash_type
	{
		$$ = type_mkHash($2, $4);
	}
	;

type_specifier:
	  scalar_type_specifier array_or_hash_type
	{
		if ($2) {
			L_set_baseType($2, $1);
			$$ = $2;
		} else {
			$$ = $1;
		}
	}
	| struct_specifier array_or_hash_type
	{
		if ($2) {
			L_set_baseType($2, $1);
			$$ = $2;
		} else {
			$$ = $1;
		}
	}
	;

scalar_type_specifier:
	  T_STRING	{ $$ = L_string; }
	| T_INT		{ $$ = L_int; }
	| T_FLOAT	{ $$ = L_float; }
	| T_POLY	{ $$ = L_poly; }
	| T_WIDGET	{ $$ = L_widget; }
	| T_VOID	{ $$ = L_void; }
	| T_TYPE	{ $$ = $1.t; ckfree($1.s); }
	;

struct_specifier:
	  T_STRUCT T_ID "{" struct_decl_list "}"
	{
		REVERSE(VarDecl, next, $4);
		$$ = L_struct_store($2, $4);
		ckfree($2);
	}
	| T_STRUCT "{" struct_decl_list "}"
	{
		REVERSE(VarDecl, next, $3);
		(void)L_struct_store(NULL, $3);  // to sanity check member types
		$$ = type_mkStruct(NULL, $3);
	}
	| T_STRUCT T_ID
	{
		$$ = L_struct_lookup($2, FALSE);
		ckfree($2);
	}
	;

struct_decl_list:
	  struct_decl
	| struct_decl_list struct_decl
	{
		APPEND(VarDecl, next, $2, $1);
		$$ = $2;
		$$->node.loc = @1;
	}
	;

struct_decl:
	  struct_declarator_list ";"	{ $$->node.loc.end = @2.end; }
	;

struct_declarator_list:
	  type_specifier declarator_list
	{
		VarDecl *v;
		for (v = $2; v; v = v->next) {
			L_set_declBaseType(v, $1);
		}
		$$ = $2;
		$$->node.loc = @1;
	}
	;

list:
	  list_element
	| list "," list_element
	{
		APPEND(Expr, b, $1, $3);
		$$ = $1;
	}
	| list ","
	;

list_element:
	  expr %prec HIGHEST
	{
		$$ = ast_mkBinOp(L_OP_LIST, $1, NULL, @1, @1);
	}
	| expr "=>" expr %prec HIGHEST
	{
		Expr *kv = ast_mkBinOp(L_OP_KV, $1, $3, @1, @3);
		$$ = ast_mkBinOp(L_OP_LIST, kv, NULL, @1, @3);
	}
	;

string_literal:
	  T_STR_LITERAL
	{
		$$ = ast_mkConst(L_string, $1, @1, @1);
	}
	| interpolated_expr T_STR_LITERAL
	{
		Expr *right = ast_mkConst(L_string, $2, @2, @2);
		$$ = ast_mkBinOp(L_OP_INTERP_STRING, $1, right, @1, @2);
	}
	| here_doc_backtick T_STR_LITERAL
	{
		Expr *right = ast_mkConst(L_string, $2, @2, @2);
		$$ = ast_mkBinOp(L_OP_INTERP_STRING, $1, right, @1, @2);
	}
	;

here_doc_backtick:
	  T_START_BACKTICK T_STR_BACKTICK
	{
		Expr *left  = ast_mkConst(L_string, $1, @1, @1);
		Expr *right = ast_mkUnOp(L_OP_CMDSUBST, NULL, @2, @2);
		right->str = $2;
		$$ = ast_mkBinOp(L_OP_INTERP_STRING, left, right, @1, @2);
	}
	| here_doc_backtick T_START_BACKTICK T_STR_BACKTICK
	{
		Expr *middle = ast_mkConst(L_string, $2, @2, @2);
		Expr *right  = ast_mkUnOp(L_OP_CMDSUBST, NULL, @3, @3);
		right->str = $3;
		$$ = ast_mkTrinOp(L_OP_INTERP_STRING, $1, middle, right,
				  @1, @3);
	}
	;

cmdsubst_literal:
	  T_STR_BACKTICK
	{
		$$ = ast_mkUnOp(L_OP_CMDSUBST, NULL, @1, @1);
		$$->str = $1;
	}
	| interpolated_expr T_STR_BACKTICK
	{
		$$ = ast_mkUnOp(L_OP_CMDSUBST, $1, @1, @2);
		$$->str = $2;
	}
	;

regexp_literal:
	  T_RE
	{
		$$ = ast_mkRegexp($1, @1, @1);
	}
	| interpolated_expr_re T_RE
	{
		Expr *right = ast_mkConst(L_string, $2, @2, @2);
		$$ = ast_mkBinOp(L_OP_INTERP_RE, $1, right, @1, @2);
	}
	;

regexp_literal_mod:
	  regexp_literal T_RE_MODIFIER
	{
		/* Note: the scanner catches illegal modifiers. */
		if (strchr($2, 'i')) $1->flags |= L_EXPR_RE_I;
		if (strchr($2, 'g')) $1->flags |= L_EXPR_RE_G;
		if (strchr($2, 'l')) $1->flags |= L_EXPR_RE_L;
		if (strchr($2, 't')) $1->flags |= L_EXPR_RE_T;
		ckfree($2);
		$$ = $1;
	}
	;

subst_literal:
	  T_SUBST
	{
		$$ = ast_mkConst(L_string, $1, @1, @1);
	}
	| interpolated_expr_re T_SUBST
	{
		Expr *right = ast_mkConst(L_string, $2, @2, @2);
		$$ = ast_mkBinOp(L_OP_INTERP_RE, $1, right, @1, @2);
	}
	;

interpolated_expr:
	  T_LEFT_INTERPOL expr T_RIGHT_INTERPOL
	{
		Expr *left = ast_mkConst(L_string, $1, @1, @1);
		$$ = ast_mkBinOp(L_OP_INTERP_STRING, left, $2, @1, @3);
	}
	| interpolated_expr T_LEFT_INTERPOL expr T_RIGHT_INTERPOL
	{
		Expr *middle = ast_mkConst(L_string, $2, @2, @2);
		$$ = ast_mkTrinOp(L_OP_INTERP_STRING, $1, middle, $3, @1, @4);
	}
	;

interpolated_expr_re:
	  T_LEFT_INTERPOL_RE expr T_RIGHT_INTERPOL_RE
	{
		Expr *left = ast_mkConst(L_string, $1, @1, @1);
		$$ = ast_mkBinOp(L_OP_INTERP_STRING, left, $2, @1, @3);
	}
	| interpolated_expr_re T_LEFT_INTERPOL_RE expr T_RIGHT_INTERPOL_RE
	{
		Expr *middle = ast_mkConst(L_string, $2, @2, @2);
		$$ = ast_mkTrinOp(L_OP_INTERP_STRING, $1, middle, $3, @1, @4);
	}
	;

dotted_id:
	  "."
	{
		$$ = ast_mkId(".", @1, @1);
	}
	| dotted_id_1
	{
		$$ = ast_mkId(Tcl_GetString($1), @1, @1);
		Tcl_DecrRefCount($1);
	}
	;

dotted_id_1:
	  "." T_ID
	{
		$$ = Tcl_NewObj();
		Tcl_IncrRefCount($$);
		Tcl_AppendToObj($$, ".", 1);
		Tcl_AppendToObj($$, $2, -1);
		ckfree($2);
	}
	| dotted_id_1 "." T_ID
	{
		Tcl_AppendToObj($1, ".", 1);
		Tcl_AppendToObj($1, $3, -1);
		$$ = $1;
		ckfree($3);
	}
	;
%%
