#!/usr/bin/env perl
#
# Copyright (C) 2005-2009 H. Appel, M. Marques, X. Andrade, D. Strubbe
#
# Originally GPL, dual-licensed under BerkeleyGW license by permission of these authors.
#
# Based on testsuite from Octopus project:
# $Id: oct-run_regression_test.pl 6053 2009-11-10 19:48:49Z dstrubbe $
# with many updates from Octopus since then.

use Getopt::Std;
use File::Basename;
use Fcntl ':mode';
use Time::HiRes qw(gettimeofday tv_interval);

sub usage {

  print <<EndOfUsage;

 Copyright (C) 2005-2011 H. Appel, M. Marques, X. Andrade, D. Strubbe

Usage: run_regression_test.pl [options]

    -n        dry-run
    -v        verbose
    -h        this usage
    -D        name of the directory where to look for the executables   
    -s        run everything serial
    -f        filename of testsuite
    -i        print inputfile
    -p        preserve working directories
    -l        copy output log to current directory
    -m        run matches only (assumes there are work directories)

Exit codes:
    0         all tests passed
    255       test skipped
    1..254    number of test failures

Report bugs to <BerkeleyGW\@civet.berkeley.edu>
EndOfUsage

  # Option -d is ignored for the moment.
  #    -d        working directory for the tests

  exit 0;
}


sub set_precision{
  my $p = $_[0];
  if($p ne "default"){
    $precnum = 1.0*$p;
  }elsif($_[1] =~ m/_single/){
    $precnum = 0.001
  } else {
    $precnum = 0.0001
  }
}

# Check whether STDOUT is a terminal. If not, no ANSI sequences are written.
if(-t STDOUT) {
    $color_start{blue}="\033[34m";
    $color_end{blue}="\033[0m";
    $color_start{red}="\033[31m";
    $color_end{red}="\033[0m";
    $color_start{green}="\033[32m";
    $color_end{green}="\033[0m";
}

if (not @ARGV) { usage; }

getopts("nlvshD:f:ipm");

$test_succeeded = 1;
# set to 0 if anything fails

# Default values
use File::Temp qw/tempdir/;

# Handle options
$opt_h && usage;

my $exec_directory;
if($opt_D) {
 $exec_directory = $opt_D;
 if($exec_directory !~ /^\//){
  $exec_directory = $ENV{PWD}."/$exec_directory";
 }
} else {
 $exec_directory = $ENV{PWD}."/../../bin";
}

$aexec = $ENV{EXEC};
$global_np = $ENV{BGW_TEST_NPROCS};

if(!$opt_s) {
# MPI stuff
    $mpiexec = $ENV{MPIEXEC};
    $machinelist = $ENV{MACHINELIST};
    if ("$mpiexec" eq "") { $mpiexec = `which mpiexec 2> /dev/null`; }
    chomp($mpiexec);
    if( "$mpiexec" eq "" ) { print "No mpiexec found: running in serial.\n\n"; }
    
# mpiexec without arguments (to check if it is available)
    $mpiexec_raw = $mpiexec;
    $mpiexec_raw =~ s/\ (.*)//;

# default number of processors is 1
    $np = 1;
} else {
    $np = serial;
}

$output = "out";
$pipe_input = 0;
$cmd_line = 0;

# This variable counts the number of failed testcases.
$failures = 0;

$tempdirpath = $ENV{TEMPDIRPATH};
if ("$tempdirpath" eq "") { $tempdirpath = '/tmp'; }

set_precision("default", $exe);

# testsuite
open(TESTSUITE, "<".$opt_f ) or die "cannot open testsuite file '$opt_f'.\n";

$return_value = 0;

while ($_ = <TESTSUITE>) {

  if($return_value != 0) { 
      print "\nSkipping subsequent steps due to nonzero exit code.\n\n";
      exit $failures;
  }

  # skip comments
  next if /^#/;

  if ( $_ =~ /^Test\s*:\s*(.*)\s*$/) {
    $test{"name"} = $1;
    if(!$opt_i) {
	print "$color_start{blue} ***** $test{\"name\"} ***** $color_end{blue} \n\n";
	print "Using test file  : $opt_f \n";
    }

  } elsif ( $_ =~ /^Enabled\s*:\s*(.*)\s*$/) {
    %test = ();
    $enabled = $1;
    $enabled =~ s/^\s*//;
    $enabled =~ s/\s*$//;
    $test{"enabled"} = $enabled;

    if( $enabled eq "Yes" ) {
	$workdir = tempdir("$tempdirpath/BGW.XXXXXX");
	chomp($workdir);

	if(!$opt_i) {
	    print "Using workdir    : $workdir \n";
	    if($opt_p) {
		print "Workdir will be saved.\n";
	    }
	}

	if (!$opt_m) {
	    system ("rm -rf $workdir");
	    mkdir $workdir;
	}
	
    } else {
	if (!$opt_p && !$opt_m && $test_succeeded) { system ("rm -rf $workdir"); }
	if ( $enabled eq "No") {
	    print "Test disabled: skipping test\n\n";
	    exit 255;
	} else {
	    die "Unknown option 'Enabled = $enabled' in testsuite file.\n\n";
	}
    }

  } else {

    if ( $enabled eq "") {
      die "Testsuite file must set Enabled tag before any others (except Test).\n\n";
    }
    if ( $_ =~ /^Executable\s*:\s*(.*)\s*$/) {
	$exe = "$exec_directory/$1";

	if( ! -x $exe) {
	  print "\nSkipping test: executable $exe not available.\n\n";
	  if (!$opt_p && !$opt_m && $test_succeeded) { system ("rm -rf $workdir"); }
	  if($failures == 0) {
	      exit 255;
	  } else {
	      exit $failures;
	      # if a previous step has failed, mark as failed not skipped
	  }
        }
    }

    if ( $_ =~ /^Processors\s*:\s*(.*)\s*$/) {
	$np = $1;
    }

    # for debugging purposes, to halt test after the part you are studying has run
    if ( $_ =~ "STOP TEST") {
	print "\nSTOP TEST\n";
	exit 1;
    }

    if ( $_ =~ /^Banner\s*:\s*(.*)\s*$/) {
	$banner = $1;
	$len = length($banner);
	print "\n";
	print '+'; print '-'x($len+2); print "+\n";
	print "| $banner |\n";
	print '+'; print '-'x($len+2); print "+\n";
	print "\n";
    }

    if ( $_ =~ /^Copy\s*:\s*(\S*)\s*(\S*)\s*$/) {
	$return_value = 0;

	$input_base = $1;
	$input_file = dirname($opt_f) . "/" . $input_base;
	$input_newname = $2;
	if($input_newname eq "") {
	  $input_newname = $input_base;
	}

	if( -f $input_file ) {
	  if( !$pipe_input && !$cmd_line) {
	    print "\n\nCopying file : $input_file \n";
    	    system("cp $input_file $workdir/$input_newname");
  	    # Ensure that the input file is writable so that it can
  	    # be overwritten by the next test.
  	    $mode = (stat "$workdir/$input_newname")[2];
  	    chmod $mode|S_IWUSR, "$workdir/$input_newname";
	  }
	} else {
	  die "could not find input file: $input_file\n";
	}

    }

    if ( $_ =~ /^Command\s*:\s*(.*)\s*$/) {
	$command = "cd $workdir; $1";

	print "\nRunning command : $command\n";
	$return_value = system($command);

	if($return_value != 0) {
	    print "Command failed with exit code $return_value.\n";
	    $failures++;
	}
    }

    if ( $_ =~ /^Arguments\s*:\s*(.*)\s*$/) {
	$command = "cd $workdir; $aexec $exe $1";

	print "\nRunning command : $command\n";
	$return_value = system($command);

	if($return_value != 0) {
	    print "Command failed with exit code $return_value.\n";
	    $failures++;
	}
    }

    if ( $_ =~ /^Unpack\s*:\s*(.*)\s*$/) {
	$command = "tar xzf " . dirname($opt_f) . "/$1 -C $workdir";

	print "\nUnpacking archive : $command\n";
	$return_value = system($command);

	if($return_value != 0) {
	    print "Unpack failed with exit code $return_value.\n";
	    $failures++;
	}
    }

    if ( $_ =~ /^Output\s*:\s*(.*)\s*$/) {
	$output = $1;
    }

    if ( $_ =~ /^Input\s*:\s*(\S*)\s*(\S*)\s*$/) {
	$return_value = 0;

	$input_base = $1;
        $input_none = ($input_base eq "NONE");
        if ( !$input_none ){
	  $input_file = dirname($opt_f) . "/" . $input_base;
        } else {
          $input_file = "";
        }

	$input_newname = $2;
	$pipe_input = ($input_newname eq "PIPE");
	$cmd_line = ($input_newname eq "CMDLINE");
	if($input_newname eq "") {
	  $input_newname = $input_base;
	}

        if( !$input_none ) {
	  if( -f $input_file ) {
	    if( !$pipe_input && !$cmd_line) {
	      print "\n\nUsing input file : $input_file \n";
      	    system("cp $input_file $workdir/$input_newname");
    	    # Ensure that the input file is writable so that it can
    	    # be overwritten by the next test.
    	    $mode = (stat "$workdir/$input_newname")[2];
    	    chmod $mode|S_IWUSR, "$workdir/$input_newname";
	    } else {
	      system("cp $input_file $workdir/");
	    }
	  } else {
	    die "could not find input file: $input_file\n";
	  }
        }

	if ( !$opt_m ) {
	  if ( !$opt_n ) {
	    print "\nStarting test run ...\n";

	    if ($pipe_input) {
		$input_text = "< $input_base";
	    } elsif($cmd_line) {
		$input_text = "$input_base";
	    } else {
		$input_text = ""; 
	    }

	    # serial or MPI run?
	    if ( -x "$mpiexec_raw" && $np ne "serial" && !$opt_s) {
	      if("$global_np" ne "") {
		  $np = $global_np;
	      }
	      if ( "$mpiexec" =~ /ibrun/ ) {
		$specify_np = "";
		$my_nslots = "MY_NSLOTS=$np";
	      } else {
		$specify_np = "-n $np";
		$my_nslots = "";
	      }
	      $command_line = "cd $workdir; $my_nslots $mpiexec $specify_np $machinelist $aexec $exe $input_text > $output";
	    } else {
	      $command_line = "cd $workdir; $aexec $exe $input_text > $output";
	    }

	    print "Executing: " . $command_line . "\n";

	    $test_start = [gettimeofday];
	    $return_value = system("$command_line");
	    $test_end   = [gettimeofday];

	    $elapsed = tv_interval($test_start, $test_end);
	    printf("\tElapsed time: %8.1f s\n\n", $elapsed);

            if($return_value == 0) { 
		print "Finished test run.\n\n"; 
		printf "%-40s%s", " Execution", ": \t [ $color_start{green}  OK  $color_end{green} ] \n"; 

	    } else { 
		print "\n\nTest run failed with exit code $return_value.\n\n"; 
		printf "%-40s%s", " Execution", ": \t [ $color_start{red} FAIL $color_end{red} ] \n\n"; 
		$failures++; 
		$test_succeeded = 0; 
	    } 

	  } else {
	    if(!$opt_i) {
	      $command_line = "cd $workdir; $aexec $exe $input_text > $output";
	      print $command_line . "\n";
	    }
	  }
	  $test{"run"} = 1;
	  if ($opt_l) {
	    system ("cat $workdir/$output >> out.log");
	  }
	}

	# file for shell script with matches
	$mscript = "$workdir/matches.sh";
	open(SCRIPT, ">$mscript") or die "could not create script file\n";
	# write skeleton for script
	print SCRIPT "#\!/bin/bash\n\n";
	close(SCRIPT);
	chmod 0755, $mscript;

	$pipe_input = 0;
	$cmd_line = 0;
    }

    if ( $_ =~ /^Precision\s*:\s*(.*)\s*$/) {
	set_precision($1, $exe) ;
    }

    if ( $_ =~ /^match/ && !$opt_n && $return_value == 0) {
	if(run_match_new($_)){
	  printf "%-40s%s", "$name", ": \t [ $color_start{green}  OK  $color_end{green} ] \t (Calculated value = $value)\n";
	  if ($opt_v) { print_hline(); }
	} else {
	  printf "%-40s%s", "$name", ": \t [ $color_start{red} FAIL $color_end{red} ] \n";
	  print_hline();
	  $test_succeeded = 0;
	  $failures++;
	}
    }
  
  }

}

if (!$opt_p && !$opt_m && $test_succeeded) { system ("rm -rf $workdir"); }

print "\n";
close(TESTSUITE);

exit $failures;


sub find_executables(){
  my $name;

  open(TESTSUITE, "<".$opt_f ) or die "cannot open testsuite file '$opt_f'.\n";
  while ($_ = <TESTSUITE>) {
    if ( $_ =~ /^Programs\s*:\s*(.*)\s*$/) {
      my $i = 0;
      foreach my $program (split(/;/, $1)) {
	$program =  "$program$exec_suffix";
	$program =~ s/^\s+//;
	foreach my $x (@execs) {
	  $valid = $program cmp $x;
	  if(!$valid) {
	    $executables[$i] = "$exec_directory/$x";
	    $i = $i+1;
	  }
	}
      }
    }

  }
  close(TESTSUITE);

  # Die if no suitable executable was found.
  if( @executables == 0 ){
    print stderr "$color_start{blue} ***** $name ***** $color_end{blue} \n\n";
    print stderr "$color_start{red}No valid executable$color_end{red} found for $opt_f\n";
    print stderr "Skipping ... \n\n";
    exit 255;
  }
}

sub run_match_new(){
  die "Have to run before matching" if !$test{"run"} && !opt_m;

  # parse match line
  my $line, $match, $name, $pre_command, $ref_value;
  $line = @_[0];
  $line =~ s/\\;/_COLUMN_/g;
  ($match, $name, $pre_command, $ref_value) = split(/;/, $line);
  $pre_command =~ s/_COLUMN_/;/g;
  $ref_value =~ s/^\s*//;
  $ref_value =~ s/\s*$//;

  # parse command
  $pre_command =~ /\s*(\w+)\s*\((.*)\)/;

  my $func = $1;
  my $params = $2;

  # parse parameters
  $params =~ s/\\,/_COMMA_/g;
  my @par = split(/,/, $params);
  for($params=0; $params <= $#par; $params++){
    $par[$params] =~ s/_COMMA_/,/g;
    $par[$params] =~ s/^\s*//;
    $par[$params] =~ s/\s*$//;
  }

  if($func eq "SHELL"){ # function SHELL(shell code)
    $pre_command = $par[0];

  }elsif($func eq "LINE") { # function LINE(filename, line, field)
    if($par[1] < 0) { # negative number means from end of file
      $line_num = "`wc -l $par[0] | awk '{print \$1}'`";
      $pre_command = "awk -v n=$line_num '(NR==n+$par[1]+1) {printf \$$par[2]}' $par[0]";
    } else {
      $pre_command = "awk '(NR==$par[1]) {printf \$$par[2]}' $par[0]";
    }
  }elsif($func eq "GREP") { # function GREP(filename, 're', field <, offset>)
    my $off = 1*$par[3];
    # -a means even if the file is considered binary due to a stray funny character, it will work
    $pre_command = "grep -a -A$off $par[1] $par[0]";
    $pre_command .= " | awk '(NR==$off+1) {printf \$$par[2]}'";
    # if there are multiple occurrences found by grep, we will only be taking the first one via awk

  }else{ # error
    printf stderr "Unknown command '$func'\n";
    return 0;
  }

  # append the command and the regexp also to the shell script matches.sh in the
  # current archive directory
  open(SCRIPT, ">>$mscript");
  print SCRIPT "
echo '", "="x4, " [ $name - pre command ]'
$pre_command
echo
echo '", "-"x4, " [ $name - ref value   ]'
echo $ref_value
export LINE=`$pre_command`
perl -e 'print \"Match: \".(abs(\$ENV{LINE}-($ref_value)) <= $precnum ? \"OK\" : \"FAILED\");'
echo
echo";
  close(SCRIPT);

  $value = `cd $workdir; $pre_command`;

  $success = ("$value" ne "") && (abs(($value)-($ref_value)) <= $precnum);

  if(!$success || $opt_v) {
    print_hline();
    print "Match".$name.":\n\n";
    print "   Calculated value : ".$value."\n";
    print "   Reference value  : ".$ref_value."\n";
    print "   Difference       : ".abs($ref_value - $value)."\n";
    print "   Tolerance        : ".$precnum."\n\n";
  }

  return $success;
}

sub print_hline(){
  print "\n-----------------------------------------\n\n";
}
