#!/usr/bin/python

#-------------------------------------------------------------------------------
#
#   gsphere.py
#   generates a sphere of G-vectors given lattice parameters and energy cutoff
#   written by G. Samsonidze (September 2009)
#
#-------------------------------------------------------------------------------

from common import eps9, inf9, boolean_input

def main(argv = None):
   if argv is None:
      argv = sys.argv
   argc = len(argv)
   self = "gsphere.py"
   if argv[0][-len(self):] != self:
      print("\n   Rename script to %s\n" % self)
      return 1
   if argc != 3:
      print("\n   Usage: %s ifn ofn\n" % self)
      print("   Description: reads lattice vectors (in bohr), energy cutoff (in Ry),")
      print("   FFT grid (set to zero to determine automatically) and flag (true or")
      print("   false) for sorting the G-vectors from file ifn, generates the G-space")
      print("   and writes it to file ofn. The number of G-vectors is roughly equal to")
      print("   N = V * E^(3/2) / (6 * pi^2) where V is the volume of the unit cell")
      print("   (in bohr^3) and E is the energy cutoff (in Ry).\n")
      return 2
   ifn = argv[1]
   ofn = argv[2]
   ierr = 0
   try:
      h = open(ifn, 'r')
   except:
      ierr = 1
   if ierr == 0:
      a = []
      for i in range(3):
         a.append([])
         s = h.readline()
         t = s.split()
         for j in range(3):
            a[i].append(float(t[j]))
      s = h.readline()
      t = s.split()
      ecut = float(t[0])
      s = h.readline()
      t = s.split()
      nr = [int(t[0]), int(t[1]), int(t[2])]
      s = h.readline()
      t = s.split()
      ierr, fsort = boolean_input(t[0])
      h.close()
   if ierr != 0:
      print("\n   Error: unable to read file %s\n" % ifn)
      return 2

   al = math.sqrt(a[0][0] * a[0][0] + a[0][1] * a[0][1] + a[0][2] * a[0][2])
   for i in range(3):
      for j in range(3):
         a[i][j] /= al
   vol = 0.0
   vol += a[0][0] * (a[1][1] * a[2][2] - a[2][1] * a[1][2])
   vol -= a[0][1] * (a[1][0] * a[2][2] - a[2][0] * a[1][2])
   vol += a[0][2] * (a[1][0] * a[2][1] - a[2][0] * a[1][1])
   bl = 2.0 * math.pi / al
   b = [[], [], []]
   b[0].append((a[1][1] * a[2][2] - a[1][2] * a[2][1]) / vol)
   b[0].append((a[1][2] * a[2][0] - a[1][0] * a[2][2]) / vol)
   b[0].append((a[1][0] * a[2][1] - a[1][1] * a[2][0]) / vol)
   b[1].append((a[2][1] * a[0][2] - a[2][2] * a[0][1]) / vol)
   b[1].append((a[2][2] * a[0][0] - a[2][0] * a[0][2]) / vol)
   b[1].append((a[2][0] * a[0][1] - a[2][1] * a[0][0]) / vol)
   b[2].append((a[0][1] * a[1][2] - a[0][2] * a[1][1]) / vol)
   b[2].append((a[0][2] * a[1][0] - a[0][0] * a[1][2]) / vol)
   b[2].append((a[0][0] * a[1][1] - a[0][1] * a[1][0]) / vol)
   vol = abs(vol) * al * al * al
   gcutm = ecut / (bl * bl)

   if nr[0] < 1 or nr[1] < 1 or nr[2] < 1:
      for i in range(3):
         nr[i] = int(2.0 * math.sqrt(gcutm) * math.sqrt(a[i][0] * a[i][0] + a[i][1] * a[i][1] + a[i][2] * a[i][2])) + 1
   nr6 = []
   for i in range(3):
      nr6.append(nr[i])
      while not check_nr_size(nr6[i], 6):
         nr6[i] += 1
   nr3 = []
   for i in range(3):
      nr3.append(nr[i])
      while not check_nr_size(nr3[i], 3):
         nr3[i] += 1

   ng = 0
   g_cryst = []
   g_cart = []
   e_kin = []
   for i in range(-nr[0] / 2, nr[0] / 2 + 1):
      for j in range(-nr[1] / 2, nr[1] / 2 + 1):
         for k in range(-nr[2] / 2, nr[2] / 2 + 1):
            gx = float(i) * b[0][0] + float(j) * b[1][0] + float(k) * b[2][0]
            gy = float(i) * b[0][1] + float(j) * b[1][1] + float(k) * b[2][1]
            gz = float(i) * b[0][2] + float(j) * b[1][2] + float(k) * b[2][2]
            g2 = gx * gx + gy * gy + gz * gz
            if bl * bl * g2 < ecut:
               ng += 1
               g_cryst.append([i, j, k])
               g_cart.append([bl * gx, bl * gy, bl * gz])
               e_kin.append(bl * bl * g2)
   g_sort = []
   for i in range(ng):
      g_sort.append(i)
   if fsort:
      e_sort = []
      for i in range(ng):
         e_sort.append(e_kin[i])
      e_sort.sort()
      for i in range(ng):
         j = e_sort.index(e_kin[i])
         e_sort[j] = inf9
         g_sort[j] = i

   ierr = 0
   try:
      h = open(ofn, 'w')
   except:
      ierr = 1
   if ierr == 0:
      h.write('\n')
      for i in range(3):
         s = ' a%1i =' % (i + 1)
         for j in range(3):
            s += ' %13.9f' % (al * a[i][j])
         s += '\n'
         h.write(s)
      h.write('\n')
      for i in range(3):
         s = ' b%1i =' % (i + 1)
         for j in range(3):
            s += ' %13.9f' % (bl * b[i][j])
         s += '\n'
         h.write(s)
      h.write('\n')
      s = ' volume = %.9f\n' % vol
      h.write(s)
      h.write('\n')
      s = ' cutoff = %.9f\n' % ecut
      h.write(s)
      h.write('\n')
      s = ' grid = ('
      for i in range(3):
         s += ' %i' % nr[i]
      s += ' ) -- minimal\n'
      h.write(s)
      s = '        ('
      for i in range(3):
         s += ' %i' % nr6[i]
      s += ' ) -- factors 2, 3, 5, 7, 1*11, 1*13\n'
      h.write(s)
      s = '        ('
      for i in range(3):
         s += ' %i' % nr3[i]
      s += ' ) -- factors 2, 3, 5\n'
      h.write(s)
      h.write('\n')
      s = ' ng = %i\n' % ng
      h.write(s)
      h.write('\n')
      h.write('     n   g1    g2    g3     gx        gy        gz        ekin\n')
      h.write(' --------------------------------------------------------------------\n')
      for i in range(ng):
         s = ' %5i' % (i + 1)
         for j in range(3):
            s += ' %5i' % g_cryst[g_sort[i]][j]
         for j in range(3):
            s += ' %9.3f' % g_cart[g_sort[i]][j]
         s += ' %15.9f' % e_kin[g_sort[i]]
         s += '\n'
         h.write(s)
      h.write('\n')
      h.close()
   if ierr != 0:
      print("\n   Error: unable to write file %s\n" % ofn)
      return 2
   return 0

def check_nr_size(nr, nf):
   factor = [2, 3, 5, 7, 11, 13]
   power = [0, 0, 0, 0, 0, 0]
   remainder = nr
   for i in range(nf):
      j = int(math.log(remainder) / math.log(factor[i])) + 1
      for k in range(j):
         if remainder % factor[i] == 0:
            remainder /= factor[i]
            power[i] += 1
   return remainder == 1 and power[4] <= 1 and power[5] <= 1

if __name__ == "__main__":
   import sys
   import math
   sys.exit(main())

