#!/usr/bin/python

#-------------------------------------------------------------------------------
#
#   average.py
#   averages the potential on the faces or in the volume of the unit cell box
#   written by G. Samsonidze (April 2009)
#
#-------------------------------------------------------------------------------

from common import rydberg, inf9

def main(argv = None):
   if argv is None:
      argv = sys.argv
   argc = len(argv)
   self = "average.py"
   if argv[0][-len(self):] != self:
      print("\n   Rename script to %s\n" % self)
      return 1
   if not (argc == 3 or argc == 6):
      print("\n   Usage: %s ivfn ivff [xpos ypos zpos]\n" % self)
      print("   ivfn = input volumetric file name")
      print("   ivff = input volumetric file format (cube)")
      print("   xpos = position of yz plane from 0 to 1")
      print("   ypos = position of zx plane from 0 to 1")
      print("   zpos = position of xy plane from 0 to 1\n")
      print("   if xpos ypos zpos are absent average in the volume\n")
      return 2
   ivfn = argv[1]
   ivff = argv[2].lower()
   faces = False
   volume = False
   if argc == 3:
      volume = True
   if argc == 6:
      faces = True
   if faces:
      try:
         xpos = float(argv[3])
      except:
         print("\n   Error: invalid xpos %s\n" % argv[3])
         return 2
      try:
         ypos = float(argv[4])
      except:
         print("\n   Error: invalid ypos %s\n" % argv[4])
         return 2
      try:
         zpos = float(argv[5])
      except:
         print("\n   Error: invalid zpos %s\n" % argv[5])
         return 2
   if ivff == 'cube':
      ierr, na, aspecies, acharge, aposition, grid, origin, step, vol = cub_read(ivfn)
   else:
      print("\n   Error: invalid input format %s\n" % ivff)
      return 2
   if ierr != 0:
      print("\n   Error: unable to read file %s\n" % ivfn)
      return 2
   if faces:
      ipos = int(round(xpos * float(grid[0])))
      jpos = int(round(ypos * float(grid[1])))
      kpos = int(round(zpos * float(grid[2])))
   navg = 0
   vavg = 0.0
   vmin = inf9
   vmax = -inf9
   if volume:
      for i in range(grid[0]):
         for j in range(grid[1]):
            for k in range(grid[2]):
               vcur = vol[i][j][k]
               vavg += vcur
               if vcur < vmin:
                  vmin = vcur
               if vcur > vmax:
                  vmax = vcur
      navg += grid[0]*grid[1]*grid[2]
   if faces:
      if ipos >=0 and ipos < grid[0]:
         i = ipos
         for j in range(grid[1]):
            for k in range(grid[2]):
               vcur = vol[i][j][k]
               vavg += vcur
               if vcur < vmin:
                  vmin = vcur
               if vcur > vmax:
                  vmax = vcur
         navg += grid[1]*grid[2]
      if jpos >=0 and jpos < grid[1]:
         j = jpos
         for i in range(grid[0]):
            for k in range(grid[2]):
               vcur = vol[i][j][k]
               vavg += vcur
               if vcur < vmin:
                  vmin = vcur
               if vcur > vmax:
                  vmax = vcur
         navg += grid[0]*grid[2]
      if kpos >=0 and kpos < grid[2]:
         k = kpos
         for i in range(grid[0]):
            for j in range(grid[1]):
               vcur = vol[i][j][k]
               vavg += vcur
               if vcur < vmin:
                  vmin = vcur
               if vcur > vmax:
                  vmax = vcur
         navg += grid[0]*grid[1]
   if navg > 0:
      vavg /= float(navg)
   vavg *= rydberg
   vmin *= rydberg
   vmax *= rydberg
   print("\n   Average potential = %.6f eV" % vavg)
   print("   Minimum potential = %.6f eV" % vmin)
   print("   Maximum potential = %.6f eV\n" % vmax)
   return 0

def cub_read(file):
   ierr = 0
   na = 0
   aspecies = []
   acharge = []
   aposition = []
   grid = []
   origin = []
   step = []
   vol = [[[]]]
   try:
      h = open(file, 'r')
   except:
      ierr = 1
   if ierr == 0:
      for i in range(2):
         s = h.readline()
      s = h.readline()
      t = s.split()
      na = int(t[0])
      origin = []
      for j in range(3):
         origin.append(float(t[j + 1]))
      grid = []
      step = []
      for i in range(3):
         s = h.readline()
         t = s.split()
         grid.append(int(t[0]))
         step.append([])
         for j in range(3):
            step[i].append(float(t[j + 1]))
      for i in range(na):
         s = h.readline()
         t = s.split()
         aspecies.append(int(t[0]))
         acharge.append(float(t[1]))
         aposition.append([])
         for j in range(3):
            aposition[i].append(float(t[j + 2]))
      n = grid[0] * grid[1] * ((grid[2] - 1) / 6 + 1)
      i = 0
      j = 0
      k = 0
      for m in range(n):
         s = h.readline()
         t = s.split()
         for l in range(6):
            if k < grid[2]:
               vol[i][j].append(float(t[l]))
               k += 1
         if k == grid[2]:
            k = 0
            j += 1
            if j < grid[1]:
               vol[i].append([])
            else:
               k = 0
               j = 0
               i += 1
               if i < grid[0]:
                  vol.append([[]])
      h.close()
   return ierr, na, aspecies, acharge, aposition, grid, origin, step, vol

if __name__ == "__main__":
   import sys
   import math
   sys.exit(main())

