!==========================================================================
!
! Routines:
!
! (1) genwf_disk()  Originally By (JRD)             Last Modified 11/2009 (JRD)
!
!     On entry:
!     q0  = current q-vector
!     rq  = current k point in irr. zone
!     irq = its index
!     iflagq0 = whether q0 is the "zero" vector
!
!     On exit:
!     vwfn%ev and vwfn%zv hold eigenvalues and wavefunctions (valence)
!     cwfn%ec and cwfn%zc hold eigenvalues and wavefunctions (conduction)
!
!     with proper phases and ordering for the k-point rq (given the
!     data on disk for irr. zone)
!
!     subroutine generates valence band wavefunctions for rq(irq)
!     and conduction band wavefunctions for rq(irq) from the
!     wavefunctions available for rk in the irreducible wedge of the
!     bz
!
!     i   rq                 rq(irq)
!     o c nkptv,...,ev       valence band wavefunctions for rq+q
!     and associated data
!     o c nkptc,...,ec       conduction band wavefunctions for rq
!     and associated data
!
!==========================================================================


#include "f_defs.h"

subroutine genwf_disk(syms,gvec,crys,kp,kpq,irq,rq,q0,vwfn,pol,cwfn,iflagq0,ivin)

  use global_m
  use gmap_m
  use sort_m
  implicit none

  type (symmetry), intent(in) :: syms
  type (gspace), intent(in) :: gvec
  type (crystal), intent(in) :: crys
  type (kpoints), target, intent(in) :: kp
  type (kpoints), target, intent(in) :: kpq
  integer, intent(in) :: irq
  real(DP), intent(in) :: rq(3)
  real(DP), intent(in) :: q0(3)
  type (valence_wfns), intent(inout) :: vwfn
  type (polarizability), intent(inout) :: pol
  type (conduction_wfns), intent(inout) :: cwfn
  integer, intent(in) :: iflagq0
  integer, intent(in) :: ivin
  
  real(DP) :: tsec(2)
  integer :: nkpt
  integer, allocatable :: isortc(:)
  real(DP), allocatable :: eig(:,:)
  SCALAR, allocatable :: zin(:,:),zinc(:,:)
  SCALAR, allocatable :: zintemp(:,:)
  type(kpoints), pointer :: kp_point
  
  character :: filename*20, filenamev*20, filenamevq*20
  character :: tmpstr*120
  integer :: irk,itq,itqq,i,j,k,nkpt2
  integer :: n,ig,itpc,itpv,iband
  integer :: naddc,jj,icount
  integer :: ik,ikrqq,ikrq,kgq(3),kgqq(3)
  integer, allocatable :: ind(:),isorti(:)
  real(DP), allocatable :: xnorm(:,:)
  real(DP) :: xnormc(2),qk(3),del(3),rqq(3),xnormcc(2)
  real(DP), allocatable :: ekin(:)
  real(DP) :: rkmatch(3)
  SCALAR, allocatable :: ph(:)

  integer, save :: irq_old=0

  PUSH_SUB(genwf_disk)
  
  if(peinf%inode.eq.0) then
    write(filenamev,'(a)') 'INT_VWFN'
    write(filenamevq,'(a)') 'INT_VWFQ'
  endif

  SAFE_ALLOCATE(isortc, (gvec%ng))
  SAFE_ALLOCATE(eig, (cwfn%nband,kp%nspin))
  SAFE_ALLOCATE(xnorm, (1,kp%nspin))
  
  if(peinf%inode.eq.0) call timacc(16,1,tsec)

! rqq = rq + q0   (i.e. rqq = current kpoint in irr. zone + q0)

  rqq(1:3)=rq(1:3)+q0(1:3)

! If this is the "zero" q vector, look in WFNq, otherwise in WFN.
  if(iflagq0.eq.1 .and. pol%iqexactlyzero .eq. 0) then
    kp_point => kpq
  else
    kp_point => kp
  endif
    
! This is what we used to do for q->0, but it is better to unfold the shifted grid.
!    ikrqq = 0
!    do ik=1,kpq%nrk
!      if(all(abs(rqq(1:3) - kpq%rk(1:3,ik)) .lt. TOL_Small)) then
!        ikrqq=ik
!        itqq=1
!        kgqq(:)=0
!        rkmatch(:)=kpq%rk(:,ik)
!      endif
!    enddo
!         
!    if(ikrqq == 0) then
!      write(tmpstr,'(a,3f12.6,a)') "Could not find point matching (", rqq(1:3), ") in WFNq."
!      call die(tmpstr)
!    endif

! SIB: look for rqq in the list
! of kpoints rotated by the symmetries.  If found, it puts
! its index in ikrqq, rkmatch has the vector, itqq is the index
! of the symmetry that worked, and kgqq is an "umklapp" vector (i.e.
! integer 3-vector) so that rqq = symm*kvec + kgqq.

  ikrqq = 0 
  ik_loop: do ik = 1, kp_point%nrk
    do itqq = 1, syms%ntran
      qk(1:3) = matmul(syms%mtrx(1:3, 1:3, itqq), kp_point%rk(1:3, ik))
      do i = 1, 3
        del(i) = rqq(i) - qk(i)
        if(del(i).ge.0.0d0) kgqq(i) = del(i) + TOL_Small
        if(del(i).lt.0.0d0) kgqq(i) = del(i) - TOL_Small
      enddo
      if(all(abs(del(1:3)-kgqq(1:3)) .lt. TOL_Small)) then
        ikrqq=ik
        rkmatch(1:3) = kp_point%rk(1:3, ikrqq)
        exit ik_loop
      endif
    enddo
  enddo ik_loop
  
  if(ikrqq == 0) then
    write(tmpstr,'(a,3f8.3,a)') &
      'genwf_disk: No match for rqq point:', rqq, ' in file WFN or WFNq.'
    call die(tmpstr)
  endif

!
! SIB:  proc 0 will read from unit itpv and
! skip as many records as needed until it reads all the ikrqq`th valence
! wavefunctions. It then broadcasts the valence data (if MPI).
!

  if(peinf%inode.eq.0) then
    if(iflagq0.ne.1 .or. pol%iqexactlyzero .ne. 0) then
      itpv=200028
      call open_file(itpv,file=filenamev,form='unformatted',status='old')
    else
      itpv=300028
      call open_file(itpv,file=filenamevq,form='unformatted',status='old')
    endif
    if(ikrqq.gt.1) then
      do i=1,ikrqq-1
        read(itpv)
        do j=1,(vwfn%nband+pol%ncrit)
          read(itpv)
        enddo
      enddo
    endif

! SIB: isortc(j) from disk tells us what the index of the j`th g-vector
! is in the array of g-vectors in gvec

    read(itpv) nkpt,(isortc(j),j=1,nkpt), &
      ((eig(j,k),j=1,(vwfn%nband+pol%ncrit)),k=1,kp%nspin),(qk(i),i=1,3)
  endif

#ifdef MPI
  call MPI_Bcast(nkpt, 1, MPI_INTEGER,0, MPI_COMM_WORLD,mpierr)
  call MPI_Bcast(isortc,nkpt,MPI_INTEGER,0,MPI_COMM_WORLD,mpierr)
  call MPI_Bcast(eig,cwfn%nband*kp%nspin,MPI_REAL_DP,0,MPI_COMM_WORLD,mpierr)
  call MPI_Bcast(qk, 3, MPI_REAL_DP, 0, MPI_COMM_WORLD, mpierr)
#endif

  SAFE_ALLOCATE(zintemp, (nkpt,kp%nspin))
  SAFE_ALLOCATE(zin, (nkpt,kp%nspin))
  
  do i=1,(vwfn%nband+pol%ncrit)
    zintemp=0D0
    if (peinf%inode .eq. 0) then
      read(itpv) ((zintemp(j,k),j=1,nkpt),k=1,kp%nspin)
    endif

#ifdef MPI
    call MPI_Bcast(zintemp, nkpt*kp%nspin, MPI_SCALAR, 0, MPI_COMM_WORLD, mpierr)
#endif

!        if (peinf%nvown(i) .eq. 1) then
    if (i .eq. ivin) then
!          zin(((peinf%nvindex(i)-1)*nkpt+1):((peinf%nvindex(i)) &
!          *nkpt),:)=zintemp(:,:)
      zin(:,:)=zintemp(:,:)
    endif
  enddo

  if(peinf%inode == 0) call close_file(itpv)

  SAFE_DEALLOCATE(zintemp)
  
  if (ivin .eq. -1) then 
    SAFE_DEALLOCATE(zin)
    SAFE_DEALLOCATE(xnorm)
    SAFE_DEALLOCATE(eig)
    SAFE_DEALLOCATE(isortc)
    irq_old=irq
    POP_SUB(genwf_disk)
    return
  endif

! Check kpoint

  if(any(abs(rkmatch(1:3) - qk(1:3)) .gt. TOL_Small)) call die('genwf_disk: rkmatch')

! Compute kinetic energies for rqq+g

  SAFE_ALLOCATE(ekin, (gvec%ng))
  do i=1,gvec%ng
    qk(:)=rqq(:)+gvec%k(:,i)
    ekin(i)=DOT_PRODUCT(qk,MATMUL(crys%bdot,qk))
  enddo

!     sort array ekin to ascending order, store indices in array vwfn%isort
!     WARNING: one should not assume that in the case of
!     q-->0 the orders as provided below and as read in from
!     WFNq file is the same (sorting may change....)
!     ===> need to call gmap also for q-->0 (see below)
!     EKC
!     we initialize vwfn%isort to the appropriate array
!     before reading in. this way we do not get zeroes in the array
!     these are the valence wave-functions that do not need
!     to be changed

  call sortrx_D(gvec%ng,ekin,vwfn%isort)

  SAFE_ALLOCATE(isorti, (gvec%ng))
  do i=1,gvec%ng
    isorti(vwfn%isort(i))=i
  enddo
  do i=1,nkpt
    isorti(isortc(i))=i
  enddo
  
  vwfn%nkptv=nkpt

! SIB: put read eigenvalues into vwfn%ev(band,spin).
! Set xnorm(1:(vwfn%nband+pol%ncrit),1:nspin)=0

  if (irq .ne. irq_old) then
    SAFE_ALLOCATE(vwfn%ev, ((vwfn%nband+pol%ncrit),kp%nspin))
    vwfn%ev(1:(vwfn%nband+pol%ncrit),:) = eig(1:(vwfn%nband+pol%ncrit),:)
  endif
  xnorm(:,:) = 0.0d0
  SAFE_DEALLOCATE(eig)

! JRD: Map planewave components for rq+q, to those of rk
! (even for q--> 0)
!
! SIB: get phases (ph) and indices (ind) for g-vectors
! gvec%k(:,vwfn%isort(1:nkpt2))+kgqq

  nkpt2=vwfn%nkptv
  SAFE_ALLOCATE(ind, (vwfn%nkptv))
  SAFE_ALLOCATE(ph, (vwfn%nkptv))
  call gmap(gvec,syms,nkpt2,itqq,kgqq,vwfn%isort,isorti,ind,ph,.true.)

! loop over valence band wavefunctions

  SAFE_ALLOCATE(vwfn%zv, (vwfn%nkptv,kp%nspin))
  n=1

! loop over components of zv
!
! XAV: vwfn%zv(ig) corresponds really to the
! vwfn%isort(ig) G-vector (between 1 and ng)
! The subroutine gmap assumes that, as read from WFNq or WFN,
! zin(ig) corresponds really to isortc(ig) G-vector !!!
 
  do ig=1,vwfn%nkptv
    vwfn%zv(ig,:)=zin(ind(ig),:)*ph(ig)
    xnorm(n,:)=xnorm(n,:)+abs(vwfn%zv(ig,:))**2
  enddo !ig

! SIB:  checks that zin vectors have norm greater than 1e-6, and then
! normalizes them to have unit square modulus.

  do k = 1, kp%nspin
    n=1
    if(xnorm(n,k).lt.TOL_Small)  then
      xnormc(k)=0.0d0
      xnormcc(k)=0.0d0
      do ig=1,vwfn%nkptv
        xnormc(k)=xnormc(k)+abs(zin(ind(ig),k))**2
        xnormcc(k)=xnormcc(k)+abs(zin(ind(ig),k)*ph(ig))**2
      enddo
      xnormc(k)=sqrt(xnormc(k))
      write(0,8000) ik,n,xnormc(k),xnormcc(k),k
8000  format(/,'In genwf_disk, read file 15 state nrk,band=',2i5,/, &
        'xnormc(k)=',f10.5,'xnormcc(k)=',f10.5,'spin=',i1/)
#ifndef CPLX
      write(0,*) '**Are you doing a supercell? If so you should use complex version with complex wavefunctions!**'
#endif
      call die('genwf_disk: zin norm')
    endif
    xnorm(n,k)=sqrt(xnorm(n,k))
  end do !k (over spin)

  n=1
  do ig=1,vwfn%nkptv
    vwfn%zv(ig,:)=vwfn%zv(ig,:)/xnorm(n,:)
  enddo

  SAFE_DEALLOCATE(zin)
  SAFE_DEALLOCATE(ind)
  SAFE_DEALLOCATE(ph)
  SAFE_DEALLOCATE(xnorm)
  if(peinf%inode.eq.0) call timacc(16,2,tsec)

!------------------------------------------------------------
! Generate conduction-band wavefunctions for rq
! find rk, r, g0 such that rq=r(rk)+g0
! SIB: This seems redundant, but find a k-point and symmetry so that
! rq = sym%mtrx(:,:,itq)*kp%rk(irk,:) + kgq, where kgq is integer 3-vec

  if(irq .ne. irq_old) then
    
    if(peinf%inode.eq.0) call timacc(17,1,tsec)
    
    ikrq = 0
    irk_loop: do irk = 1, kp%nrk
      do itq = 1, syms%ntran
        qk(1:3) = matmul(syms%mtrx(1:3, 1:3, itq), kp%rk(1:3, irk))
        do i = 1, 3
          del(i) = rq(i) - qk(i)
          if(del(i) .ge. 0.0d0) kgq(i) = del(i) + TOL_Small
          if(del(i) .lt. 0.0d0) kgq(i) = del(i) - TOL_Small
        enddo
        if(all(abs(del(1:3)-kgq(1:3)) .lt. TOL_Small)) then
          ikrq = irk 
          exit irk_loop
        endif
      enddo
    enddo irk_loop
    
    if(ikrq == 0) call die('genwf_disk: kgq mismatch')
    
! write out rq, it and kgq
  
    if(peinf%inode.eq.0) then
      write(6,7000) (rq(i),i=1,3),irk,(kp%rk(i,irk),i=1,3),itq,(kgq(i),i=1,3)
7000  format(1x,'rq=',3f7.3,1x,'irk=',i5,1x,' rk=',3f7.3,1x,'it=',i5,1x,'kg0=',3i3)
    endif
  
! SIB:  if we already found this k-point last time, get its qk, nkpt,
! and isortc(:).  Otherwise, open from unit itpc=128+peinf%inode,
! skip irk-1 records, and read in information (qk,cwfn%ec,nkpt,isortc),
! JRD: Now need open
  
    if(peinf%inode.lt.10000) then
      write(filename,'(a,i4.4)') 'INT_CWFN_', peinf%inode
    else
      call die('too many nodes required')
    endif
  
    itpc=100028+peinf%inode
    call open_file(itpc,file=filename,form='unformatted',status='old')
    
    if(irk.gt.1) then
      icount=0
      do i=1,irk-1
        icount=icount+peinf%ncownt+1
      enddo
      do i=1,icount
        read(itpc)
      enddo
    endif
    
    SAFE_ALLOCATE(cwfn%ec, (cwfn%nband,kp%nspin))
    read(itpc) (qk(i),i=1,3),((cwfn%ec(j,k),j=1,cwfn%nband),k=1,kp%nspin),nkpt,(isortc(j),j=1,nkpt)
  
! Check kpoint (again ...  boring...)
! Check that kp%rk(:,irk) = qk  (why it wouldn`t is a mystery!)
  
    if(any(abs(kp%rk(1:3, irk) - qk(1:3)) .gt. TOL_Small)) &
      call die('genwf_disk: qk mismatch')
  
    cwfn%nkptc=nkpt
  
! Compute inverse to isort
! isortc orders   |kp%rk+G|^2
! It is not necessarily the same order than |rq+G|^2
! (particularly if umklapp, ie kgq non zero)
  
    do i=1,gvec%ng
      qk(:)=rq(:)+gvec%k(:,i)
      ekin(i)=DOT_PRODUCT(qk,MATMUL(crys%bdot,qk))
    enddo
  
! sort array ekin to ascending order
! store indices in array isort

    call sortrx_D(gvec%ng,ekin,cwfn%isort)
    
    do i=1,gvec%ng
      cwfn%ekin(i) = ekin(cwfn%isort(i))
      isorti(cwfn%isort(i))=i !to initialize
    enddo
    do i=1,nkpt
      isorti(isortc(i))=i
    enddo
    SAFE_DEALLOCATE(isortc)
    SAFE_DEALLOCATE(ekin)
!
!     map planewave components for rq to those of rk
!     compute phases
!     We do not the isorti related to kp%rk BUT the cwfn%isort related to rq
!
    nkpt2=cwfn%nkptc
    SAFE_ALLOCATE(ind, (cwfn%nkptc))
    SAFE_ALLOCATE(ph, (cwfn%nkptc))
    call gmap(gvec,syms,nkpt2,itq,kgq,cwfn%isort,isorti,ind,ph,.true.)
    SAFE_DEALLOCATE(isorti)
!
!     generate conduction band wavefunctions
!     loop over wavefunctions
!     read conduction band from tape one by one
!
    SAFE_ALLOCATE(cwfn%zc, (peinf%ncownt*cwfn%nkptc,kp%nspin))
    SAFE_ALLOCATE(zinc, (cwfn%nkptc,kp%nspin))
  
    do n=1,peinf%ncownt
      
      read(itpc) iband,nkpt2,((zinc(jj,k),jj=1,nkpt2),k=1,kp%nspin)
      
      if(nkpt2.ne.cwfn%nkptc) call die('genwf_disk: nkptc mismatch')
      if(iband.ne.peinf%invindexc(n)) call die('genwf_disk: invindexc mismatch')
      
      naddc=(n-1)*cwfn%nkptc
!
!     loop over components of wfns
!     note that only conduction band wfns are stored-
!     they start in the 1st position with the state nvband+1
!
      do ig=1,cwfn%nkptc
        cwfn%zc(naddc+ig,:)=ph(ig)*zinc(ind(ig),:)
      enddo
    enddo  ! n (cond-bands per node [ncownt] loop)

    call close_file(itpc)

    SAFE_DEALLOCATE(zinc)
    SAFE_DEALLOCATE(ind)
    SAFE_DEALLOCATE(ph)
!
!     end generation of conduction band wavefunctions for rq
!
  
    if(peinf%inode.eq.0) call timacc(17,2,tsec)

  endif !irq_old == irq

  irq_old=irq

  POP_SUB(genwf_disk)
  
  return
end subroutine genwf_disk
