!===============================================================================
!
! Program:
!
! (1) openmp_main              Originally By gsm        Last Modified 7/15/2010 (gsm)
!
! This program tests OpenMP multithreading within MPI process.
!
!===============================================================================

#include "f_defs.h"

program openmp_main

  use global_m
  use misc_m
  implicit none

!------------------------
! Parameters

  integer, parameter :: n = 134217728 ! 2^27 = 128*1024^2

!------------------------
! Local variables

  integer :: i, nslices, nmpinode, ithread, nthreads
  real(DP) :: mem_b, mem_mb, a, b, tsec(2)
  character(len=16) :: s1, s2, s3, s4

!------------------------
! External functions

  call peinfo_init()

!------------------------
! Determine available memory

  call procmem(mem_b,nmpinode)
  mem_mb=mem_b/1024.0d0**2

!------------------------
! Initialize OpenMP

#ifdef OMP
!$OMP PARALLEL PRIVATE(ithread)
  ithread = omp_get_thread_num()
!$OMP END PARALLEL
  nthreads = omp_get_max_threads()
#else
  ithread = 0
  nthreads = 1
#endif

!------------------------
! Write header

  write(s1,101) peinf%npes
  write(s2,101) nmpinode
  write(s3,102) mem_mb
  write(s4,101) nthreads
  if (peinf%inode.eq.0) then
    write(6,201) TRUNC(s1)
    write(6,202) TRUNC(s2)
    write(6,203) TRUNC(s3)
    write(6,204) TRUNC(s4)
  endif

!------------------------
! Test performance

  call timacc(0,0,tsec)

  call timacc(1,1,tsec)

  b = 0.0d0
!$OMP PARALLEL PRIVATE(ithread, i, a) SHARED(nthreads, b)
#ifdef OMP
  ithread = omp_get_thread_num()
#endif
  a = 0.0d0
!$OMP DO
  do i = 1, n
    a = a + LOG(dble(i))
  enddo
!$OMP END DO
  a = a / dble(n)
!$OMP CRITICAL
  write(s1,101) ithread
  write(s2,103) a
  if (peinf%inode.eq.0) then
    write(6,205) TRUNC(s1), TRUNC(s2)
  endif
  b = b + a
!$OMP END CRITICAL
!$OMP END PARALLEL
  write(s1,103) b
  if (peinf%inode.eq.0) then
    write(6,206) TRUNC(s1)
  endif
  
  call timacc(1,2,tsec)

  call timacc(1,3,tsec,nslices)
  
  write(s1,104) tsec(1)
  write(s2,104) tsec(2)
  if (peinf%inode.eq.0) then
    write(6,207) TRUNC(s1), TRUNC(s2)
  endif

!------------------------
! Finish

#ifdef MPI
  call MPI_Finalize(mpierr)
#endif

101 format(i16)
102 format(f16.1)
103 format(f16.6)
104 format(f16.3)
  
201 format(/,1x,"Total number of MPI processes:",1x,a)
202 format(1x,"Number of MPI processes per node:",1x,a)
203 format(1x,"Amount of memory per MPI process:",1x,a,1x,"MB")
204 format(1x,"Number of OpenMP threads per MPI process:",1x,a,/)
205 format(1x,"OpenMP thread:",1x,a,1x,"Local sum:",1x,a)
206 format(1x,"Global sum:",1x,a,/)
207 format(1x,"Elapsed time:",1x,"CPU",1x,a,1x,"sec",";", &
      1x,"WALL",1x,a,1x,"sec",/)
  
end program openmp_main
