#include "f_defs.h"

!-----------------------------------------------------------------------
!
!  nk(3)    : number of unit cell repetitions
!
!  pstate   : index of excited state to be plotted
!
!  rhole(3) : coordinate of hole, with respect to lattice vectors
!
!  avec(3,3): coordinates of lattice vectors in cartesian components
!
!  ntau     : number of atoms in the unit cell (ntau < 1000)
!  tau(3,1:ntau) : coordinates of atoms in the unit cell
!
!
!-----------------------------------------------------------------------
subroutine inread(xct,nk,pstate,rhole,avec)

  use global_m
  implicit none
  
  type (xctinfo), intent(out) :: xct
  integer, intent(out) :: nk(3),pstate
  real(DP), intent(out) :: rhole(3),avec(3,3)

  character*256 :: blockword,keyword,line,errmsg
  integer :: ii,jj,ntau,iostat
  real(DP) :: tau(3,1000)
  
  PUSH_SUB(inread)

#ifdef MPI
  ! Non-root nodes should wait for root to read the whole file.
  ! That way, we can be sure root gets a chance to write errors before
  ! any die call is issued by another node. Root calls MPI_Barrier below.
  if(peinf%inode /= 0) call MPI_Barrier(MPI_COMM_WORLD, mpierr)
#endif

  call open_file(8,file='plotxct.inp',form='formatted',status='old')

!     Set default values
  nk(:)=0
  pstate=0
  rhole(:)=0.d0
  xct%shift(:)=0.d0
  ntau=0
  tau=0.d0
  xct%freplacebz=.false.
  xct%fwritebz=.false.
  xct%nn=0

!     Never ending loop...
  do while(0.eq.0)

!       Actually the loop ends when the end of the file is reached
    read(8,'(a256)',iostat=iostat) line
    if(iostat < 0) exit

!       Skip comment lines
    if(len_trim(line).eq.0) cycle
    if(line(1:1).eq.'#') cycle

!       Determine keyword:
    keyword=line(1:scan(line," ")-1)
    line=adjustl(line(scan(line," ")+1:256))

    if(trim(keyword).eq.'begin') then
      blockword=line(1:scan(line," ")-1)
      ii=0
      do while(trim(line).ne.'end')
        read(8,'(a256)',iostat=iostat) line
        if(iostat /= 0) then
          write(errmsg,'(3a)') 'The end of the file was reached while reading elements of the ', &
            trim(blockword),' block.'
          call die(errmsg, only_root_writes = .true.)
        endif
        if(trim(line).ne.'end') then
          ii=ii+1
          if (ii.gt.1000.and.trim(blockword).eq.'atom_positions') then
            if(peinf%inode == 0) write(0,'(i6,a)') ii, ' > 1000'
            call die('too many atoms', only_root_writes = .true.)
          endif
          if(trim(blockword).eq.'lattice_vectors') then
            read(line,*,err=112) (avec(ii,jj),jj=1,3)
          elseif(trim(blockword).eq.'atom_positions') then
            read(line,*,err=112) (tau(jj,ii),jj=1,3)
          else
            write(errmsg,'(3a)') 'Unexpected blockword ', trim(blockword), ' was found in plotxct.inp.'
            call die(errmsg, only_root_writes = .true.)
          end if
        end if
      end do
      if(trim(blockword).eq.'atom_positions') ntau=ii
    elseif(trim(keyword).eq.'restrict_kpoints') then
      read(line,*,err=110) xct%nn
    elseif(trim(keyword).eq.'plot_state') then
      read(line,*,err=110) pstate
    elseif(trim(keyword).eq.'hole_position') then
      read(line,*,err=110) (rhole(ii),ii=1,3)
    elseif(trim(keyword).eq.'q_shift') then
      read(line,*,err=110) (xct%shift(ii),ii=1,3)
    elseif(trim(keyword).eq.'supercell_size') then
      read(line,*,err=110) (nk(ii),ii=1,3)
    elseif(trim(keyword).eq.'fullbz_replace') then
      xct%freplacebz=.true.
    elseif(trim(keyword).eq.'fullbz_write') then
      xct%fwritebz=.true.
    else
      write(errmsg,'(3a)') 'Unexpected keyword ', trim(keyword), ' was found in plotxct.inp.'
      call die(errmsg, only_root_writes = .true.)
    end if
  enddo
  
  call close_file(8)
  
  if(pstate.eq.0) then
    call die("The plot_state keyword could not be found.", only_root_writes = .true.)
  endif
  
  if(nk(1)*nk(2)*nk(3).eq.0) then
    if(peinf%inode == 0) write(0,'(a,3i6)') 'kgrid = ', nk(1:3)
    call die("kgrid cannot be zero", only_root_writes = .true.)
  endif
  
  if(peinf%inode.eq.0) write(6,*)
  
#ifdef MPI
  ! root lets the others go after it is done reading (see beginning of function)
  if(peinf%inode == 0) call MPI_Barrier(MPI_COMM_WORLD, mpierr)
#endif

  POP_SUB(inread)
  
  return
  
110 write(errmsg,'(3a)') 'Unexpected characters were found while reading the value for the keyword ', &
      trim(keyword), '. '
  call die(errmsg, only_root_writes = .true.)
  
112 write(errmsg,'(3a)') 'Unexpected characters were found while reading elements of the ', &
      trim(blockword),' block.'
  call die(errmsg, only_root_writes = .true.)
  
end subroutine inread
